{-# OPTIONS -#include "HsUnix.h" #-}
{-# LINE 1 "IO.hsc" #-}
{-# OPTIONS -fffi #-}
{-# LINE 2 "IO.hsc" #-}
-----------------------------------------------------------------------------
-- |
-- Module      :  System.Posix.IO
-- Copyright   :  (c) The University of Glasgow 2002
-- License     :  BSD-style (see the file libraries/base/LICENSE)
-- 
-- Maintainer  :  libraries@haskell.org
-- Stability   :  provisional
-- Portability :  non-portable (requires POSIX)
--
-- POSIX IO support
--
-----------------------------------------------------------------------------

module System.Posix.IO (
    -- * Input \/ Output

    -- ** Standard file descriptors
    stdInput, stdOutput, stdError,

    -- ** Opening and closing files
    OpenMode(..),
    OpenFileFlags(..), defaultFileFlags,
    openFd, createFile,
    closeFd,

    -- ** Reading\/writing data
    -- |Programmers using the 'fdRead' and 'fdWrite' API should be aware that
    -- EAGAIN exceptions may occur for non-blocking IO!

    fdRead, fdWrite,

    -- ** Seeking
    fdSeek,

    -- ** File options
    FdOption(..),
    queryFdOption,
    setFdOption,

    -- ** Locking
    FileLock,
    LockRequest(..),
    getLock,  setLock,
    waitToSetLock,

    -- ** Pipes
    createPipe,

    -- ** Duplicating file descriptors
    dup, dupTo,


{-# LINE 55 "IO.hsc" #-}
    -- ** Converting file descriptors to\/from Handles
    handleToFd,

{-# LINE 58 "IO.hsc" #-}
    fdToHandle,  

  ) where

import System.IO
import System.IO.Error
import System.Posix.Types
import System.Posix.Internals

import Foreign
import Foreign.C
import Data.Bits


{-# LINE 72 "IO.hsc" #-}
import GHC.IOBase
import GHC.Handle hiding (fdToHandle, openFd)
import qualified GHC.Handle

{-# LINE 76 "IO.hsc" #-}


{-# LINE 81 "IO.hsc" #-}


{-# LINE 83 "IO.hsc" #-}

-- -----------------------------------------------------------------------------
-- Pipes
-- |The 'createPipe' function creates a pair of connected file descriptors. The first
-- component is the fd to read from, the second is the write end.
-- Although pipes may be bidirectional, this behaviour is not portable and
-- programmers should use two separate pipes for this purpose.

createPipe :: IO (Fd, Fd)
createPipe =
  allocaArray 2 $ \p_fd -> do
    throwErrnoIfMinus1_ "createPipe" (c_pipe p_fd)
    rfd <- peekElemOff p_fd 0
    wfd <- peekElemOff p_fd 1
    return (Fd rfd, Fd wfd)

-- -----------------------------------------------------------------------------
-- Duplicating file descriptors

dup :: Fd -> IO Fd
dup (Fd fd) = do r <- throwErrnoIfMinus1 "dup" (c_dup fd); return (Fd r)

dupTo :: Fd -> Fd -> IO Fd
dupTo (Fd fd1) (Fd fd2) = do
  r <- throwErrnoIfMinus1 "dupTp" (c_dup2 fd1 fd2)
  return (Fd r)

-- -----------------------------------------------------------------------------
-- Opening and closing files

stdInput, stdOutput, stdError :: Fd
stdInput   = Fd (0)
{-# LINE 115 "IO.hsc" #-}
stdOutput  = Fd (1)
{-# LINE 116 "IO.hsc" #-}
stdError   = Fd (2)
{-# LINE 117 "IO.hsc" #-}

data OpenMode = ReadOnly | WriteOnly | ReadWrite

data OpenFileFlags =
 OpenFileFlags {
    append    :: Bool,
    exclusive :: Bool,
    noctty    :: Bool,
    nonBlock  :: Bool,
    trunc     :: Bool
 }

defaultFileFlags :: OpenFileFlags
defaultFileFlags =
 OpenFileFlags {
    append    = False,
    exclusive = False,
    noctty    = False,
    nonBlock  = False,
    trunc     = False
  }

openFd :: FilePath
       -> OpenMode
       -> Maybe FileMode -- Just x => O_CREAT, Nothing => must exist
       -> OpenFileFlags
       -> IO Fd
openFd name how maybe_mode (OpenFileFlags append exclusive noctty
				nonBlock truncate) = do
   withCString name $ \s -> do
    fd <- throwErrnoIfMinus1 "openFd" (c_open s all_flags mode_w)
    return (Fd fd)
  where
    all_flags  = creat .|. flags .|. open_mode

    flags =
       (if append    then (8)   else 0) .|.
{-# LINE 154 "IO.hsc" #-}
       (if exclusive then (2048)     else 0) .|.
{-# LINE 155 "IO.hsc" #-}
       (if noctty    then (32768)   else 0) .|.
{-# LINE 156 "IO.hsc" #-}
       (if nonBlock  then (4) else 0) .|.
{-# LINE 157 "IO.hsc" #-}
       (if truncate  then (1024)    else 0)
{-# LINE 158 "IO.hsc" #-}

    (creat, mode_w) = case maybe_mode of 
			Nothing -> (0,0)
			Just x  -> ((512), x)
{-# LINE 162 "IO.hsc" #-}

    open_mode = case how of
		   ReadOnly  -> (0)
{-# LINE 165 "IO.hsc" #-}
		   WriteOnly -> (1)
{-# LINE 166 "IO.hsc" #-}
		   ReadWrite -> (2)
{-# LINE 167 "IO.hsc" #-}

createFile :: FilePath -> FileMode -> IO Fd
createFile name mode
  = openFd name WriteOnly (Just mode) defaultFileFlags{ trunc=True } 

closeFd :: Fd -> IO ()
closeFd (Fd fd) = throwErrnoIfMinus1_ "closeFd" (c_close fd)

-- -----------------------------------------------------------------------------
-- Converting file descriptors to/from Handles


{-# LINE 179 "IO.hsc" #-}
handleToFd :: Handle -> IO Fd
handleToFd h = withHandle "handleToFd" h $ \ h_ -> do
  -- converting a Handle into an Fd effectively means
  -- letting go of the Handle; it is put into a closed
  -- state as a result. 
  let fd = haFD h_
  flushWriteBufferOnly h_
  unlockFile (fromIntegral fd)
    -- setting the Handle's fd to (-1) as well as its 'type'
    -- to closed, is enough to disable the finalizer that
    -- eventually is run on the Handle.
  return (h_{haFD= (-1),haType=ClosedHandle}, Fd (fromIntegral fd))

fdToHandle :: Fd -> IO Handle
fdToHandle fd = GHC.Handle.fdToHandle (fromIntegral fd)

{-# LINE 195 "IO.hsc" #-}


{-# LINE 202 "IO.hsc" #-}

-- -----------------------------------------------------------------------------
-- Fd options

data FdOption = AppendOnWrite
	      | CloseOnExec
	      | NonBlockingRead
	      | SynchronousWrites

queryFdOption :: Fd -> FdOption -> IO Bool
queryFdOption (Fd fd) opt = do
  r <- throwErrnoIfMinus1 "queryFdOption" (c_fcntl_read fd flag)
  return ((r .&. opt_val) /= 0)
 where
  flag    = case opt of
	      CloseOnExec       -> (1)
{-# LINE 218 "IO.hsc" #-}
	      other		-> (3)
{-# LINE 219 "IO.hsc" #-}

  opt_val = case opt of
	      CloseOnExec       -> (1)
{-# LINE 222 "IO.hsc" #-}
	      AppendOnWrite     -> (8)
{-# LINE 223 "IO.hsc" #-}
              NonBlockingRead   -> (4)
{-# LINE 224 "IO.hsc" #-}
	      SynchronousWrites -> (128)
{-# LINE 225 "IO.hsc" #-}

setFdOption :: Fd -> FdOption -> Bool -> IO ()
setFdOption (Fd fd) opt val = do
  r <- throwErrnoIfMinus1 "setFdOption" (c_fcntl_read fd getflag)
  let r' | val       = r .|. opt_val
	 | otherwise = r .&. (complement opt_val) 
  throwErrnoIfMinus1_ "setFdOption" (c_fcntl_write fd setflag r')
 where
  (getflag,setflag)= case opt of
	      CloseOnExec       -> ((1),(2)) 
{-# LINE 235 "IO.hsc" #-}
	      other		-> ((3),(4))
{-# LINE 236 "IO.hsc" #-}
  opt_val = case opt of
	      CloseOnExec       -> (1)
{-# LINE 238 "IO.hsc" #-}
	      AppendOnWrite     -> (8)
{-# LINE 239 "IO.hsc" #-}
              NonBlockingRead   -> (4)
{-# LINE 240 "IO.hsc" #-}
	      SynchronousWrites -> (128)
{-# LINE 241 "IO.hsc" #-}

-- -----------------------------------------------------------------------------
-- Seeking 

mode2Int :: SeekMode -> CInt
mode2Int AbsoluteSeek = (0)
{-# LINE 247 "IO.hsc" #-}
mode2Int RelativeSeek = (1)
{-# LINE 248 "IO.hsc" #-}
mode2Int SeekFromEnd  = (2)
{-# LINE 249 "IO.hsc" #-}

fdSeek :: Fd -> SeekMode -> FileOffset -> IO FileOffset
fdSeek (Fd fd) mode off =
  throwErrnoIfMinus1 "fdSeek" (c_lseek fd off (mode2Int mode))

-- -----------------------------------------------------------------------------
-- Locking

data LockRequest = ReadLock
                 | WriteLock
                 | Unlock

type FileLock = (LockRequest, SeekMode, FileOffset, FileOffset)

getLock :: Fd -> FileLock -> IO (Maybe (ProcessID, FileLock))
getLock (Fd fd) lock =
  allocaLock lock $ \p_flock -> do
    throwErrnoIfMinus1_ "getLock" (c_fcntl_lock fd (7) p_flock)
{-# LINE 267 "IO.hsc" #-}
    result <- bytes2ProcessIDAndLock p_flock
    return (maybeResult result)
  where
    maybeResult (_, (Unlock, _, _, _)) = Nothing
    maybeResult x = Just x

allocaLock :: FileLock -> (Ptr CFLock -> IO a) -> IO a
allocaLock (lockreq, mode, start, len) io = 
  allocaBytes (24) $ \p -> do
{-# LINE 276 "IO.hsc" #-}
    ((\hsc_ptr -> pokeByteOff hsc_ptr 20))   p (lockReqToInt lockreq :: CShort)
{-# LINE 277 "IO.hsc" #-}
    ((\hsc_ptr -> pokeByteOff hsc_ptr 22)) p (fromIntegral (mode2Int mode) :: CShort)
{-# LINE 278 "IO.hsc" #-}
    ((\hsc_ptr -> pokeByteOff hsc_ptr 0))  p start
{-# LINE 279 "IO.hsc" #-}
    ((\hsc_ptr -> pokeByteOff hsc_ptr 8))    p len
{-# LINE 280 "IO.hsc" #-}
    io p

lockReqToInt :: LockRequest -> CShort
lockReqToInt ReadLock  = (1)
{-# LINE 284 "IO.hsc" #-}
lockReqToInt WriteLock = (3)
{-# LINE 285 "IO.hsc" #-}
lockReqToInt Unlock    = (2)
{-# LINE 286 "IO.hsc" #-}

bytes2ProcessIDAndLock :: Ptr CFLock -> IO (ProcessID, FileLock)
bytes2ProcessIDAndLock p = do
  req   <- ((\hsc_ptr -> peekByteOff hsc_ptr 20))   p
{-# LINE 290 "IO.hsc" #-}
  mode  <- ((\hsc_ptr -> peekByteOff hsc_ptr 22)) p
{-# LINE 291 "IO.hsc" #-}
  start <- ((\hsc_ptr -> peekByteOff hsc_ptr 0))  p
{-# LINE 292 "IO.hsc" #-}
  len   <- ((\hsc_ptr -> peekByteOff hsc_ptr 8))    p
{-# LINE 293 "IO.hsc" #-}
  pid   <- ((\hsc_ptr -> peekByteOff hsc_ptr 16))    p
{-# LINE 294 "IO.hsc" #-}
  return (pid, (int2req req, int2mode mode, start, len))
 where
  int2req :: CShort -> LockRequest
  int2req (1) = ReadLock
{-# LINE 298 "IO.hsc" #-}
  int2req (3) = WriteLock
{-# LINE 299 "IO.hsc" #-}
  int2req (2) = Unlock
{-# LINE 300 "IO.hsc" #-}
  int2req _ = error $ "int2req: bad argument"

  int2mode :: CShort -> SeekMode
  int2mode (0) = AbsoluteSeek
{-# LINE 304 "IO.hsc" #-}
  int2mode (1) = RelativeSeek
{-# LINE 305 "IO.hsc" #-}
  int2mode (2) = SeekFromEnd
{-# LINE 306 "IO.hsc" #-}
  int2mode _ = error $ "int2mode: bad argument"

setLock :: Fd -> FileLock -> IO ()
setLock (Fd fd) lock = do
  allocaLock lock $ \p_flock ->
    throwErrnoIfMinus1_ "setLock" (c_fcntl_lock fd (8) p_flock)
{-# LINE 312 "IO.hsc" #-}

waitToSetLock :: Fd -> FileLock -> IO ()
waitToSetLock (Fd fd) lock = do
  allocaLock lock $ \p_flock ->
    throwErrnoIfMinus1_ "waitToSetLock" 
	(c_fcntl_lock fd (9) p_flock)
{-# LINE 318 "IO.hsc" #-}

-- -----------------------------------------------------------------------------
-- fd{Read,Write}

fdRead :: Fd -> ByteCount -> IO (String, ByteCount)
fdRead _fd 0 = return ("", 0)
fdRead (Fd fd) nbytes = do
    allocaBytes (fromIntegral nbytes) $ \ bytes -> do
    rc    <-  throwErrnoIfMinus1Retry "fdRead" (c_read fd bytes nbytes)
    case fromIntegral rc of
      0 -> ioError (IOError Nothing EOF "fdRead" "EOF" Nothing)
      n -> do
       s <- peekCStringLen (bytes, fromIntegral n)
       return (s, n)

fdWrite :: Fd -> String -> IO ByteCount
fdWrite (Fd fd) str = withCStringLen str $ \ (strPtr,len) -> do
    rc <- throwErrnoIfMinus1Retry "fdWrite" (c_write fd strPtr (fromIntegral len))
    return (fromIntegral rc)
