{-# OPTIONS -#include "HsBase.h" #-}
{-# LINE 1 "CPUTime.hsc" #-}
-----------------------------------------------------------------------------
{-# LINE 2 "CPUTime.hsc" #-}
-- |
-- Module      :  System.CPUTime
-- Copyright   :  (c) The University of Glasgow 2001
-- License     :  BSD-style (see the file libraries/core/LICENSE)
-- 
-- Maintainer  :  libraries@haskell.org
-- Stability   :  provisional
-- Portability :  portable
--
-- The standard CPUTime library.
--
-----------------------------------------------------------------------------

module System.CPUTime 
	(
         getCPUTime,       -- :: IO Integer
	 cpuTimePrecision  -- :: Integer
        ) where

import Prelude

import Data.Ratio


{-# LINE 28 "CPUTime.hsc" #-}


{-# LINE 30 "CPUTime.hsc" #-}
import Foreign
import Foreign.C


{-# LINE 34 "CPUTime.hsc" #-}

{-# LINE 35 "CPUTime.hsc" #-}


{-# LINE 37 "CPUTime.hsc" #-}
-- -----------------------------------------------------------------------------
-- |Computation 'getCPUTime' returns the number of picoseconds CPU time
-- used by the current program.  The precision of this result is
-- implementation-dependent.

getCPUTime :: IO Integer
getCPUTime = do


{-# LINE 46 "CPUTime.hsc" #-}
-- getrusage() is right royal pain to deal with when targetting multiple
-- versions of Solaris, since some versions supply it in libc (2.3 and 2.5),
-- while 2.4 has got it in libucb (I wouldn't be too surprised if it was back
-- again in libucb in 2.6..)
--
-- Avoid the problem by resorting to times() instead.
--

{-# LINE 54 "CPUTime.hsc" #-}
    allocaBytes (72) $ \ p_rusage -> do
{-# LINE 55 "CPUTime.hsc" #-}
    getrusage (0) p_rusage
{-# LINE 56 "CPUTime.hsc" #-}

    let ru_utime = ((\hsc_ptr -> hsc_ptr `plusPtr` 0)) p_rusage
{-# LINE 58 "CPUTime.hsc" #-}
    let ru_stime = ((\hsc_ptr -> hsc_ptr `plusPtr` 8)) p_rusage
{-# LINE 59 "CPUTime.hsc" #-}
    u_sec  <- ((\hsc_ptr -> peekByteOff hsc_ptr 0))  ru_utime :: IO CTime
{-# LINE 60 "CPUTime.hsc" #-}
    u_usec <- ((\hsc_ptr -> peekByteOff hsc_ptr 4)) ru_utime :: IO CTime
{-# LINE 61 "CPUTime.hsc" #-}
    s_sec  <- ((\hsc_ptr -> peekByteOff hsc_ptr 0))  ru_stime :: IO CTime
{-# LINE 62 "CPUTime.hsc" #-}
    s_usec <- ((\hsc_ptr -> peekByteOff hsc_ptr 4)) ru_stime :: IO CTime
{-# LINE 63 "CPUTime.hsc" #-}

    return ((fromIntegral u_sec * 1000000 + fromIntegral u_usec + 
             fromIntegral s_sec * 1000000 + fromIntegral s_usec) 
		* 1000000)

type CRUsage = ()
foreign import ccall unsafe getrusage :: CInt -> Ptr CRUsage -> IO CInt

{-# LINE 88 "CPUTime.hsc" #-}


{-# LINE 122 "CPUTime.hsc" #-}

{-# LINE 123 "CPUTime.hsc" #-}

-- |The 'cpuTimePrecision' constant is the smallest measurable difference
-- in CPU time that the implementation can record, and is given as an
-- integral number of picoseconds.

cpuTimePrecision :: Integer
cpuTimePrecision = round ((1000000000000::Integer) % fromIntegral (clockTicks))


{-# LINE 132 "CPUTime.hsc" #-}
clockTicks :: Int
clockTicks =

{-# LINE 135 "CPUTime.hsc" #-}
    (100)
{-# LINE 136 "CPUTime.hsc" #-}

{-# LINE 140 "CPUTime.hsc" #-}

{-# LINE 141 "CPUTime.hsc" #-}
