#define _GNU_SOURCE 1
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>

#define error(who)	 (perror(who))

#define AIO_RD	(1<<0)
#define AIO_WR	(1<<1)
#define AIO_EX	(1<<2)

typedef void (*handler)(int fd, int mask);

typedef struct {
  int     fd;
  handler hdlr;
} Socket ;

fd_set fdMask;

static handler handlers[FD_SETSIZE];

static int lastSocket= -1;

static void sigpipe(int signum)
{
  fprintf(stderr, "sigpipe\n");
}

static void sigio(int signum)
{
  fd_set rd= fdMask;
  fd_set wr= fdMask;
  fd_set ex= fdMask;
  fprintf(stderr, "sigio\n");
  {
    struct timeval tv= { 0, 0 };
    int n= select(lastSocket + 1, &rd, &wr, &ex, &tv);
    if (n < 0)
      perror("select");
    else if (n > 0)
      {
	for (n= lastSocket; n >= 0 ; --n)
	  {
	    int mask= (  (FD_ISSET(n, &rd) ? AIO_RD : 0)
		       | (FD_ISSET(n, &wr) ? AIO_WR : 0)
		       | (FD_ISSET(n, &ex) ? AIO_EX : 0));
	    if (mask)
	      handlers[n](n, mask);
	  }
      }
  }
}

static void handle(int s, handler hdlr)
{
  handlers[s]= hdlr;
}

static void enable(int s)
{
  int arg= 1;
  fprintf(stderr, "%d enabling notification\n", s);
  if (lastSocket < s) lastSocket= s;
  FD_SET(s, &fdMask);
  arg= 1;
  if (ioctl(s, FIONBIO,  (char *)&arg))        error("ioctl(FIONBIO)");
  arg= 1;
  if (ioctl(s, FIOASYNC, (char *)&arg))        error("ioctl(FIOASYNC)");
  if (fcntl(s, F_SETOWN, getpid()))            error("fcntl(F_SETOWN)");
  if (fcntl(s, F_SETOWN, getpid()))            error("fcntl(F_SETOWN)");
  if (fcntl(s, F_SETSIG, SIGIO))               error("fcntl(F_SETSIG)");
}

static void disable(int s)
{
  int arg= 0;
  fprintf(stderr, "%d disabling notification\n", s);
  arg= 0;
  if (ioctl(s, FIONBIO,  (char *)&arg))        error("ioctl(FIONBIO)");
  arg= 0;
  if (ioctl(s, FIOASYNC, (char *)&arg))        error("ioctl(FIOASYNC)");
  if (fcntl(s, F_SETOWN, getpid()))            error("fcntl(F_SETOWN)");
  handle(s, 0);
  FD_CLR(s, &fdMask);
  while ((lastSocket >= 0) && (!FD_ISSET(lastSocket, &fdMask)))
    --lastSocket;
}

static char *maskString(int mask)
{
  switch (mask)
    {
    case 0 ... 7:
      {
	static char *modes[]= {
	  "---", "--R", "-W-", "-WR", "X--", "X-R", "XW-", "XWR"
	};
	return modes[mask];
      }
    }
  return "BAD";
}

static void dataHandler(int fd, int mask)
{
  fprintf(stderr, "handling: data %d %d %s\n", fd, mask, maskString(mask));
  if (mask & AIO_RD)
    {
      fprintf(stderr, "%d read possible\n", fd);
      {
	char buf[4096];
	int n= read(fd, (void *)buf, sizeof(buf));
	if (n < 0) error("read");
	if (n == 0)
	  {
	    fprintf(stderr, "%d EOF detected\n", fd);
	  }
	else
	  {
	    char *ptr= buf;
	    fprintf(stderr, "%d received: ", fd);
	    while (n--)
	      putchar(*ptr++);
	  }
      }
    }
  if (mask & AIO_WR)
    {
      fprintf(stderr, "%d write possible\n", fd);
    }
  if (mask & AIO_EX)
    {
      fprintf(stderr, "%d exception\n", fd);
      {
	int optval= 0;
	socklen_t optlen= sizeof(optval);
	if (getsockopt(fd, SOL_SOCKET, SO_ERROR, &optval, &optlen))
	  error("getsockopt");
	if (optlen != 4)
	  fprintf(stderr, "getsockopt: optlen is not 4!\n");
	fprintf(stderr, "exception condition: %s\n", strerror(optval));
      }
    }
}

static void connectHandler(int fd, int mask)
{
  int optval= 0;
  socklen_t optlen= sizeof(optval);
  fprintf(stderr, "handling: connect %d %s\n", fd, maskString(mask));
  if (getsockopt(fd, SOL_SOCKET, SO_ERROR, &optval, &optlen))
    error("getsockopt");
  if (optlen != 4)
    fprintf(stderr, "getsockopt: optlen is not 4!\n");
  if (optval != 0)
    fprintf(stderr, "handled connect: %s\n", strerror(optval));
  fprintf(stderr, "%d connected\n", fd);
  handle(fd, dataHandler);
  {
    char buf[]= "SOCKET 2 ME DUDE!\n";
    fprintf(stderr, "%d sending: %s", fd, buf);
    send(fd, (void *)buf, strlen(buf), 0);
  }
}

static void acceptHandler(int fd, int mask)
{
  int conn= 0;
  struct sockaddr_in saddr;
  int len= sizeof(saddr);
  fprintf(stderr, "handling: accept %d %s\n", fd, maskString(mask));
  conn= accept(fd, &saddr, &len);
  if (conn < 0) error("accept");
  fprintf(stderr, "%d accepted\n", conn); 
  handle(conn, dataHandler);
  enable(conn);
}

static void bindSocket(int s, unsigned int host, int port)
{
  struct sockaddr_in saddr;

  memset(&saddr, 0, sizeof(saddr));
  saddr.sin_family= AF_INET;
  saddr.sin_port= htons((short)port);
  saddr.sin_addr.s_addr= host;

  if (bind(s, (struct sockaddr*) &saddr, sizeof(saddr))) error("bind");
}

static int listenOn(int port)
{
  int s= socket(AF_INET, SOCK_STREAM, 0);
  int one= 1;
  if (s < 0) error("socket");
  if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (char *)&one, sizeof(one)))
    error("setsockopt");
  bindSocket(s, INADDR_ANY, port);
  handle(s, acceptHandler);
  enable(s);
  fprintf(stderr, "%d listening\n", s);
  if (listen(s, 5)) error("listen");
  return s;
}

static int connectTo(unsigned int host, int port)
{
  struct sockaddr_in saddr;
  int one= 1;
  int s= socket(AF_INET, SOCK_STREAM, 0);

  if (s < 0) error("socket");
  if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (char *)&one, sizeof(one)))
    error("setsockopt");

  memset(&saddr, 0, sizeof(saddr));
  saddr.sin_family= AF_INET;
  saddr.sin_port= htons((short)port);
  saddr.sin_addr.s_addr= host;

  fprintf(stderr, "%d connecting\n", s);
  handle(s, connectHandler);
  enable(s);
  {
    int err= 0;
    do
      {
	err= connect(s, (struct sockaddr*) &saddr, sizeof(saddr));
      }
    while ((err < 0) && (errno == EINTR));
    if (err < 0) error("connect");
  }
  return s;
}

int main()
{
  {
    struct sigaction sa;
    sigset_t mask;

    sigemptyset(&mask);
    sigaddset(&mask, SIGIO);
    sa.sa_handler= sigio;
    sa.sa_mask= mask;
    sa.sa_flags= SA_RESTART;
    sigaction(SIGIO, &sa, 0);

    sigemptyset(&mask);
    sigaddset(&mask, SIGPIPE);
    sa.sa_handler= sigpipe;
    sa.sa_mask= mask;
    sa.sa_flags= SA_RESTART;
    sigaction(SIGPIPE, &sa, 0);
  }

  {
    int l= listenOn(4321);
    int c= connectTo(INADDR_LOOPBACK, 4321);
    sleep(1);
    disable(c);
    close(c);
    sleep(1);
    disable(l);
    close(l);
  }

  return 0;
}
