#ifdef MATHML
/*
 *
 *  (c) COPYRIGHT MIT and INRIA, 1996.
 *  Please first read the full copyright statement in file COPYRIGHT.
 *
 */

/*
 * This module contains editing functions for handling MathML objects.
 *
 * Author: I. Vatton
 *         R. Guetari (W3C/INRIA) - Windows routines.
 */

/* Included headerfiles */
#define THOT_EXPORT
#include "amaya.h"
#include "css.h"
#include "trans.h"
#include "undo.h"
#include "interface.h"
#include "MathML.h"
#ifdef GRAPHML
#include "GraphML.h"
#endif

#define FormMaths 0
#define MenuMaths 1
#define MAX_MATHS  2

#ifdef _WINDOWS
#define iconMath   21 
#define iconMathNo 21 
#else /* _WINDOWS */
static ThotIcon	   iconMath;
static ThotIcon	   iconMathNo;

#include "Math.xpm"
#include "MathNo.xpm"
#include "Bmath.xpm"
#include "root.xpm"
#include "sqrt.xpm"
#include "frac.xpm"
#include "subsup.xpm"
#include "sup.xpm"
#include "sub.xpm"
#include "overunder.xpm"
#include "over.xpm"
#include "under.xpm"
#include "fence.xpm"
#include "mscript.xpm"
#include "matrix.xpm"
#include "greek.xpm"
#endif /* _WINDOWS */

static int      MathButton;
static Pixmap	mIcons[14];
static int	MathsDialogue;
static ThotBool	InitMaths;
static ThotBool	IsLastDeletedElement = FALSE;
static Element	LastDeletedElement = NULL;

#include "html2thot_f.h"
#include "fetchXMLname_f.h"
#include "HTMLtable_f.h"
#include "MathMLbuilder_f.h"
#include "styleparser_f.h"
#include "trans_f.h"
#ifdef _WINDOWS
#include "wininclude.h"
#endif /* _WINDOWS */

/*----------------------------------------------------------------------
   SplitTextInMathML
   Split element el and the enclosing element (MO, MI, MN or MTEXT).
   Parameter index indicates the position where the text has to be split.
   Return the text element created within the next enclosing element.
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static Element      SplitTextInMathML (Document doc, Element el, int index, ThotBool *mrowCreated)
#else
static Element      SplitTextInMathML (doc, el, index, mrowCreated)
Document            doc;
Element             el;
int                 index;
ThotBool            *mrowCreated;
#endif
{
  Element            added, parent, row;
  ElementType        elType;
  int                oldStructureChecking;
  ThotBool           withinMrow;

  /* do not check the Thot abstract tree against the structure schema while
     changing the structure */
  oldStructureChecking = TtaGetStructureChecking (doc);
  TtaSetStructureChecking (0, doc);

  /* get the parent element (MO, MN, MI or MTEXT) */
  parent = TtaGetParent (el);

  /* check whether the parent is a child of a MROW */
  row = TtaGetParent (parent);
  elType = TtaGetElementType (row);
  withinMrow = (elType.ElTypeNum == MathML_EL_MROW ||
                elType.ElTypeNum == MathML_EL_FencedExpression);

  /* split the text element */
  if (withinMrow)
     TtaRegisterElementReplace (el, doc);
  TtaSplitText (el, index-1, doc);
  /* take the second part of the split text */
  TtaNextSibling (&el);

  *mrowCreated = FALSE;
  if (!withinMrow)
    {
      /* generate a new MROW element */
      elType.ElTypeNum = MathML_EL_MROW;
      row = TtaNewElement (doc, elType);
      TtaInsertSibling (row, parent, TRUE, doc);
      TtaRegisterElementCreate (row, doc);
      TtaRegisterElementDelete (parent, doc);
      TtaRemoveTree (parent, doc);
      /* move the parent into the new MROW */
      TtaInsertFirstChild (&parent, row, doc);
      /* the MROW element has been registered in the Undo queue. There is
	 no need to register the elements that will be created within
	 this MROW element */
      *mrowCreated = TRUE;
    }

  /* duplicate the parent element (MO, MN, MI or MTEXT) */
  elType = TtaGetElementType (parent);
  added = TtaNewElement (doc, elType);
  TtaInsertSibling (added, parent, FALSE, doc);
  /* move the second part of text into the duplicated parent */
  TtaRemoveTree (el, doc);
  TtaInsertFirstChild (&el, added, doc);
  if (withinMrow)
     TtaRegisterElementCreate (added, doc);
  /* resume structure checking */
  TtaSetStructureChecking ((ThotBool)oldStructureChecking, doc);
  return (el);
}

/*----------------------------------------------------------------------
  DeleteIfPlaceholder
  
  Delete element el if it's a placeholder.
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void       DeleteIfPlaceholder (Element* el, Document doc, ThotBool record)
#else
static void       DeleteIfPlaceholder (el, doc, record)
Element* el;
Document doc;
ThotBool record;

#endif
{
Attribute	attr;
ElementType	elType;
AttributeType	attrType;

     if (*el == NULL)
       return;
     elType = TtaGetElementType (*el);
     if (elType.ElTypeNum == MathML_EL_Construct)
	{
        attrType.AttrSSchema = elType.ElSSchema;
        attrType.AttrTypeNum = MathML_ATTR_IntPlaceholder;
	attr = TtaGetAttribute (*el, attrType);
	if (attr != NULL)
	   /* this element is a placeholder. Delete it */
	   {
	   if (record)
	      TtaRegisterElementDelete (*el, doc);
	   TtaDeleteTree (*el, doc);
	   *el = NULL;
	   }
	}
}

/*----------------------------------------------------------------------
  InsertPlaceholder
  
  Return the new placeholder, if one has been created. Return NULL if
  no placeholder created.
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static Element       InsertPlaceholder (Element el, ThotBool before, Document doc, ThotBool record)
#else
static Element       InsertPlaceholder (el, before, doc, record)
Element el;
ThotBool before;
Document doc;
ThotBool record;

#endif
{
Element		sibling, placeholderEl;
ElementType	elType;
Attribute	attr;
AttributeType	attrType;
ThotBool		createConstruct, oldStructureChecking;

     placeholderEl = NULL;

     if (!ElementNeedsPlaceholder (el))
	/* this element does not need placeholders.  Delete its previous
	   and next siblings if they are place holders */
	{
	sibling = el;
	TtaPreviousSibling (&sibling);
	DeleteIfPlaceholder (&sibling, doc, record);
	sibling = el;
	TtaNextSibling (&sibling);
	DeleteIfPlaceholder (&sibling, doc, record);
	}
     else
	/* this element needs placeholders.  Create placeholders if the
	   previous and/or next sibling are absent or need placeholders too */
	{
	createConstruct = TRUE;
	sibling = el;
	if (before)
	   TtaPreviousSibling (&sibling);
	else
	   TtaNextSibling (&sibling);
	if (sibling != NULL)
	   if (!ElementNeedsPlaceholder (sibling))
	      createConstruct = FALSE;
	if (createConstruct)
	   {
	   elType = TtaGetElementType (el);
	   elType.ElTypeNum = MathML_EL_Construct;
	   placeholderEl = TtaNewElement (doc, elType);
	   /* do not check the Thot abstract tree against the structure */
	   /* schema while inserting the Placeholder */
	   oldStructureChecking = TtaGetStructureChecking (doc);
	   TtaSetStructureChecking (0, doc);
	   TtaInsertSibling (placeholderEl, el, before, doc);
	   /* resume structure checking */
	   TtaSetStructureChecking (oldStructureChecking, doc);
           attrType.AttrSSchema = elType.ElSSchema;
           attrType.AttrTypeNum = MathML_ATTR_IntPlaceholder;
           attr = TtaNewAttribute (attrType);
           TtaAttachAttribute (placeholderEl, attr, doc);
           TtaSetAttributeValue (attr, MathML_ATTR_IntPlaceholder_VAL_yes_,
				 placeholderEl, doc);
	   if (record)
	      TtaRegisterElementCreate (placeholderEl, doc);
	   }
	}
     return placeholderEl;
}

/*----------------------------------------------------------------------
  CreateParentMROW
  If element el is not a child of a MROW and if it has at least one
  sibling that is not a Construct (place holder), create an enclosing MROW,
  except if el is a child of a MFENCED element.
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void	   CreateParentMROW (Element el, Document doc)
#else
static void        CreateParentMROW (el, doc)
Element	el;
Document doc;

#endif
{
  Element            sibling, row, parent, firstChild, lastChild, next,
                     previous;
  ElementType        elType;
  int                nChildren;

  /* check whether the parent is a row */
  parent = TtaGetParent (el);
  if (parent == NULL)
     return;
  elType = TtaGetElementType (parent);
  if (elType.ElTypeNum != MathML_EL_MROW &&
      elType.ElTypeNum != MathML_EL_FencedExpression)
	 {
	 sibling = TtaGetFirstChild (parent);
	 nChildren = 0;
	 firstChild = sibling;
	 while (sibling != NULL)
	    {
	    elType = TtaGetElementType (sibling);
	    if (elType.ElTypeNum != MathML_EL_Construct)
	       nChildren++;
	    TtaNextSibling (&sibling);
	    }
	 if (nChildren > 1)
	    {
	      /* generate a new row element to include these elements */
	      elType.ElTypeNum = MathML_EL_MROW;
	      row = TtaNewElement (doc, elType);
	      lastChild = TtaGetLastChild (parent);
	      TtaInsertSibling (row, lastChild, FALSE, doc);
	      TtaRegisterElementCreate (row, doc);
	      sibling = firstChild;
	      previous = NULL;
	      while (sibling != NULL)
		{
		next = sibling;
		TtaNextSibling (&next);
		TtaRegisterElementDelete (sibling, doc);
	        TtaRemoveTree (sibling, doc);
	        /* move the old element into the new MROW */
		if (previous == NULL)
	           TtaInsertFirstChild (&sibling, row, doc);
		else
		   TtaInsertSibling (sibling, previous, FALSE, doc);
		previous = sibling;
		if (next == row)
		   sibling = NULL;
		else
		   sibling = next;
		}
	    }
	 }
}


/*----------------------------------------------------------------------
   RemoveAttr
   Remove attribute of type attrTypeNum from element el, if it exists
 -----------------------------------------------------------------------*/
#ifdef __STDC__
static void RemoveAttr (Element el, Document doc, int attrTypeNum)
#else /* __STDC__*/
static void RemoveAttr (el, doc, attrTypeNum)
     Element el;
     Document doc;
     int attrTypeNum;
#endif /* __STDC__*/
{
  ElementType	elType;
  AttributeType attrType;
  Attribute	attr;

  elType = TtaGetElementType (el);
  attrType.AttrSSchema = elType.ElSSchema;
  attrType.AttrTypeNum = attrTypeNum;
  attr = TtaGetAttribute (el, attrType);
  if (attr != NULL)
      TtaRemoveAttribute (el, attr, doc);
}


/*----------------------------------------------------------------------
   RegenerateFencedSeparators
   el must be a FencedExpression element.
   Delete all existing FencedSeparator elements in el and create new
   ones according to the value of attribute separators of parent MFENCED.
 -----------------------------------------------------------------------*/
#ifdef __STDC__
static void RegenerateFencedSeparators (Element el, Document doc, ThotBool record)
#else /* __STDC__*/
static void RegenerateFencedSeparators (el, doc, record)
     Element el;
     Document doc;
     ThotBool record;

#endif /* __STDC__*/
{
  Element	child, next;
  ElementType	elType;

  /* Delete all existing FencedSeparator elements that are children of the
     FencedExpression element */
  child = TtaGetFirstChild (el);
  while (child != NULL)
     {
     next = child;
     TtaNextSibling (&next);
     elType = TtaGetElementType (child);
     if (elType.ElTypeNum == MathML_EL_FencedSeparator)
       {
        if (record)
	  TtaRegisterElementDelete (child, doc);
	TtaDeleteTree (child, doc);
       }
     child = next;
     }
  /* Create all FencedSeparator elements, according to attribute separators */
  CreateFencedSeparators (el, doc, record);
}

/*----------------------------------------------------------------------
   CreateMathConstruct
   Create a MathML construct at the current position
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void         CreateMathConstruct (int construct)
#else
static void         CreateMathConstruct (construct)
int                 construct;

#endif
{
  Document           doc;
  Element            sibling, last, el, row, fence, symbol, child, leaf,
		     placeholderEl, parent, new;
  ElementType        newType, elType, symbType;
  SSchema            docSchema, mathSchema;
  int                c1, c2, i, j, len, oldStructureChecking;
  ThotBool	     before, ParBlock, surround, insertSibling,
		     selectFirstChild, displayTableForm, mrowCreated;

      doc = TtaGetSelectedDocument ();
      TtaGiveLastSelectedElement (doc, &last, &c2, &j);
      TtaGiveFirstSelectedElement (doc, &sibling, &c1, &i); 
    
      /* Get the type of the first selected element */
      elType = TtaGetElementType (sibling);
      docSchema = TtaGetDocumentSSchema (doc);

      if (construct == 1)
	/* Math button */
	{
	if (ustrcmp(TtaGetSSchemaName (elType.ElSSchema), TEXT("HTML")) == 0)
	   /* selection is in an HTML element */
	   {
           newType.ElTypeNum = HTML_EL_Math;
           newType.ElSSchema = docSchema;
           TtaCreateElement (newType, doc);
	   }
#ifdef GRAPHML
	if (ustrcmp(TtaGetSSchemaName (elType.ElSSchema), "GraphML") == 0)
	   /* selection is in a GraphML element */
	   {
           newType.ElTypeNum = GraphML_EL_Math;
           newType.ElSSchema = elType.ElSSchema;
	   TtaAskFirstCreation ();
           TtaCreateElement (newType, doc);
	   }
#endif /* GRAPHML */
	return;
	}

      surround = (last != sibling || 
		  (c1 < i) || 
		  (c1 == 0 && i == 0 && (TtaGetElementVolume (sibling) != 0))
		 );
      
      TtaSetDisplayMode (doc, DeferredDisplay);

      /* By default, the new element will be inserted before the selected
	 element */
      before = TRUE;

      TtaOpenUndoSequence (doc, sibling, last, c1, c2);
      mrowCreated = FALSE;

      /* Check whether the selected element is a MathML element */
      if (ustrcmp(TtaGetSSchemaName (elType.ElSSchema), TEXT("MathML")) == 0)
	{
	  /* the selection concerns a MathML element */
	  mathSchema = elType.ElSSchema;
	  if (elType.ElTypeNum == MathML_EL_TEXT_UNIT && c1 > 1)
	    /* the first selected element is a character string */
	    {
	      len = TtaGetTextLength (sibling);
	      if (c1 > len)
		/* the caret is at the end of that character string */
		/* create the new element after the character string */
		before = FALSE;
	      else
		/* split the character string before the first selected char */
		sibling = SplitTextInMathML (doc, sibling, c1, &mrowCreated);
	    }
	}
      else
	  /* the selection is not in a MathML element */
	{
	  /* get the MathML schema for this document or associate it to the
	     document if it is not associated yet */
	  mathSchema = TtaNewNature (docSchema, TEXT("MathML"), TEXT("MathMLP"));
	  if (ustrcmp (TtaGetSSchemaName (elType.ElSSchema), TEXT("HTML")) == 0 &&
	      elType.ElTypeNum != HTML_EL_Math)
	    /* the current selection is in an HTML element, but it's not
	       a Math element */
	    {
	      if (elType.ElTypeNum == HTML_EL_TEXT_UNIT && c1 > 1)
		{
		  len = TtaGetTextLength (sibling);
		  if (c1 > len)
		      /* the caret is at the end of that character string */
		      {
		      /* the new element has to be created after the character
			 string */
		      before = FALSE;
		      el = sibling;
		      TtaNextSibling (&el);
		      if (el == NULL)
			/* the character string is the last child of its
			   parent */
			/* create an empty character string after the
			   Math element to come */
			{
		        el = TtaNewTree (doc, elType, _EMPTYSTR_);
		        TtaInsertSibling (el, sibling, FALSE, doc);
			TtaRegisterElementCreate (el, doc);
			}
		      }
		  else
		    {
		      /* split the text to insert the Math element */
		      TtaRegisterElementReplace (sibling, doc);
		      TtaSplitText (sibling, c1-1, doc);
		      /* take the second part of the split text element */
		      TtaNextSibling (&sibling);
		      TtaRegisterElementCreate (sibling, doc);
		    }
		}

	      if (before)
		{
		  el = sibling;
		  TtaPreviousSibling (&el);
		  if (el != NULL)
		    {
		      newType = TtaGetElementType (el);
		      if (newType.ElTypeNum == HTML_EL_Math)
			{
			  /* insert at the end of the previous MathML element*/
			  before = FALSE;
			  sibling = TtaGetLastChild (el);		      
			}
		    }
		}
	      else
		{
		  el = sibling;
		  TtaNextSibling (&el);
		  if (el != NULL)
		    {
		      newType = TtaGetElementType (el);
		      if (newType.ElTypeNum == HTML_EL_Math)
			{
			  /* insert at the beginning of the next MathML element */
			  before = TRUE;
			  sibling = TtaGetFirstChild (el);		      
			}
		    }
		}
	    }

	  if (ustrcmp (TtaGetSSchemaName (elType.ElSSchema), TEXT("HTML")) == 0 &&
	      elType.ElTypeNum == HTML_EL_Math)
	    /* the current selection is in an HTML element, and it's a
	       Math element */
	    {
	      /* search the first MathML element */
		sibling = TtaGetFirstChild (sibling);
	      if (before)
		el = TtaGetFirstChild (sibling);
	      else
		el = TtaGetLastChild (sibling);		
	      if (el != NULL)
		sibling = el;
	    }
	  else
	    {
	      surround = FALSE;
	      insertSibling = TRUE;
	      /* try to create a Math element at the current position */
	      elType.ElSSchema = TtaGetSSchema (TEXT("HTML"), doc);
	      elType.ElTypeNum = HTML_EL_Math;
	      if (TtaCanInsertSibling (elType, sibling, before, doc))
		 /* create a new Math element as a sibling */
	         insertSibling = TRUE;
	      else if (TtaCanInsertFirstChild (elType, sibling, doc))
		 /* create a new Math element as a child */
	         insertSibling = FALSE;
	      else
                 /* cannot insert a Math element here */
		 {
#ifdef GRAPHML
		 elType = TtaGetElementType (sibling);
		 if (ustrcmp (TtaGetSSchemaName (elType.ElSSchema), "GraphML") == 0)
		    /* selection is within a GraphML element */
		    {
		    elType.ElTypeNum = GraphML_EL_Math;
	            if (TtaCanInsertSibling (elType, sibling, before, doc))
	               /* insert the new Math element as a sibling element */
	                insertSibling = TRUE;
	            else if (TtaCanInsertFirstChild (elType, sibling,doc))
	               /* insert the new Math element as a child element */
	               insertSibling = FALSE;
	            else if (TtaCanInsertSibling (elType,
					TtaGetParent (sibling), before, doc))
			{
			sibling = TtaGetParent (sibling);
			insertSibling = TRUE;
			}
		    else
			sibling = NULL;
		    }
		 else
		    /* not within a GraphML element */
#endif
		    /* cannot insert any MathML element here */
                    sibling = NULL;
		 }
              if (sibling == NULL)
		/* cannot insert a math element here */
                 {
                 TtaSetDisplayMode (doc, DisplayImmediately);
                 TtaCloseUndoSequence (doc);
                 return;
                 }
              else
                 {
		 /* create a Math element */
                 el = TtaNewTree (doc, elType, _EMPTYSTR_);
                 if (insertSibling)
                    /* insert the new Math element as a sibling element */
                    TtaInsertSibling (el, sibling, before, doc);
                 else
                    /* insert the new Math element as a child element */
                    TtaInsertFirstChild (&el, sibling, doc);
                 sibling = TtaGetFirstChild (el);
		 /* register the new Math element in the Undo queue */
		 TtaRegisterElementCreate (el, doc);
                 }
	    }
	}

      elType = TtaGetElementType (sibling);
      newType.ElSSchema = mathSchema;
      selectFirstChild = TRUE;
      ParBlock = FALSE;
      switch (construct)
	{
	case 1:	/* create a Math element */
	  /* handled above */
	  break;
	case 2:
	  newType.ElTypeNum = MathML_EL_MROOT;
	  selectFirstChild = FALSE;	/* select the Index component */
	  break;
	case 3:
	  newType.ElTypeNum = MathML_EL_MSQRT;
	  break;
	case 4:
	  newType.ElTypeNum = MathML_EL_MFRAC;
	  break;
	case 5:
	  newType.ElTypeNum = MathML_EL_MSUBSUP;
	  break;
	case 6:
	  newType.ElTypeNum = MathML_EL_MSUB;
	  break;
	case 7:
	  newType.ElTypeNum = MathML_EL_MSUP;
	  break;
	case 8:
	  newType.ElTypeNum = MathML_EL_MUNDEROVER;
	  break;
	case 9:
	  newType.ElTypeNum = MathML_EL_MUNDER;
	  break;
	case 10:
	  newType.ElTypeNum = MathML_EL_MOVER;
	  break;
	case 11:
	  newType.ElTypeNum = MathML_EL_MROW;
	  ParBlock = TRUE;
	  selectFirstChild = FALSE;	/* select the second component */
	  break;
	case 12:
	  newType.ElTypeNum = MathML_EL_MMULTISCRIPTS;
	  break;
	case 13:     /* MTABLE */
	  displayTableForm = TtaIsSelectionEmpty ();
	  if (displayTableForm)
	    /* ask the user about the number of rows and columns to be created */
	    {
	      NumberRows = 2;
	      NumberCols = 2;
#  ifdef _WINDOWS
              CreateMatrixDlgWindow (BaseDialog, TableForm, TableCols,
				     TableRows, NumberCols, NumberRows);
#  else  /* !_WINDOWS */
	      TtaNewForm (BaseDialog + TableForm, TtaGetViewFrame (doc, 1),
			  TtaGetMessage (1, BMatrix), TRUE, 1, 'L', D_CANCEL);
	      TtaNewNumberForm (BaseDialog + TableCols, BaseDialog + TableForm,
				TtaGetMessage (AMAYA, AM_COLS), 1, 50, TRUE);
	      TtaNewNumberForm (BaseDialog + TableRows, BaseDialog + TableForm,
				TtaGetMessage (AMAYA, AM_ROWS), 1, 200, TRUE);
	      TtaSetNumberForm (BaseDialog + TableCols, NumberCols);
	      TtaSetNumberForm (BaseDialog + TableRows, NumberRows);
	      TtaSetDialoguePosition ();
	      TtaShowDialogue (BaseDialog + TableForm, FALSE);
	      /* wait for an answer */
	      TtaWaitShowDialogue ();
	      if (!UserAnswer)
		/* the user decided to abort the command */
		{
		  TtaCloseUndoSequence (doc);
		  return;
		}
#  endif /* !_WINDOWS */
	    }
	  else
	    {
	      NumberRows = 0;
	      NumberCols = 0;
	    }

	  newType.ElTypeNum = MathML_EL_MTABLE;
	  selectFirstChild = FALSE;	/* select the second component */
	  break;
	default:
	  TtaCloseUndoSequence (doc);
	  return;
	}
      if (!surround || !TransformIntoType (newType, doc))
	{
	  TtaUnselect (doc);
          el = TtaNewTree (doc, newType, _EMPTYSTR_);
	  /* do not check the Thot abstract tree against the structure */
	  /* schema while changing the structure */
	  oldStructureChecking = TtaGetStructureChecking (doc);
	  TtaSetStructureChecking (0, doc);
	  
	  if (elType.ElTypeNum == MathML_EL_MROW ||
	      elType.ElTypeNum == MathML_EL_MathML)
	    {
	      /* the selected element is a MROW or the MathML element */
	      row = sibling;
	      if (before)
		sibling = TtaGetFirstChild (row);
	      else
		sibling = TtaGetLastChild (row);
	      if (sibling == NULL)
		{
		  /* replace the empty MROW by the new element*/
		  TtaInsertSibling (el, row, TRUE, doc);
		  TtaRegisterElementCreate (el, doc);
		  TtaRegisterElementDelete (row, doc);
		  TtaRemoveTree (row, doc);
		}
	      else
		{
		  /* check whether the selected element is a Construct */
		  elType = TtaGetElementType (sibling);
		  if (elType.ElTypeNum == MathML_EL_Construct)
		    {
		    TtaInsertFirstChild (&el, sibling, doc);
		    RemoveAttr (el, doc, MathML_ATTR_IntPlaceholder);
		    }
		  else
		    TtaInsertSibling (el, sibling, before, doc);
		  TtaRegisterElementCreate (el, doc);
		}
	    }
	  else if (elType.ElTypeNum == MathML_EL_Construct)
	    {
	      /* replace the Construct element */
	      TtaInsertFirstChild (&el, sibling, doc);
	      RemoveAttr (el, doc, MathML_ATTR_IntPlaceholder);
	      TtaRegisterElementCreate (el, doc);
	    }
	  else
	    {
	      /* the selected element is not a MROW */
	      if (elType.ElTypeNum == MathML_EL_TEXT_UNIT)
		/* go up to the MN, MI, MO or M_TEXT element */
		sibling = TtaGetParent (sibling);
	      /* insert the new element */
	      TtaInsertSibling (el, sibling, before, doc);
	      if (!mrowCreated)
		 TtaRegisterElementCreate (el, doc);
	    }
	  
	  TtaSetDocumentModified (doc);
	  if (ParBlock)
	    /* the user wants to create a parenthesized block */
	    /* create two MF elements, as the first and last child of the new
	       MROW */
	    {
	      child = TtaGetFirstChild (el);
	      if (child != NULL)
		{
		  newType.ElTypeNum = MathML_EL_MF;
		  fence = TtaNewElement (doc, newType);
		  TtaInsertSibling (fence, child, TRUE, doc);
		  symbType.ElSSchema = mathSchema;
		  symbType.ElTypeNum = MathML_EL_SYMBOL_UNIT;
		  symbol = TtaNewElement (doc, symbType);
		  TtaSetGraphicsShape (symbol, '(', doc);
		  TtaInsertFirstChild (&symbol, fence, doc);

		  fence = TtaNewElement (doc, newType);
		  TtaInsertSibling (fence, child, FALSE, doc);
		  symbol = TtaNewElement (doc, symbType);
		  TtaSetGraphicsShape (symbol, ')', doc);
		  TtaInsertFirstChild (&symbol, fence, doc);
		}
	    }
	  
	  CreateParentMROW (el, doc);

	  if (newType.ElTypeNum == MathML_EL_MTABLE &&
	      (NumberRows > 1 || NumberCols > 1))
	    {
	      /* create the required number of columns and rows in the table */
	      if (NumberCols > 1)
		{
		  elType.ElTypeNum = MathML_EL_MTD;
		  child = TtaSearchTypedElement (elType, SearchInTree, el);
		  while (NumberCols > 1)
		    {
		      new = TtaNewTree (doc, elType, _EMPTYSTR_);
		      TtaInsertSibling (new, child, FALSE, doc);
		      NumberCols--;
		    }
		}
	      if (NumberRows > 1)
		{
		  elType.ElTypeNum = MathML_EL_MTR;
		  row = TtaSearchTypedElement (elType, SearchInTree, el);
		  while (NumberRows > 1)
		    {
		      new = TtaNewTree (doc, elType, _EMPTYSTR_);
		      TtaInsertSibling (new, row, FALSE, doc);
		      NumberRows--;
		    }
		}
	      CheckAllRows (el, doc);
	    }

	  /* if the new element is a child of a FencedExpression element,
	     create the associated FencedSeparator elements */
	  parent = TtaGetParent (el);
	  elType = TtaGetElementType (parent);
	  if (elType.ElTypeNum == MathML_EL_FencedExpression)
	    RegenerateFencedSeparators (parent, doc, TRUE);

	 /* insert placeholders before and/or after the new element if
	    they are needed */
	  placeholderEl = InsertPlaceholder (el, TRUE, doc, !mrowCreated);
	  placeholderEl = InsertPlaceholder (el, FALSE, doc, !mrowCreated);

	  TtaSetDisplayMode (doc, DisplayImmediately);
	  /* check the Thot abstract tree against the structure schema. */
	  TtaSetStructureChecking ((ThotBool)oldStructureChecking, doc);
	  
	  /* selected the leaf in the first (or second) child of the new
	     element */
	  child = TtaGetFirstChild (el);
	  if (!selectFirstChild)
	     /* get the second child */
	     TtaNextSibling (&child);
	  leaf = NULL;
	  while (child != NULL)
	    {
	      leaf = child;
	      child = TtaGetFirstChild (child);
	    }
	  if (leaf)
	    TtaSelectElement (doc, leaf);
	}

      TtaCloseUndoSequence (doc);
}

/*----------------------------------------------------------------------
   CallbackMaths: manage Maths dialogue events.
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void         CallbackMaths (int ref, int typedata, STRING data)
#else
static void         CallbackMaths (ref, typedata, data)
int                 ref;
int                 typedata;
STRING              data;

#endif
{
  Document           doc;

  switch (ref - MathsDialogue)
    {
    case FormMaths:
      /* the user has clicked the DONE button in the Math dialog box */
      InitMaths = FALSE;
      TtaDestroyDialogue (ref);	   
      break;

    case MenuMaths:
      /* the user has selected an entry in the math menu */
      doc = TtaGetSelectedDocument ();
      if (!doc || !TtaGetDocumentAccessMode (doc))
	/* the document is in ReadOnly mode */
	return;

      if ((int) data == 13)
	/* the user asks for the Symbol palette */
	{
	  TtcDisplayGreekKeyboard (doc, 1);
	  return;
	}
      else if (doc > 0)
	/* there is a selection */
        CreateMathConstruct (((int) data) +1);
      break;

    default:
      break;
    }
}

/*----------------------------------------------------------------------
   CreateMathMenu creates the maths menus.           
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void         CreateMathMenu (Document doc, View view)
#else
static void         CreateMathMenu (doc, view)
Document            doc;
View                view;
#endif
{
   if (!TtaGetDocumentAccessMode (doc))
     /* the document is in ReadOnly mode */
     return;

# ifndef _WINDOWS
  if (!InitMaths)
    {
      InitMaths = TRUE;

      /* Dialogue box for the Math palette */
      TtaNewSheet (MathsDialogue + FormMaths, TtaGetViewFrame (doc, view), 
		   TtaGetMessage (AMAYA, AM_BUTTON_MATH),
		   0, NULL, TRUE, 1, 'L', D_DONE);
      TtaNewIconMenu (MathsDialogue + MenuMaths, MathsDialogue + FormMaths, 0,
		   NULL, 14, mIcons, FALSE);
      TtaSetMenuForm (MathsDialogue + MenuMaths, 0);
      TtaSetDialoguePosition ();
    }
  TtaShowDialogue (MathsDialogue + FormMaths, TRUE);
# else /* _WINDOWS */
  CreateMathDlgWindow (TtaGetViewFrame (doc, view), MathsDialogue, TtaGetThotWindow (GetWindowNumber (doc, view)));
# endif /* _WINDOWS */
}
#endif /* MATHML */


/*----------------------------------------------------------------------
   AddMathButton        
  ----------------------------------------------------------------------*/
#ifdef __STDC__
void                AddMathButton (Document doc, View view)
#else
void                AddMathButton (doc, view)
Document            doc;
View                view;
#endif
{
#ifdef MATHML
  MathButton = TtaAddButton (doc, 1, iconMath, CreateMathMenu, "CreateMathMenu",
			     TtaGetMessage (AMAYA, AM_BUTTON_MATH),
			     TBSTYLE_BUTTON, TRUE);
#endif /* MATHML */
}

/*----------------------------------------------------------------------
  SwitchIconMath
  ----------------------------------------------------------------------*/
#ifdef __STDC__
void              SwitchIconMath (Document doc, View view, ThotBool state)
#else  /* __STDC__ */
void              SwitchIconMath (doc, view, state)
Document          doc;
 View             view;
ThotBool          state;
#endif /* __STDC__ */
{
#ifdef MATHML
  if (state)
    TtaChangeButton (doc, view, MathButton, iconMath, state);
  else
    TtaChangeButton (doc, view, MathButton, iconMathNo, state);
#endif /* MATHML */
}

#ifdef MATHML
/*----------------------------------------------------------------------
  CreateMath
  ----------------------------------------------------------------------*/
#ifdef __STDC__
void                CreateMath (Document document, View view)
#else  /* __STDC__ */
void                CreateMath (document, view)
Document            document;
View                view;
 
#endif /* __STDC__ */
{
   CreateMathConstruct (1);
}

/*----------------------------------------------------------------------
  CreateMROOT
  ----------------------------------------------------------------------*/
#ifdef __STDC__
void                CreateMROOT (Document document, View view)
#else  /* __STDC__ */
void                CreateMROOT (document, view)
Document            document;
View                view;
 
#endif /* __STDC__ */
{
   CreateMathConstruct (2);
}

/*----------------------------------------------------------------------
  CreateMSQRT
  ----------------------------------------------------------------------*/
#ifdef __STDC__
void                CreateMSQRT (Document document, View view)
#else  /* __STDC__ */
void                CreateMSQRT (document, view)
Document            document;
View                view;
 
#endif /* __STDC__ */
{
   CreateMathConstruct (3);
}

/*----------------------------------------------------------------------
  CreateMFRAC
  ----------------------------------------------------------------------*/
#ifdef __STDC__
void                CreateMFRAC (Document document, View view)
#else  /* __STDC__ */
void                CreateMFRAC (document, view)
Document            document;
View                view;
 
#endif /* __STDC__ */
{
   CreateMathConstruct (4);
}

/*----------------------------------------------------------------------
  CreateMSUBSUP
  ----------------------------------------------------------------------*/
#ifdef __STDC__
void                CreateMSUBSUP (Document document, View view)
#else  /* __STDC__ */
void                CreateMSUBSUP (document, view)
Document            document;
View                view;
 
#endif /* __STDC__ */
{
   CreateMathConstruct (5);
}

/*----------------------------------------------------------------------
  CreateMSUB
  ----------------------------------------------------------------------*/
#ifdef __STDC__
void                CreateMSUB (Document document, View view)
#else  /* __STDC__ */
void                CreateMSUB (document, view)
Document            document;
View                view;
 
#endif /* __STDC__ */
{
   CreateMathConstruct (6);
}

/*----------------------------------------------------------------------
  CreateMSUP
  ----------------------------------------------------------------------*/
#ifdef __STDC__
void                CreateMSUP (Document document, View view)
#else  /* __STDC__ */
void                CreateMSUP (document, view)
Document            document;
View                view;
 
#endif /* __STDC__ */
{
   CreateMathConstruct (7);
}

/*----------------------------------------------------------------------
  CreateMUNDEROVER
  ----------------------------------------------------------------------*/
#ifdef __STDC__
void                CreateMUNDEROVER (Document document, View view)
#else  /* __STDC__ */
void                CreateMUNDEROVER (document, view)
Document            document;
View                view;
 
#endif /* __STDC__ */
{
   CreateMathConstruct (8);
}

/*----------------------------------------------------------------------
  CreateMUNDER
  ----------------------------------------------------------------------*/
#ifdef __STDC__
void                CreateMUNDER (Document document, View view)
#else  /* __STDC__ */
void                CreateMUNDER (document, view)
Document            document;
View                view;
 
#endif /* __STDC__ */
{
   CreateMathConstruct (9);
}

/*----------------------------------------------------------------------
  CreateMOVER
  ----------------------------------------------------------------------*/
#ifdef __STDC__
void                CreateMOVER (Document document, View view)
#else  /* __STDC__ */
void                CreateMOVER (document, view)
Document            document;
View                view;
 
#endif /* __STDC__ */
{
   CreateMathConstruct (10);
}

/*----------------------------------------------------------------------
  CreateMROW
  ----------------------------------------------------------------------*/
#ifdef __STDC__
void                CreateMROW (Document document, View view)
#else  /* __STDC__ */
void                CreateMROW (document, view)
Document            document;
View                view;
 
#endif /* __STDC__ */
{
   CreateMathConstruct (11);
}

/*----------------------------------------------------------------------
  CreateMMULTISCRIPTS
  ----------------------------------------------------------------------*/
#ifdef __STDC__
void                CreateMMULTISCRIPTS (Document document, View view)
#else  /* __STDC__ */
void                CreateMMULTISCRIPTS (document, view)
Document            document;
View                view;
 
#endif /* __STDC__ */
{
   CreateMathConstruct (12);
}

/*----------------------------------------------------------------------
  CreateMTABLE
  ----------------------------------------------------------------------*/
#ifdef __STDC__
void                CreateMTABLE (Document document, View view)
#else  /* __STDC__ */
void                CreateMTABLE (document, view)
Document            document;
View                view;
 
#endif /* __STDC__ */
{
   CreateMathConstruct (13);
}

/*----------------------------------------------------------------------
   InitMathML initializes MathML context.           
  ----------------------------------------------------------------------*/
void                InitMathML ()
{
#  ifndef _WINDOWS 
   iconMath = TtaCreatePixmapLogo (Math_xpm);
   iconMathNo = TtaCreatePixmapLogo (MathNo_xpm);
#  ifdef AMAYA_JAVA
   TtaRegisterPixmap("Math", iconMath);
   TtaRegisterPixmap("MathNo", iconMathNo);
#  endif /* AMAYA_JAVA */
   mIcons[0] = TtaCreatePixmapLogo (Bmath_xpm);
   mIcons[1] = TtaCreatePixmapLogo (root_xpm);
   mIcons[2] = TtaCreatePixmapLogo (sqrt_xpm);
   mIcons[3] = TtaCreatePixmapLogo (frac_xpm);
   mIcons[4] = TtaCreatePixmapLogo (subsup_xpm);
   mIcons[5] = TtaCreatePixmapLogo (sub_xpm);
   mIcons[6] = TtaCreatePixmapLogo (sup_xpm);
   mIcons[7] = TtaCreatePixmapLogo (overunder_xpm);
   mIcons[8] = TtaCreatePixmapLogo (under_xpm);
   mIcons[9] = TtaCreatePixmapLogo (over_xpm);
   mIcons[10] = TtaCreatePixmapLogo (fence_xpm);
   mIcons[11] = TtaCreatePixmapLogo (mscript_xpm);
   mIcons[12] = TtaCreatePixmapLogo (matrix_xpm);
   mIcons[13] = TtaCreatePixmapLogo (greek_xpm);
#  endif /* _WINDOWS */
  MathsDialogue = TtaSetCallback (CallbackMaths, MAX_MATHS);
  KeyboardsLoadResources ();
}

/*----------------------------------------------------------------------
 CheckMROW
 If el is a MROW element with only one child, remove the MROW element
 and replace it by its child.
 -----------------------------------------------------------------------*/
#ifdef __STDC__
static void CheckMROW (Element* el, Document doc)
#else /* __STDC__*/
static void CheckMROW (el, doc)
  Element* el;
  Document doc;
#endif /* __STDC__*/
{
  Element	child, firstChild, next;
  ElementType	elType;
  AttributeType	attrType;
  Attribute	attr;
  int		nChildren;
  int           oldStructureChecking;

  elType = TtaGetElementType (*el);
  if (elType.ElTypeNum == MathML_EL_MROW)
     /* the parent of the deleted element is a MROW */
     {
     firstChild = TtaGetFirstChild (*el);
     child = firstChild;
     /* count all children that are not placeholders */
     nChildren = 0;
     while (child != NULL && nChildren < 2)
	{
	elType = TtaGetElementType (child);
	if (elType.ElTypeNum != MathML_EL_Construct)
	   /* this is not a Construct */
	   nChildren++;
	else
	   {
	   attrType.AttrSSchema = elType.ElSSchema;
	   attrType.AttrTypeNum = MathML_ATTR_IntPlaceholder;
	   attr = TtaGetAttribute (child, attrType);
	   if (attr == NULL)
	      /* this is not a placeholder */
	      nChildren++;
	   }
        TtaNextSibling (&child);
	}

     if (nChildren == 1)
       /* there is only one element that is not a placeholder in the MROW.
          Remove the MROW */
       {
       TtaSetDisplayMode (doc, DeferredDisplay);
       oldStructureChecking = TtaGetStructureChecking (doc);
       TtaSetStructureChecking (0, doc);
       TtaRegisterElementDelete (*el, doc);
       child = firstChild;
       while (child != NULL)
	  {
	  next = child;
	  TtaNextSibling (&next);
          TtaRemoveTree (child, doc);
          TtaInsertSibling (child, *el, TRUE, doc);
	  TtaRegisterElementCreate (child, doc);
	  child = next;
	  }
       TtaDeleteTree (*el, doc);
       *el = NULL;
       TtaSetStructureChecking ((ThotBool)oldStructureChecking, doc);
       TtaSetDisplayMode (doc, DisplayImmediately);
       }
     }
}


/*----------------------------------------------------------------------
   GetCharType
   returns the type of character c (MN, MI or MO).
 -----------------------------------------------------------------------*/
#ifdef __STDC__
static int GetCharType (UCHAR_T c, CHAR_T alphabet)
#else /* __STDC__*/
static int GetCharType (c, alphabet)
UCHAR_T c;
CHAR_T  alphabet;
#endif /* __STDC__*/
{
  int	ret;

  ret = MathML_EL_MO;
  if (alphabet == 'L')
     /* ISO-Latin 1 */
     {
     if (c >= '0' && c <= '9')
        ret = MathML_EL_MN;
     else if ((c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z') || c == ' ')
        ret = MathML_EL_MI;
     else if (((int) c) >= 192 && ((int) c) <= 255 &&
	      ((int) c) != 215 && ((int) c) != 247)
	ret = MathML_EL_MI;
     else
        ret = MathML_EL_MO;
     }
  else if (alphabet == 'G')
     /* Symbol character set */
     {
     if (c >= '0' && c <= '9')
        ret = MathML_EL_MN;
     else if ((c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z'))
        ret = MathML_EL_MI;
     else
        ret = MathML_EL_MO;
     }
  return ret;
}

/*----------------------------------------------------------------------
   MathSetAttributes
   Set attributes of element el according to its content.
 -----------------------------------------------------------------------*/
#ifdef __STDC__
static void MathSetAttributes (Element el, Document doc, Element* selEl)
#else /* __STDC__*/
static void MathSetAttributes (el, doc, selEl)
     Element el;
     Document doc;
     Element* selEl;
#endif /* __STDC__*/
{
  ElementType	elType, parentType;
  Element	parent, grandParent;

  elType = TtaGetElementType (el);
  if (elType.ElTypeNum == MathML_EL_MI)
     /* is it really an identifier? */
     {
     /* try to separate function names, identifiers and plain text */
     /****** TO DO ******/
     }
  if (elType.ElTypeNum == MathML_EL_MO)
     {
     SetIntAddSpaceAttr (el, doc);
     parent = TtaGetParent (el);
     if (parent != NULL)
	{
	parentType = TtaGetElementType (parent);
	if (parentType.ElTypeNum != MathML_EL_Base &&
	    parentType.ElTypeNum != MathML_EL_UnderOverBase)
	   SetIntVertStretchAttr (el, doc, 0, selEl);
	else
	   {
	   grandParent = TtaGetParent (parent);
	   if (grandParent != NULL)
	      SetIntVertStretchAttr (grandParent, doc, parentType.ElTypeNum, selEl);
	   }
	}
     }
  else
     /* it's not an operator (mo). Remove all attributes that can be set only
       on operators, except if it's a mstyle element. */
     if (elType.ElTypeNum == MathML_EL_MSTYLE)
     {
     RemoveAttr (el, doc, MathML_ATTR_IntAddSpace);
     RemoveAttr (el, doc, MathML_ATTR_form);
     RemoveAttr (el, doc, MathML_ATTR_fence);
     RemoveAttr (el, doc, MathML_ATTR_separator);
     RemoveAttr (el, doc, MathML_ATTR_lspace);
     RemoveAttr (el, doc, MathML_ATTR_rspace);
     RemoveAttr (el, doc, MathML_ATTR_separator);
     RemoveAttr (el, doc, MathML_ATTR_stretchy);
     RemoveAttr (el, doc, MathML_ATTR_symmetric);
     RemoveAttr (el, doc, MathML_ATTR_maxsize);
     RemoveAttr (el, doc, MathML_ATTR_minsize);
     RemoveAttr (el, doc, MathML_ATTR_largeop);
     RemoveAttr (el, doc, MathML_ATTR_movablelimits);
     RemoveAttr (el, doc, MathML_ATTR_accent);
     }
  if (elType.ElTypeNum == MathML_EL_MI)
     SetFontstyleAttr (el, doc);
  else
     RemoveAttr (el, doc, MathML_ATTR_IntFontstyle);		
}

/*----------------------------------------------------------------------
   MergeMathEl
   merge element el2 with element el
 -----------------------------------------------------------------------*/
#ifdef __STDC__
static void MergeMathEl (Element el, Element el2, ThotBool before, Document doc)
#else /* __STDC__*/
static void MergeMathEl (el, el2, before, doc)
     Element el;
     Element el2;
     ThotBool before;
     Document doc;
#endif /* __STDC__*/
{
  Element	textEl2, nextEl, prevEl;

  TtaRegisterElementReplace (el, doc);
  textEl2 = TtaGetFirstChild (el2);
  if (before)
     prevEl = NULL;
  else
     prevEl = TtaGetLastChild (el);
  while (textEl2 != NULL)
     {
     nextEl = textEl2;
     TtaNextSibling (&nextEl);
     TtaRemoveTree (textEl2, doc);
     if (prevEl == NULL)
	TtaInsertFirstChild (&textEl2, el, doc);
     else
        TtaInsertSibling (textEl2, prevEl, FALSE, doc);
     prevEl = textEl2;
     textEl2 = nextEl;
     }
  TtaRegisterElementDelete (el2, doc);
  TtaDeleteTree (el2, doc);
  MathSetAttributes (el, doc, NULL);
}

/*----------------------------------------------------------------------
   TextLength
   return the total length of the text contained in element el.
 -----------------------------------------------------------------------*/
#ifdef __STDC__
static int TextLength (Element el)
#else /* __STDC__*/
static int TextLength (el)
     Element el;
#endif /* __STDC__*/
{
  int len;
  Element child;

  len = 0;
  child = TtaGetFirstChild (el);
  while (child != NULL)
    {
    len += TtaGetTextLength (child);
    TtaNextSibling (&child);
    }
  return len;
}

/*----------------------------------------------------------------------
   ClosestLeaf
   return the Closest TEXT element for element el;
 -----------------------------------------------------------------------*/
#ifdef __STDC__
static Element ClosestLeaf (Element el, int* pos)
#else /* __STDC__*/
static Element ClosestLeaf (el, pos)
     Element el;
     int* pos;
#endif /* __STDC__*/
{
   Element	elem, prev, next, child, leaf, parent;
   ElementType	elType;

   elem = NULL;
   leaf = NULL;
   prev = NULL;
   next = NULL;
   parent = el;
   do
      {
      prev = parent;  TtaPreviousSibling (&prev);
      if (prev != NULL)
	 {
         elType = TtaGetElementType (prev);
         if (elType.ElTypeNum == MathML_EL_FencedSeparator)
	    /* avoid selecting FencedSeparator elements */
	    TtaPreviousSibling (&prev);
	 }
      if (prev == NULL)
	 {
	 next = parent;  TtaNextSibling (&next);
	 if (next != NULL)
	    {
	    elType = TtaGetElementType (next);
	    if (elType.ElTypeNum == MathML_EL_FencedSeparator)
	       /* avoid selecting FencedSeparator elements */
	       TtaNextSibling (&next);
	    }
	 if (next == NULL)
	    parent = TtaGetParent (parent);
	 }
      }
   while (next == NULL && prev == NULL && parent != NULL);
   if (prev != NULL)
      {
      child = prev;
      while (child != NULL)
	{
	leaf = child;
	child = TtaGetLastChild (child);
	}
      }
   else
      if (next != NULL)
        {
	child = next;
	while (child != NULL)
	   {
	   leaf = child;
	   child = TtaGetFirstChild (child);
	   }
	}
   if (leaf != NULL)
      {
      elem = leaf;
      elType = TtaGetElementType (leaf);
      if (elType.ElTypeNum == MathML_EL_TEXT_UNIT)
	if (prev != NULL)
	   *pos = TtaGetTextLength (leaf) + 1;
	else
	   *pos = 1;
      }
   return elem;
}

/*----------------------------------------------------------------------
   ParseMathString
   The content of an element MTEXT, MI, MO, or MN, has been modified
   or created.
   Parse the new content and create the appropriate MI, MO, MN elements
   according to the new contents.
 -----------------------------------------------------------------------*/
#ifdef __STDC__
static void ParseMathString (Element theText, Element theElem, Document doc)
#else /* __STDC__*/
static void ParseMathString (theText, theElem, doc)
     Element theText;
     Element theElem;
     Document doc;
#endif /* __STDC__*/

{
  Element	el, selEl, prevEl, nextEl, textEl, newEl, lastEl,
		firstEl, newSelEl, prev, next, parent, placeholderEl;
  ElementType	elType, elType2;
  SSchema	MathMLSchema;
  int		firstSelChar, lastSelChar, newSelChar, len, totLen, i, j,
		start;
  CHAR_T		alphabet, c;
  Language	lang;
#define TXTBUFLEN 200
  UCHAR_T         text[TXTBUFLEN];
  CHAR_T		language[TXTBUFLEN];
  UCHAR_T         mathType[TXTBUFLEN];
  int           oldStructureChecking;

  /* get the current selection */
  TtaGiveFirstSelectedElement (doc, &selEl, &firstSelChar, &lastSelChar);
  newSelEl = NULL;

  elType = TtaGetElementType (theElem);
  MathMLSchema = elType.ElSSchema;

  prevEl = NULL;
  el = theElem;
  TtaPreviousSibling (&el);
  if (el != NULL)
     {
     elType = TtaGetElementType (el);
     if (elType.ElSSchema == MathMLSchema &&
	 (elType.ElTypeNum == MathML_EL_MTEXT ||
	 elType.ElTypeNum == MathML_EL_MI ||
	 elType.ElTypeNum == MathML_EL_MO ||
	 elType.ElTypeNum == MathML_EL_MN ||
	 elType.ElTypeNum == MathML_EL_MS))
	prevEl = el;
     }
  nextEl = NULL;
  el = theElem;
  TtaNextSibling (&el);
  if (el != NULL)
     {
     elType = TtaGetElementType (el);
     if (elType.ElSSchema == MathMLSchema &&
	 (elType.ElTypeNum == MathML_EL_MTEXT ||
	 elType.ElTypeNum == MathML_EL_MI ||
	 elType.ElTypeNum == MathML_EL_MO ||
	 elType.ElTypeNum == MathML_EL_MN ||
	 elType.ElTypeNum == MathML_EL_MS))
	nextEl = el;
     }

  i = 0;
  totLen = 0;
  elType = TtaGetElementType (theElem);
  textEl = theText;
  if (textEl != NULL)
       {
       len = TtaGetTextLength (textEl);
       /* selection */
       if (selEl == textEl)
	 {
	   newSelChar = totLen + firstSelChar;
	   newSelEl = textEl;
	   TtaUnselect (doc);
	 }
       /* get the content and analyze it */
       if (len > 0)
          {
          len = TXTBUFLEN - totLen;
          TtaGiveTextContent (textEl, &text[i], &len, &lang);
          alphabet = TtaGetAlphabet (lang);
	  for (j = 0; j < len; j++)
	     {
	     language[i+j] = lang;
	     mathType[i+j] = (UCHAR_T) GetCharType (text[i+j], alphabet);
	     }
	  i+= len;
	  totLen += len;
	  }
       }

  /* try to identify numbers like: 0.123  1,000,000  2.1e10 */
  for (i = 1; i < totLen; i++)
    {
    if ((text[i] == ',' || text[i] == '.' || text[i] == 'e') &&
         mathType[i-1] == MathML_EL_MN &&
	 i < totLen-1 &&mathType[i+1] == MathML_EL_MN)
	/* comma or point between two digits: the comma or point is part of
	   the number */
	  {
	  mathType[i] = (UCHAR_T) MathML_EL_MN;
	  i++;
	  }
    }

  TtaOpenUndoSequence (doc, selEl, selEl, firstSelChar, lastSelChar);
  TtaSetDisplayMode (doc, DeferredDisplay);
  oldStructureChecking = TtaGetStructureChecking (doc);
  TtaSetStructureChecking (0, doc);
  firstEl = NULL;
  start = 0;
  lastEl = NULL;
  if (totLen == 0)
    /* the character string is empty. Remove the parent element (MI, MN, MO...)
       if the parent does not contain any other element */
    {
    el = theText;
    TtaPreviousSibling (&el);
    if (el == NULL)
       {
       el = theText;
       TtaNextSibling (&el);
       if (el == NULL)
	  /* the text element has no sibling, delete its parent */
	  {
	  if (newSelEl != NULL)
	     newSelEl = ClosestLeaf (theElem, &newSelChar);

	  prev = theElem;
	  TtaPreviousSibling (&prev);
	  if (prev == NULL)
	     {
	     next = theElem;
	     TtaNextSibling (&next);
	     }

	  parent = TtaGetParent (theElem);
	  TtaRegisterElementDelete (theElem, doc);
	  TtaDeleteTree (theElem, doc);
	  theElem = NULL;

	  /* The deletion of the parent element may require a Placeholder
	     instead of the deleted element */
	  placeholderEl = NULL;
	  if (prev != NULL)
	    placeholderEl = InsertPlaceholder (prev, FALSE, doc, TRUE);
	  else if (next != NULL)
	    placeholderEl = InsertPlaceholder (next, TRUE, doc, TRUE);
	  if (placeholderEl != NULL)
	     newSelEl = placeholderEl;   

	  /* if the deleted element is a child of a FencedExpression element,
	     upate the associated FencedSeparator elements */
	  elType = TtaGetElementType (parent);
	  if (elType.ElTypeNum == MathML_EL_FencedExpression)
	    RegenerateFencedSeparators (parent, doc, TRUE);

	  CheckMROW (&parent, doc);
	  if (parent != NULL)
	   if (TtaGetFirstChild (parent) == NULL)
	      {
	      elType.ElTypeNum = MathML_EL_Construct;
	      newEl = TtaNewElement (doc, elType);
	      TtaInsertFirstChild (&newEl, parent, doc);
	      if (newSelEl != NULL)
	         newSelEl = newEl;
	      }
	  firstEl = NULL;
	  prevEl = NULL;
	  nextEl = NULL;
	  }
       }
    }
  else
    /* the modified character string is not empty. Parse it */
    for (i = 1; i <= totLen; i++)
     if (mathType[i] != mathType[i-1] ||
	 language[i] != language[i-1] ||
	 mathType[i-1] == MathML_EL_MO ||
	 i == totLen)
       /* create a new element */
       {
       if (lastEl == NULL)
	  {
	  elType = TtaGetElementType (theElem);
	  if (elType.ElTypeNum != mathType[i-1] ||
	      mathType[i-1] == MathML_EL_MO)
	     {
	     prev = theText;
	     TtaPreviousSibling (&prev);
	     if (prev != NULL)
		{
		textEl = prev;
		TtaPreviousSibling (&prev);
		newEl = TtaNewElement (doc, elType);
		TtaInsertSibling (newEl, theElem, TRUE, doc);
		prevEl = newEl;
		TtaRegisterElementDelete (textEl, doc);
		TtaRemoveTree (textEl, doc);
		TtaInsertFirstChild (&textEl, newEl, doc);
		while (prev != NULL)
		   {
		   next = textEl;
		   textEl = prev;
		   TtaPreviousSibling (&prev);
		   TtaRegisterElementDelete (textEl, doc);
		   TtaRemoveTree (textEl, doc);
		   TtaInsertSibling (textEl, next, TRUE, doc);
		   }
		MathSetAttributes (newEl, doc, &newSelEl);
		TtaRegisterElementCreate (newEl, doc);
		}
	     TtaRegisterElementReplace (theElem, doc);
	     ChangeTypeOfElement (theElem, doc, mathType[i-1]);
	     }
	  next = theText;
	  TtaNextSibling (&next);
	  if (next != NULL)
	     {
	     textEl = next;
	     TtaNextSibling (&next);
	     newEl = TtaNewElement (doc, elType);
	     TtaInsertSibling (newEl, theElem, FALSE, doc);
	     nextEl = newEl;
	     TtaRegisterElementDelete (textEl, doc);
	     TtaRemoveTree (textEl, doc);
	     TtaInsertFirstChild (&textEl, newEl, doc);
	     while (next != NULL)
		{
	        prev = textEl;
		textEl = next;
		TtaNextSibling (&next);
		TtaRegisterElementDelete (textEl, doc);
		TtaRemoveTree (textEl, doc);
		TtaInsertSibling (textEl, prev, FALSE, doc);
		}
	     MathSetAttributes (newEl, doc, &newSelEl);
	     TtaRegisterElementCreate (newEl, doc);
	     }
	  newEl = theElem;
	  textEl = theText;
	  firstEl = theElem;
	  }
       else
	  {
          elType.ElTypeNum = mathType[i-1];
          newEl = TtaNewElement (doc, elType);
	  TtaInsertSibling (newEl, lastEl, FALSE, doc);
	  elType.ElTypeNum = MathML_EL_TEXT_UNIT;
	  textEl = TtaNewElement (doc, elType);
	  TtaInsertFirstChild (&textEl, newEl, doc);
	  TtaRegisterElementCreate (newEl, doc);
          }
       while (text[start] == ' ')
	  start++;
       j = i - 1;
       while (text[j] == ' ' && j > start)
	  j--;
       j++;
       c = text[j];
       text[j] = '\0';
       TtaSetTextContent (textEl, &text[start], language[start], doc);
       text[j] = c;
       lastEl = newEl;
       if (newSelEl != NULL)
	  {
	  newSelEl = textEl;
	  if (newSelChar < j)
	     if (newSelChar < start)
		newSelChar = 1;
	     else
		newSelChar -= start;
	  }
       MathSetAttributes (newEl, doc, &newSelEl);
       start = i;

       if (mathType[i-1] == MathML_EL_MO)
	  /* the new element is an operator */
	  {
	  /* if the new element contains a single SYMBOL, placeholders may
	     be needed before and/or after that operator */
	  placeholderEl = InsertPlaceholder (newEl, TRUE, doc, TRUE);
	  placeholderEl = InsertPlaceholder (newEl, FALSE, doc, TRUE);
	  /* the new contents may be an horizontally stretchable symbol */
	  if (newEl != NULL)
	    {
	    parent = TtaGetParent (newEl);
	    elType = TtaGetElementType (parent);
	    if (elType.ElTypeNum == MathML_EL_UnderOverBase ||
		elType.ElTypeNum == MathML_EL_Underscript ||
		elType.ElTypeNum == MathML_EL_Overscript)
	       SetSingleIntHorizStretchAttr (parent, doc, &newSelEl);
	    }
	  }
       }

  /* try to merge the first element processed with its previous sibling */
  if (prevEl != NULL && firstEl != NULL)
    {
    elType = TtaGetElementType (prevEl);
    if (elType.ElTypeNum != MathML_EL_MO)
      /* Don't merge operators */
      {
      elType2 = TtaGetElementType (firstEl);
      if (elType.ElTypeNum == elType2.ElTypeNum &&
          elType.ElSSchema == elType2.ElSSchema)
         {
         if (newSelEl == prevEl)
	    newSelEl = firstEl;
         else if (newSelEl == firstEl)
	    newSelChar += TextLength (prevEl);
         MergeMathEl (firstEl, prevEl, TRUE, doc);
	 }
       }
    }
  /* try to merge the last element processed with its next sibling */
  if (nextEl != NULL && lastEl != NULL)
    {
    elType = TtaGetElementType (nextEl);
    if (elType.ElTypeNum != MathML_EL_MO)
      /* Don't merge operators */
      {
      elType2 = TtaGetElementType (lastEl);
      if (elType.ElTypeNum == elType2.ElTypeNum &&
          elType.ElSSchema == elType2.ElSSchema)
         {
         if (newSelEl == nextEl)
	    {
	    newSelEl = lastEl;
	    newSelChar += TextLength (lastEl);
	    }
         MergeMathEl (lastEl, nextEl, FALSE, doc);
	 }
       }
    }

  if (firstEl != NULL)
     {
     /* if we are in a FencedExpression element, upate the associated
	FencedSeparator elements */
     parent = TtaGetParent (firstEl);
     elType = TtaGetElementType (parent);
     if (elType.ElTypeNum == MathML_EL_FencedExpression)
       RegenerateFencedSeparators (parent, doc, TRUE);

     /* Create a MROW element that encompasses the new elements if necessary */
     CreateParentMROW (firstEl, doc);
     }

  TtaSetStructureChecking ((ThotBool)oldStructureChecking, doc);
  TtaSetDisplayMode (doc, DisplayImmediately);
  TtaCloseUndoSequence (doc);

  /* set a new selection */
  if (newSelEl != NULL)
     {
     elType = TtaGetElementType (newSelEl);
     if (elType.ElTypeNum == MathML_EL_TEXT_UNIT)
        TtaSelectString (doc, newSelEl, newSelChar, newSelChar-1);
     else
	TtaSelectElement (doc, newSelEl);
     }
}

/*----------------------------------------------------------------------
   MathStringModified
   The content of an element MTEXT, MI, MO, MN, or MS has been modified.
   Parse the new content and create the appropriate MI, MO, MN elements.
 -----------------------------------------------------------------------*/
#ifdef __STDC__
void MathStringModified (NotifyOnTarget *event)
#else /* __STDC__*/
void MathStringModified (event)
     NotifyOnTarget *event;
#endif /* __STDC__*/
{
  ParseMathString (event->target, event->element, event->document);
}

/*----------------------------------------------------------------------
 NewMathString
 An new text string has been created in a MathML element.
 Parse its contents.
 -----------------------------------------------------------------------*/
#ifdef __STDC__
void NewMathString (NotifyElement *event)
#else /* __STDC__*/
void NewMathString(event)
     NotifyElement *event;
#endif /* __STDC__*/
{
   RemoveAttribute (event->element, event->document, MathML_ATTR_EntityName);
   if (TtaGetTextLength (event->element) > 0)
      ParseMathString (event->element, TtaGetParent (event->element),
		       event->document);
}

/*----------------------------------------------------------------------
 MathElementPasted
 An element has been pasted in a MathML structure.
 Create placeholders before and after the pasted elements if necessary.
 -----------------------------------------------------------------------*/
#ifdef __STDC__
void MathElementPasted (NotifyElement *event)
#else /* __STDC__*/
void MathElementPasted(event)
     NotifyElement *event;
#endif /* __STDC__*/
{
   Element	placeholderEl, parent;
   ElementType	elType;
  int           oldStructureChecking;

   elType = TtaGetElementType (event->element);
   oldStructureChecking = TtaGetStructureChecking (event->document);
   TtaSetStructureChecking (0, event->document);

   /* if the new element is a child of a FencedExpression element,
      create the associated FencedSeparator elements */
   parent = TtaGetParent (event->element);
   elType = TtaGetElementType (parent);
   if (elType.ElTypeNum == MathML_EL_FencedExpression)
     RegenerateFencedSeparators (parent, event->document, FALSE/******/);

   /* create placeholders before and/or after the new element */
   placeholderEl = InsertPlaceholder (event->element, TRUE, event->document,
				      FALSE/****/);
   placeholderEl = InsertPlaceholder (event->element, FALSE, event->document,
				      FALSE/****/);

   TtaSetStructureChecking ((ThotBool)oldStructureChecking, event->document);
}


/*----------------------------------------------------------------------
 MathElementWillBeDeleted
 This function is called by the DELETE command for each selected element
 and for all their descendants.
 -----------------------------------------------------------------------*/
#ifdef __STDC__
ThotBool MathElementWillBeDeleted (NotifyElement *event)
#else /* __STDC__*/
ThotBool MathElementWillBeDeleted(event)
     NotifyElement *event;
#endif /* __STDC__*/
{
  /* TTE_STANDARD_DELETE_LAST_ITEM indicates the last element to be
     deleted, but function MathElementWillBeDeleted is called afterwards
     for all decendants of this last selected element, without
     TTE_STANDARD_DELETE_LAST_ITEM.
     Function MathElementDeleted is called only for the selected elements,
     not for their descendants */
  if (!IsLastDeletedElement)
     {
     IsLastDeletedElement = (event->position == TTE_STANDARD_DELETE_LAST_ITEM);
     if (IsLastDeletedElement)
	LastDeletedElement = event->element;
     }
  else
     {
     if (!TtaIsAncestor (event->element, LastDeletedElement))
	{
	LastDeletedElement = NULL;
	IsLastDeletedElement = False;
	}
     }
  return FALSE; /* let Thot perform normal operation */
}


/*----------------------------------------------------------------------
  ----------------------------------------------------------------------*/
#ifdef __STDC__
void                DeleteMColumn (Document document, View view)
#else  /* __STDC__ */
void                DeleteMColumn (document, view)
Document            document;
View                view;

#endif /* __STDC__ */
{
   Element             el, cell, colHead, selCell, leaf;
   ElementType         elType;
   AttributeType       attrType;
   Attribute           attr;
   Document            refDoc;
   CHAR_T                name[50];
   int                 firstchar, lastchar, len;
   ThotBool            selBefore;

   /* get the first selected element */
   TtaGiveFirstSelectedElement (document, &el, &firstchar, &lastchar);
   if (el != NULL)
     {
       elType = TtaGetElementType (el);
       if (elType.ElSSchema == GetMathMLSSchema (document))
	 {
	   if (elType.ElTypeNum == MathML_EL_MTD)
	     cell = el;
	   else
	     {
	       elType.ElTypeNum = MathML_EL_MTD;
	       cell = TtaGetTypedAncestor (el, elType);
	     }
	   if (cell != NULL)
	     {
	       /* prepare the new selection */
	       selCell = cell;
	       TtaNextSibling (&selCell);
	       if (selCell)
		  selBefore = FALSE;
	       else
		  {
		  selCell = cell;
		  TtaPreviousSibling (&selCell);
		  selBefore = TRUE;
		  }
	       /* get current column */
	       attrType.AttrSSchema = elType.ElSSchema;
	       attrType.AttrTypeNum = MathML_ATTR_MRef_column;
	       attr = TtaGetAttribute (cell, attrType);
	       if (attr != NULL)
		 {
		   TtaGiveReferenceAttributeValue (attr, &colHead, name,
						   &refDoc);
		   TtaOpenUndoSequence (document, el, el, firstchar,
					lastchar);
		   /* remove column */
		   RemoveColumn (colHead, document, FALSE, TRUE);
		   
		   /* set new selection */
		   if (selBefore)
		      leaf = TtaGetLastLeaf (selCell);
		   else
		      leaf = TtaGetFirstLeaf (selCell);
		   elType = TtaGetElementType (leaf);
		   if (elType.ElTypeNum == MathML_EL_TEXT_UNIT)
		     if (selBefore)
		        {
			len = TtaGetTextLength (leaf);
		        TtaSelectString (document, leaf, len+1, len);
			}
		     else
		        TtaSelectString (document, leaf, 1, 0);
		   else
		     TtaSelectElement (document, leaf);

		   TtaCloseUndoSequence (document);
		   TtaSetDocumentModified (document);
		 }
	     }
	 }
     }
}

/*----------------------------------------------------------------------
 MathElementDeleted
 An element has been deleted in a MathML structure.
 Create the necessary placeholders.
 Remove the enclosing MROW element if it has only one child.
 -----------------------------------------------------------------------*/
#ifdef __STDC__
void MathElementDeleted (NotifyElement *event)
#else /* __STDC__*/
void MathElementDeleted(event)
     NotifyElement *event;
#endif /* __STDC__*/
{
   Element	sibling, placeholderEl, parent, child, grandChild;
   ElementType	parentType;
   int		i, newTypeNum;
   int          oldStructureChecking;

   parent = event->element; /* parent of the deleted element */
   parentType = TtaGetElementType (parent);

   if (parentType.ElTypeNum == MathML_EL_FencedExpression)
      /* a child of a FencedExpression element has been deleted,
         re-generate all FencedSeparator elements in that FencedExpression */
     RegenerateFencedSeparators (parent, event->document, FALSE/*****/);

   /* If there are several successive placeholders at the place where the
      element has been deleted, remove all unneeded placeholders.
      If the deletion makes a placeholder necessary at the position of the
      deleted element, insert a placeholder at that position */
   sibling = TtaGetFirstChild (parent); /* first sibling of the deleted
					   element */
   if (event->position == 0)
      {
      /* the first child of parent has been deleted.
	 Create a placeholder before the new first child */
      if (sibling != NULL)
        placeholderEl = InsertPlaceholder (sibling, TRUE, event->document,
					   FALSE/******/);
      }
   else if (IsLastDeletedElement)
      {
      for (i = 1; i < event->position && sibling != NULL; i++)
         TtaNextSibling (&sibling);
      if (sibling != NULL)
         placeholderEl = InsertPlaceholder (sibling, FALSE, event->document,
					    FALSE/*****/);
      }
   IsLastDeletedElement = FALSE;
   LastDeletedElement = NULL;

   /* If there is an enclosing MROW that is no longer needed, remove
      that MROW */
   CheckMROW (&parent, event->document); /******/

   /* The deletion of this component may lead to a structure change for its
      siblings and its parent */
   newTypeNum = 0;
   switch (event->elementType.ElTypeNum)
      {
      case MathML_EL_Index:		/* an Index has been deleted */
	/* transform the MROOT into a MSQRT */
	newTypeNum = MathML_EL_MSQRT;
	break;

      case MathML_EL_Numerator:		/* a Numerator has been deleted */
      case MathML_EL_Denominator:	/* a Denominator has been deleted */
	/* remove the enclosing MFRAC */
	newTypeNum = -1;
	break;

      case MathML_EL_Subscript:		/* a Subscript has been deleted */
	if (parentType.ElTypeNum == MathML_EL_MSUBSUP)
	   /* a Subscript in a MSUBSUP. Transform the MSUBSUP into a MSUP */
	   newTypeNum = MathML_EL_MSUP;
	else if (parentType.ElTypeNum == MathML_EL_MSUB)
	   /* a Subscript in a MSUB. Remove the MSUB and the Base */
	   newTypeNum = -1;
	break;

      case MathML_EL_Superscript:	/* a Superscript has been deleted */
	if (parentType.ElTypeNum == MathML_EL_MSUBSUP)
	   /* a Superscript in a MSUBSUP. Transform the MSUBSUP into a MSUB */
	   newTypeNum = MathML_EL_MSUB;
	else if (parentType.ElTypeNum == MathML_EL_MSUP)
	   /* a Superscript in a MSUP. Remove the MSUP and the Base */
	   newTypeNum = -1;
	break;

      case MathML_EL_Underscript:	/* an Underscript has been deleted */
	if (parentType.ElTypeNum == MathML_EL_MUNDEROVER)
	   /* an Underscript in a MUNDEROVER. Transform the MUNDEROVER into
	      a MOVER */
	   newTypeNum = MathML_EL_MOVER;
	else if (parentType.ElTypeNum == MathML_EL_MUNDER)
	   /* an Underscript in a MUNDER. Remove the MUNDER and the
	      UnderOverBase */
	   newTypeNum = -1;
	break;

      case MathML_EL_Overscript:	/* an Overscript has been deleted */
	if (parentType.ElTypeNum == MathML_EL_MUNDEROVER)
	   /* an Overscript in a MUNDEROVER. Transform the MUNDEROVER into
	      a MUNDER */
	   newTypeNum = MathML_EL_MUNDER;
	else if (parentType.ElTypeNum == MathML_EL_MOVER)
	   /* an Overscript in a MOVER. Remove the MOVER and the
	      UnderOverBase */
	   newTypeNum = -1;
	break;

      default:
	break;
      }

   oldStructureChecking = TtaGetStructureChecking (event->document);
   TtaSetStructureChecking (0, event->document);
   if (newTypeNum > 0)
      /* transform the parent element */
      ChangeTypeOfElement (parent, event->document, newTypeNum);
   else if (newTypeNum < 0)
      /* put the content of the single sibling of the deleted element
	 instead of the parent element */
      {
      child = TtaGetFirstChild (parent);
      if (child != NULL)
	{
	grandChild = TtaGetFirstChild (child);
	if (grandChild != NULL)
	   {
	   TtaRemoveTree (grandChild, event->document);
	   TtaInsertSibling (grandChild, parent, TRUE, event->document);
	   TtaDeleteTree (parent, event->document);
	   placeholderEl = InsertPlaceholder (grandChild, FALSE, event->document, FALSE/******/);
	   placeholderEl = InsertPlaceholder (grandChild, TRUE, event->document, FALSE/******/);
	   }
	}
      }
   TtaSetStructureChecking ((ThotBool)oldStructureChecking, event->document);
}

/*----------------------------------------------------------------------
 FenceModified
 The opening or closing fence element in a MFENCED element has been modified
 by the user. Update the corresponding open or close attribute.
 -----------------------------------------------------------------------*/
#ifdef __STDC__
void FenceModified (NotifyOnValue *event)
#else /* __STDC__*/
void FenceModified(event)
     NotifyOnValue *event;
#endif /* __STDC__*/
{
  Element	mfencedEl;
  ElementType	elType;
  AttributeType	attrType;
  Attribute	attr;
  UCHAR_T         text[2];

  mfencedEl = TtaGetParent (event->element);
  elType = TtaGetElementType (event->element);
  if (elType.ElTypeNum == MathML_EL_OpeningFence)
     attrType.AttrTypeNum = MathML_ATTR_open;
  else
     attrType.AttrTypeNum = MathML_ATTR_close;
  attrType.AttrSSchema = elType.ElSSchema;
  attr = TtaGetAttribute (mfencedEl, attrType);
  if (attr == NULL)
     /* no attribute open on this MFENCED element. Create one */
     {
     attr =  TtaNewAttribute (attrType);
     TtaAttachAttribute (mfencedEl, attr, event->document);
     }
  text[0] = (UCHAR_T) event->value;
  text[1] = '\0';
  TtaSetAttributeText (attr, text, mfencedEl, event->document);
}

/*----------------------------------------------------------------------
   LinkAttrInMenu
   Called by Thot when building the Attribute menu.
   Prevent Thot from including attribute link in the menu
  ----------------------------------------------------------------------*/
#ifdef __STDC__
ThotBool            LinkAttrInMenu (NotifyAttribute * event)
#else  /* __STDC__ */
ThotBool            LinkAttrInMenu (event)
NotifyAttribute    *event;
 
#endif /* __STDC__ */
{
   return TRUE;
}

/*----------------------------------------------------------------------
 MathAttrFontsizeCreated
 An attribute fontsize has been created or updated by the user.
 -----------------------------------------------------------------------*/
#ifdef __STDC__
void MathAttrFontsizeCreated (NotifyAttribute *event)
#else /* __STDC__*/
void MathAttrFontsizeCreated(event)
     NotifyAttribute *event;
#endif /* __STDC__*/
{
#define buflen 200
  STRING           value = (STRING) TtaGetMemory (sizeof (CHAR_T) * buflen);
  int              length;

  value[0] = EOS;
  length = TtaGetTextAttributeLength (event->attribute);
  if (length >= buflen)
     length = buflen - 1;
  if (length > 0)
     TtaGiveTextAttributeValue (event->attribute, value, &length);
  /* associate a CSS property font-size with the element */
  SetFontsize (event->document, event->element, value);
}
 
 
/*----------------------------------------------------------------------
 MathAttrFontsizeDelete
 The user is deleting an attribute fontsize.
 -----------------------------------------------------------------------*/
#ifdef __STDC__
ThotBool MathAttrFontsizeDelete (NotifyAttribute *event)
#else /* __STDC__*/
ThotBool MathAttrFontsizeDelete(event)
     NotifyAttribute *event;
#endif /* __STDC__*/
{
  /* ask the CSS handler to remove the effect of the CSS property font-size */
  /* in the statement below, "10pt" is meaningless. It's here just to
     make the CSS parser happy */
  ParseHTMLSpecificStyle (event->element, TEXT("font-size: 10pt"), event->document,
			  TRUE);
  return FALSE; /* let Thot perform normal operation */
}


/*----------------------------------------------------------------------
 MathAttrFontfamilyCreated
 An attribute fontfamily has been created or modified by the user.
 -----------------------------------------------------------------------*/
#ifdef __STDC__
void MathAttrFontfamilyCreated (NotifyAttribute *event)
#else /* __STDC__*/
void MathAttrFontfamilyCreated (event)
     NotifyAttribute *event;
#endif /* __STDC__*/
{
  STRING           value = (STRING) TtaGetMemory (sizeof (CHAR_T) * buflen);
  int              length;

  value[0] = EOS;
  length = TtaGetTextAttributeLength (event->attribute);
  if (length >= buflen)
     length = buflen - 1;
  if (length > 0)
     TtaGiveTextAttributeValue (event->attribute, value, &length);  
  SetFontfamily (event->document, event->element, value);
}


/*----------------------------------------------------------------------
 MathAttrFontfamilyDelete
 The user is deleting an attribute fontfamily.
 -----------------------------------------------------------------------*/
#ifdef __STDC__
ThotBool MathAttrFontfamilyDelete (NotifyAttribute *event)
#else /* __STDC__*/
ThotBool MathAttrFontfamilyDelete (event)
     NotifyAttribute *event;
#endif /* __STDC__*/
{
  /* ask the CSS handler to remove the effect of property font-family */
  /* in the statement below, "serif" is meaningless. It's here just to
     make the CSS parser happy */
  ParseHTMLSpecificStyle (event->element, TEXT("font-family: serif"), event->document,
			  TRUE);
  return FALSE; /* let Thot perform normal operation */
}

/*----------------------------------------------------------------------
 MathAttrColorCreated
 An attribute color has been created or modified by the user.
 -----------------------------------------------------------------------*/
#ifdef __STDC__
void MathAttrColorCreated (NotifyAttribute *event)
#else /* __STDC__*/
void MathAttrColorCreated (event)
     NotifyAttribute *event;
#endif /* __STDC__*/
{
  STRING           value = (STRING) TtaGetMemory (sizeof (CHAR_T) * buflen);
  int              length;

  value[0] = EOS;
  length = TtaGetTextAttributeLength (event->attribute);
  if (length >= buflen)
     length = buflen - 1;
  if (length > 0)
     TtaGiveTextAttributeValue (event->attribute, value, &length);  
  HTMLSetForegroundColor (event->document, event->element, value);
}


/*----------------------------------------------------------------------
 MathAttrColorDelete
 The user is deleting an attribute color.
 -----------------------------------------------------------------------*/
#ifdef __STDC__
ThotBool MathAttrColorDelete (NotifyAttribute *event)
#else /* __STDC__*/
ThotBool MathAttrColorDelete(event)
     NotifyAttribute *event;
#endif /* __STDC__*/
{
  HTMLResetForegroundColor (event->document, event->element);
  return FALSE; /* let Thot perform normal operation */
}

/*----------------------------------------------------------------------
 MathAttrBackgroundCreated
 An attribute background has been created or modified by the user.
 -----------------------------------------------------------------------*/
#ifdef __STDC__
void MathAttrBackgroundCreated (NotifyAttribute *event)
#else /* __STDC__*/
void MathAttrBackgroundCreated (event)
     NotifyAttribute *event;
#endif /* __STDC__*/
{
  STRING           value = (STRING) TtaGetMemory (sizeof (CHAR_T) * buflen);
  int              length;

  value[0] = EOS;
  length = TtaGetTextAttributeLength (event->attribute);
  if (length >= buflen)
     length = buflen - 1;
  if (length > 0)
     TtaGiveTextAttributeValue (event->attribute, value, &length);  
  HTMLSetBackgroundColor (event->document, event->element, value);
}


/*----------------------------------------------------------------------
 MathAttrBackgroundDelete
 The user is deleting an attribute background.
 -----------------------------------------------------------------------*/
#ifdef __STDC__
ThotBool MathAttrBackgroundDelete (NotifyAttribute *event)
#else /* __STDC__*/
ThotBool MathAttrBackgroundDelete(event)
     NotifyAttribute *event;
#endif /* __STDC__*/
{
  HTMLResetBackgroundColor (event->document, event->element);
  return FALSE; /* let Thot perform normal operation */
}
 
/*----------------------------------------------------------------------
 AttrOpenCloseChanged
 Attribute open or close in a MFENCED element has been modified or deleted
 by the user. Update the corresponding fence element.
 -----------------------------------------------------------------------*/
#ifdef __STDC__
void AttrOpenCloseChanged (NotifyAttribute *event)
#else /* __STDC__*/
void AttrOpenCloseChanged (event)
     NotifyAttribute *event;
#endif /* __STDC__*/
{
  Element	fence, content;
  int		length;
  UCHAR_T         text[8];

  if (event->attributeType.AttrTypeNum == MathML_ATTR_open)
     fence = TtaGetFirstChild (event->element);
  else
     fence = TtaGetLastChild (event->element);
  if (fence != NULL)
    {
    content = TtaGetFirstChild (fence);
    if (content != NULL)
      {
      if (event->attribute == NULL)
	/* Attribute has been deleted */
	if (event->attributeType.AttrTypeNum == MathML_ATTR_open)
	   text[0] = '(';	/* default value for open */
	else
	   text[0] = ')';	/* default value for close */
      else
	/* attribute has been modified, get its new value */
	{
        length = 7;
        TtaGiveTextAttributeValue (event->attribute, text, &length);
	}
      /* set the content of the fence element */
      TtaSetGraphicsShape (content, text[0], event->document);
      }
    }
}

/*----------------------------------------------------------------------
 FencedSeparatorModified
 The content of a FenceSeparator element has been modified by the user
 in a MFENCED element.  Update the corresponding separators attribute.
 -----------------------------------------------------------------------*/
#ifdef __STDC__
void FencedSeparatorModified (NotifyOnTarget *event)
#else /* __STDC__*/
void FencedSeparatorModified(event)
     NotifyOnTarget *event;
#endif /* __STDC__*/
{
  Element	mfencedEl, fencedExpEl, child, content;
  Attribute	attr;
  ElementType	elType;
  AttributeType	attrType;
  int		i, len;
  Language	lang;
  UCHAR_T         text[32];

  fencedExpEl = TtaGetParent (event->element);
  if (fencedExpEl == NULL)
     return;
  mfencedEl = TtaGetParent (fencedExpEl);
  if (mfencedEl == NULL)
     return;
  i = 0;
  child = TtaGetFirstChild (fencedExpEl);
  while (child != NULL)
     {
     elType = TtaGetElementType (child);
     if (elType.ElTypeNum == MathML_EL_FencedSeparator)
	{
	content = TtaGetFirstChild (child);
        len = 31 - i;
        TtaGiveTextContent (content, &text[i], &len, &lang);
	i++;
	}
     TtaNextSibling (&child);
     }
  text[i] = '\0';
  /* if the last character is repeated, delete the repeated characters */
  if (i > 1)
     {
     i--;
     while (text[i-1] == text[i] && i > 0)
	i--;
     text[i+1] = '\0';
     }
  elType = TtaGetElementType (event->element);
  attrType.AttrSSchema = elType.ElSSchema;
  attrType.AttrTypeNum = MathML_ATTR_separators;
  attr = TtaGetAttribute (mfencedEl, attrType);
  if (attr == NULL)
     /* no attribute separators on this MFENCED element. Create one */
     {
     attr = TtaNewAttribute (attrType);
     TtaAttachAttribute (mfencedEl, attr, event->document);
     }
  /* set the value of the separators attribute */
  TtaSetAttributeText (attr, text, mfencedEl, event->document);
}


/*----------------------------------------------------------------------
 AttrSeparatorsChanged
 An attribute separators has been created, modified or deleted by the user
 for a MFENCED element. Update the corresponding FenceSeparator elements.
 -----------------------------------------------------------------------*/
#ifdef __STDC__
void AttrSeparatorsChanged (NotifyAttribute *event)
#else /* __STDC__*/
void AttrSeparatorsChanged (event)
     NotifyAttribute *event;
#endif /* __STDC__*/
{
  Element	child, fencedExpression;
  ElementType	elType;

  /* get the first child of the MFENCED element */
  child = TtaGetFirstChild (event->element);
  if (child == NULL)
     return;
  /* search the FencedExpression element among the children of MFENCED */
  fencedExpression = NULL;
  do
    {
    elType = TtaGetElementType (child);
    if (elType.ElTypeNum == MathML_EL_FencedExpression)
       fencedExpression = child;
    else
       TtaNextSibling (&child);
    }
  while (fencedExpression == NULL && child != NULL);
  if (fencedExpression != NULL)
    RegenerateFencedSeparators (fencedExpression, event->document, FALSE/****/);
}

#endif /* MATHML */
