/******************************************************************************
*									      *
*	File:     dgpsip.c						      *
*	Author:   Wolfgang Rupprecht <wolfgang@wsrcc.com>	              *
*	Created:  Sun Jan 24 20:36:39 PST 1999				      *
*	Contents: DGPS over ip -- get RTCM signals from a remote tcp source   *
*									      *
*	$Id: dgpsip.c,v 1.43 1999/08/25 19:20:19 wolfgang Exp $
******************************************************************************/

/*
 *  Copyright (c) 1999 Wolfgang Rupprecht
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  			    NO WARRANTY
 *
 *    BECAUSE THE PROGRAM IS LICENSED FREE OF CHARGE, THERE IS NO
 *  WARRANTY FOR THE PROGRAM, TO THE EXTENT PERMITTED BY APPLICABLE
 *  LAW.  EXCEPT WHEN OTHERWISE STATED IN WRITING THE COPYRIGHT
 *  HOLDERS AND/OR OTHER PARTIES PROVIDE THE PROGRAM "AS IS" WITHOUT
 *  WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, BUT
 *  NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE.  THE ENTIRE RISK AS TO THE
 *  QUALITY AND PERFORMANCE OF THE PROGRAM IS WITH YOU.  SHOULD THE
 *  PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL NECESSARY
 *  SERVICING, REPAIR OR CORRECTION.
 *
 *    IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN
 *  WRITING WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY
 *  MODIFY AND/OR REDISTRIBUTE THE PROGRAM AS PERMITTED ABOVE, BE
 *  LIABLE TO YOU FOR DAMAGES, INCLUDING ANY GENERAL, SPECIAL,
 *  INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OR
 *  INABILITY TO USE THE PROGRAM (INCLUDING BUT NOT LIMITED TO LOSS OF
 *  DATA OR DATA BEING RENDERED INACCURATE OR LOSSES SUSTAINED BY YOU
 *  OR THIRD PARTIES OR A FAILURE OF THE PROGRAM TO OPERATE WITH ANY
 *  OTHER PROGRAMS), EVEN IF SUCH HOLDER OR OTHER PARTY HAS BEEN
 *  ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "config.h"

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <math.h>
#include <sys/types.h>		/* for serial hacking */
#include <fcntl.h>
#include <sys/termios.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <signal.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netinet/tcp.h>
#include <math.h>
#include <errno.h>		/* for errno, EAGAIN */

#include "dgpsip.h"
#include "rtcm.h"

#define BUFSZ     (4 * 1024)
#define NMEABUFSZ (4 * 1024)

int             verbose = 0;

void            usage(void);
int             openserial(u_char * tty, int blocksz, int ttybaud);
void            fullwrite(int fd, u_char * buf, int cnt, int nonfatal);
int             tcpopen(u_char * host, u_char * servp, u_int af_family);
void            procstats(int nmeafd, int infd, FILE * posf, u_char *, u_int, u_int);
void            logsats(FILE * posf);
void            satpicture(u_char *);
void            writesvfile(u_char *);
void            readsvfile(u_char *);
char           *fgets_nc(u_char * buf, int size, FILE * ifp);

u_char         *nowarranty = "\n\
  BECAUSE THE PROGRAM IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY\n\
FOR THE PROGRAM, TO THE EXTENT PERMITTED BY APPLICABLE LAW.  EXCEPT WHEN\n\
OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES\n\
PROVIDE THE PROGRAM \" AS IS \" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS\n\
OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF\n\
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE ENTIRE RISK AS\n\
TO THE QUALITY AND PERFORMANCE OF THE PROGRAM IS WITH YOU.  SHOULD THE\n\
PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL NECESSARY SERVICING,\n\
REPAIR OR CORRECTION.\n\
\n\
  IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING\n\
WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR\n\
REDISTRIBUTE THE PROGRAM AS PERMITTED ABOVE, BE LIABLE TO YOU FOR DAMAGES,\n\
INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING\n\
OUT OF THE USE OR INABILITY TO USE THE PROGRAM (INCLUDING BUT NOT LIMITED\n\
TO LOSS OF DATA OR DATA BEING RENDERED INACCURATE OR LOSSES SUSTAINED BY\n\
YOU OR THIRD PARTIES OR A FAILURE OF THE PROGRAM TO OPERATE WITH ANY OTHER\n\
PROGRAMS), EVEN IF SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE\n\
POSSIBILITY OF SUCH DAMAGES.\n\n\
";

int             binpgm = 1;	/* default to binary pgm files 3x smaller,
				 * easier to load */

int
main(int argc,
     char **argv)
{
    u_char         *ttyout = "/dev/gps",
                   *ttynme = NULL,
                   *dgpshost = "dgps.wsrcc.com",
                   *dgpsport = "rtcm-sc104",
                   *svtrackfile = "svtrack.pgm",
                    buf[BUFSZ];
    int             cnt,
                    c,
                    i,
                    dflag = 0,
                    infd = -1,
                    outfd = -1,
                    nmeafd = -1,
                    nmebaud = 4800,
                    slp = 1,
                    wflag = 0,
                    pflag = 0,
                    rflag = 0,
                    tflag = 0,
                    needpsn = 0,
                    profile = 0,
                    af_family = AF_UNSPEC,
                    ttybaud = 4800;
    FILE           *posf = NULL;

    while ((c = getopt(argc, argv, "c:dh:n:N:o:p:rs:v:wP:BS:64")) != EOF) {
	switch (c) {
	case 'c':		/* capture GPS output to file */
	    if ((posf = fopen(optarg, "w")) == NULL) {
		perror("open logfile");
		exit(1);
	    }
	    break;

	case 'd':		/* decode */
	    dflag = 1;
	    ttyout = NULL;
	    break;

	case 'h':		/* dgps host */
	    dgpshost = optarg;
	    break;

	case 'p':		/* dgps port */
	    dgpsport = optarg;
	    break;

	case 'o':		/* gps serial ttyout */
	    ttyout = optarg;
	    break;

	case 'n':		/* gps serial NMEA input */
	    ttynme = optarg;
	    break;

	case 'N':		/* serial NMEA input speed */
	    nmebaud = atoi(optarg);
	    break;

	case 'r':		/* relay flag */
	    rflag = 1;
	    break;

	case 's':		/* serial ttyout speed */
	    ttybaud = atoi(optarg);
	    break;

	case 'v':		/* verbose */
	    verbose = atoi(optarg);
	    break;

	case 'w':		/* turn off nowarranty printf at startup */
	    wflag = 1;
	    break;

	case 'B':		/* turn off binary-pgm flag */
	    binpgm = 0;
	    break;

	case 't':		/* do SVtrack processing */
	    tflag = 1;
	    break;

	case 'S':		/* SVtrack file name */
	    svtrackfile = optarg;
	    break;

	case 'P':		/* profile cntr - developer only flag */
	    pflag = 1;
	    profile = atoi(optarg);
	    break;

	case '6':		/* ipv6 */
	    af_family = AF_INET6;
	    break;

	case '4':		/* ipv4 */
	    af_family = AF_INET;
	    break;

	case '?':
	default:
	    usage();
	    break;
	}
    }

    argc -= optind;
    argv += optind;

    if (argc > 0) {
	fprintf(stderr, "Extra args on command line.\n");
	for (; argc > 0; argc--) {
	    fprintf(stderr, " %s", *argv++);
	}
	fprintf(stderr, "\n");
	usage();		/* never returns */
    }
    if (verbose >= 1)
	fprintf(stderr, "%s %s\n", PACKAGE, VERSION);

    if (wflag == 0) {
	fprintf(stderr, "%s", nowarranty);
    }
    if (ttyout) {
	outfd = openserial(ttyout, 1, ttybaud);
	if (outfd < 0) {
	    perror("open tty");
	    exit(1);
	}
	if (!ttynme || (strcmp(ttyout, ttynme) == 0)) {
	    nmeafd = outfd;
	} else {
	    nmeafd = openserial(ttynme, 1, nmebaud);
	    if (nmeafd < 0) {
		perror("open NMEA tty");
		exit(1);
	    }
	}
	read(nmeafd, buf, BUFSZ);	/* flush */
    }
    if (tflag) {
	readsvfile(svtrackfile);/* if we have an old svfile read it now. */
	writesvfile(svtrackfile);	/* and do the first write right away */
    }
    while (infd < 0) {

	initialise();		/* init rtcm decoder */
	needpsn = 1;

	infd = tcpopen(dgpshost, dgpsport, af_family);
	if (infd < 0) {
	    fprintf(stderr,
	       "Can't open network input stream to %s:%s sleeping %d sec\n",
		    dgpshost, dgpsport, slp);
	    sleep(slp);
	    slp *= 2;
	    if (slp > 60)
		slp = 60;
	} else {
	    {
		u_char          hn[256];

		gethostname(hn, sizeof(hn));
		snprintf(buf, sizeof(buf), "HELO %s %s %s%s\r\n", hn, PACKAGE,
			 VERSION,
			 (outfd < 0) ? "\r\nB" : (rflag ? "\r\nR" : ""));
		fullwrite(infd, buf, strlen(buf), 0);
	    }
	    slp = 1;
	    while ((cnt = read(infd, buf, sizeof(buf))) > 0) {
		if (dflag) {
		    for (i = 0; i < cnt; i++) {
			new_byte(buf[i]);
		    }
		}
		if (outfd >= 0) {
		    fullwrite(outfd, buf, cnt, 1);
		    procstats(nmeafd, infd, posf, svtrackfile, tflag, needpsn);
		    needpsn = 0;
		}
		if (verbose >= 10) {
		    fprintf(stderr, "Read %d bytes\n", cnt);
		} else if (verbose >= 3) {
		    fprintf(stderr, ".");
		}
		if (pflag) {
		    profile -= cnt;
		    if (profile < 0) {
			exit(0);
		    }
		}
	    }
	    close(infd);
	    infd = -1;
	    fprintf(stderr,
		    "Network input stream to %s:%s Closed.  Reopening.\n",
		    dgpshost, dgpsport);
	}
    }
    exit(0);
}

#define MAXELV    91
#define MAXAZI   361
#define MAXINTEN  55
u_char          sigpict[MAXELV * MAXAZI] = {
    0
};

 /* do signal picture stuff */

void
satpicture(u_char * svtrackfile)
{
    int             i;
    static int      update = 0;

    for (i = 0; i < NUMSV; i++) {
	if (n.sv[i].time) {
	    if (n.time < (n.sv[i].time + 10)) {
		if ((n.sv[i].elev < MAXELV) &&
		    (n.sv[i].azi < MAXAZI)) {
		    sigpict[n.sv[i].azi +
			    (((MAXELV - 1) - n.sv[i].elev) * MAXAZI)]
			= n.sv[i].sig;
		}
	    } else {
		/*
	         * clear the old data before the time wraps around and
	         * confuses us
	         */
		n.sv[i].time = 0;
		n.sv[i].azi = 0;
		n.sv[i].elev = 0;
		n.sv[i].sig = 0;
	    }
	}
    }

    if ((update++ % 16 /* 128 */ ) == 0) {
	writesvfile(svtrackfile);
    }
}

void
writesvfile(u_char * svtrackfile)
{
    FILE           *svfile;
    int             i,
                    cnt;

    svfile = fopen(svtrackfile, "w");
    if (!svfile) {
	fprintf(stderr, "Can't open svtrack for writing.  No old tracks saved.\n");
	return;
    }
    /* 0 is black */
    fprintf(svfile,
	    "%s\n# Satellite raster\n%d %d\n%d\n",
	    binpgm ? "P5" : "P2",
	    MAXAZI, MAXELV, MAXINTEN);
    if (binpgm) {
	cnt = fwrite(sigpict, 1, sizeof(sigpict), svfile);
	if (cnt != sizeof(sigpict)) {
	    fprintf(stderr, "short write (%d/%d)\n",
		    cnt, sizeof(sigpict));
	    fclose(svfile);
	    return;
	}
    } else {
	for (i = 0; i <= sizeof(sigpict); i++) {
	    fprintf(svfile, "%d\n", sigpict[i]);
	}
    }
    fclose(svfile);
}

char           *
fgets_nc(u_char * buf,
	 int size,
	 FILE * ifp)
{
    char           *ret;

    ret = fgets(buf, size, ifp);
    while (ret && (buf[0] == '#')) {
	ret = fgets(buf, size, ifp);
    }
    return ret;
}

void
readsvfile(u_char * svtrackfile)
{
    FILE           *svfile;
    int             maxint,
                    cnt,
                    rows,
                    cols,
                    i,
                    bin = 1;
    u_char          buf[BUFSZ];

    svfile = fopen(svtrackfile, "r");
    if (!svfile) {
	fprintf(stderr, "Can't open svtrack for reading.  No old tracks.\n");
	return;
    }
    fgets_nc(buf, sizeof(buf), svfile);
    if (strcmp("P2\n", buf)) {
	if (strcmp("P5\n", buf)) {
	    fprintf(stderr,
		    "input files must be PGM, P2 or P5 type (was: %s)\n",
		    buf);
	    fclose(svfile);
	    return;
	} else {
	    bin = 1;
	}
    } else {
	bin = 0;
    }

    fgets_nc(buf, sizeof(buf), svfile);
    cnt = sscanf(buf, "%d %d", &cols, &rows);
    if (cnt != 2) {
	fprintf(stderr, "couldn't parse rows/cols spec\n");
	fclose(svfile);
	return;
    }
    if ((rows != MAXELV) && (cols != MAXAZI)) {
	fprintf(stderr, "pgm must be a %dx%d array (was: %d %d)\n",
		MAXELV, MAXAZI, rows, cols);
	fclose(svfile);
	return;
    }
    fgets_nc(buf, sizeof(buf), svfile);
    cnt = sscanf(buf, "%d", &maxint);
    if (cnt != 1) {
	fprintf(stderr, "couldn't parse max intensity spec\n");
	fclose(svfile);
	return;
    }
    fprintf(stderr, "Reading old tracks (%s).\n", bin ? "BIN" : "ASCII");
    if (bin) {
	cnt = fread(sigpict, 1, sizeof(sigpict), svfile);
	if (cnt != sizeof(sigpict)) {
	    fprintf(stderr, "short read (%d/%d)\n",
		    cnt, sizeof(sigpict));
	    fclose(svfile);
	    return;
	}
    } else {
	int             x;

	for (i = 0; i <= sizeof(sigpict); i++) {
	    fscanf(svfile, "%d ", &x);
	    sigpict[i] = x;
	}
    }
    fclose(svfile);
}

void
logsats(FILE * posf)
{
    int             i;

    if (!posf) {
	return;
    }
    if (verbose > 4)
	fprintf(stderr, "%ld -----\n", n.time);

    for (i = 0; i < NUMSV; i++) {
	if (n.time < (n.sv[i].time + 10)) {
	    fprintf(posf, "%ld %d %d %d %d\n",
		    n.sv[i].time,
		    i, n.sv[i].azi, n.sv[i].elev, n.sv[i].sig);
	    if (verbose > 4)
		fprintf(stderr, "%ld %d %d %d %d\n",
			n.sv[i].time,
			i, n.sv[i].azi, n.sv[i].elev, n.sv[i].sig);
	}
    }
}

void
procstats(int nmeafd,
	  int infd,
	  FILE * posf,
	  u_char * svtrackfile,
	  u_int tflag,
	  u_int needpsn)
{
    int             ncnt,
                    rd;
    double          x,
                    y,
                    h,
                    sdevlat,
                    sdevlon,
                    sdevheight;
    struct llpoint  point;
    static char     nmeabuf[NMEABUFSZ];
    static int      nmeaind = 0,
                    pts = -16,
                    cutoff = 1;
    static struct llpoint
                    sum = {0.0, 0.0, 0.0},
                    sumdif = {0.0, 0.0, 0.0};

    ncnt = read(nmeafd, nmeabuf + nmeaind, NMEABUFSZ - nmeaind);
    if (ncnt > 0) {
#if 0
	if (posf != NULL) {
	    fwrite(nmeabuf + nmeaind, ncnt, (size_t) 1, posf);
	}
#endif
	nmeaind += ncnt;
	while ((rd = nmeaRead(nmeabuf, nmeaind, &point)) > 0) {
	    if (point.fix > 0) {

		if (tflag) {
		    satpicture(svtrackfile);
		}
		logsats(posf);

		/* prime the pumps */
		if (pts <= 0) {
		    if (verbose >= 2) {
			printf("R %0.8f %0.8f %0.2f -- %d - - - %0.1f %0.1f\n",
			       point.lat, point.lon, point.height, pts,
			       point.hdop, point.numsats);
			fflush(stdout);
		    }
		    pts++;
		    continue;
		}
		sum.lat += point.lat;
		sum.lon += point.lon;
		sum.height += point.height;
		sum.hdop += point.hdop;
		sum.numsats += point.numsats;

		x = point.lat - (sum.lat / pts);
		sumdif.lat += x * x;
		y = point.lon - (sum.lon / pts);
		sumdif.lon += y * y;
		h = point.height - (sum.height / pts);
		sumdif.height += h * h;

		if ((pts >= cutoff) || needpsn) {
		    u_char          rptbuf[BUFSZ];
#define CUTOFF_LINEAR_PT 1024
		    if (needpsn) {
			needpsn = 0;
		    } else if (cutoff < CUTOFF_LINEAR_PT) {
			cutoff *= 2;
		    } else {
			cutoff += CUTOFF_LINEAR_PT;
		    }


		    sdevlat = sqrt(sumdif.lat / (pts - 1));
		    sdevlon = sqrt(sumdif.lon / (pts - 1));
		    sdevheight = sqrt(sumdif.height / (pts - 1));
		    snprintf(rptbuf, BUFSZ,
			     "R %0.8f %0.8f %0.2f -- %d %0.2e %0.2e %0.2e %0.1f %0.1f\n",
			     sum.lat / pts, sum.lon / pts,
			     sum.height / pts, pts,
			     sdevlat, sdevlon, sdevheight,
			     sum.hdop / pts,
			     sum.numsats / pts);

		    if (verbose >= 2) {
			printf("%s", rptbuf);
		    }
#if 1				/* only send 2 digit on the wire */
		    snprintf(rptbuf, BUFSZ,
			     "R %0.2f %0.2f %0.2f -- %d %0.2e %0.2e %0.2e %0.1f %0.1f\n",
			     sum.lat / pts, sum.lon / pts,
			     sum.height / pts, pts,
			     sdevlat, sdevlon, sdevheight,
			     sum.hdop / pts,
			     (double) (sum.numsats) / pts);
#endif
		    if (cutoff >= 16) {
			fullwrite(infd, rptbuf,
				  strlen(rptbuf), 1);
		    }
		}
		pts++;
	    }
	    nmeaind -= rd;
	    memmove(nmeabuf, nmeabuf + rd, nmeaind);
	}
    }
}

int
openserial(u_char * tty, int blocksz, int ttybaud)
{
    int             fd;
    struct termios  termios;

    fd = open(tty, O_RDWR | O_NONBLOCK);
    if (fd < 0) {
	perror("open");
	return (-1);
    }
    if (tcgetattr(fd, &termios) < 0) {

	perror("tcgetattr");
	return (-1);
    }
    termios.c_iflag = 0;
    termios.c_oflag = 0;	/* (ONLRET) */
    termios.c_cflag = CS8 | CLOCAL | CREAD;
    termios.c_lflag = 0;
    {
	int             cnt;

	for (cnt = 0; cnt < NCCS; cnt++)
	    termios.c_cc[cnt] = -1;
    }
    termios.c_cc[VMIN] = blocksz;
    termios.c_cc[VTIME] = 2;

#if (B4800 != 4800)
    /*
     * Only paleolithic systems need this.
     */

    switch (ttybaud) {
    case 300:
	ttybaud = B300;
	break;
    case 1200:
	ttybaud = B1200;
	break;
    case 2400:
	ttybaud = B2400;
	break;
    case 4800:
	ttybaud = B4800;
	break;
    case 9600:
	ttybaud = B9600;
	break;
    case 19200:
	ttybaud = B19200;
	break;
    default:
	ttybaud = B4800;
	break;
    }
#endif

    if (cfsetispeed(&termios, ttybaud) != 0) {
	perror("cfsetispeed");
	return (-1);
    }
    if (cfsetospeed(&termios, ttybaud) != 0) {
	perror("cfsetospeed");
	return (-1);
    }
    if (tcsetattr(fd, TCSANOW, &termios) < 0) {
	perror("tcsetattr");
	return (-1);
    }
#if WANT_BLOCKING_READ
    if (fcntl(fd, F_SETFL, 0) == -1) {
	perror("fcntl: set nonblock");
    }
#endif
    return (fd);
}

void
fullwrite(int fd,
	  u_char * buf,
	  int cnt,
	  int nonfatal)
{
    int             ret;

    while (cnt > 0) {
	ret = write(fd, buf, cnt);
	if (ret > 0) {
	    cnt -= ret;
	    buf += ret;
	} else {
	    perror("error writing");
	    /*
	     * Don't overreact to a write on a temporarily full tty line.
	     * This happens after a net disruption when a burst of data comes
	     * in for the tty.  We should really just drop it before the write.
	     */
	    if (nonfatal && (errno == EAGAIN))
		return;
	    exit(1);
	}
    }
}

void
usage(void)
{
    fprintf(stderr, "dgpsip receives DGPS over IP and sends to serial port\n");
    fprintf(stderr, "Usage: dgpsip [OPTIONS]\n");
    fprintf(stderr, "  Options are:\n");
    fprintf(stderr, "    -c capture_file  saves GPS NMEA data to file\n");
    fprintf(stderr, "    -d    decode RTCM and print to stdout\n");
    fprintf(stderr, "       this turns off tty output, use -o to turn it on again\n");
    fprintf(stderr, "    -h DGPS_host  sets name of server to connect to\n");
    fprintf(stderr, "       default is dgps.wsrcc.com\n");
    fprintf(stderr, "    -p DGPS_port  sets IP port #/name to connect to\n");
    fprintf(stderr, "       default is IANA assigned port rtcm-sc104 (2101)\n");
    fprintf(stderr, "    -o serial_port  sets name of serial output device\n");
    fprintf(stderr, "       default is /dev/gps - a link to /dev/tty??\n");
    fprintf(stderr, "    -s serial_speed  sets serial output baud rate\n");
    fprintf(stderr, "       default is 4800 baud\n");
    fprintf(stderr, "    -v verblevel  turns on extra output\n");
    fprintf(stderr, "       see README for details\n");
    fprintf(stderr, "    -w turns off startup warranty message\n");
    exit(2);
}

/* Are we missing ipv6 or vital libs? */
#if !defined(HAVE_GETADDRINFO) || !defined(HAVE_NETINET6_IN6_H)

int
tcpopen(u_char * host,
	u_char * servp,
	u_int af_family)
{
    int             sockt,
                    port;
    struct hostent *hp;
    struct servent *sp;
    struct sockaddr_in rem_sockaddr_in;

    if (af_family == AF_INET6) {
	fprintf(stderr, "Fatal: IPv6 not available.\n");
	exit(1);
    }
    memset((char *) &rem_sockaddr_in, 0, sizeof(rem_sockaddr_in));

    if ((rem_sockaddr_in.sin_addr.s_addr = inet_addr(host)) != INADDR_NONE) {
	hp = gethostbyaddr((const char *) &rem_sockaddr_in.sin_addr,
			   sizeof(rem_sockaddr_in.sin_addr), AF_INET);
	if (hp == NULL) {
	    /*
             * herror("gethostbyaddr"); not an error for numeric
             * addresses
             */
	} else {
	    bcopy(hp->h_addr, (char *) &rem_sockaddr_in.sin_addr, hp->h_length);
	}
    } else {
	hp = gethostbyname(host);
	if (hp == NULL) {
	    herror("gethostbyname");
	    return (-1);
	}
	bcopy(hp->h_addr, (char *) &rem_sockaddr_in.sin_addr, hp->h_length);
    }
    sockt = socket(AF_INET, SOCK_STREAM, 0);
    if (sockt < 0) {
	perror("socket/tcp");
	return (-1);
    }
    port = htons(atoi(servp));
    if (!port) {
	sp = getservbyname(servp, "tcp");
	if (sp == 0) {
	    if (strcmp(servp, "rtcm-sc104") == 0) {
		sp = getservent();
		if (sp) {
		    sp->s_port = htons(2101);
		}
	    }
	}
	if (sp == 0) {
	    fprintf(stderr, "unknown service %s/%s\n", servp,
		    "tcp");
	    close(sockt);
	    return (-1);
	}
	port = sp->s_port;
    }
#ifndef SOCKADDR_IN_HAS_NO_SINLEN
    rem_sockaddr_in.sin_len = sizeof(rem_sockaddr_in);
#endif
    rem_sockaddr_in.sin_family = AF_INET;
    rem_sockaddr_in.sin_port = port;

    if (connect(sockt, (const struct sockaddr *) & rem_sockaddr_in,
		sizeof(rem_sockaddr_in)) == -1) {
	if (verbose >= 5)
	    perror("connect");
	close(sockt);
	return (-1);
    }
    return (sockt);
}

#else				/* INET6 */

static const char *
sockaddr_ntop(struct sockaddr * sa)
{
    void           *addr;
    static char     addrbuf[INET6_ADDRSTRLEN];

    switch (sa->sa_family) {
    case AF_INET:
	addr = &((struct sockaddr_in *) sa)->sin_addr;
	break;
    case AF_INET6:
	addr = &((struct sockaddr_in6 *) sa)->sin6_addr;
	break;
    default:
	return NULL;
    }
    return (char *) inet_ntop(sa->sa_family, addr, addrbuf, sizeof(addrbuf));
}


int
tcpopen(u_char * hostname,
	u_char * portp,
	u_int af_family)
{
    int             sockt = -1,
                    error;
    struct addrinfo hints,
                   *res,
                   *lres;

    memset(&hints, 0, sizeof(hints));
    hints.ai_flags = AI_CANONNAME;
    hints.ai_family = af_family;
    hints.ai_socktype = SOCK_STREAM;
    hints.ai_protocol = 0;
    error = getaddrinfo(hostname, portp, &hints, &res);
    if (error) {
	fprintf(stderr, "%s: %s\n", hostname, gai_strerror(error));
	return (-1);
    }
    for (lres = res; lres; lres = lres->ai_next) {
	sockt = socket(lres->ai_family, lres->ai_socktype, lres->ai_protocol);
	if (sockt < 0) {
	    continue;
	}
	if (connect(sockt, lres->ai_addr, lres->ai_addrlen) == -1) {
	    close(sockt);
	    sockt = -1;
	    continue;
	}
	break;
    }

    if ((sockt >= 0) && (verbose >= 1))
	fprintf(stderr, "Opening %s connection to %s %s\n",
		(lres->ai_family == AF_INET6) ? "IPv6" : "IPv4",
		sockaddr_ntop(lres->ai_addr),
		lres->ai_canonname ? lres->ai_canonname : "");

    freeaddrinfo(res);
    return (sockt);
}
#endif


/* end */
