/* valalambdaexpression.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valalambdaexpression.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/readonlylist.h>
#include <vala/valablock.h>
#include <vala/valasourcereference.h>
#include <vala/valamethod.h>
#include <vala/valacodevisitor.h>




struct _ValaLambdaExpressionPrivate {
	ValaExpression* _expression_body;
	ValaBlock* _statement_body;
	ValaMethod* _method;
	GeeList* parameters;
};

#define VALA_LAMBDA_EXPRESSION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_LAMBDA_EXPRESSION, ValaLambdaExpressionPrivate))
enum  {
	VALA_LAMBDA_EXPRESSION_DUMMY_PROPERTY
};
static void vala_lambda_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_lambda_expression_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_lambda_expression_real_is_pure (ValaExpression* base);
static gpointer vala_lambda_expression_parent_class = NULL;
static void vala_lambda_expression_finalize (ValaCodeNode* obj);



/**
 * Creates a new lambda expression.
 *
 * @param expression_body  expression body
 * @param source_reference reference to source code
 * @return                 newly created lambda expression
 */
ValaLambdaExpression* vala_lambda_expression_construct (GType object_type, ValaExpression* expression_body, ValaSourceReference* source_reference) {
	ValaLambdaExpression* self;
	g_return_val_if_fail (expression_body != NULL, NULL);
	g_return_val_if_fail (source_reference != NULL, NULL);
	self = ((ValaLambdaExpression*) (g_type_create_instance (object_type)));
	vala_code_node_set_source_reference (((ValaCodeNode*) (self)), source_reference);
	vala_lambda_expression_set_expression_body (self, expression_body);
	return self;
}


ValaLambdaExpression* vala_lambda_expression_new (ValaExpression* expression_body, ValaSourceReference* source_reference) {
	return vala_lambda_expression_construct (VALA_TYPE_LAMBDA_EXPRESSION, expression_body, source_reference);
}


/**
 * Creates a new lambda expression with statement body.
 *
 * @param statement_body   statement body
 * @param source_reference reference to source code
 * @return                 newly created lambda expression
 */
ValaLambdaExpression* vala_lambda_expression_construct_with_statement_body (GType object_type, ValaBlock* statement_body, ValaSourceReference* source_reference) {
	ValaLambdaExpression* self;
	g_return_val_if_fail (statement_body != NULL, NULL);
	g_return_val_if_fail (source_reference != NULL, NULL);
	self = ((ValaLambdaExpression*) (g_type_create_instance (object_type)));
	vala_lambda_expression_set_statement_body (self, statement_body);
	vala_code_node_set_source_reference (((ValaCodeNode*) (self)), source_reference);
	return self;
}


ValaLambdaExpression* vala_lambda_expression_new_with_statement_body (ValaBlock* statement_body, ValaSourceReference* source_reference) {
	return vala_lambda_expression_construct_with_statement_body (VALA_TYPE_LAMBDA_EXPRESSION, statement_body, source_reference);
}


/**
 * Appends implicitly typed parameter.
 *
 * @param param parameter name
 */
void vala_lambda_expression_add_parameter (ValaLambdaExpression* self, const char* param) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (param != NULL);
	gee_collection_add (((GeeCollection*) (self->priv->parameters)), param);
}


/**
 * Returns copy of parameter list.
 *
 * @return parameter list
 */
GeeList* vala_lambda_expression_get_parameters (ValaLambdaExpression* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return ((GeeList*) (gee_read_only_list_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, self->priv->parameters)));
}


static void vala_lambda_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaLambdaExpression * self;
	self = ((ValaLambdaExpression*) (base));
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_lambda_expression (visitor, self);
	vala_code_visitor_visit_expression (visitor, ((ValaExpression*) (self)));
}


static void vala_lambda_expression_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaLambdaExpression * self;
	self = ((ValaLambdaExpression*) (base));
	g_return_if_fail (visitor != NULL);
	if (self->priv->_method == NULL) {
		if (self->priv->_expression_body != NULL) {
			vala_code_node_accept (((ValaCodeNode*) (self->priv->_expression_body)), visitor);
			vala_code_visitor_visit_end_full_expression (visitor, self->priv->_expression_body);
		} else {
			if (self->priv->_statement_body != NULL) {
				vala_code_node_accept (((ValaCodeNode*) (self->priv->_statement_body)), visitor);
			}
		}
	} else {
		vala_code_node_accept (((ValaCodeNode*) (self->priv->_method)), visitor);
	}
}


static gboolean vala_lambda_expression_real_is_pure (ValaExpression* base) {
	ValaLambdaExpression * self;
	self = ((ValaLambdaExpression*) (base));
	return FALSE;
}


ValaExpression* vala_lambda_expression_get_expression_body (ValaLambdaExpression* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_expression_body;
}


void vala_lambda_expression_set_expression_body (ValaLambdaExpression* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_expression_body = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_expression_body == NULL ? NULL : (self->priv->_expression_body = (vala_code_node_unref (self->priv->_expression_body), NULL))), _tmp2);
}


ValaBlock* vala_lambda_expression_get_statement_body (ValaLambdaExpression* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_statement_body;
}


void vala_lambda_expression_set_statement_body (ValaLambdaExpression* self, ValaBlock* value) {
	ValaBlock* _tmp2;
	ValaBlock* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_statement_body = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_statement_body == NULL ? NULL : (self->priv->_statement_body = (vala_code_node_unref (self->priv->_statement_body), NULL))), _tmp2);
}


ValaMethod* vala_lambda_expression_get_method (ValaLambdaExpression* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_method;
}


void vala_lambda_expression_set_method (ValaLambdaExpression* self, ValaMethod* value) {
	ValaMethod* _tmp2;
	ValaMethod* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_method = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_method == NULL ? NULL : (self->priv->_method = (vala_code_node_unref (self->priv->_method), NULL))), _tmp2);
}


static void vala_lambda_expression_class_init (ValaLambdaExpressionClass * klass) {
	vala_lambda_expression_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_lambda_expression_finalize;
	g_type_class_add_private (klass, sizeof (ValaLambdaExpressionPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_lambda_expression_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_lambda_expression_real_accept_children;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_lambda_expression_real_is_pure;
}


static void vala_lambda_expression_instance_init (ValaLambdaExpression * self) {
	self->priv = VALA_LAMBDA_EXPRESSION_GET_PRIVATE (self);
	self->priv->parameters = ((GeeList*) (gee_array_list_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, g_direct_equal)));
}


static void vala_lambda_expression_finalize (ValaCodeNode* obj) {
	ValaLambdaExpression * self;
	self = VALA_LAMBDA_EXPRESSION (obj);
	(self->priv->_expression_body == NULL ? NULL : (self->priv->_expression_body = (vala_code_node_unref (self->priv->_expression_body), NULL)));
	(self->priv->_statement_body == NULL ? NULL : (self->priv->_statement_body = (vala_code_node_unref (self->priv->_statement_body), NULL)));
	(self->priv->_method == NULL ? NULL : (self->priv->_method = (vala_code_node_unref (self->priv->_method), NULL)));
	(self->priv->parameters == NULL ? NULL : (self->priv->parameters = (gee_collection_object_unref (self->priv->parameters), NULL)));
	VALA_CODE_NODE_CLASS (vala_lambda_expression_parent_class)->finalize (obj);
}


GType vala_lambda_expression_get_type (void) {
	static GType vala_lambda_expression_type_id = 0;
	if (vala_lambda_expression_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaLambdaExpressionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_lambda_expression_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaLambdaExpression), 0, (GInstanceInitFunc) vala_lambda_expression_instance_init, NULL };
		vala_lambda_expression_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaLambdaExpression", &g_define_type_info, 0);
	}
	return vala_lambda_expression_type_id;
}




