<?php
// $Horde: horde/lib/Perms.php,v 1.14 2001/11/07 09:21:32 chuck Exp $

// Return codes
/** @constant PERMS_OK Operation succeeded. */
define('PERMS_OK', 0);

/** @constant PERMS_ERROR Operation failed. */
define('PERMS_ERROR', -1);

/** @constant PERMS_ERROR_PARAMS Bad or missing parameters ($params). */
define('PERMS_ERROR_PARAMS', -2);

/** @constant PERMS_ERROR_CONNECT Connection failure. */
define('PERMS_ERROR_CONNECT', -3);

/** @constant PERMS_ERROR_AUTH Authentication failure. */
define('PERMS_ERROR_AUTH', -4);

/** @constant PERMS_ERROR_EMPTY Empty retrieval result. */
define('PERMS_ERROR_EMPTY', -5);


// Permissions constants (bitmasks)
/** @constant _PERMS_NONE No rights. */
define('_PERMS_NONE', 0);

/** @constant _PERMS_SHOW Existence of object is known - object is shown to user. */
define('_PERMS_SHOW', 1);

/** @constant _PERMS_READ Contents of the object can be read. */
define('_PERMS_READ', 2);

/** @constant _PERMS_EDIT Contents of the object can be edited. */
define('_PERMS_EDIT', 4);

/** @constant _PERMS_DELETE The object can be deleted. */
define('_PERMS_DELETE', 8);


/**
 * The Perms:: class provides a common abstracted interface into the
 * various backends for the Horde permissions system. It also includes
 * all of the functions for retrieving, storing, and checking
 * permissions.
 *
 * Permissions are based on a generic, granular model. The Horde
 * permission system doesn't need to be associated with, or control,
 * the objects being restricted - as long as you control all access to
 * them through Horde interfaces, or take other steps.
 *
 * We just expect some kind of unique identifier and a userid, and
 * check that against the storage being used. So if we used the ids
 * assigned to MCAL calendar events, and made sure they were unique
 * across servers, we could store permissions for those events in a
 * completely seperate SQL database.
 *
 * @author  Chuck Hagenbuch <chuck@horde.org>
 * @version $Revision: 1.14 $
 * @since   Horde 1.3
 * @package horde.perms
 */
class Perms {

    /** The character to seperate levels of permissions hierarchy with. */
    var $path_seperator = '|';


    /**
     * Attempts to return a concrete Perms instance based on $driver.
     * 
     * @param string $driver The type of concrete Perms subclass to return.
     *                This is based on the storage driver ($driver).  The
     *                code is dynamically included.
     * @param optional array $params A hash containing any additional
     *                configuration or connection parameters a subclass
     *                might need.
     *
     * @return object Perms The newly created concrete Perms instance, or false
     *         on an error.
     */
    function &factory($driver, $params = array())
    {
        $driver = strtolower(basename($driver));
        if (empty($driver) || (strcmp($driver, 'none') == 0)) {
            return new Perms;
        }

        @include_once dirname(__FILE__) . '/Perms/' . $driver . '.php';
        $class = 'Perms_' . $driver;
        if (class_exists($class)) {
            return new $class($params);
        } else {
            return false;
        }
    }

    /**
     * Attempts to return a reference to a concrete Perms instance based on
     * $driver. It will only create a new instance if no Perms instance
     * with the same parameters currently exists.
     *
     * This should be used if multiple permissions sources (and, thus,
     * multiple Perms instances) are required.
     *
     * This method must be invoked as: $var = &Perms::singleton()
     *
     * @param string $driver The type of concrete Perms subclass to return.
     *                This is based on the storage driver ($driver). The
     *                code is dynamically included.
     * @param optional array $params A hash containing any additional
     *                configuration or connection parameters a subclass
     *                might need.
     *
     * @return object Perms The concrete Perms reference, or false on an error.
     */
    function &singleton($driver, $user = '', $params = array())
    {
        static $instances;

        if (!isset($instances)) $instances = array();

        $signature = md5(strtolower($driver) . '][' . $user . '][' . implode('][', $params));
        if (!isset($instances[$signature])) {
            $instances[$signature] = &Perms::factory($driver, $user, $params);
        }

        return $instances[$signature];
    }

    /**
     * Find out what rights the given user has to the specified object.
     * 
     * @param string $objectpath The full path to the piece of
     *                           content/whatever to check the permissions of.
     * @param string $user The user to check for.
     * @param int    $default What to return if nothing is found in the db.
     * 
     * @return int Any permissions the user has, or $default if there
     *             are none.
     */
    function check($objectpath, $user, $default = _PERMS_NONE)
    {
        return $default;
    }

    /**
     * Find out if the user has the specified rights to the given object.
     * 
     * @param string $objectpath The full path to the piece of
     *                           content/whatever to check the permissions of.
     * @param string $user The user to check for.
     * @param int    $perm The permission level that needs to be checked for.
     * 
     * @return boolean True if the user has the specified permissions, and
     *                 false otherwise.
     */
    function hasPerm($objectpath, $user, $perm)
    {
        return ($this->check($objectpath, $user) | $perm);
    }

    /**
     * Add the permission(s) specified to the given user and object.
     *
     * @param string $objectpath The full path to the piece of
     *                           content/whatever to add permissions to.
     * @param string $user The user to add permissions for.
     * @param int    $perm The permission to be added.
     */
    function addPerms($objectpath, $user, $perm)
    {
        return PERMS_OK;
    }

    /**
     * Remove the permission(s) specified from the given user and object.
     *
     * @param string $objectpath The full path to the piece of
     *                           content/whatever to remove permissions from.
     * @param string $user The user to remove permissions from.
     * @param int    $perm The permission to be removed.
     */
    function removePerms($objectpath, $user, $perm)
    {
        return PERMS_OK;
    }

    /**
     * Set the object to be showable to the given user.
     *
     * @param string $objectpath The full path to the piece of
     *                           content/whatever to make showable.
     * @param string $user The user to assign the permissions to.
     */
    function setShowable($objectpath, $user)
    {
        $this->addPerms($objectpath, $user, _PERMS_SHOW);
    }

    /**
     * Set the object to not be showable to the given user.
     *
     * @param string $objectpath The full path to the piece of
     *                           content/whatever to make unshowable.
     * @param string $user The user to assign the permissions to.
     */
    function setUnshowable($objectpath, $user)
    {
        $this->removePerms($objectpath, $user, _PERMS_SHOW);
    }

    /**
     * Set the object to be readable by the given user.
     *
     * @param string $objectpath The full path to the piece of
     *                           content/whatever to make readable.
     * @param string $user The user to assign the permissions to.
     */
    function setReadable($objectpath, $user)
    {
        $this->addPerms($objectpath, $user, _PERMS_READ);
    }

    /**
     * Set the object to be unreadable by the given user.
     *
     * @param string $objectpath The full path to the piece of
     *                           content/whatever to make unreadable.
     * @param string $user The user to assign the permissions to.
     */
    function setUnreadable($objectpath, $user)
    {
        $this->removePerms($objectpath, $user, _PERMS_READ);
    }

    /**
     * Set the object to be editable by the given user.
     *
     * @param string $objectpath The full path to the piece of
     *                           content/whatever to make editable.
     * @param string $user The user to assign the permissions to.
     */
    function setEditable($objectpath, $user)
    {
        $this->addPerms($objectpath, $user, _PERMS_EDIT);
    }

    /**
     * Set the object to be uneditable by the given user.
     *
     * @param string $objectpath The full path to the piece of
     *                           content/whatever to make uneditable.
     * @param string $user The user to assign the permissions to.
     */
    function setUneditable($objectpath, $user)
    {
        $this->removePerms($objectpath, $user, _PERMS_EDIT);
    }

    /**
     * Set the object to be deleteable by the given user.
     *
     * @param string $objectpath The full path to the piece of
     *                           content/whatever to make deleteable.
     * @param string $user The user to assign the permissions to.
     */
    function setDeleteable($objectpath, $user)
    {
        $this->addPerms($objectpath, $user, _PERMS_DELETE);
    }

    /**
     * Set the object to be undeleteable by the given user.
     *
     * @param string $objectpath The full path to the piece of
     *                           content/whatever to make undeleteable.
     * @param string $user The user to assign the permissions to.
     */
    function setUndeleteable($objectpath, $user)
    {
        $this->removePerms($objectpath, $user, _PERMS_DELETE);
    }

}
?>
