/*	$Id: vilter-spamd.c,v 1.30 2006/02/16 08:50:56 mbalmer Exp $	*/

/*
 * Copyright (c) 2003, 2004, 2005, 2006 Marc Balmer <marc@msys.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sys/stat.h>

#include <netinet/in.h>
#include <arpa/inet.h>

#include <ctype.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>

#include "pathnames.h"
#include "smtp-vilter.h"

#ifdef __linux__
#include "strlfunc.h"
#endif

#define TIMEOUT 	60
#define MAXBUF		1024
#define MAXSTR		64
#define MAXTRIES	256

int	 spamd_port;
char	*spamd_host;
char	*bind_addr;
int	 spamd_tries;
int	 spamd_timeout;
off_t	 spamd_maxsize;

int
vilter_scan(char *fn, size_t fnlen, char *chroot, double *score, double *threshold)
{
	int fd;
	int try;
	int nread;
	FILE *fp;
	struct sockaddr_in server_sockaddr;
	struct sockaddr_in bind_sockaddr;
	struct timeval tv, *to;
	struct hostent *hostent;
	SOCK *sp;
	char buf[MAXBUF];
	int buflen;
	struct stat statbuf;
	char status[6];
	double s, t;

	if (stat(fn, &statbuf)) {
		syslog(LOG_ERR, "spamd: can't stat file to scan");
		return SCAN_ERROR;
	}
	if ((spamd_maxsize > 0) && (statbuf.st_size > spamd_maxsize)) {
		if (verbose)
			warnx("spamd: skipping large message of %lld bytes", (long long) statbuf.st_size);
		return SCAN_OK;
	}
	if ((fd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1) {
		syslog(LOG_ERR, "spamd: unable to obtain network");
		return SCAN_ERROR;
	}
	if (bind_addr != NULL) {
		bzero((char *)&server_sockaddr, sizeof(server_sockaddr));
		bind_sockaddr.sin_family = AF_INET;

		if (inet_aton(bind_addr, &bind_sockaddr.sin_addr) == 0) {
			hostent = gethostbyname(bind_addr);
			if (!hostent) {
				syslog(LOG_ERR, "spamd: unknown bind address: %s", bind_addr);
				close(fd);
				return SCAN_ERROR;
			}
			bind_sockaddr.sin_family = hostent->h_addrtype;
			memcpy(&bind_sockaddr.sin_addr, hostent->h_addr, hostent->h_length);
		}
		if (bind(fd, (struct sockaddr *)&bind_sockaddr, sizeof(bind_sockaddr))) {
			syslog(LOG_ERR, "spam: can't bind to address %s", bind_addr);
			close(fd);
			return SCAN_ERROR;
		}
	}
	bzero((char *)&server_sockaddr, sizeof(server_sockaddr));

	server_sockaddr.sin_family = AF_INET;
	server_sockaddr.sin_port = htons(spamd_port);

	if (inet_aton(spamd_host, &server_sockaddr.sin_addr) == 0) {
		hostent = gethostbyname(spamd_host);
		if (!hostent) {
			syslog(LOG_ERR, "spamd: unknown host: %s", spamd_host);
			close(fd);
			return SCAN_ERROR;
		}
		server_sockaddr.sin_family = hostent->h_addrtype;
		memcpy(&server_sockaddr.sin_addr, hostent->h_addr, hostent->h_length);
	}
	try = 0;

	while (connect(fd, (struct sockaddr *)&server_sockaddr, sizeof(server_sockaddr)) == -1) {
		++try;

		if (try == spamd_tries) {
			syslog(LOG_ERR, "spamd: unable to connect socket");
			close(fd);
			return SCAN_ERROR;
		} else
			sleep(1);
	}
	if (spamd_timeout > 0) {
		tv.tv_sec = spamd_timeout;
		tv.tv_usec = 0;
		to = &tv;
	} else
		to = NULL;

	snprintf(buf, sizeof(buf), "CHECK SPAMC/1.2\r\nContent-length: %lld\r\n\r\n",
		(long long)statbuf.st_size);
	buflen = strlen(buf);
		
	if (to_send(fd, buf, buflen, 0, to) != buflen) {
		syslog(LOG_ERR, "spamd: timeout");
		close(fd);
		return SCAN_ERROR;
	}
	if ((fp = fopen(fn, "r")) == NULL) {
		syslog(LOG_ERR, "spamd: error opening file to scan");
		close(fd);
		return SCAN_ERROR;
	}
	while ((nread = fread(buf, 1, sizeof(buf), fp)) > 0)
		if (to_send(fd, buf, nread, 0, to) != nread) {
			syslog(LOG_ERR, "spamd: error sending file");
			close(fd);
			fclose(fp);
			return SCAN_ERROR;
		}
	fclose(fp);	

	if (shutdown(fd, SHUT_WR)) {
		syslog(LOG_ERR, "spamd: error shutting down writing on socket");
		close(fd);
		return SCAN_ERROR;
	}
	if ((sp = fdsock(fd)) == NULL) {
		syslog(LOG_ERR, "spamd: error buffering socket");
		close(fd);
		return SCAN_ERROR;
	}
	if (to_readln(buf, sizeof(buf), sp, to) <= 0) {
		syslog(LOG_ERR, "spamd: lost header - no response to CHECK cmd");
		sclose(sp);
		return SCAN_ERROR;
	}

	/* SPAMD/1.1 <result_code> <msg> */

	if (to_readln(buf, sizeof(buf), sp, to) <= 0) {
		syslog(LOG_ERR, "spamd: spamd: lost report - no response to CHECK cmd");
		sclose(sp);
		return SCAN_ERROR;
	}

	/* Spam: True|False ; score / threshold */

	sclose(sp);
	sscanf(buf, "Spam: %5s ; %lf / %lf", status, &s, &t);

	if (score != NULL)
		*score = s;
	if (threshold != NULL)
		*threshold = t;

	if (!strcmp(status, "True"))
		return SCAN_SPAM;
	else
		return SCAN_OK;

	return SCAN_ERROR;
}

char *
vilter_name(void)
{
	return "SpamAssassin Daemon (spamd)";
}

int
vilter_type(void)
{
	return BE_SCAN_SPAM | BE_SCAN_PER_RCPT;
}

void
vilter_exit(void)
{
	if (verbose)
		warnx("spamd: vilter_exit()");

	free(spamd_host);
}
