<?php
/**
 * The base class representing Kolab objects stored in the server
 * database.
 *
 * $Horde: framework/Kolab_Server/lib/Horde/Kolab/Server/Object.php,v 1.2.2.3 2008/09/22 16:20:59 wrobel Exp $
 *
 * @package Kolab_Server
 */

/** Define the different Kolab object types */
define('KOLAB_OBJECT_ADDRESS',          'Horde_Kolab_Server_Object_address');
define('KOLAB_OBJECT_ADMINISTRATOR',    'Horde_Kolab_Server_Object_administrator');
define('KOLAB_OBJECT_DOMAINMAINTAINER', 'Horde_Kolab_Server_Object_domainmaintainer');
define('KOLAB_OBJECT_GROUP',            'Horde_Kolab_Server_Object_group');
define('KOLAB_OBJECT_MAINTAINER',       'Horde_Kolab_Server_Object_maintainer');
define('KOLAB_OBJECT_SHAREDFOLDER',     'Horde_Kolab_Server_Object_sharedfolder');
define('KOLAB_OBJECT_USER',             'Horde_Kolab_Server_Object_user');
define('KOLAB_OBJECT_SERVER',           'Horde_Kolab_Server_Object_server');

/** Define the possible Kolab object attributes */
define('KOLAB_ATTR_DN',        'dn');
define('KOLAB_ATTR_SN',        'sn');
define('KOLAB_ATTR_CN',        'cn');
define('KOLAB_ATTR_FN',        'fn');
define('KOLAB_ATTR_MAIL',      'mail');
define('KOLAB_ATTR_UID',       'uid');
define('KOLAB_ATTR_DELETED',   'kolabDeleteFlag');
define('KOLAB_ATTR_IMAPHOST',  'kolabImapServer');
define('KOLAB_ATTR_HOMESERVER','kolabHomeServer');
define('KOLAB_ATTR_IPOLICY',   'kolabInvitationPolicy');
define('KOLAB_ATTR_FBPAST',    'kolabFreeBusyPast');
define('KOLAB_ATTR_FBFUTURE',  'kolabFreeBusyFuture');

/**
 * This class provides methods to deal with Kolab objects stored in
 * the Kolab db.
 *
 * $Horde: framework/Kolab_Server/lib/Horde/Kolab/Server/Object.php,v 1.2.2.3 2008/09/22 16:20:59 wrobel Exp $
 *
 * Copyright 2008 The Horde Project (http://www.horde.org/)
 *
 * See the enclosed file COPYING for license information (LGPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/lgpl.html.
 *
 * @author  Gunnar Wrobel  <wrobel@pardus.de>
 * @package Kolab_Server
 */
class Horde_Kolab_Server_Object {

    /**
     * Link into the Kolab server.
     *
     * @var Kolab_Server
     */
    var $_db;

    /**
     * DN of this object on the Kolab server.
     *
     * @var string
     */
    var $_dn;

    /**
     * The cached result
     *
     * @var mixed
     */
    var $_cache = false;

    /**
     * The LDAP filter to retrieve this object type.
     *
     * @var string
     */
    var $filter = '';

    /**
     * The attributes supported by this class.
     *
     * @var array
     */
    var $_supported_attributes = array();

    /**
     * Initialize the Kolab Object. Provide either the DN or a
     * LDAP search result.
     *
     * @param Horde_Kolab_Server $db     The link into the Kolab db.
     * @param string             $dn     DN of the object.
     * @param array              $data   A possible array of data for the object
     */
    function Horde_Kolab_Server_Object(&$db, $dn = null, $data = null)
    {
        $this->_db = &$db;
        if (empty($dn)) {
            if (empty($data) || !isset($data['dn'])) {
                $this->_cache = PEAR::raiseError(_('Specify either the DN or a search result!'));
                return;
            }
			if (is_array($data['dn'])) {
				$this->_dn = $data['dn'][0];
			} else {
				$this->_dn = $data['dn'];
			}
			$this->_cache = $data;
        } else {
            $this->_dn = $dn;
        }
    }

    /**
     * Attempts to return a concrete Horde_Kolab_Server_Object instance based on $type.
     *
     * @param mixed  $type    The type of the Horde_Kolab_Server_Object subclass to
     *                        return.
     * @param string $dn      DN of the object
     * @param array  $storage A link to the Kolab_Server class handling read/write.
     * @param array  $data    A possible array of data for the object
     *
     * @return Horde_Kolab_Server_Object|PEAR_Error The newly created concrete
     *                                 Horde_Kolab_Server_Object instance.
     */

	function &factory($type, $dn, &$storage, $data = null)
	{
        if (in_array($type, array(KOLAB_OBJECT_ADDRESS, KOLAB_OBJECT_ADMINISTRATOR,
                                  KOLAB_OBJECT_DOMAINMAINTAINER, KOLAB_OBJECT_GROUP,
                                  KOLAB_OBJECT_MAINTAINER, KOLAB_OBJECT_SHAREDFOLDER,
                                  KOLAB_OBJECT_USER, KOLAB_OBJECT_SERVER))) {
            $type = substr($type, 26);
        }

        $type = basename($type);

        if (file_exists(dirname(__FILE__) . '/Object/' . $type . '.php')) {
            include_once dirname(__FILE__) . '/Object/' . $type . '.php';
        }

        $class = 'Horde_Kolab_Server_Object_' . $type;
        if (class_exists($class)) {
            $object = &new $class($storage, $dn, $data);
        } else {
            $object = PEAR::raiseError('Class definition of ' . $class . ' not found.');
        }

        return $object;
	}

    /**
     * Does the object exist?
     */
    function exists()
    {
        $this->_read();
        if (!$this->_cache || is_a($this->_cache, 'PEAR_Error')) {
            return false;
        }
        return true;
    }

    /**
     * Read the object into the cache
     */
    function _read()
    {
        $this->_cache = $this->_db->read($this->_dn, $this->_supported_attributes);
    }

    /**
     * Get the specified attribute of this object
     *
	 * @param string $attr The attribute to read
	 *
     * @return string the value of this attribute
     */
    function get($attr)
    {
        if ($attr != KOLAB_ATTR_DN) {
            if (!in_array($attr, $this->_supported_attributes)) {
                return PEAR::raiseError(sprintf(_("Attribute \"%s\" not supported!"),
                                                $attr));
            }
            if (!$this->_cache) {
                $this->_read();
            }
            if (is_a($this->_cache, 'PEAR_Error')) {
                return $this->_cache;
            }
        }

		switch ($attr) {
		case KOLAB_ATTR_DN:
			return $this->_getDn();
		case KOLAB_ATTR_FN:
			return $this->_getFn();
		case KOLAB_ATTR_SN:
		case KOLAB_ATTR_CN:
		case KOLAB_ATTR_MAIL:
		case KOLAB_ATTR_UID:
		case KOLAB_ATTR_IMAPHOST:
		case KOLAB_ATTR_HOMESERVER:
			return $this->_get($attr, true);
		default:
			return $this->_get($attr, false);
		}
    }

    /**
     * Get the specified attribute of this object
     *
	 * @param string  $attr   The attribute to read
	 * @param boolean $single Should a single value be returned
	 *                        or are multiple values allowed?
	 *
     * @return string the value of this attribute
     */
    function _get($attr, $single = true)
    {
        if (isset($this->_cache[$attr])) {
			if ($single && is_array($this->_cache[$attr])) {
				return $this->_cache[$attr][0];
			} else {
				return $this->_cache[$attr];
			}
        }
        return false;
    }

    /**
     * Get the DN of this object
     *
     * @return string the DN of this object
     */
    function _getDn()
    {
        return $this->_dn;
    }

    /**
     * Get the "first name" attribute of this object
     *
     * @return string the "first name" of this object
     */
    function _getFn()
    {
        $sn = $this->_get(KOLAB_ATTR_SN, true);
        $cn = $this->_get(KOLAB_ATTR_CN, true);
        return trim(substr($cn, 0, strlen($cn) - strlen($sn)));
    }

    /**
     * Get the groups for this object
     *
     * @return mixed An array of group ids or a PEAR Error in case of
     *               an error.
     */
    function getGroups()
    {
        return array();
    }

    /**
     * Returns the server url of the given type for this user.
     *
     * This method can be used to encapsulate multidomain support.
     *
     * @return string|PEAR_Error The server url or empty.
     */
    function getServer($server_type)
    {
        return PEAR::raiseError('Not implemented!');
    }
};
