/*
 * dstumbler v1.0 [nodes.c]
 * by h1kari - (c) Dachb0den Labs 2001
 */

/*
 * Copyright (c) 2001 Dachb0den Labs.
 *      David Hulton <h1kari@dachb0den.com>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by David Hulton.
 * 4. Neither the name of the author nor the names of any co-contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY David Hulton AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL David Hulton OR THE VOICES IN HIS HEAD
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>
#include <errno.h>

#ifdef __OpenBSD__
#include <curses.h>
#else
#include <ncurses.h>
#endif

#include "dstumbler.h"
#include "screen.h"

/*
 * poll the wi driver or parse monitor mode output to find a node
 */
void
parse_node(const char *iface)
{
  int ap, node;
  char bssid[MACSIZE];
  struct node_s new;

  if(!monmode)
    return;

  memset((char *)&new, 0, sizeof(struct node_s));

  if(mon_parse_node(&new, bssid) == -1)
    return;

  /* fill in cached gps coordinates */
  getgps(&new.ns, &new.ew);

  if((ap = find_node_ap(bssid)) == -1)
    return;

  if((node = find_node(ap, new.mac)) == -1)
  {
    /* set timestamps */
    gettimeofday(&new.firstseen, NULL);
    gettimeofday(&new.lastseen, NULL);

    add_node(ap, &new);

    if(nodemode && ap == aps_cur)
    {
      redraw_nodes();

      /* if we're in autoselect mode, make sure we refresh stats for this ap */
      if(autosel)
        redraw_node_graph();
    }
  }
  else
  {
    if(new.weped)
      aps[ap]->node[node]->weped = new.weped;
    if(new.adhoc)
      aps[ap]->node[node]->adhoc = new.adhoc;
    if(new.keyed)
      aps[ap]->node[node]->keyed = new.keyed;
    if(new.interval)
      aps[ap]->node[node]->interval = new.interval;
    if(new.srate)
      aps[ap]->node[node]->srate = new.srate;

    if(new.keyid && aps[ap]->weped)
    {
      aps[ap]->node[node]->keyid = new.keyid;
      aps[ap]->wepid = 1;
      redraw_aps();
    }

    gettimeofday(&new.lastseen, NULL);

    /* update the found entry with quality info and timestamp and gps */
    memcpy((char *)aps[ap]->node[node]->quality, (char *)new.quality,
     sizeof(aps[ap]->node[node]->quality));
    memcpy((char *)&aps[ap]->node[node]->lastseen, (char *)&new.lastseen,
     sizeof(aps[ap]->node[node]->lastseen));
    memcpy((char *)&aps[ap]->node[node]->ns, (char *)&new.ns,
     sizeof(aps[ap]->node[node]->ns));
    memcpy((char *)&aps[ap]->node[node]->ew, (char *)&new.ew,
     sizeof(aps[ap]->node[node]->ew));

    aps[ap]->node[node]->datalen += new.datalen;

    /* write log */
    add_node_log(ap, aps[ap]->node[node], &new);

    if(ap == aps_cur)
    {
      if(node != node_new || autosel)
        NODENEW(node);

      if(nodemode)
      {
        redraw_nodes();

        if(node == aps[aps_cur]->node_cur || autosel)
          redraw_node_graph();
      }
    }
  }
}

/*
 * search through the aps linked list to find the ap this node is associated
 * with
 */
int
find_node_ap(const char *bssid)
{
  int i, found = -1;

  for(i = 0; i < aps_len; i++)
  {
    if(memcmp(aps[i]->bssid, bssid, MACSIZE) == 0)
    {
      found = i;
      break;
    }
  }

  return found;
}

/*
 * search through the nodes for this ap and find if this node already exists
 */
int
find_node(int ap, const char *mac)
{
  int i, found = -1;

  for(i = 0; i < aps[ap]->node_len; i++)
  {
    if(memcmp(aps[ap]->node[i]->mac, mac, MACSIZE) == 0)
      found = i;

    memset((char *)aps[ap]->node[i]->quality, 0,
     sizeof(aps[ap]->node[i]->quality));
  }

  return found;
}

/*
 * set the any mode on the specified node
 */
void
node_setany(u_char *mac, int any)
{
  int i, found;

  for(i = 0; i < aps_len; i++)
  {
    if((found = find_node(i, mac)) != -1)
      aps[i]->node[found]->any = any ? 1 : 2;
  }

  if(nodemode)
    redraw_nodes();
}

/*
 * set the srate mode on the specified node
 */
void
node_setsrate(u_char *mac, int srate)
{
  int i, found;

  for(i = 0; i < aps_len; i++)
  {
    if((found = find_node(i, mac)) != -1)
      aps[i]->node[found]->srate = srate;
  }

  if(nodemode)
    redraw_nodes();
}

/*
 * redraw the ap screen with all of the aps in the aps linked list
 */
#define PADDING_LEN (NODE_SCR_COLS - 40)
#define MFG_FMT_LEN 7
#define MAC_STR_LEN 17
#define DATALEN_LEN 12
#define RATE_BUF_LEN 8
void
redraw_nodes(void)
{
  int i, j, node_color, node_color_bold, code, off = 0, wrote = 0;
  char datalen[DATALEN_LEN], rate_buf[RATE_BUF_LEN];
  struct node_s *ptr;

  if(aps_len < 1)
    return;

  werase(node_scr);
  off = (nodechange || !autosel) ? aps[aps_cur]->node_cur : node_new;

  if(off < node_win)
    node_win = off;
  else if(off > (node_win + NODE_SCR_LINES - 1))
    node_win = off - (NODE_SCR_LINES - 1);

  for(i = node_win; i < aps[aps_cur]->node_len; i++)
  {
    ptr = aps[aps_cur]->node[i];

    if(wrote++)
      waddch(node_scr, '\n');

    if(i == aps[aps_cur]->node_cur)
    {
      node_color = NODE_COLOR_SEL;
      node_color_bold = NODE_COLOR_SEL_BOLD;
    }
    else
    {
      node_color = NODE_COLOR;
      node_color_bold = NODE_COLOR_BOLD;
    }

    /* figure out how many symbols our quality is to determine color code */
    if(ptr->quality[0] < 6)
      code = COLOR_CODE_LOW;
    else if(ptr->quality[0] > 5 && ptr->quality[0] < 9)
      code = COLOR_CODE_LOW_BOLD;
    else if(ptr->quality[0] > 8 && ptr->quality[0] < 12)
      code = COLOR_CODE_MED;
    else if(ptr->quality[0] > 11 && ptr->quality[0] < 15)
      code = COLOR_CODE_MED_BOLD;
    else if(ptr->quality[0] > 14 && ptr->quality[0] < 18)
      code = COLOR_CODE_HI;
    else if(ptr->quality[0] > 17)
      code = COLOR_CODE_HI_BOLD;

    wattrset(node_scr, 0);

    /*
     * sorry for the messyness... better then using a loop
     */
    waddch(node_scr, ((autosel ? i == node_new : i == aps[aps_cur]->node_cur) ?
     NODE_CUR | code : (i == node_new ? NODE_NEW | code : ' ')));

    wattrset(node_scr, node_color_bold);
    wprintw(node_scr, " [");
    wattrset(node_scr, node_color);
    snprintf(datalen, DATALEN_LEN, "%d", ptr->datalen);
    wprintw(node_scr, "%s", datalen);
    wattrset(node_scr, node_color_bold);
    wprintw(node_scr, "]");

    if(ptr->srate)
      snprintf(rate_buf, RATE_BUF_LEN, "%d.%d", ptr->srate / 2, ptr->srate % 2);
    else
      rate_buf[0] = '\0';

    for(j = 0; j < (PADDING_LEN - strlen(datalen) - strlen(rate_buf)); j++)
      waddch(node_scr, ' ');

    wattrset(node_scr, node_color);
    wprintw(node_scr, "%s", rate_buf);
    wattrset(node_scr, node_color_bold);
    wprintw(node_scr, " (");
    wattrset(node_scr, node_color);

    if(resolvmfg)
    {
      char mfg[MFG_STRLEN], mfg_fmt[MFG_FMT_LEN];
      mfgresolve(mfg, ptr->mac);
      snprintf(mfg_fmt, MFG_FMT_LEN - 1, "%%-%ds", MAC_STR_LEN);
      wprintw(node_scr, mfg_fmt, mfg);
    }
    else
      wprintw(node_scr, "%02x:%02x:%02x:%02x:%02x:%02x",
       ptr->mac[0], ptr->mac[1], ptr->mac[2],
       ptr->mac[3], ptr->mac[4], ptr->mac[5]);

    wattrset(node_scr, node_color_bold);
    wprintw(node_scr, ")");
    waddch(node_scr, ptr->any == 1 ? NODE_ANY | NODE_COLOR_ANY :
     (ptr->any == 2 ? NODE_SET | NODE_COLOR_SET : ' '));
    waddch(node_scr, ptr->keyed == 1 ? NODE_KEYED | NODE_COLOR_KEYED :
     (ptr->keyed == 2 ? NODE_OPEN | NODE_COLOR_OPEN : ' '));
    waddch(node_scr, ptr->adhoc == 1 ? NODE_ADHOC | NODE_COLOR_ADHOC :
     (ptr->adhoc == 2 ? NODE_BSS | NODE_COLOR_BSS : ' '));
    waddch(node_scr, ptr->weped == 1 || ptr->keyid ?
     (ptr->keyid ? ptr->keyid + '0' : NODE_WEP) | NODE_COLOR_WEPED :
     (ptr->weped == 2 ? NODE_NOWEP | NODE_COLOR_NOWEP : ' '));
    wattrset(node_scr, node_color);
    wprintw(node_scr, "%03d", ptr->quality[0]);
    waddch(node_scr, ':' | node_color_bold);
    wprintw(node_scr, "%03d", ptr->quality[1]);
    waddch(node_scr, ':' | node_color_bold);
    wprintw(node_scr, "%03d", ptr->quality[2]);

    if(wrote == NODE_SCR_LINES)
      break;
  }

  redraw.node_scr++;
}

/*
 * redraw signal to noise graph for current node
 */
void
redraw_node_graph(void)
{
  static int wrote = 0;
  int i;
  struct node_s *ptr;

  if(aps_len < 1)
    return;

  if(!aps[aps_cur]->node_len)
  {
    werase(graph_scr);
    redraw.graph_scr++;
    return;
  }

  ptr = aps[aps_cur]->node[autosel ? node_new : aps[aps_cur]->node_cur];

  if(nodechange || apchange || (autosel && nodenew))
  {
    werase(graph_scr);
    wrote = 0;

    for(i = (ptr->log_len > GRAPH_SCR_LINES ?
     ptr->log_len - GRAPH_SCR_LINES : 0); i < ptr->log_len; i++)
      print_graph(ptr->log[i]->quality, wrote++);
  }
  else
    print_graph(ptr->quality, wrote++);

  redraw.graph_scr++;
}

/*
 * add ap to the linked list (along with a log of the addition)
 */
#define LOG_INIT_LEN (sizeof(struct log_s *) * PREALLOC_LOG)
void
add_node(int ap, const struct node_s *node)
{
  int i;
  struct node_s *ptr;

  /* reallocate aps pointer array if we're reaching the end */
  if(aps[ap]->node_len == aps[ap]->node_max)
  {
    struct node_s **new_nodes;

    if((new_nodes = (struct node_s **)realloc(aps[ap]->node,
     (aps[ap]->node_max + PREALLOC_APS) * sizeof(struct node_s *))) == NULL)
    {
      alert("error: unable to reallocate memory: %s", strerror(errno));
      exitclean(2);
    }

    aps[ap]->node = new_nodes;

    for(i = aps[ap]->node_max; i < (aps[ap]->node_max + PREALLOC_APS); i++)
      aps[ap]->node[i] = NULL;

    aps[ap]->node_max += PREALLOC_APS;
  }

  if((aps[ap]->node[aps[ap]->node_len] =
   (struct node_s *)malloc(sizeof(struct node_s))) == NULL)
  {
    alert("error: unable to allocate memory: %s", strerror(errno));
    exitclean(2);
  }

  memcpy((char *)aps[ap]->node[aps[ap]->node_len], (char *)node,
   sizeof(struct node_s));

  ptr = aps[ap]->node[aps[ap]->node_len];

  /* now pre-allocate log pointer array */
  if((ptr->log = (struct log_s **)malloc(LOG_INIT_LEN)) == NULL)
  {
    alert("error: unable to allocate memory: %s", strerror(errno));
    exitclean(2);
  }

  memset((char *)ptr->log, 0, LOG_INIT_LEN);
  ptr->log_max = PREALLOC_LOG;
  ptr->log_len = 0;

  add_node_log(ap, ptr, node);

  if(ap == aps_cur)
    NODENEW(aps[ap]->node_len);

  aps[ap]->node_len++;
}

/*
 * add log to the specified node
 */
void
add_node_log(int ap, struct node_s *node, const struct node_s *info)
{
  int i;
  struct log_s *logptr;

  /* reallocate log pointer array if we're reaching the end */
  if(node->log_len == node->log_max)
  {
    struct log_s **new_log;

    if((new_log = (struct log_s **)realloc(node->log,
     (node->log_max + PREALLOC_LOG) * sizeof(struct log_s *))) == NULL)
    {
      alert("error: unable to reallocate memory: %s", strerror(errno));
      exitclean(2);
    }

    node->log = new_log;

    for(i = node->log_max; i < (node->log_max + PREALLOC_LOG); i++)
      node->log[i] = NULL;

    node->log_max += PREALLOC_LOG;
  }

  /* make first log entry */
  if((node->log[node->log_len] = (struct log_s *)malloc(sizeof(struct log_s)))
   == NULL)
  {
    alert("error: unable to allocate memory: %s", strerror(errno));
    exitclean(2);
  }

  memset((char *)node->log[node->log_len], 0, sizeof(struct log_s));

  logptr = node->log[node->log_len];

  logptr->ns.dir = info->ns.dir;
  logptr->ns.coordinates = info->ns.coordinates;
  logptr->ew.dir = info->ew.dir;
  logptr->ew.coordinates = info->ew.coordinates;

  logptr->quality[0] = info->quality[0];
  logptr->quality[1] = info->quality[1];
  logptr->quality[2] = info->quality[2];

  logptr->seen.tv_sec = info->lastseen.tv_sec;
  logptr->seen.tv_usec = info->lastseen.tv_usec; 

  if(uselog)
    log_node(log_fd, aps[ap], node, logptr);

  node->log_len++;
}
