<?php
/**
 * @package    Zend_Pdf
 * @subpackage UnitTests
 */


/** Zend_Pdf */
require_once 'Zend/Pdf.php';

/** PHPUnit Test Case */
require_once 'PHPUnit/Framework/TestCase.php';


/**
 * @package    Zend_Pdf
 * @subpackage UnitTests
 */
class Zend_Pdf_DrawingTest extends PHPUnit_Framework_TestCase
{
    public function testDrawing()
    {
        $pdf = new Zend_Pdf();

        // Add new page generated by Zend_Pdf object (page is attached to the specified the document)
        $pdf->pages[] = ($page1 = $pdf->newPage('A4'));

        // Add new page generated by Zend_Pdf_Page object (page is not attached to the document)
        $pdf->pages[] = ($page2 = new Zend_Pdf_Page(Zend_Pdf_Page::SIZE_LETTER_LANDSCAPE));

        // Create new font
        $font = Zend_Pdf_Font::fontWithName(Zend_Pdf_Font::FONT_HELVETICA);

        // Apply font and draw text
        $page1->setFont($font, 36);
        $page1->setFillColor(Zend_Pdf_Color_Html::color('#9999cc'));
        $page1->drawText('Helvetica 36 text string', 60, 500);

        // Use font object for another page
        $page2->setFont($font, 24);
        $page2->drawText('Helvetica 24 text string', 60, 500);

        // Use another font
        $page2->setFont(Zend_Pdf_Font::fontWithName(Zend_Pdf_Font::FONT_TIMES), 32);
        $page2->drawText('Times-Roman 32 text string', 60, 450);

        // Draw rectangle
        $page2->setFillColor(new Zend_Pdf_Color_GrayScale(0.8));
        $page2->setLineColor(new Zend_Pdf_Color_GrayScale(0.2));
        $page2->setLineDashingPattern(array(3, 2, 3, 4), 1.6);
        $page2->drawRectangle(60, 400, 400, 350);

        // Draw circle
        $page2->setLineDashingPattern(Zend_Pdf_Page::LINE_DASHING_SOLID);
        $page2->setFillColor(new Zend_Pdf_Color_Rgb(1, 0, 0));
        $page2->drawCircle(85, 375, 25);

        // Draw sectors
        $page2->drawCircle(200, 375, 25, 2*M_PI/3, -M_PI/6);
        $page2->setFillColor(new Zend_Pdf_Color_Cmyk(1, 0, 0, 0));
        $page2->drawCircle(200, 375, 25, M_PI/6, 2*M_PI/3);
        $page2->setFillColor(new Zend_Pdf_Color_Rgb(1, 1, 0));
        $page2->drawCircle(200, 375, 25, -M_PI/6, M_PI/6);

        // Draw ellipse
        $page2->setFillColor(new Zend_Pdf_Color_Html('Red'));
        $page2->drawEllipse(250, 400, 400, 350);
        $page2->setFillColor(new Zend_Pdf_Color_Cmyk(1, 0, 0, 0));
        $page2->drawEllipse(250, 400, 400, 350, M_PI/6, 2*M_PI/3);
        $page2->setFillColor(new Zend_Pdf_Color_Rgb(1, 1, 0));
        $page2->drawEllipse(250, 400, 400, 350, -M_PI/6, M_PI/6);

        // Draw and fill polygon
        $page2->setFillColor(new Zend_Pdf_Color_Rgb(1, 0, 1));
        $x = array();
        $y = array();
        for ($count = 0; $count < 8; $count++) {
            $x[] = 140 + 25*cos(3*M_PI_4*$count);
            $y[] = 375 + 25*sin(3*M_PI_4*$count);
        }
        $page2->drawPolygon($x, $y,
                            Zend_Pdf_Page::SHAPE_DRAW_FILL_AND_STROKE,
                            Zend_Pdf_Page::FILL_METHOD_EVEN_ODD);

        // Draw line
        $page2->setLineWidth(0.5);
        $page2->drawLine(60, 375, 400, 375);

        $pdf->save(dirname(__FILE__) . '/_files/output.pdf');
        unset($pdf);

        $pdf1 = Zend_Pdf::load(dirname(__FILE__) . '/_files/output.pdf');
        $this->assertTrue($pdf1 instanceof Zend_Pdf);
        unset($pdf1);

        unlink(dirname(__FILE__) . '/_files/output.pdf');
    }

    public function testImageDrawing()
    {
        $pdf = new Zend_Pdf();

        // Add new page generated by Zend_Pdf object (page is attached to the specified the document)
        $pdf->pages[] = ($page = $pdf->newPage('A4'));


        $stampImagePNG = Zend_Pdf_Image::imageWithPath(dirname(__FILE__) . '/_files/stamp.png');
        $this->assertTrue($stampImagePNG instanceof Zend_Pdf_Resource_Image);

        $page->saveGS();
        $page->clipCircle(250, 500, 50);
        $page->drawImage($stampImagePNG, 200, 450, 300, 550);
        $page->restoreGS();


        $stampImageTIFF = Zend_Pdf_Image::imageWithPath(dirname(__FILE__) . '/_files/stamp.tif');
        $this->assertTrue($stampImageTIFF instanceof Zend_Pdf_Resource_Image);

        $page->saveGS();
        $page->clipCircle(325, 500, 50);
        $page->drawImage($stampImagePNG, 275, 450, 375, 550);
        $page->restoreGS();

        if (function_exists('gd_info')) {
            $info = gd_info();
            $jpegSupported = $info['JPG Support'];
        } else {
            $jpegSupported = false;
        }
        if ($jpegSupported) {
            $stampImageJPG = Zend_Pdf_Image::imageWithPath(dirname(__FILE__) . '/_files/stamp.jpg');

            $this->assertTrue($stampImageJPG instanceof Zend_Pdf_Resource_Image);

            $page->saveGS();
            $page->clipCircle(287.5, 440, 50);
            $page->drawImage($stampImageJPG, 237.5, 390, 337.5, 490);
            $page->restoreGS();

            $page->saveGS();
            $page->clipCircle(250, 500, 50);
            $page->clipCircle(287.5, 440, 50);
            $page->drawImage($stampImagePNG, 200, 450, 300, 550);
            $page->restoreGS();
        }

        $pdf->save(dirname(__FILE__) . '/_files/output.pdf');
        unset($pdf);

        $pdf1 = Zend_Pdf::load(dirname(__FILE__) . '/_files/output.pdf');
        $this->assertTrue($pdf1 instanceof Zend_Pdf);
        unset($pdf1);

        unlink(dirname(__FILE__) . '/_files/output.pdf');
    }

    public function testFontDrawing()
    {
        $pdf = new Zend_Pdf();

        $fontsList = array(Zend_Pdf_Font::FONT_COURIER,
                          Zend_Pdf_Font::FONT_COURIER_BOLD,
                          Zend_Pdf_Font::FONT_COURIER_BOLD_ITALIC,
                          Zend_Pdf_Font::FONT_COURIER_BOLD_OBLIQUE,
                          Zend_Pdf_Font::FONT_COURIER_ITALIC,
                          Zend_Pdf_Font::FONT_COURIER_OBLIQUE,
                          Zend_Pdf_Font::FONT_HELVETICA,
                          Zend_Pdf_Font::FONT_HELVETICA_BOLD,
                          Zend_Pdf_Font::FONT_HELVETICA_BOLD_ITALIC,
                          Zend_Pdf_Font::FONT_HELVETICA_BOLD_OBLIQUE,
                          Zend_Pdf_Font::FONT_HELVETICA_ITALIC,
                          Zend_Pdf_Font::FONT_HELVETICA_OBLIQUE,
                          Zend_Pdf_Font::FONT_TIMES,
                          Zend_Pdf_Font::FONT_TIMES_BOLD,
                          Zend_Pdf_Font::FONT_TIMES_BOLD_ITALIC,
                          Zend_Pdf_Font::FONT_TIMES_ITALIC,
                          Zend_Pdf_Font::FONT_TIMES_ROMAN);

        $titleFont = Zend_Pdf_Font::fontWithName(Zend_Pdf_Font::FONT_COURIER_BOLD_OBLIQUE);

        foreach ($fontsList as $fontName) {
            // Add new page generated by Zend_Pdf object (page is attached to the specified the document)
            $pdf->pages[] = ($page = $pdf->newPage(Zend_Pdf_Page::SIZE_A4_LANDSCAPE));

            $font = Zend_Pdf_Font::fontWithName($fontName);
            $this->assertTrue($font instanceof Zend_Pdf_Resource_Font);

            $page->setFont($titleFont, 10);
            $page->drawText($font->getFontName(Zend_Pdf_Font::NAME_POSTSCRIPT, 'en') . ':', 100, 400);

            $page->setFont($font, 20);
            $page->drawText("'The quick brown fox jumps over the lazy dog'", 100, 360);

            $ascent = $font->getAscent();
            $this->assertTrue( abs(1 - $font->getCoveredPercentage('ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxwz')) < 0.00001 );
            $descent = $font->getDescent();

            $font->getFontName(Zend_Pdf_Font::NAME_FULL, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_FAMILY, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_PREFERRED_FAMILY, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_STYLE, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_PREFERRED_STYLE, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_DESCRIPTION, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_SAMPLE_TEXT, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_ID, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_VERSION, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_POSTSCRIPT, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_CID_NAME, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_DESIGNER, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_DESIGNER_URL, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_MANUFACTURER, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_VENDOR_URL, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_COPYRIGHT, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_TRADEMARK, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_LICENSE, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_LICENSE_URL, 'en');

            $type       = $font->getFontType();
            $lineGap    = $font->getLineGap();
            $lineHeight = $font->getLineHeight();
            $this->assertTrue($font->getResource() instanceof Zend_Pdf_Element_Object);
            $font->getStrikePosition();
            $font->getStrikeThickness();
            $font->getUnderlinePosition();
            $font->getUnitsPerEm();
            $font->widthForGlyph(10);
        }

        $nonAlphabeticalPhonts =
                 array(Zend_Pdf_Font::FONT_SYMBOL =>
                                "\x00\x20\x00\x21\x22\x00\x00\x23\x22\x03\x00\x25\x00\x26\x22\x0b\x00\x28\x00\x29\x22\x17\x00\x2b\x00\x2c\x22\x12\x00\x2e\x00\x2f\x00\x30\x00\x31\x00\x32\x00\x33\x00\x34\x00\x35\x00\x36\x00\x37\x00\x38\x00\x39\x00\x3a\x00\x3b\x00\x3c\x00\x3d\x00\x3e\x00\x3f\x22\x45\x03\x91\x03\x92\x03\xa7\x22\x06\x03\x95\x03\xa6",
                       Zend_Pdf_Font::FONT_ZAPFDINGBATS =>
                                "\x00\x20\x27\x01\x27\x02\x27\x03\x27\x04\x26\x0e\x27\x06\x27\x07\x27\x08\x27\x09\x26\x1b\x26\x1e\x27\x0c\x27\x0d\x27\x0e\x27\x0f\x27\x10\x27\x11\x27\x12\x27\x13\x27\x14\x27\x15\x27\x16\x27\x17\x27\x18\x27\x19\x27\x1a");
        foreach ($nonAlphabeticalPhonts as $fontName => $example) {
            // Add new page generated by Zend_Pdf object (page is attached to the specified the document)
            $pdf->pages[] = ($page = $pdf->newPage(Zend_Pdf_Page::SIZE_A4_LANDSCAPE));

            $font = Zend_Pdf_Font::fontWithName($fontName);
            $this->assertTrue($font instanceof Zend_Pdf_Resource_Font);

            $page->setFont($titleFont, 10);
            $page->drawText($font->getFontName(Zend_Pdf_Font::NAME_POSTSCRIPT, 'en') . ':', 100, 400);

            $page->setFont($font, 20);
            $page->drawText($example, 100, 360, 'UTF-16BE');

            $ascent = $font->getAscent();
            $this->assertTrue( abs(1 - $font->getCoveredPercentage($example, 'UTF-16BE')) < 0.00001 );
            $descent = $font->getDescent();

            $font->getFontName(Zend_Pdf_Font::NAME_FULL, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_FAMILY, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_PREFERRED_FAMILY, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_STYLE, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_PREFERRED_STYLE, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_DESCRIPTION, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_SAMPLE_TEXT, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_ID, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_VERSION, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_POSTSCRIPT, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_CID_NAME, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_DESIGNER, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_DESIGNER_URL, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_MANUFACTURER, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_VENDOR_URL, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_COPYRIGHT, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_TRADEMARK, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_LICENSE, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_LICENSE_URL, 'en');

            $type       = $font->getFontType();
            $lineGap    = $font->getLineGap();
            $lineHeight = $font->getLineHeight();
            $this->assertTrue($font->getResource() instanceof Zend_Pdf_Element_Object);
            $font->getStrikePosition();
            $font->getStrikeThickness();
            $font->getUnderlinePosition();
            $font->getUnitsPerEm();
            $font->widthForGlyph(10);
        }

        $TTFFontsList = array('VeraBd.ttf',
                              'VeraBI.ttf',
                              'VeraIt.ttf',
                              'VeraMoBd.ttf',
                              'VeraMoBI.ttf',
                              'VeraMoIt.ttf',
                              'VeraMono.ttf',
                              'VeraSeBd.ttf',
                              'VeraSe.ttf',
                              'Vera.ttf');

        foreach ($TTFFontsList as $fontName) {
            // Add new page generated by Zend_Pdf object (page is attached to the specified the document)
            $pdf->pages[] = ($page = $pdf->newPage(Zend_Pdf_Page::SIZE_A4_LANDSCAPE));

            $font = Zend_Pdf_Font::fontWithPath(dirname(__FILE__) . '/_fonts/' . $fontName);
            $this->assertTrue($font instanceof Zend_Pdf_Resource_Font);

            $page->setFont($titleFont, 10);
            $page->drawText($font->getFontName(Zend_Pdf_Font::NAME_POSTSCRIPT, 'en') . ':', 100, 400);

            $page->setFont($font, 20);
            $page->drawText("'The quick brown fox jumps over the lazy dog'", 100, 360);

            $ascent = $font->getAscent();
            $this->assertTrue( abs(1 - $font->getCoveredPercentage('ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxwz')) < 0.00001 );
            $descent = $font->getDescent();

            $font->getFontName(Zend_Pdf_Font::NAME_FULL, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_FAMILY, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_PREFERRED_FAMILY, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_STYLE, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_PREFERRED_STYLE, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_DESCRIPTION, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_SAMPLE_TEXT, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_ID, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_VERSION, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_POSTSCRIPT, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_CID_NAME, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_DESIGNER, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_DESIGNER_URL, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_MANUFACTURER, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_VENDOR_URL, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_COPYRIGHT, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_TRADEMARK, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_LICENSE, 'en');
            $font->getFontName(Zend_Pdf_Font::NAME_LICENSE_URL, 'en');

            $type       = $font->getFontType();
            $lineGap    = $font->getLineGap();
            $lineHeight = $font->getLineHeight();
            $this->assertTrue($font->getResource() instanceof Zend_Pdf_Element_Object);
            $font->getStrikePosition();
            $font->getStrikeThickness();
            $font->getUnderlinePosition();
            $font->getUnitsPerEm();
            $font->widthForGlyph(10);
        }

        $pdf->save(dirname(__FILE__) . '/_files/output.pdf');
        unset($pdf);

        $pdf1 = Zend_Pdf::load(dirname(__FILE__) . '/_files/output.pdf');
        $this->assertTrue($pdf1 instanceof Zend_Pdf);
        unset($pdf1);

        unlink(dirname(__FILE__) . '/_files/output.pdf');
    }
}
