/***************************************************************************
                          rappwin.h  -  description
                             -------------------
    begin                : Mon Sep 27 1999
    copyright            : (C) 1999 by Andreas Mustun
    email                : andrew@ribbonsoft.com
 ***************************************************************************/

#ifndef RAPPWIN_H
#define RAPPWIN_H

#include <qlist.h>
#include <qmainwindow.h>
#include <qprinter.h>

#include "ractiondef.h"
#include "rprgdef.h"

#define TOOLBAR_LEFT_FIRST 0

#define   TOOLBAR_MAIN   0

#define   TOOLBAR_POINT  1
#define   TOOLBAR_LINE   2
#define   TOOLBAR_ARC    3
#define   TOOLBAR_CIRCLE 4
#define   TOOLBAR_TEXT   5
#define   TOOLBAR_DIM    6
#define	  TOOLBAR_MEASURE 7
#define   TOOLBAR_TAG   10
#define   TOOLBAR_TAG2  15
#define   TOOLBAR_EDIT  20

#define   TOOLBAR_SNAP  30

#define   TOOLBAR_CAM   40

#define TOOLBAR_LEFT_LAST  40

#define TOOLBAR_FILE  70
#define TOOLBAR_ZOOM  80
#define TOOLBAR_UNDO  90
#define TOOLBAR_SHOW  100
#define TOOLBAR_ATTRIB 110

class QFrame;
class QPopupMenu;
class QToolButton;

class RCadWidget;
class RComboBox;
class RDimDlg;
class RGraphic;
class RIntro;
class RInputPanel;
class RLibraryBrowser;
class RSimulationPanel;
class RSplitter;
class RStatusPanel;
class RToolBar;
class RWidget;


class RAppWin : public QMainWindow
{
  Q_OBJECT
protected:
  RAppWin();

public:
  ~RAppWin();

  static RAppWin* getRAppWin();
  RGraphic* currentDoc();
  RGraphic* currentOut();

	void init();

  void reinit();

  /*! \return Pointer to the CAD Widget at the left side document (RGraphic)
   */
  RCadWidget* getCadWidget1() { return cadWidget1; }

  /*! \return Pointer to the CAD Widget at the right side (RGraphic)
   */
  RCadWidget* getCadWidget2() { return cadWidget2; }

  /*! \return Pointer to Status Panel (RStatusPanel)
   */
  RStatusPanel* getStatusPanel() { return statusPanel; }

  /*! \return Pointer to Input Panel (RInputPanel)
   */
  RInputPanel* getInputPanel() { return inputPanel; }

  /*! \return Pointer to Input Panel 2 (RInputPanel)
   */
  RInputPanel* getInputPanel2() { return inputPanel2; }

  /*! \return Pointer to dimensionDlg (RDimDlg)
   */
  RDimDlg* getDimensionDlg() { return dimensionDlg; }

  /*! \return Pointer to Library browser (RLibraryBrowser)
   */
  RLibraryBrowser* getLibraryBrowser();

  /*! \return Pointer to Clipboard
   */
  RGraphic* getClipboard() { return clipboard; }
  
  /** \return Pointer to printer */
  QPrinter* getPrinter() { return printer; }

  void updatePrintMetrics();
  
  /*! \return paper width
  */
  int getPaperWidth() { return paperWidth; }
  /*! \return paper height
  */
  int getPaperHeight() { return paperHeight; }
  /*! \return paper width in mm
  */
  int getPaperWidthMM() { return paperWidthMM; }
  /*! \return paper height in mm
  */
  int getPaperHeightMM() { return paperHeightMM; }

  /*! \return Chosen page size
  */
  QPrinter::PageSize getPageSize();

  void activateDocument(RGraphic* _graphic);

  void clearRecentFiles();
  void addRecentFile(QString recentFile, bool updateMenu=true);
  QString getRecentFile(int _index);
  void updateRecentFileMenu();
  void updateAttribToolbar();

private:

  /*!  Test validity of a document
      \return true if the document with index _docIndex is valid
      \param _docIndex index of document to test
   */
  bool isDocumentValid(int _docIndex);
  
  /*!  Test validity of the output
      \return true if the output is valid
   */
  bool isOutputValid();

  /*!  Get current mode
      \return The current Mode 'n' for Construction, 'm' Comparison, 's' Simulation
   */
  char getMode() { return mode; }


  /** Gets the pixmap for a given edit-function
   */
  QPixmap getPixmapForEditFunction( int _fkt );

public slots:
  void show();
  void showIntro();
  void closeIntro();
  void changeMenu(int _num, bool _back=false);
  void backMenu();
  
  /*! Gets the current menu */
  int getCurrentMenu() { return currentMenu; }

  void enableAll(bool _enable);
  void enableMenus(bool _enable);
  void enableToolBars(bool _enable);

  void resizeChildren();

  /*! Synchronize Zoom of the graphics (from CAD to CAM view) */
  void synchronizeZoom() { synchronizeZoom(false); }
  void synchronizeZoom(bool _back);

  /*! Shows only left graphic view */
  void setConstructionMode() { setMode('n'); }

  /*! Shows left and right graphic view */
  void setComparisonMode() { setMode('m'); }

  /*! Shows only right graphic view */
  void setSimulationMode() { setMode('s'); }

  void setMode(char _mode);

  int  fileNew();
  int  fileNew(bool _load);
  int  fileClose() { return fileClose(true); }
  int  fileClose(bool _ask);
  void closeEvent(QCloseEvent* ce);
  int  fileCloseOutput();
  
  bool fileOpen() { return fileOpen(false); }
  bool fileOpen(const QString& _fileName) { return fileOpen(_fileName, false); }


  bool fileOpen(bool add);

  /*! 
      Opens a file by its name
      \param _fileName the name (with path) of the file to open
      \return true: successful / false: file does not exist
   */
  bool fileOpen(const QString& _fileName, bool add);
  
  /**
      Adds a file to the loaded graphic.
   */
  bool fileAdd() { return fileOpen(true); }
  
  /**
      Adds a file to the loaded graphic.
   */
  bool fileAdd(const QString& _fileName) { return fileOpen(_fileName, true); }

  /*! 
      Saves the current document as a file (DXF / CXD in future versions).
      Asks user for file name (file selection box)
   */
  bool fileSaveAs();

  /*!
      Saves the current document as a file (DXF / CXD in future versions).
      Only asks user for file name (file selection box) if there's no name
      available
   */
  bool fileSave();

  /*! 
      Saves the current document as a file (DXF / CXD in future versions).
      File name is given.
      \param _fileName the name of the file to save (with path)
   */
  bool fileSave(const QString& _fileName);

  /*! Shows printer setup dialog
   */
  void filePrintSetup();

  /*! Shows the print preview
   */
  void filePrintPreview();

  /*! Prints this file
   */
  void filePrint() { filePrint( true );  }
  
  /*! Prints this file
   */
  void filePrint( bool _setup );
  
  /*! Quits the application
   */
  void fileQuit();


  /*! 
      Opens a file which is in the recent list by its index
      \param _id identification number
   */
  void fileOpenRecent(int id);


  /*! User Action: Redraw graphic */
  void zoomRedraw();
  
  /*! User Action: Autozoom */
  void zoomAuto();
  
  /*! User Action: Autozoom */
  void zoomAuto(bool _update);

  /*! User Action: Zoom to page */
  void zoomToPage();

  /*! User Action: Increase Zoom by 1.5 */
  void zoomIncrease();

  /*! User Action: Decrease Zoom by 1.5 */
  void zoomDecrease();

  /*! 
      User Action: Zoom window
   */
  void zoomWindow();

  /*! 
      User Action: Panzoom
   */
  void zoomPan();

  /*! User Action: Zoom to all tagged elements
   */
  void zoomTagged();

  /*! User Action: Zoom 1:1
   */
  void zoomOne();

  /** Show/hide layer list
  */
  void viewLayerList();

  /** Show/hide library browser
  */
  void viewLibraryBrowser();

  /** Show/hide grid
  */
  void viewGrid();

  /**
   *  User Action: Show/hide grid
   */
  void showGrid(bool _on);

  /**
   *  User Action: Show/hide layer list
   */
  void showLayerList(bool _on);

  /**
   *  User Action: Show/hide object browser
   */
  void showLibraryBrowser(bool _on);
  
  /*!
      User Action: Create single points
   */
  void pointsSingle();

  /*! 
      User Action: Create normal lines
   */
  void linesNormal();

  /*!
      User Action: Create horizontal lines
   */
  void linesHor();

  /*!
      User Action: Create vertical lines
   */
  void linesVer();

  /*!
      User Action: Create lines with a given angle
   */
  void linesAngle();

  /*! 
      User Action: Create rectangles
   */
  void linesRect();

  /*! 
      User Action: Create Parallels
   */
  void linesPara();

  /*! 
      User Action: Create Bisectors
   */
  void linesBisector();

  /*! 
      User Action: Create Tangents (Point - Arc)
   */
  void linesTan1();

  /*! 
      User Action: Create Tangents (Arc - Arc)
   */
  void linesTan2();

  /*!
      User Action: Create Lines orthogonal to others
   */
  void linesOrtho();

  /*!
      User Action: Create Lines orthogonal with angle to others
   */
  void linesOrtho2();


  /*! 
      User Action: Create Arcs (Center, Radius, Angle1, Angle2)
   */
  void arcsCRAA();

  /*! 
      User Action: Create Arcs (Three Points)
   */
  void arcsPPP();

  /*! 
      User Action: Create Arcs (Point - Arc)
   */
  void arcsPara();


  /*! User Action: Create Circles (Center, Radius) */
  void circlesCR();

  /*! User Action: Create Circles (Center, Linepoint) */
  void circlesCL();

  /*! User Action: Create Circles (Three points) */
  void circles3P();


  /*! 
      User Action: Create Texts
   */
  void textsNormal();

  /*! User Action: Create Dimensions
   */
  void dimsPara();

  /*! User Action: Create horiz. Dimensions
   */
  void dimsHor();

  /*! User Action: Create vert. Dimensions
   */
  void dimsVer();

  /*! User Action: Create angle Dimensions
   */
  void dimsAngle();
  
  /*! User Action: Create Diameter Dimensions
   */
  void dimsDiam();
  
  /*! User Action: Create Radius Dimensions
   */
  void dimsRad();

  /*! User Action: Create Arrow Dimension
   */
  void dimsArrow();

  /*! User Action: Create Hatchings
   */
  void hatching();


  /*! User Action: Measure distance between 2 points
   */
  void measurePtPt();
  /*! User Action: Measure distance between a line an a point
   */
  void measureLnPt();
  /*! User Action: Measure distance between a line an a point
   */
  void measureParallelLines();
  /*! User Action: Measure distance between a line an a point
   */
  void measureAngleLnLn();
  /*! User Action: Edit the geometric data of an Element
   */
  void measureEditElement();
	/*! User Action: calculate area                                               
   */                                                                           
  void measureArea();  

  
  /*! User Action: Tag nothing
   */
  void tagNothing();
  
  /*! User Action: Tag all
   */
  void tagAll();
  
  /*! User Action: (Un-)tag single elements
   */
  void tagElement();
  
  /*! User Action: (Un-)tag contours
   */
  void tagContour();
  
  /*! User Action: Untag range
   */
  void untagRange();
  
  /*! User Action: Tag range
   */
  void tagRange();
  
  /*! User Action: Untag elements intersected by a line
   */
  void untagInters();
  
  /*! User Action: Tag elements intersected by a line
   */
  void tagInters();
  
  /*! User Action: Invert selection
   */
  void tagInvert();
  
  /*! User Action: Tag Layer
   */
  void tagLayer();
  
  /*! User Action: Tag double elements
   */
  void tagDoubles();
  
  /*! User Action: Tag enough
   */
  void tagEnough();
  
  
  /*! User Action: Edit: move
   */
  void editMove();
  
  /*! User Action: Edit: rotate
   */
  void editRotate();
  
  /*! User Action: Edit: scale 
   */
  void editScale();
  
  /*! User Action: Edit: mirror
   */
  void editMirror();
  
  /*! User Action: Edit: move/rotate
   */
  void editMoveRotate();
  
  /*! User Action: Edit: rotate/rotate
   */
  void editRotateRotate();
  
  /*! User Action: Edit: trim
   */
  void editTrim();
  
  /*! User Action: Edit: trim 2
   */
  void editTrim2();

  /*! User Action: Edit: trim 3
   */
  void editTrim3();
  
  /*! User Action: Edit: cut
   */
  void editCut();

  /*! User Action: Edit: cut2
   */
  void editCut2();

  /*! User Action: Edit: stretch
   */
  void editStretch();
  
  /*! User Action: Edit: round
   */
  void editRound();
  
  /*! User Action: Edit: bevel
   */
  void editBevel();
  
  /*! User Action: Edit: textelements
   */
  void editTextElements();
  
  /*! User Action: Edit: Text
   */
  void editText();
  
  /*! User Action: Edit: delete
   */
  void editDelete();
  
  /*! User Action: Edit: layer
   */
  void editLayer();
  
  /*! User Action: Edit: zero
   */
  void editZero();

  /*! User Action: Edit: attributes
   */
  void editAttrib();

  /*! User Action: Edit: clip
   */
  void editClip();

  /** Edit copy */
  void editClipCopy();

  /** Edit cut */
  void editClipCut();

  /** Edit paste */
  void editClipPaste();

  /** Edit insert part from library browser */
  void editInsertPart();

  /** Edit convert drawing to font */
  void editConvertToFont();
  
  /*! Undo */
  void undo();
  
  /*! Redo */
  void redo();


  /** Choose color */
  void chooseColor(int _col);

  /** Choose width */
  void chooseWidth(int _width);

  /** Choose style */
  void chooseStyle(int _style);


  /*! 
      User Action: Snap to nothing (free positioning)
   */
  void snapFree();

  /*! 
      User Action: Snap to grid points
   */
  void snapGrid();

  /*! 
      User Action: Snap to end points
   */
  void snapEndpoints();

  /*! 
      User Action: Snap to center points
   */
  void snapCenters();

  /*! 
      User Action: Snap to middle points (not center points!)
   */
  void snapMiddles();

  /*! 
      User Action: Snap to intersection points (auto)
   */
  void snapAutoInter();

  /*! 
      User Action: Snap to intersection points (manual, two elements given)
   */
  void snapManInter();

  /*! 
      User Action: Snap to distance points (point on an element with a given distance)
   */
  void snapDist();

  /*! 
      User Action: Snap to points on elements
   */
  void snapOnElement();

  /*!
      User Action: Keyboard input x/y
   */
  void snapKeyXY();

  /*!
      User Action: Keyboard input alpha/radius
   */
  void snapKeyAR();


  /*!  Untoggle all buttons in Snap menu except the one with index "_exception"
      \param _exception index of button which should not get untoggled
   */
  void untoggleSnapMenu(int _exception);


  /*!  Automatic creation of NC-Files
      User Action: 
   */
  void camAuto();

  /*!  Manual creation of NC-Files
      User Action: 
   */
  void camManual();

  /** Toggle simulation panel
  */
  void toggleSimulationPanel();


  /*! 
      User Action: Configure Machine Generator
   */
  void configMachineGenerator();

  /*! Options -> Preferences
  */
  void optionsPreferences();

  /*! Options -> Drawing
  */
  void optionsDrawing();

  /*! Debug -> countour check
  */
  void debugContourCheck();


  /*! 
      User Action: Show the "About"-Window
   */
  void helpAbout();

  /*! User Action: Help -> Manual */
  void helpManual();

  /*! User Action: Help -> Manual */
  void helpManual(const QString& _what );


  /*! Show Document with ID "_id"
      \param _id ID of document to show
   */
  void showDocument(int _id);

  /*! Show current Document (Document with ID "currentId")
      Used for activation of new or loaded Documents
      \sa hideDocuments(int _exception);
   */
  void showCurrentDocument();

  /*! Hide all documents except the one with ID "_exception"
      (all Documents which are not used currently)
      \param _exception ID of Document which should not get hided or -1 to hide ALL Documents
      \sa showCurrentDocument();
   */
  void hideDocuments(int _exception=-1);


  /*! Update Caption

   */
  void updateCaption();

  /*! Show the Main Menu

   */
  void showMainMenu()   { changeMenu(TOOLBAR_MAIN);   }


  /*! Show the Point Menu

   */
  void showPointMenu()  { changeMenu(TOOLBAR_POINT);  }

  /*! Show the Line Menu
   */
  void showLineMenu()   { changeMenu(TOOLBAR_LINE);   }

  /*! Show the Arc Menu
   */
  void showArcMenu()    { changeMenu(TOOLBAR_ARC);    }

  /*! Show the Circle Menu
   */
  void showCircleMenu() { changeMenu(TOOLBAR_CIRCLE); }

  /*! Show the Text Menu
   */
  void showTextMenu()   { changeMenu(TOOLBAR_TEXT);   }

  /*! Show the Dimension Menu
   */
  void showDimMenu()   { changeMenu(TOOLBAR_DIM);   }

  /*! Show the Measure Menu
   */
  void showMeasureMenu()   { changeMenu(TOOLBAR_MEASURE);   }

  /*! Show the Tag Menu
   */
  void showTagMenu();
  
  
  /*! Show the extended Tag Menu (for edit functions)
      This menu includes the button [Enough]
   */
  void showTag2Menu();


  /*! Show the Edit Menu
   */
  void showEditMenu()    { changeMenu(TOOLBAR_EDIT);    }


  /*! Show the Snap Menu
   */
  void showSnapMenu()   { changeMenu(TOOLBAR_SNAP);  }


  /*! Show the CAM Menu
   */
  void showCamMenu()    { changeMenu(TOOLBAR_CAM);    }


  /*! Show the Main Toolbar
   */
  void toolBarMainShow(bool _show=true);

  /*! Show the Point Toolbar
   */
  void toolBarPointShow(bool _show=true);

  /*! Show the Line Toolbar
   */
  void toolBarLineShow(bool _show=true);

  /*! Show the Arc Toolbar
   */
  void toolBarArcShow(bool _show=true);

  /*! Show the Circle Toolbar
   */
  void toolBarCircleShow(bool _show=true);

  /*! Show the Text Toolbar
   */
  void toolBarTextShow(bool _show=true);

  /*! Show the Dimension Toolbar
   */
  void toolBarDimShow(bool _show=true);

  /*! Show the Measure Toolbar
   */
  void toolBarMeasureShow(bool _show=true);

  /*! Show the Tag Toolbar
   */
  void toolBarTagShow(bool _show=true);
  
  /*! Show the extended Tag Toolbar
   */
  void toolBarTag2Show(bool _show=true);
  
  /*! Show the Edit Toolbar
   */
  void toolBarEditShow(bool _show=true);

  /*! Show the Snap Toolbar
   */
  void toolBarSnapShow(bool _show=true);

  /*! Show the CAM Toolbar
   */
  void toolBarCamShow(bool _show=true);
  
protected:

  /*! Create the Toolbars
   */
  void createToolBars();

  /*! Create the Toolbars for CAM Expert
   */
  void createToolBarsCamExpert();

  /*! Create the Toolbars for Text Expert
   */
  void createToolBarsTextExpert();
  
  /*! Hide the Toolbar on the left
   */
  void hideToolBarsLeft();

  /*! Hide the Toolbar at the top
   */
  void hideToolBarsTop();


  /*! Create Menus
   */
  void createMenus();

  /*! Event: Resize
   */
  void resizeEvent(QResizeEvent*);

  /*! Event Mouse Move/Click
   */
  void moveEvent(QMoveEvent*);

  /*! Event Mouse Release
   */
  void mouseReleaseEvent(QMouseEvent* _ev);

  /** Events for keys */
  virtual void keyPressEvent( QKeyEvent * _ev );
  virtual void keyReleaseEvent (QKeyEvent* _ev);


  /*! Clear all Documents
   */
  void clearAllDocuments();

  /*! Get free Document Index (First Index which is available)
      Create a new document in list 'graphicDoc' sets curDocInd to new index
      \return  index of new graphic, which is initialized by a new graphic<br>
               -1 if no free document was found
   */
  int  getFreeDocumentIndex();

  /*! Get a new Document Index
      Create a new document in list 'graphicDoc'
      sets curDocInd to new index
      \return  index of new graphic, which is initialized by a new graphic<br>
               -1 if no free document was found
   */
  int  getNewDocumentIndex();

public:

  /*! Cursor Magnifying Glass
  */
  QCursor*      curGlass;         // mouse cursor glass
  /*! Cursor Cross
  */
  QCursor*      curCross;         // mouse cursor cross
  
private:
  /*! Intro widget
  */
  RIntro*       intro;
  /*! Printer Object
  */
  QPrinter*     printer;
  
  /*! Print metrics
  */
  QPaintDeviceMetrics* printMetrics;
  int paperWidth, paperHeight, paperWidthMM, paperHeightMM;
	
	/*! Shows if the user has initialized the printer <code>(=true)</code>
	*/
	bool printerInitialized;

  /*! Popup Menu for file handling
  */
  QPopupMenu*   fileMenu;
  /*! Popup Menu for editing
  */
  QPopupMenu*   editMenu;
  /*! Popup Menu Zoom (Zoom functions)
  */
  QPopupMenu*   zoomMenu;
  /*! Popup menu View
  */
  QPopupMenu*   viewMenu;
  /*! Popup Menu Snap (Snap functions)
  */
  QPopupMenu*   snapMenu;
  /*! Popup Menu CAM (CAM functions)
  */
  QPopupMenu*   camMenu;
  /*! Popup Menu Window (windows)
  */
  QPopupMenu*   windowMenu;
  /*! Popup Menu Options
  */
  QPopupMenu*   optionsMenu;
  /*! Popup menu Debug
  */
  QPopupMenu*   debugMenu;
  /*! Popup Menu Help
  */
  QPopupMenu*   helpMenu;

  /** The internal clipboard */
  RGraphic* clipboard;


  /*! Mode:<br>
      'm' : Comparison <br>
      'n' : Construction <br>
      's' : Simulation
   */
  char          mode;

  /*! Splitter (split window into two parts)
  */
  RSplitter*    splitter;

  /*! Graphic widget for left side
  */
  RCadWidget*   cadWidget1;

  /*! The graphic documents
  */
  RGraphic**    graphicDoc;

  /*! Number of current document
  */
  int           curDocInd;

  /*! Counter for new docs ("Document 1")
  */
  int           docNumber;

  /*! Graphics Widget for right side
  */
  RCadWidget*   cadWidget2;

  /*! The graphic which was outputed
  */
  RGraphic*     graphicOut;

  /*! Simulation Panel for both sides:
  */
  RSimulationPanel* simPanel;

  /*! Input Panel / the widget for user inputs
  */
  RInputPanel*  inputPanel;

  /*! Input Panel 2 / the widget for user inputs for snap functions
  */
  RInputPanel*  inputPanel2;

  /*! dimensionDlg: Lets the user set the Dimensiontext manually
  */
  RDimDlg *dimensionDlg;

  /*! Status Panel / Widget for Status viewing
  */
  RStatusPanel*  statusPanel;

  /*! Tool-Menu (left) / Tool bar left
  */
  RWidget*      fToolBar;

    /** Tool button back
    */
    QToolButton*  bBack;
    /*! Tool button submenu points
    */
    QToolButton*  bSubmenuPoints;
    /*! Tool button submenu points
    */
    QToolButton*  bSubmenuLines;
    /*! Tool button submenu arcs
    */
    QToolButton*  bSubmenuArcs;
    /*! Tool button submenu arcs
    */
    QToolButton*  bSubmenuCircles;
    /*! Tool button submenu texts
    */
    QToolButton*  bSubmenuTexts;
    /*! Tool button submenu dimension
    */
    QToolButton*  bSubmenuDims;
    /*! Tool button submenu measure
    */
    QToolButton*  bSubmenuMeasure;
    /*! Tool button submenu hatchings
    */
    QToolButton*  bSubmenuHatch;
    /*! Tool button submenu tag
    */
    QToolButton*  bSubmenuTag;
    /*! Tool button submenu edit
    */
    QToolButton*  bSubmenuEdit;
    /*! Tool button submenu cam
    */
    QToolButton*  bSubmenuCam;


    /*! Tool button cam automatic
    */
    QToolButton*  bCamAuto;
    /*! Tool button cam manual
    */
    QToolButton*  bCamManual;

    /*! Create single points
    */
    QToolButton*  bPointsSingle;

    /*! Create single lines
    */
    QToolButton*  bLinesNormal;

    /*! Create horizontal lines
    */
    QToolButton*  bLinesHor;

    /*! Create vertical lines
    */
    QToolButton*  bLinesVer;

    /*! Create lines with a given angle
    */
    QToolButton*  bLinesAngle;

    /*! Create rectangles
    */
    QToolButton*  bLinesRect;
    /*! Create parallels
    */
    QToolButton*  bLinesPara;
    /*! Create bisectors
    */
    QToolButton*  bLinesBisector;
    /*! Create tangents P-C
    */
    QToolButton*  bLinesTan1;
    /*! Create tangents C-C
    */
    QToolButton*  bLinesTan2;
    /*! Create ortho lines
    */
    QToolButton*  bLinesOrtho;
    /*! Create ortho lines with angle
    */
    QToolButton*  bLinesOrtho2;

    /*! Create arcs with center, radius, angles
    */
    QToolButton*  bArcsCRAA;
    /*! Create arcs with three points
    */
    QToolButton*  bArcsPPP;
    /*! Create arcs parallel to others
    */
    QToolButton*  bArcsPara;

    /*! Create circles with center, radius */
    QToolButton*  bCirclesCR;
    /*! Create circles with center, linepoint */
    QToolButton*  bCirclesCL;
    /*! Create circles with 3 points */
    QToolButton*  bCircles3P;

    /*! Create parallel dimensions
    */
    QToolButton*  bDimsPara;
    /*! Create horizontal dimensions
    */
    QToolButton*  bDimsHor;
    /*! Create vertical dimensions
    */
    QToolButton*  bDimsVer;
    /*! Create circular dimensions
    */
    QToolButton*  bDimsAngle;
    /*! Create diameter dimensions
    */
    QToolButton*  bDimsDiam;
    /*! Create radius dimensions
    */
    QToolButton*  bDimsRad;
    /*! Create arrow dimensions
    */
    QToolButton*  bDimsArrow;

    /*! Measure Point-Point distance
    */
    QToolButton*  bMeasurePtPt;
    /*! Measure Line-Point distance
    */
    QToolButton*  bMeasureLnPt;
    /*! Measure Line-Point distance
    */
    QToolButton*  bMeasureParallelLines;
    /*! Measure Line-Point distance
    */
    QToolButton*  bMeasureAngleLnLn;
    /*! Edit the geometric data of an element
    */
    QToolButton*  bMeasureEditElement;
		/*! Measure the area defined throup 3 or more points                        
    */                                                                          
    QToolButton*  bMeasureArea;

    /*! Snap nothing (free)
    */
    QToolButton*  bSnapFree;
    /*! Snap to grid
    */
    QToolButton*  bSnapGrid;
    /*! Snap to endpoints
    */
    QToolButton*  bSnapEndpoints;
    /*! Snap to centers
    */
    QToolButton*  bSnapCenters;
    /*! Snap to middles
    */
    QToolButton*  bSnapMiddles;
    /*! Snap to auto intersections
    */
    QToolButton*  bSnapAutoInter;
    /*! Snap to manual intersections
    */
    QToolButton*  bSnapManInter;
    /*! Snap to points with distance on element
    */
    QToolButton*  bSnapDist;
    /*! Snap to points on elements
    */
    QToolButton*  bSnapOnElement;
    /*! Snap Keyboard X/Y
    */
    QToolButton*  bSnapKeyXY;
    /*! Snap Keyboard Alpha/Radius
    */
    QToolButton*  bSnapKeyAR;
    
    /*! Tag all
    */

    QToolButton*  bTagAll;
    /*! Tag nothing
    */
    QToolButton*  bTagNothing;
    /*! Tag element
    */
    QToolButton*  bTagElement;
    /*! Tag contour
    */
    QToolButton*  bTagContour;
    /*! Untag range
    */
    QToolButton*  bUntagRange;
    /*! Tag range
    */
    QToolButton*  bTagRange;
    /*! Untag elements intersected by a line
    */
    QToolButton*  bUntagInters;
    /*! Tag elements intersected by a line
    */
    QToolButton*  bTagInters;
    /*! Invert tagged elements
    */
    QToolButton*  bTagInvert;
    /*! Tag layer
    */
    QToolButton*  bTagLayer;
    /*! Tag double elements
    */
    QToolButton*  bTagDoubles;
    /*! Tag enough (extended)
    */
    QToolButton*  bTagEnough;
    
    
    /*! Edit: move
    */
    QToolButton*  bEditMove;
  
    /*! Edit: rotate
     */
    QToolButton*  bEditRotate;
    
    /*! Edit: scale 
     */
    QToolButton*  bEditScale;
    
    /*! Edit: mirror
     */
    QToolButton*  bEditMirror;
    
    /*! Edit: move/rotate
     */
    QToolButton*  bEditMoveRotate;
    
    /*! Edit: rotate/rotate
     */
    QToolButton*  bEditRotateRotate;
    
    /*! Edit: trim
     */
    QToolButton*  bEditTrim;
    
    /*! Edit: trim 2
     */
    QToolButton*  bEditTrim2;

    /*! Edit: trim 3
     */
    QToolButton*  bEditTrim3;
    
    /*! Edit: cut
     */
    QToolButton*  bEditCut;

    /*! Edit: cut2
     */
    QToolButton*  bEditCut2;

    /*! Edit: stretch
     */
    QToolButton*  bEditStretch;
    
    /*! Edit: round
     */
    QToolButton*  bEditRound;
    
    /*! Edit: bevel
     */
    QToolButton*  bEditBevel;
    
    /*! Edit: textelements
     */
    QToolButton*  bEditTextElements;
    
    /*! Edit: Text
     */
    QToolButton*  bEditText;
    
    /*! Edit: delete
     */
    QToolButton*  bEditDelete;
    
    /*! Edit: layer
     */
    QToolButton*  bEditLayer;
    
    /*! Edit: zero
     */
    QToolButton*  bEditZero;

    /*! Edit: attributes
     */
    QToolButton*  bEditAttrib;

    /*! Edit: clip
     */
    QToolButton*  bEditClip;
    
    
  /*! Tool bar file  (top)
  */
  QToolBar*     toolBarFile;
    /*! tool button file new
    */
    QToolButton*  bFileNew;
    /*! tool button file load
    */
    QToolButton*  bFileOpen;
    /*! tool button file close
    */
    QToolButton*  bFileClose;
    /*! tool button file output close
    */
    QToolButton*  bFileCloseOutput;
    /*! tool button file save
    */
    QToolButton*  bFileSave;
    /*! tool button file print preview
    */
    QToolButton*  bFilePrintPreview;
    /*! tool button file print
    */
    QToolButton*  bFilePrint;

  /*! tool bar zoom  (top)
  */
  QToolBar*     toolBarZoom;
    /*! tool button auto zoom */
    QToolButton*  bZoomAuto;
    /*! tool button zoom to page */
    QToolButton*  bZoomToPage;
    /*! tool button redraw
    */
    QToolButton*  bZoomRedraw;
    /*! tool button increase zoom
    */
    QToolButton*  bZoomIncrease;
    /*! tool button decrease zoom
    */
    QToolButton*  bZoomDecrease;
    /*! tool button zoom window
    */
    QToolButton*  bZoomWindow;
    /*! tool button pan zoom
    */
    QToolButton*  bZoomPan;
    /*! tool button zoom tagged
    */
    QToolButton*  bZoomTagged;
    /*! tool button zoom 1:1
    */
    QToolButton*  bZoomOne;
  
 /*! tool bar undo  (top)
 */
  QToolBar*     toolBarUndo;
    /** Edit Copy  */
    QToolButton* bClipCopy;
    /** Edit Cut  */
    QToolButton* bClipCut;
    /** Edit Paste  */
    QToolButton* bClipPaste;
    /** Undo-Button */
    QToolButton* bUndo;
    /** Redo-Button */
    QToolButton* bRedo;
    
 /*! tool bar show (top)
 */
  QToolBar*     toolBarShow;
    /*! Show grid
    */
    QToolButton* bShowGrid;
    /*! Show layer list
    */
    QToolButton* bShowLayerList;
    /*! Show object browser
    */
    QToolButton* bShowLibraryBrowser;

 /*! tool bar attributes (top)
 */
  QToolBar*     toolBarAttrib;
    /*! Attribute color
    */
    RComboBox* cbAttribColor;
    /*! Attribute width
    */
    RComboBox* cbAttribWidth;
    /*! Attribute style
    */
    RComboBox* cbAttribStyle;

  /*! ID of current menu
  */
  int           currentMenu;
  
  /*! ID's of last menus
  */
  int           lastMenu[DEF_MENUBUF];
  
  /*! Index of current last menu
  */
  int           ilastMenu;

  /** List of recent files. */
  QList<QString> recentFiles;

protected:
  /*! Pointer to the one and only Application Window
  */
  static RAppWin* theOneAndOnlyRAppWin;
  
  /*! Central widget
  */
  RWidget* centralWidget;

  /** Key down?
  */
  bool keyDown;
};


#endif

// EOF
