/*
 * Brian Carrier [carrier@sleuthkit.org]
 * Copyright (c) 2003 Brian Carrier.  All rights reserved
 *
 * mm_part - functions to sort generic partition structures
 *
 * mmtools is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * mmtools is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with mactime; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>

#include "mm_tools.h"
#include "mymalloc.h"

/* Add a partition to a sorted list 
 *
 * the structure is returned
 */
MM_PART *
mm_part_add(MM_INFO *mm, daddr_t start, daddr_t len, char *desc, 
		int8_t table, int8_t slot)
{
	MM_PART *part = (MM_PART *)mymalloc(sizeof(MM_PART));
	MM_PART *cur_part = mm->part_list;	

	/* set the values */
	part->next = NULL;
	part->prev = NULL;
	part->start = start;
	part->len = len;
	part->desc = desc;
	part->table_num = table;
	part->slot_num = slot;

	/* is this the first entry in the list */
	if (mm->part_list == NULL) {
		mm->part_list = part;
		mm->first_part = 0;
		mm->last_part = 0;

		return part;
	}

	/* Cycle through to find the correct place to put it into */
	while (cur_part) { 

		/* The one to add starts before this partition */
		if (cur_part->start > part->start) {
			part->next = cur_part;
			part->prev = cur_part->prev;
			if (part->prev)
				part->prev->next = part;
			cur_part->prev = part;

			/* If the current one was the head, set this to the head */
			if (cur_part == mm->part_list)
				mm->part_list = part;

			mm->last_part++;
			break;
		}

		/* the one to add is bigger then current and the list is done */
		else if (cur_part->next == NULL) {
			cur_part->next = part;
			part->prev = cur_part;

			mm->last_part++;
			break;
		}

		/* The one to add fits in between this and the next */
		else if (cur_part->next->start > part->start) {
			part->prev = cur_part;
			part->next = cur_part->next;
			cur_part->next->prev = part;
			cur_part->next = part;

			mm->last_part++;
			break;
		}

		cur_part = cur_part->next;
	}

	return part;
}

/* 
 * cycle through the sorted list and add unallocated entries
 * to the unallocated areas of disk
 */
void
mm_part_unused(MM_INFO *mm)
{
	MM_PART *part = mm->part_list;
	daddr_t prev_end = 0;

	/* prev_ent is set to where the previous entry stopped  plus 1*/
	while (part) {

		if (part->start > prev_end) {
			char *str = mymalloc(12);
			snprintf(str, 12, "Unallocated");
			mm_part_add(mm, prev_end, part->start - prev_end, 
			  str, -1, -1);
		}

		prev_end = part->start + part->len;
		part = part->next;
	}

	return;
}

/* 
 * free the buffer with the description 
 */
void
mm_part_free(MM_INFO *mm)
{
	MM_PART *part = mm->part_list;

	while (part) {
		if (part->desc)
			free(part->desc);
		part = part->next;
	}

	return;
}
