/***************************************************************************
                          strat_map.c  -  description
                             -------------------
    begin                : Fri Jul 27 2001
    copyright            : (C) 2001 by Michael Speck
    email                : kulkanie@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <SDL.h>

#ifdef WITH_SOUND
#include <SDL_mixer.h>
#include "audio.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include "tools.h"
#include "sdl.h"
#include "dynlist.h"
#include "gui.h"
#include "theme.h"
#include "config.h"
#include "nation.h"
#include "unit.h"
#include "ai_action.h"
#include "player.h"
#include "date.h"
#include "map.h"
#include "scenario.h"
#include "engine_tools.h"
#include "slot.h"
#include "engine.h"
#include "window.h"
#include "strat_map.h"

extern Sdl sdl;
extern Map map;
extern Scen scen;

/*
====================================================================
Strategic map data.
====================================================================
*/
int screen_x, screen_y; /* position on screen */
int width, height; /* size of map picture */
int x_offset, y_offset; /* offset for first strat map tile 0,0 in strat_map */
int strat_tile_width;
int strat_tile_height; /* size of a shrinked map tile */
int strat_tile_x_offset;
int strat_tile_y_offset; /* offsets that wil be add to one's position for the next tile */
SDL_Surface *strat_map = 0; /* picture assembled by create_strat_map() */
SDL_Surface *fog_layer = 0; /* used to buffer fog before addind to strat map */
SDL_Surface *unit_layer = 0; /* layer with unit flags indicating there position */
SDL_Surface *flag_layer = 0; /* layer with all flags; white frame means normal; gold frame means objective */
int tile_count; /* equals map::def::tile_count */
SDL_Surface **strat_tile_pic = 0; /* shrinked normal map tiles; created in strat_map_init() */
SDL_Surface *strat_flags = 0; /* resized flag picture containing the nation flags */
int strat_flag_width, strat_flag_height; /* size of resized strat flag */

/*
====================================================================
LOCALS
====================================================================
*/

/*
====================================================================
Get size of strategic map depending on scale where
scale == 1: normal size
scale == 2: size / 2
scale == 3: size / 3
...
====================================================================
*/
int scaled_strat_map_width( int scale )
{
    return ( ( map.width - 1 ) * map.def->tile_x_offset / scale );
}
int scaled_strat_map_height( int scale )
{
    return ( ( map.height - 1 ) * map.def->tile_height / scale );
}

/*
====================================================================
Updates the picture offset for the strategic map in all map tiles.
====================================================================
*/
void update_strat_pic_offset()
{
    int x, y;
    for ( x = 0; x < map.width; x++ )
        for ( y = 0;  y < map.height; y++ )
            map_tile( x, y )->strat_pic_offset =   strat_tile_width
                                                 * map_tile( x, y )->pic_offset
                                                 / map.def->tile_width;
}

/*
====================================================================
PUBLIC
====================================================================
*/

/*
====================================================================
Update the bitmap containing the strategic map.
====================================================================
*/
void strat_map_update( Engine *engine, int flags )
{
    int i, j;
    Unit *unit;

    /* clear screen */
    DEST( sdl.screen, 0, engine->upper_bar->height, sdl.screen->w,
          sdl.screen->h - engine->upper_bar->height - engine->lower_bar->height );
    fill_surf( 0x0 );

    /* delete old pictures */
    if ( flags & RECREATE_STRAT_MAP ) {
        FREE( strat_map );
        FREE( fog_layer );
    }
    if ( flags & RECREATE_UNIT_LAYER )
        FREE( unit_layer );
    if ( flags & RECREATE_FLAG_LAYER )
        FREE( flag_layer );

    /* measures */
    width = ( map.width - 1 ) * strat_tile_x_offset;
    x_offset = -strat_tile_width + strat_tile_x_offset;
    height = ( map.height - 1 ) * strat_tile_height;
    y_offset = -strat_tile_height / 2;

    /* create strat_map & unit layer */
    if ( flags & RECREATE_STRAT_MAP ) {
        strat_map = create_surf( width, height, SDL_SWSURFACE );
        FULL_DEST( strat_map );
        fill_surf( 0x0 );
        fog_layer = create_surf( width, height, SDL_SWSURFACE );
        FULL_DEST( fog_layer );
        fill_surf( 0x0 );
    }
    if ( flags & RECREATE_UNIT_LAYER ) {
        unit_layer = create_surf( width, height, SDL_SWSURFACE );
        FULL_DEST( unit_layer );
        fill_surf( 0x0 );
    }
    if ( flags & RECREATE_FLAG_LAYER ) {
        flag_layer = create_surf( width, height, SDL_SWSURFACE );
        FULL_DEST( flag_layer );
        fill_surf( 0x0 );
    }

    /* create map */
    if ( flags & RECREATE_STRAT_MAP ) {
        /* first gather all fogged tiles without alpha */
        for ( j = 0; j < map.height; j++ )
            for ( i = 0; i < map.width; i++ )
                if ( ( engine->status == DEPLOYING_UNITS && !mask_tile( i, j )->deploy ) || ( engine->status != DEPLOYING_UNITS && mask_tile( i, j )->fog ) ) {
                    DEST( fog_layer,
                          x_offset + i * strat_tile_x_offset,
                          y_offset + j * strat_tile_height + ( i & 1 ) * strat_tile_y_offset,
                          strat_tile_width, strat_tile_height );
                    SOURCE( strat_tile_pic[map_tile( i, j )->prop_id],
                            map_tile( i, j )->strat_pic_offset, 0 );
                    blit_surf();
                }
        /* now add this fog map with transparency to strat_map */
        FULL_DEST( strat_map );
        FULL_SOURCE( fog_layer );
        alpha_blit_surf( 255 - FOG_ALPHA );
        /* now add unfogged map tiles */
        for ( j = 0; j < map.height; j++ )
            for ( i = 0; i < map.width; i++ )
                if ( ( engine->status == DEPLOYING_UNITS && mask_tile( i, j )->deploy ) || ( engine->status != DEPLOYING_UNITS && !mask_tile( i, j )->fog ) ) {
                    DEST( strat_map,
                          x_offset + i * strat_tile_x_offset,
                          y_offset + j * strat_tile_height + ( i & 1 ) * strat_tile_y_offset,
                          strat_tile_width, strat_tile_height );
                    SOURCE( strat_tile_pic[map_tile( i, j )->prop_id],
                            map_tile( i, j )->strat_pic_offset, 0 );
                    blit_surf();
                }
    }

    /* draw tiles; fogged tiles are copied transparent with the inverse of FOG_ALPHA */
    for ( j = 0; j < map.height; j++ )
        for ( i = 0; i < map.width; i++ ) {
            /* flag layer */
            if ( flags & RECREATE_FLAG_LAYER )
                if ( map_tile( i, j )->nation_id != -1 ) {
                    /* draw frame and flag only if objective */
                    if ( map_tile( i, j )->obj ) {
                        DEST( flag_layer,
                              x_offset + i * strat_tile_x_offset,
                              y_offset + j * strat_tile_height + ( i & 1 ) * strat_tile_y_offset,
                              strat_tile_width, strat_tile_height );
                        fill_surf( 0xffff00 );
                        /* add flag */
                        DEST( flag_layer,
                              x_offset + i * strat_tile_x_offset + 1,
                              y_offset + j * strat_tile_height + ( i & 1 ) * strat_tile_y_offset + 1,
                              strat_flag_width, strat_flag_height );
                        SOURCE( strat_flags, 0, map_tile( i, j )->nation_id * strat_flag_height );
                        blit_surf();
                    }
                }
            /* unit layer */
            if ( flags & RECREATE_UNIT_LAYER ) {
                unit = 0;
                if ( engine->air_mode && map_tile( i, j )->air_unit )
                    unit = map_tile( i, j )->air_unit;
                if ( !engine->air_mode && map_tile( i, j )->unit )
                    unit = map_tile( i, j )->unit;
                if ( mask_tile( i, j )->fog ) unit = 0;
                if ( unit ) {
                    /* nation flag has a small frame of one pixel */
                    DEST( unit_layer,
                          x_offset + i * strat_tile_x_offset + 1,
                          y_offset + j * strat_tile_height + ( i & 1 ) * strat_tile_y_offset + 1,
                          strat_flag_width, strat_flag_height );
                    SOURCE( strat_flags, 0, unit->sel_prop->nation * strat_flag_height );
                    blit_surf();
                }
            }
        }
    /* center */
    screen_x = ( sdl.screen->w - width ) / 2;
    screen_y = ( sdl.screen->h - height - engine->lower_bar->height - engine->upper_bar->height ) / 2 + engine->upper_bar->height;
    /* draw strat map */
    DEST( sdl.screen, screen_x, screen_y, width, height );
    SOURCE( strat_map, 0, 0 );
    blit_surf();
    /* add flag layer */
    DEST( sdl.screen, screen_x, screen_y, width, height );
    SOURCE( flag_layer, 0, 0 );
    blit_surf();
    /* add unit layer */
    DEST( sdl.screen, screen_x, screen_y, width, height );
    SOURCE( unit_layer, 0, 0 );
    blit_surf();
    refresh_screen( 0, engine->upper_bar->height, sdl.screen->w,
                    sdl.screen->h - engine->upper_bar->height - engine->lower_bar->height );
}

/*
====================================================================
Is called after scenario was loaded in init_engine() and creates
the strategic map tile pictures, flags and the strat_map.
====================================================================
*/
void strat_map_init( Engine *engine )
{
    Map_Def *def = map.def;
    Uint32 ckey;
    int i, j, x, y;
    int strat_tile_count;
    Uint32 pixel;
    int hori_scale, vert_scale;
    int scale;

    /* scale normal geometry so it fits the screen */
    /* try horizontal */
    hori_scale = 1;
    while( scaled_strat_map_width( hori_scale ) > sdl.screen->w ) hori_scale++;
    vert_scale = 1;
    while( scaled_strat_map_height( vert_scale ) > sdl.screen->h - engine->upper_bar->height - engine->lower_bar->height )
        vert_scale++;
    /* use greater scale */
    if ( hori_scale > vert_scale )
        scale = hori_scale;
    else
        scale = vert_scale;
    /* get strat map tile size */
    strat_tile_width = def->tile_width / scale;
    strat_tile_height = def->tile_height / scale;
    strat_tile_x_offset = def->tile_x_offset / scale;
    strat_tile_y_offset = def->tile_y_offset / scale;

    /* create strat tile array */
    tile_count = def->tile_count;
    strat_tile_pic = calloc( tile_count, sizeof( SDL_Surface* ) );

    /* create strat tiles */
    for ( i = 0; i < tile_count; i++ ) {
        /* how many tiles are rowed? */
        strat_tile_count = def->tiles[i]->pic->w / def->tile_width;
        /* create strat_pic */
        strat_tile_pic[i] = create_surf( strat_tile_count * strat_tile_width,
                                         strat_tile_height,
                                         SDL_SWSURFACE );
        /* clear to color key */
        ckey = get_pixel( def->tiles[i]->pic, 0, 0 );
        FULL_DEST( strat_tile_pic[i] );
        fill_surf( ckey );
        SDL_SetColorKey( strat_tile_pic[i], SDL_SRCCOLORKEY, ckey );
        /* copy pixels from pic to strat_pic if strat_fog is none transparent */
        for ( j = 0; j < strat_tile_count; j++ )
            for ( x = 0; x < strat_tile_width; x++ )
                for ( y = 0; y < strat_tile_height; y++ ) {
                    /* we have the pixel in strat_pic by x + j * strat_fog_pic->w,y */
                    /* no we need the aquivalent pixel in tiles[i]->pic to copy it */
                    pixel = get_pixel( def->tiles[i]->pic,
                                       j * def->tile_width +
                                       def->tile_width * x / strat_tile_width,
                                       def->tile_height * y / strat_tile_height );
                    set_pixel( strat_tile_pic[i], j * strat_tile_width + x, y, pixel );
                }
    }

    /* update strat picture offset in all map tiles */
    update_strat_pic_offset();

    /* resized nation flags */
    strat_flag_width = strat_tile_width - 2;
    strat_flag_height = strat_tile_height - 2;
    strat_flags = create_surf( strat_flag_width, strat_flag_height * scen.nation_count, SDL_SWSURFACE );
    /* scale down */
    for ( i = 0; i < scen.nation_count; i++ )
        for ( x = 0; x < strat_flag_width; x++ )
            for ( y = 0; y < strat_flag_height; y++ ) {
                pixel = get_pixel( scen.nations[i]->flag_pic,
                                   scen.nations[i]->flag_pic->w * x / strat_flag_width,
                                   scen.nations[i]->flag_pic->h * y / strat_flag_height );
                set_pixel( strat_flags, x, i * strat_flag_height + y, pixel );
            }
}

/*
====================================================================
Clean up stuff that was allocated by strat_map_init()
====================================================================
*/
void strat_map_clear()
{
    int i;
    if ( strat_tile_pic ) {
        for ( i = 0; i < tile_count; i++ )
            if ( strat_tile_pic[i] ) SDL_FreeSurface( strat_tile_pic[i] );
        free( strat_tile_pic );
        strat_tile_pic = 0;
    }
    FREE( strat_flags );
    FREE( strat_map );
    FREE( fog_layer );
    FREE( unit_layer );
    FREE( flag_layer );
}

/*
====================================================================
Handle SDL input. If mouse pointer is on a map tile x,y is returned
and true.
====================================================================
*/
int get_strat_map_pos( int mouse_x, int mouse_y, int *map_x, int *map_y )
{
    int x = 0, y = 0;
    int tile_x_offset, tile_y_offset;
    int tile_x, tile_y;
    Uint32 ckey;

    /* check if outside of map */
    if ( mouse_x < screen_x || mouse_y < screen_y )
        return 0;
    if ( mouse_x >= screen_x + width || mouse_y >= screen_y + height )
        return 0;

    /* clip mouse pos to mask */
    mouse_x -= screen_x;
    mouse_y -= screen_y;

    /* get the map offset in screen from mouse position */
    x = ( mouse_x - x_offset ) / strat_tile_x_offset;
    if ( x & 1 )
        y = ( mouse_y - ( strat_tile_height >> 1 ) - y_offset ) / strat_tile_height;
    else
        y = ( mouse_y - y_offset ) / strat_tile_height;

    /* get the starting position of this strat map tile at x,y in surface strat_map */
    tile_x_offset = x_offset;
    tile_y_offset = y_offset;
    tile_x_offset += x * strat_tile_x_offset;
    tile_y_offset += y * strat_tile_height;
    if ( x & 1 )
        tile_y_offset += strat_tile_height >> 1;

    /* now test where exactly the mouse pointer is in this tile using
       strat_tile_pic[0] */
    tile_x = mouse_x - tile_x_offset;
    tile_y = mouse_y - tile_y_offset;
    ckey = get_pixel( strat_tile_pic[0], 0, 0 );
    if ( get_pixel( strat_tile_pic[0], tile_x, tile_y ) == ckey ) {

        if ( tile_y < strat_tile_y_offset && !( x & 1 ) ) y--;
        if ( tile_y >= strat_tile_y_offset && (x & 1 ) ) y++;
        x--;

    }

    /* assign */
    *map_x = x;
    *map_y = y;

    return 1;
}

