/***************************************************************************
                          map.c  -  description
                             -------------------
    begin                : Mon Jan 22 2001
    copyright            : (C) 2001 by Michael Speck
    email                : kulkanie@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <SDL.h>

#ifdef WITH_SOUND
#include <SDL_mixer.h>
#include "audio.h"
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "sdl.h"
#include "dynlist.h"
#include "tools.h"
#include "file.h"
#include "nation.h"
#include "unit.h"
#include "map.h"

extern Sdl sdl;

Map map = { 0, 0, 0, 0 };

/* --- forwarded --- */
FILE* open_map_file( char *file_name );
Map_Def* read_map_def( FILE *file );
void delete_map_def( Map_Def *map_def );
Map* create_map( int width, int height );
void cache_fogged_map_tiles( Map_Def *map_def, int fog_alpha );
void create_tiles();
void delete_tile_def( Map_Tile_Prop *tile_def );

int read_map( char *map_name )
{
    char name[256];
    char *cur_arg = 0;
    char **args = 0;
    int arg_count = 0;
    int i = 0, j = 0, x = 0, y = 0;
    int limit;
    FILE *file = 0, *def_file = 0;

    /* clear struct */
    clear_map( );

    /* open map file */
    file = open_map_file( map_name );
    if ( file == 0 ) goto failure;

    /* read definitions */
    if ( ( cur_arg = get_arg( file, "map_tile_lib", RESET_FILE_POS ) ) == 0 ) goto failure;
    def_file = open_map_file( cur_arg );
    FREE( cur_arg );
    if ( !def_file ) goto failure;
    map.def = read_map_def( def_file );
    fclose( def_file );
    if ( map.def == 0 ) goto failure;

    /* read  width */
    if ( ( cur_arg = get_arg( file, "map_width", RESET_FILE_POS ) ) == 0 ) goto failure;
    map.width = atoi( cur_arg );
    FREE( cur_arg );
    /* read  width */
    if ( ( cur_arg = get_arg( file, "map_height", RESET_FILE_POS ) ) == 0 ) goto failure;
    map.height = atoi( cur_arg );
    FREE( cur_arg );

    /* create tiles */
    create_tiles();

    /* read map */
    if ( ( args = get_arg_cluster( file, "map", &arg_count, RESET_FILE_POS, WARNING ) ) == 0 ) goto failure;
    if ( arg_count != map.width * map.height ) {
        printf("the number of defined map tiles does not match the map its size...\n");
        delete_arg_cluster( args, arg_count );
        goto failure;
    }
    for ( i = 0; i < arg_count; i++ ) {

        x = i % map.width;
        y = i / map.width;

        cur_arg = args[i];

        /* default is no flag */
        map.tiles[x][y].nation_id = map.tiles[x][y].player_id = -1;
        /* default is no mil target */
        map.tiles[x][y].obj = 0;

        /* check tile type */
        for ( j = 0; j < map.def->tile_count; j++ ) {
            if ( map.def->tiles[j]->char_id == cur_arg[0] ) {
                map.tiles[x][y].prop = map.def->tiles[j];
                map_tile( x, y )->prop_id = j;
            }
        }
        /* tile not found, used first one */
        if ( map.tiles[x][y].prop == 0 )
            map.tiles[x][y].prop = map.def->tiles[0];

        /* check picture id -- set offset */
        limit = map.tiles[x][y].prop->pic->w / map.def->tile_width - 1;
        if ( cur_arg[1] == '?' )
            /* set offset by random */
            map.tiles[x][y].pic_offset = RANDOM( 0, limit ) * map.def->tile_width;
        else
            map.tiles[x][y].pic_offset = atoi( cur_arg + 1 ) * map.def->tile_width;

        /* set name */
        map.tiles[x][y].name = strdup( map.tiles[x][y].prop->name );


    }
    delete_arg_cluster( args, arg_count );

    /* read tile names */
    if ( find_token( file, "tile_name", RESET_FILE_POS, NO_WARNING ) )
        while ( ( args = get_arg_cluster( file, "tile_name", &arg_count, FROM_CURRENT_FILE_POS, NO_WARNING ) ) != 0 ) {
            /* get name */
            strcpy( name, args[0] );
            /* get positions */
            for ( i = 1; i < arg_count; i++ ) {
                get_coord( args[i], &x, &y );
                /* set new tile name */
                FREE( map.tiles[x][y].name );
                map.tiles[x][y].name = strdup( name );
            }
            /* delete cluster */
            delete_arg_cluster( args, arg_count );
        }


    fclose( file );
    return 1;

failure:
    if ( file ) fclose( file );
    clear_map( );
    return 0;
}

void clear_map( )
{
    int i, j;

    /* remove definitions */
    if ( map.def )
        delete_map_def( map.def );

    /* remove tiles and mask */
    if ( map.tiles ) {
        for ( i = 0; i < map.width; i++ )
            if ( map.tiles[i] ) {
                for ( j = 0; j < map.height; j++ )
                    if ( map.tiles[i][j].name )
                        FREE( map.tiles[i][j].name );
                FREE( map.tiles[i]);
            }
        FREE( map.tiles );
    }
    if ( map.mask ) {
        for ( i = 0; i < map.width; i++ )
            if ( map.mask[i] )
                FREE( map.mask[i] );
        FREE( map.mask );
    }

    memset( &map, 0, sizeof( Map ) );
}

void create_tiles()
{
    int i;
    /* tiles */
    map.tiles = calloc( map.width, sizeof( Map_Tile* ) );
    for ( i = 0; i < map.width; i++ )
        map.tiles[i] = calloc( map.height, sizeof( Map_Tile ) );
    /* mask */
    map.mask = calloc( map.width, sizeof( Mask_Tile* ) );
    for ( i = 0; i < map.width; i++ )
        map.mask[i] = calloc( map.height, sizeof( Mask_Tile ) );
}

/* open map file and return file handle (adds the correct path)*/
FILE* open_map_file( char *file_name )
{
    char full_path[256];
    sprintf(full_path, "%s/maps/%s", SRC_DIR, file_name);
    printf( "parsing maps/%s\n", file_name );
    return file_open( full_path );
}

/* read map tile */
Map_Tile_Prop* read_map_tile_def( FILE *file )
{
    char *cur_arg = 0;
    char **args = 0;
    int arg_count = 0;
    int i = 0;
    Map_Tile_Prop *tile = 0;
    char path[512];

    /* create & clear */
    tile = calloc( 1, sizeof( Map_Tile_Prop ) );

    /* read  entry name */
    if ( ( cur_arg = get_arg( file, "entry_name", FROM_CURRENT_FILE_POS ) ) == 0 ) goto failure;
    tile->entry_name = strdup( cur_arg );
    FREE( cur_arg );

    /* read  name */
    if ( ( cur_arg = get_arg( file, "name", FROM_CURRENT_FILE_POS ) ) == 0 ) goto failure;
    tile->name = strdup( cur_arg );
    FREE( cur_arg );

    /* char id */
    if ( ( cur_arg = get_arg( file, "char_id", FROM_CURRENT_FILE_POS ) ) == 0 ) goto failure;
    tile->char_id = cur_arg[0];
    FREE( cur_arg );

    /* read picture */
    if ( ( cur_arg = get_arg( file, "picture", FROM_CURRENT_FILE_POS ) ) == 0 ) goto failure;
    sprintf( path, "maps/%s", cur_arg );
    tile->pic = load_surf( path, SDL_SWSURFACE );
    /* well, turning this stuff to indexed causes some problems with 32bit depth so take the */
    /* first pixel in the left upper corner as color key (instead of standard 0x0) */
    SDL_SetColorKey( tile->pic, SDL_SRCCOLORKEY, get_pixel( tile->pic, 0, 0 ) );
    FREE( cur_arg );

    /* read movement cost */
    if ( ( cur_arg = get_arg( file, "movement", FROM_CURRENT_FILE_POS ) ) == 0 ) goto failure;
    tile->mov = atoi( cur_arg );
    FREE( cur_arg );

    /* read spotting cost */
    if ( ( cur_arg = get_arg( file, "spotting", FROM_CURRENT_FILE_POS ) ) == 0 ) goto failure;
    tile->spot = atoi( cur_arg );
    FREE( cur_arg );

    /* read min entrenchment */
    if ( ( cur_arg = get_arg( file, "min_entrench", FROM_CURRENT_FILE_POS ) ) == 0 ) goto failure;
    tile->min_entr = atoi( cur_arg );
    FREE( cur_arg );

    /* read max entrenchment */
    if ( ( cur_arg = get_arg( file, "max_entrench", FROM_CURRENT_FILE_POS ) ) == 0 ) goto failure;
    tile->max_entr = atoi( cur_arg );
    FREE( cur_arg );

    /* read flags */
    if ( ( args = get_arg_cluster( file, "flags", &arg_count, FROM_CURRENT_FILE_POS, WARNING ) ) == 0 ) goto failure;
    tile->flags = 0;
    for ( i = 0; i < arg_count; i++ ) {
        cur_arg = args[i];
        /* check flags */
        if ( equal_str( "none", cur_arg ) )
            continue;
        if ( equal_str( "no_spotting", cur_arg ) )
            tile->flags = tile->flags | NO_SPOTTING;
        else
        if ( equal_str( "inf_entr_bonus", cur_arg ) )
            tile->flags = tile->flags | INF_ENTR_BONUS;
        else
        if ( equal_str( "water", cur_arg ) )
            tile->flags = tile->flags | WATER;
        else
        if ( equal_str( "airfield", cur_arg ) )
            tile->flags = tile->flags | AIRFIELD;
        else
        if ( equal_str( "harbor", cur_arg ) )
            tile->flags = tile->flags | HARBOR;
        else
        if ( equal_str( "river", cur_arg ) )
            tile->flags = tile->flags | RIVER;
        else
        if ( equal_str( "mountain", cur_arg ) )
            tile->flags = tile->flags | MOUNTAIN;
        else
        if ( equal_str( "difficult", cur_arg ) )
            tile->flags = tile->flags | DIFFICULT;
        else
            fprintf( stderr, "read_map_tile_def: unknown map tile flag '%s'\n", cur_arg );
    }
    delete_arg_cluster( args, arg_count );

    return tile;

failure:
    delete_tile_def( tile );
    return 0;
}

/* read map definitions */
Map_Def* read_map_def( FILE *file )
{
    Map_Def *def = 0;
    char path[512];
    char *cur_arg = 0;
    int i = 0;
    Map_Tile_Prop *tile_prop = 0;

    /* create & clear */
    def = calloc( 1, sizeof( Map_Def ) );

    /* read  width */
    if ( ( cur_arg = get_arg( file, "map_tile_width", RESET_FILE_POS ) ) == 0 ) goto failure;
    def->tile_width = atoi( cur_arg );
    FREE( cur_arg );
    /* read  height */
    if ( ( cur_arg = get_arg( file, "map_tile_height", RESET_FILE_POS ) ) == 0 ) goto failure;
    def->tile_height = atoi( cur_arg );
    FREE( cur_arg );
    /* read  x offset */
    if ( ( cur_arg = get_arg( file, "map_tile_x_offset", RESET_FILE_POS ) ) == 0 ) goto failure;
    def->tile_x_offset = atoi( cur_arg );
    FREE( cur_arg );
    /* read  y offset */
    if ( ( cur_arg = get_arg( file, "map_tile_y_offset", RESET_FILE_POS ) ) == 0 ) goto failure;
    def->tile_y_offset = atoi( cur_arg );
    FREE( cur_arg );

    /* read fog picture */
    if ( ( cur_arg = get_arg( file, "fog", RESET_FILE_POS ) ) == 0 ) goto failure;
    sprintf( path, "maps/%s", cur_arg );
    def->fog_pic = load_surf( path, SDL_SWSURFACE );
    FREE( cur_arg );
    /* read grid picture */
    if ( ( cur_arg = get_arg( file, "grid", RESET_FILE_POS ) ) == 0 ) goto failure;
    sprintf( path, "maps/%s", cur_arg );
    def->grid_pic = load_surf( path, SDL_SWSURFACE );
    FREE( cur_arg );
    /* read selecting frame picture */
    if ( ( cur_arg = get_arg( file, "frame", RESET_FILE_POS ) ) == 0 ) goto failure;
    sprintf( path, "maps/%s", cur_arg );
    def->sel_pic = load_surf( path, SDL_SWSURFACE );
    FREE( cur_arg );

    /* read crosshair picture */
    if ( ( cur_arg = get_arg( file, "crosshair", RESET_FILE_POS ) ) == 0 ) goto failure;
    sprintf( path, "maps/%s", cur_arg );
    def->cross_pic = load_surf( path, SDL_SWSURFACE );
    FREE( cur_arg );

    /* read explosion */
    if ( ( cur_arg = get_arg( file, "explosion", RESET_FILE_POS ) ) == 0 ) goto failure;
    sprintf( path, "maps/%s", cur_arg );
    def->expl_pic = load_surf( path, SDL_SWSURFACE );
    FREE( cur_arg );

    /* health bars */
    if ( ( cur_arg = get_arg( file, "damage_bar", RESET_FILE_POS ) ) == 0 ) goto failure;
    sprintf( path, "maps/%s", cur_arg );
    def->damage_bar_icons = load_surf( path, SDL_SWSURFACE );
    FREE( cur_arg );

#ifdef WITH_SOUND
    /* sounds */
    if ( ( cur_arg = get_arg( file, "explosion_sound", RESET_FILE_POS ) ) == 0 ) goto failure;
    def->expl_sound = sound_chunk_load( cur_arg );
    FREE( cur_arg );
    if ( ( cur_arg = get_arg( file, "select_sound", RESET_FILE_POS ) ) == 0 ) goto failure;
    def->select_sound = sound_chunk_load( cur_arg );
    FREE( cur_arg );
#endif

    /* count tiles */
    def->tile_count = count_arg( file, "entry_name" );

    /* read map tiles */
    find_token( file, "entry_name", RESET_FILE_POS, WARNING );
    def->tiles = calloc( def->tile_count, sizeof( Map_Tile_Prop* ) );
    for ( i = 0; i < def->tile_count; i++ ) {
        tile_prop = read_map_tile_def( file );
        if ( tile_prop != 0 )
            def->tiles[i] = tile_prop;
        else
            break;
    }

    /* weather names are set here right now */
    strcpy( def->weather_names[FAIR], "Fair" );
    strcpy( def->weather_names[CLOUDS], "Clouds" );
    strcpy( def->weather_names[RAIN], "Rain" );
    strcpy( def->weather_names[SNOW], "Snow" );

    /* cache fogged map tiles */
    cache_fogged_map_tiles( def, FOG_ALPHA );

    return def;

failure:
    delete_map_def( def );
    return 0;
}

/* delete map definitions */
void delete_map_def( Map_Def *map_def )
{
    int i;

    if ( map_def == 0 ) return;

    /* delete within ...*/
    if ( map_def->tiles ) {
        for ( i = 0; i < map_def->tile_count; i++ )
            if ( map_def->tiles[i] )
                delete_tile_def( map_def->tiles[i] );
        FREE( map_def->tiles );
    }
    if ( map_def->fog_pic ) SDL_FreeSurface( map_def->fog_pic );
    if ( map_def->grid_pic ) SDL_FreeSurface( map_def->grid_pic );
    if ( map_def->sel_pic ) SDL_FreeSurface( map_def->sel_pic );
    if ( map_def->cross_pic ) SDL_FreeSurface( map_def->cross_pic );
    if ( map_def->expl_pic ) SDL_FreeSurface( map_def->expl_pic );
    if ( map_def->damage_bar_icons ) SDL_FreeSurface( map_def->damage_bar_icons );

#ifdef WITH_SOUND
    if ( map_def->expl_sound ) sound_chunk_free( map_def->expl_sound );
    if ( map_def->select_sound ) sound_chunk_free( map_def->select_sound );
#endif

    /* ... and struct itself */
    FREE( map_def );
}

void delete_tile_def( Map_Tile_Prop *tile_def )
{
    if ( tile_def->entry_name ) FREE( tile_def->entry_name );
    if ( tile_def->name ) FREE( tile_def->name );
    if ( tile_def->pic ) SDL_FreeSurface( tile_def->pic );
    if ( tile_def->fogged_pic ) SDL_FreeSurface( tile_def->fogged_pic );

    FREE( tile_def );
}

/* cache fogged map tiles */
void cache_fogged_map_tiles( Map_Def *map_def, int fog_alpha )
{
    int i, j;
    int pic_count;

    if ( !map_def->fog_pic ) {
        fprintf( stderr, "cache_fogged_map_tiles: fog_pic does not exist, can't cache\n" );
        return;
    }

    for ( i = 0; i < map_def->tile_count; i++ ) {
        /* create fog tile */
        map_def->tiles[i]->fogged_pic = create_surf( map_def->tiles[i]->pic->w,
                                                     map_def->tiles[i]->pic->h, SDL_SWSURFACE );
        /* copy data */
        FULL_DEST( map_def->tiles[i]->fogged_pic );
        FULL_SOURCE( map_def->tiles[i]->pic );
        blit_surf();
        /* countt number of single tiles */
        pic_count = map_def->tiles[i]->pic->w /  map_def->tile_width;
        /* cover with fog */
        for ( j = 0; j < pic_count; j++ ) {
            DEST( map_def->tiles[i]->fogged_pic, j *  map_def->tile_width, 0,  map_def->tile_width,
                  map_def->tile_height );
            SOURCE(  map_def->fog_pic, 0, 0 );
            alpha_blit_surf( fog_alpha );
        }
    }
}

/* return map tile at x, y */
Map_Tile* map_tile( int x, int y )
{
    if ( x < 0 || y < 0 || x >= map.width || y >= map.height ) {
        fprintf( stderr, "map_tile: map tile at %i,%i doesn't exist\n", x, y);
        return 0;
    }
    return &map.tiles[x][y];
}

/* return mask tile */
Mask_Tile* mask_tile( int x, int y )
{
    if ( x < 0 || y < 0 || x >= map.width || y >= map.height ) {
        fprintf( stderr, "map_tile: mask tile at %i,%i doesn't exist\n", x, y);
        return 0;
    }
    return &map.mask[x][y];
}

/* set all flags specified to their default value */
void clear_mask( int flags )
{
    int i, j;
    for ( i = 0; i < map.width; i++ )
        for ( j = 0; j < map.height; j++ ) {
            if ( flags & F_FOG ) map.mask[i][j].fog = 1;
            if ( flags & F_INVERSE_FOG ) map.mask[i][j].fog = 0;
            if ( flags & F_SPOT ) map.mask[i][j].spot = 0;
            if ( flags & F_IN_RANGE ) map.mask[i][j].in_range = -1;
            if ( flags & F_MOUNT ) map.mask[i][j].mount = 0;
            if ( flags & F_SEA_EMBARK ) map.mask[i][j].sea_embark = 0;
            if ( flags & F_AUX ) map.mask[i][j].aux = 0;
            if ( flags & F_INFL ) map.mask[i][j].infl = 0;
            if ( flags & F_AIR_INFL ) map.mask[i][j].air_infl = 0;
            if ( flags & F_VIS_INFL ) map.mask[i][j].vis_infl = 0;
            if ( flags & F_VIS_AIR_INFL ) map.mask[i][j].vis_air_infl = 0;
            if ( flags & F_BLOCKED ) map.mask[i][j].blocked = 0;
            if ( flags & F_BACKUP ) map.mask[i][j].backup = 0;
            if ( flags & F_MELT ) map.mask[i][j].merge = 0;
            if ( flags & F_DEPLOY ) map.mask[i][j].deploy = 0;
        }
}

/* check surrounding tiles and return the one with the highest in_range value */
void get_next_pos( int x, int y, int *next_x, int *next_y )
{
    *next_x = x; *next_y = y;
    if ( mask_tile( x + 1, y )->in_range > mask_tile( *next_x, *next_y )->in_range ) {
        *next_x = x + 1;
        *next_y = y;
    }
    if ( mask_tile( x - 1, y )->in_range > mask_tile( *next_x, *next_y )->in_range ) {
        *next_x = x - 1;
        *next_y = y;
    }
    if ( mask_tile( x, y + 1 )->in_range > mask_tile( *next_x, *next_y )->in_range ) {
        *next_x = x;
        *next_y = y + 1;
    }
    if ( mask_tile( x, y - 1 )->in_range > mask_tile( *next_x, *next_y )->in_range ) {
        *next_x = x;
        *next_y = y - 1;
    }
    if ( x & 1 ) {
        if ( mask_tile( x + 1, y + 1 )->in_range > mask_tile( *next_x, *next_y )->in_range ) {
            *next_x = x + 1;
            *next_y = y + 1;
        }
        if ( mask_tile( x - 1, y + 1 )->in_range > mask_tile( *next_x, *next_y )->in_range ) {
            *next_x = x - 1;
            *next_y = y + 1;
        }
    }
    else {
        if ( mask_tile( x + 1, y - 1 )->in_range > mask_tile( *next_x, *next_y )->in_range ) {
            *next_x = x + 1;
            *next_y = y - 1;
        }
        if ( mask_tile( x - 1, y - 1 )->in_range > mask_tile( *next_x, *next_y )->in_range ) {
            *next_x = x - 1;
            *next_y = y - 1;
        }
    }
}

/* get a list of waypoints at which the unit moves along to it's destination (x,y); includes the units
starting position */
/* get_way_point checks the way for unspotted influence and ambush and cuts it down if needed */
Way_Point* get_way_points( Unit *unit, int x, int y, int *count, Unit **ambush_unit )
{
    Way_Point *way = 0;
    int i;
    int next_x, next_y;

    /* first count way points */
    /* can't loop for infinity if in_range mask is setup correctly */
    *count = 1; /* always have start position */
    next_x = x;
    next_y = y;
    while ( next_x != unit->x || next_y != unit->y ) {
        get_next_pos( next_x, next_y, &next_x, &next_y );
        (*count)++;
    }

    /* get mem */
    way = calloc( *count, sizeof( Way_Point ) );

    /* it's easiest to get positions in reverse order; so start at end of array */
    next_x = x; next_y = y;
    for ( i = *count - 1; i >= 0; i-- ) {
        way[i].x = next_x; way[i].y = next_y;
        get_next_pos( next_x, next_y, &next_x, &next_y );
    }
    /* check if last tile was unit's position */
    if ( way[0].x != unit->x || way[0].y != unit->y )
        fprintf( stderr, "get_way_points: where're u goin'??? get back here!\n" );

    /* way points */
/*    printf( "'%s': %i,%i", unit->name, way[0].x, way[0].y );
    for ( i = 1; i < *count; i++ )
        printf( " -> %i,%i", way[i].x, way[i].y );
    printf( "\n" );*/

    /* check for ambush and influence */
    /* if there is a unit in the way it must be an enemy (friends, spotted enemies are not allowed)
    so cut down way to this way_point and set engine::ambush_unit */
    /* if an unspotted tile does have influence >0 an enemy is nearby and our unit must stop */
    for ( i = 1; i < *count; i++ ) {
        /* check if on this tile a unit is waiting */
        /* if mask::blocked is set it's an own unit so don't check for ambush */
        if ( !mask_tile( way[i].x, way[i].y )->blocked ) {
            if ( map_tile( way[i].x, way[i].y )->unit )
                if ( !( unit->sel_prop->flags & FLYING ) ) {
                    *ambush_unit = map_tile( way[i].x, way[i].y )->unit;
                    break;
                }
            if ( map_tile( way[i].x, way[i].y )->air_unit )
                if ( unit->sel_prop->flags & FLYING ) {
                    *ambush_unit = map_tile( way[i].x, way[i].y )->air_unit;
                    break;
                }
        }
        /* if we get here there is no unit waiting but maybe close too the tile */
        /* therefore check tile of moving unit if it is influenced by a previously unspotted unit */
        if ( unit->sel_prop->flags & FLYING ) {
            if ( mask_tile( way[i - 1].x, way[i - 1].y )->air_infl && !mask_tile( way[i - 1].x, way[i - 1].y )->vis_air_infl )
                break;
        }
        else {
            if ( mask_tile( way[i - 1].x, way[i - 1].y )->infl && !mask_tile( way[i - 1].x, way[i - 1].y )->vis_infl )
                break;
        }
    }
    if ( i < *count )
        /* enemy in the way; cut down */
        *count = i;

    return way;
}

/*
====================================================================
Insert, Remove unit pointer from map.
====================================================================
*/
void insert_unit_to_map( Unit *unit )
{
    if ( unit->sel_prop->flags & FLYING ) {
        if ( map_tile( unit->x, unit->y )->air_unit ) {
            fprintf( stderr, "insert_unit_to_map: warning: "
                             "unit %s hasn't been removed properly from %i,%i:"
                             "overwrite it\n",
                     map_tile( unit->x, unit->y )->air_unit->name, unit->x, unit->y );
        }
        map_tile( unit->x, unit->y )->air_unit = unit;
    }
    else {
        if ( map_tile( unit->x, unit->y )->unit ) {
            fprintf( stderr, "insert_unit_to_map: warning: "
                             "unit %s hasn't been removed properly from %i,%i:"
                             "overwrite it\n",
                     map_tile( unit->x, unit->y )->unit->name, unit->x, unit->y );
        }
        map_tile( unit->x, unit->y )->unit = unit;
    }
}
void remove_unit_from_map( Unit *unit )
{
    if ( unit->sel_prop->flags & FLYING )
        map_tile( unit->x, unit->y )->air_unit = 0;
    else
        map_tile( unit->x, unit->y )->unit = 0;
}

