#include <stdlib.h>

#include <qstring.h>
#include <qxml.h>
#include <qbuffer.h>

#include "klogic.h"
#include "xnet.h"
#include "xdevice.h"
#include "deviceTypes.h"
#include "xwire.h"
#include "xmlExport.h"
#include "xmlImport.h"
#include "xmlSymbolContainer.h"
#include "klogicList.h"

/*******************************************/
/* XML Export Interface                    */
/*                                         */
/* Andreas Rostin 23.08.2001               */
/*******************************************/
int XmlObject::sym = 0;
const char * XmlObject::ATT_CIRCUIT = "CIRCUIT";
const char * XmlObject::ATT_SUBCIRCUIT = "SUBCIRCUIT";
const char * XmlObject::ATT_LIBRARY = "LIBRARY";

XmlSymbolContainer XmlObject::sFILE_TYPE("FILE_TYPE", XmlObject::FILE_TYPE);

XmlSymbolContainer XmlObject::sKLOGIC_MAIN("KLOGIC_MAIN", XmlObject::KLOGIC_MAIN);
XmlSymbolContainer XmlObject::sCIRCUIT("CIRCUIT", XmlObject::CIRCUIT);
XmlSymbolContainer XmlObject::sDEV_CONN("DEV_CONN", XmlObject::DEV_CONN);
XmlSymbolContainer XmlObject::sWIRE_CONN("WIRE_CONN", XmlObject::WIRE_CONN);
XmlSymbolContainer XmlObject::sWIRE("WIRE", XmlObject::WIRE);
XmlSymbolContainer XmlObject::sDEVICE("DEV", XmlObject::DEVICE);
XmlSymbolContainer XmlObject::sID("ID", XmlObject::ID);
XmlSymbolContainer XmlObject::sLIBDEV("LIBDEV", XmlObject::LIBDEV);

XmlSymbolContainer XmlObject::sPROGRAM_VERSION("PROGRAM_VERSION", XmlObject::PROGRAM_VERSION);
XmlSymbolContainer XmlObject::sCREATION_DATE("CREATION_DATE", XmlObject::CREATION_DATE);
XmlSymbolContainer XmlObject::sUPDATE_DATE("UPDATE_DATE", XmlObject::UPDATE_DATE);
XmlSymbolContainer XmlObject::sAUTHOR("AUTHOR", XmlObject::AUTHOR);
XmlSymbolContainer XmlObject::sCOMMENT("COMMENT", XmlObject::COMMENT);

XmlSymbolContainer XmlObject::sFUNCTION("F", XmlObject::FUNCTION);
XmlSymbolContainer XmlObject::sDELAY("DELAY", XmlObject::DELAY);
XmlSymbolContainer XmlObject::sUNDEF("UNDEF", XmlObject::UNDEF);
XmlSymbolContainer XmlObject::sINV("INV", XmlObject::INV);
XmlSymbolContainer XmlObject::sCLK_TYPE("CLK_TYPE", XmlObject::CLK_TYPE);
XmlSymbolContainer XmlObject::sMASTER("MASTER", XmlObject::MASTER);
XmlSymbolContainer XmlObject::sOSC_ON("OSC_ON", XmlObject::OSC_ON);
XmlSymbolContainer XmlObject::sOSC_OFF("OSC_OFF", XmlObject::OSC_OFF);
XmlSymbolContainer XmlObject::sOSC_STEP("OSC_STEP", XmlObject::OSC_STEP);
XmlSymbolContainer XmlObject::sSTAT("STAT", XmlObject::STAT);
XmlSymbolContainer XmlObject::sSIZE("SZ", XmlObject::SIZE);
XmlSymbolContainer XmlObject::sXPOS("X", XmlObject::XPOS);
XmlSymbolContainer XmlObject::sYPOS("Y", XmlObject::YPOS);
XmlSymbolContainer XmlObject::sRAM("RAM", XmlObject::RAM);
XmlSymbolContainer XmlObject::sTEXT_DISP("TEXT_DISP", XmlObject::TEXT_DISP);
XmlSymbolContainer XmlObject::sTEXT("TEXT", XmlObject::TEXT);
XmlSymbolContainer XmlObject::sPUSH("PUSH", XmlObject::PUSH);
XmlSymbolContainer XmlObject::sWIDTH("WIDTH", XmlObject::WIDTH);

XmlSymbolContainer XmlObject::sOUT_EQU("OUT_EQU", XmlObject::OUT_EQU);
XmlSymbolContainer XmlObject::sOUT_ID("OUT_ID", XmlObject::OUT_ID);
XmlSymbolContainer XmlObject::sIN_NAME("IN_NAME", XmlObject::IN_NAME);
XmlSymbolContainer XmlObject::sIN_ID("IN_ID", XmlObject::IN_ID);
XmlSymbolContainer XmlObject::sINTERN("INT", XmlObject::INTERN);
XmlSymbolContainer XmlObject::sTRI("TRI", XmlObject::TRI);
XmlSymbolContainer XmlObject::sPOS("POS", XmlObject::POS);
XmlSymbolContainer XmlObject::sNAME("NAME", XmlObject::NAME);
XmlSymbolContainer XmlObject::sBUSTYPE("BUSTYPE", XmlObject::BUSTYPE);

// static
void XmlObject::setSymbolSet(int new_set)
{
	sym = new_set;
}

// static
QString XmlObject::getSymbol(int symbol_idx)
{
	XmlSymbolContainer *Cnt =  XmlSymbolContainer::get(symbol_idx);
	if (Cnt) return (*Cnt)[sym];
	return "";
}

// static
bool XmlObject::symbolEquals(const QString& sSymbol, int iSymbol)
{
	XmlSymbolContainer *arg = XmlSymbolContainer::get(sSymbol);
	if (*arg == iSymbol) return true;
	return false;
}

// static
int XmlObject::getID(const QString& symbol)
{
	XmlSymbolContainer *Cnt =  XmlSymbolContainer::get(symbol);
	if (Cnt) return Cnt->getID();
	return 0;
}

// static
int XmlObject::getSymbolSet()
{
	return sym;
}

XmlObject::XmlObject(QString name)
	:XmlField(name, 0)
{
	dx = 0;
	dy = 0;
	_valid = true;
}

XmlObject::XmlObject(QString name, int level, int _dx, int _dy)
	:XmlField(name, level)
{
	dx = _dx;
	dy = _dy;
	_valid = false;
}

// virtual
XmlObject::~XmlObject()
{
}

// private virtual; to be redefined..
void XmlObject::createXML(bool)
{
	return;
}

// return value can be both, a circuit or a device
// use XDevice::devIsNet() to retrieve XDeviceNet pointer from XDevice
XDevice * XmlObject::createInstance(XDeviceNet *_net, int dx, int dy)
{
	QString content = getField();

	// xml reader does not work with string based textstreams!
	QBuffer f;
	f.open(IO_WriteOnly|IO_Truncate);
	QTextStream ts3(&f);
	ts3 << content;
	f.close();
	f.open(IO_ReadOnly);
	QTextStream ts2(&f);
	return XMLImportHandler::import(_net, dx, dy, ts2);
}

void XmlObject::setValid(bool __valid)
{
	_valid = __valid;
}

bool XmlObject::isValid()
{
	return _valid;
}

/*******************************************/
/* XML representation of a circuit         */
/*                                         */
/* because a circuit can consume a lot of  */
/* memory, this export is mainly done by   */
/* the XDeviceNet itself!                  */
/* see klogicIO::writeXML() for details    */
/*                                         */
/* Andreas Rostin 24.08.2001               */
/*******************************************/
XmlNet::XmlNet(XDeviceNet *_net, bool selected, int level, int dx, int dy)
	: XmlObject(XmlObject::getSymbol(CIRCUIT), level, dx, dy)
{
	net = _net;
	createXML(selected);
}

XmlNet::XmlNet()
	: XmlObject(XmlObject::getSymbol(CIRCUIT), 0, 0, 0)
{
	net = (XDeviceNet *)NULL;
}

void XmlNet::createXML(bool selected)
{
	if (selected && !net->isSelected()) return;
	setValid(true);
	setAttribute(XmlObject::getSymbol(ID), net->getID());
	addAttribute(XmlObject::getSymbol(FUNCTION), net->type());
	addAttribute(XmlObject::getSymbol(SIZE), net->size());
	int x = net->getPos().x() - net->getHOffset() - dx;
	if (x < 0) x = 0;
	int y = net->getPos().y() + net->getVOffset() - dy;
	if (y < 0) y = 0;
	addAttribute(XmlObject::getSymbol(XPOS), x);
	addAttribute(XmlObject::getSymbol(YPOS), y);

	XmlDevice dev(net, selected, -1, dx, dy);
	setContent(dev.getFieldContent());
}

/*******************************************/
/* XML representation of a connection      */
/*     between devices and wires           */
/*                                         */
/* Andreas Rostin 24.08.2001               */
/*******************************************/
XmlDeviceConnection::XmlDeviceConnection(XWire *_wire, bool selected, int level, int dx, int dy)
	: XmlObject(XmlObject::getSymbol(DEV_CONN), level, dx, dy)
{
	wire = _wire;
	createXML(selected);
}

void XmlDeviceConnection::createXML(bool selected)
{	int inverter;

	if (selected && !wire->isSelected()) return;

	QPoint pi1 = wire->getInputPosition(0, selected);
	QPoint pi2 = wire->getInputPosition(1, selected);
	QPoint po1 = wire->getOutputPosition(0, selected);
	QPoint po2 = wire->getOutputPosition(1, selected);
	QPoint pt;
	XDevice *dev;

	// maximum of 2 points
	if (!pi1.isNull() || !pi2.isNull() || !po1.isNull() || !po2.isNull()) {
		setValid(true);
		setAttribute(XmlObject::getSymbol(ID), wire->getID());

		if (!pi1.isNull()) {
			dev = wire->inputDevice(0);
			inverter = wire->inputIsInverted(dev);
			pt.setX(pi1.x() - dx);
			pt.setY(pi1.y() - dy);
			setField(dev, pt, inverter);
		}
		if (!pi2.isNull()) {
			dev = wire->inputDevice(1);
			inverter = wire->inputIsInverted(dev);
			pt.setX(pi2.x() - dx);
			pt.setY(pi2.y() - dy);
			setField(dev, pt, inverter);
		}
		if (!po1.isNull()) {
			dev = wire->outputDevice(0);
			inverter = wire->outputIsInverted(dev);
			pt.setX(po1.x() - dx);
			pt.setY(po1.y() - dy);
			setField(dev, pt, inverter);
		}
		if (!po2.isNull()) {
			dev = wire->outputDevice(1);
			inverter = wire->outputIsInverted(dev);
			pt.setX(po2.x() - dx);
			pt.setY(po2.y() - dy);
			setField(dev, pt, inverter);
		}
	}
}

void XmlDeviceConnection::setField(XDevice *dev, QPoint pt, int inverter)
{	XmlField field;

	field.set(XmlObject::getSymbol(POS), pt);
	field.setAttribute(XmlObject::getSymbol(ID), dev->getID());
	field.addAttribute(XmlObject::getSymbol(INV), inverter);
	addContent(field.getField());
}

/*******************************************/
/* XML representation of a connection      */
/*            between wires                */
/*                                         */
/* Andreas Rostin 24.08.2001               */
/*******************************************/
XmlWireConnection::XmlWireConnection(XWire *_wire, bool selected, int level, int dx, int dy)
	: XmlObject(XmlObject::getSymbol(WIRE_CONN), level, dx, dy)
{
	wire = _wire;
	createXML(selected);
}

void XmlWireConnection::createXML(bool selected)
{	XmlField field;

	if (selected && !wire->isSelected()) return;

	QPoint pi = wire->getIOPosition(1, selected);
	QPoint po = wire->getIOPosition(0, selected);
	QPoint pt;

	if ((!pi.isNull()) || (!po.isNull())) {
		setValid(true);
		setAttribute(XmlObject::getSymbol(ID), wire->getID());

		if (!pi.isNull()) {
			pt.setX(pi.x() - dx);
			pt.setY(pi.y() - dy);
			
			field.set(XmlObject::getSymbol(POS), pt);
			field.setAttribute(XmlObject::getSymbol(ID), wire->ioWire(1)->getID());
			addContent(field.getField());
		}

		if (!po.isNull()) {
			pt.setX(po.x() - dx);
			pt.setY(po.y() - dy);
			field.set(XmlObject::getSymbol(POS), pt);
			field.setAttribute(XmlObject::getSymbol(ID), wire->ioWire(0)->getID());
			addContent(field.getField());
		}
	}
}

/*******************************************/
/* XML representation of a XWire           */
/*                                         */
/* Andreas Rostin 23.08.2001               */
/*******************************************/
XmlWire::XmlWire(XWire *_wire, bool selected, int level, int dx, int dy)
	: XmlObject(XmlObject::getSymbol(WIRE), level, dx, dy)
{
	wire = _wire;
	createXML(selected);
}

void XmlWire::createXML(bool selected)
{	QString _content;

	if (selected && !wire->isSelected()) return;
	setValid(true);
 
	// wire properties
	setAttribute(XmlObject::getSymbol(ID), wire->getID());

	// wire content
	wire->setExportSelected(selected);
	wire->setExportDelta(dx, dy);
	*wire >> _content;

	setContent(_content);
}

/*******************************************/
/* XML representation of a XDevice         */
/*                                         */
/* Andreas Rostin 23.08.2001               */
/*******************************************/
XmlDevice::XmlDevice(XDevice *_dev, bool selected, int level, int dx, int dy)
	: XmlObject(XmlObject::getSymbol(DEVICE), level, dx, dy)
{
	dev = _dev;
	createXML(selected);
}

XmlDevice::XmlDevice()
	: XmlObject(XmlObject::getSymbol(DEVICE), 0, 0, 0)
{
	dev = (XDeviceNet *)NULL;
}

void XmlDevice::createXML(bool selected)
{	XmlField field;
	QString text;

	if (selected && !dev->isSelected()) return;
	setValid(true);

	// device attributes
	setAttribute(XmlObject::getSymbol(ID), dev->getID());
	addAttribute(XmlObject::getSymbol(FUNCTION), dev->type());
	addAttribute(XmlObject::getSymbol(SIZE), dev->size());
	addAttribute(XmlObject::getSymbol(XPOS), dev->getPos().x() - dev->getHOffset() - dx);
	addAttribute(XmlObject::getSymbol(YPOS), dev->getPos().y() + dev->getVOffset() - dy);

	// device content 
	text = dev->getText();
	field.set(XmlObject::getSymbol(TEXT), XmlField::quote(text));
	addContent(field.getField());

	if (dev->textDisplayed()) {
		field.set(XmlObject::getSymbol(TEXT_DISP), "");
		addContent(field.getField());
	}

	if (dev->type() != DeviceType::fNET &&
	    dev->type() != DeviceType::fLEDred &&
	    dev->type() != DeviceType::fLEDblue &&
	    dev->type() != DeviceType::fLEDgreen &&
	    dev->type() != DeviceType::fLEDyellow &&
	    dev->delay() != Device::DEFAULT_DELAY) { 
		field.set(XmlObject::getSymbol(DELAY), dev->delay());
		addContent(field.getField());
	}

	if (dev->undef() != Device::DEFAULT_UNDEF) { 
		field.set(XmlObject::getSymbol(UNDEF), dev->undef());
		addContent(field.getField());
	}
 
	if (dev->hasStaticInput()) {
		field.set(XmlObject::getSymbol(STAT), dev->output());
		addContent(field.getField());
	}
 
	// flipflop properties
	if (dev->type() != DeviceType::fOSC && dev->hasClock()) {
		field.set(XmlObject::getSymbol(CLK_TYPE), dev->clock());
		addContent(field.getField());

		field.set(XmlObject::getSymbol(MASTER), dev->hasMaster());
		addContent(field.getField());
	}

	// switch properties
	if (dev->type() == DeviceType::fSWI && !dev->isSwitchType()) {
		field.set(XmlObject::getSymbol(PUSH));
		addContent(field.getField());
	}

	// width property: possible only for equ, net
	if ((dev->type() == DeviceType::fEQU || dev->type() == DeviceType::fNET) && dev->width() != XDevice::DEFAULT_WIDTH) {
		field.set(XmlObject::getSymbol(WIDTH), dev->width());
		addContent(field.getField());
	}

	// default inverted output
	if (dev->isInverted()) {
		field.set(XmlObject::getSymbol(INV));
		addContent(field.getField());
	}

	// oscillator properties
	if (dev->type() == DeviceType::fOSC) {
		field.set(XmlObject::getSymbol(CLK_TYPE), dev->clock());
		addContent(field.getField());

		if (dev->oszOn() != Device::DEFAULT_OSC_CNT) {
			field.set(XmlObject::getSymbol(OSC_ON), dev->oszOn());
			addContent(field.getField());
		}
	 
		if (dev->oszOff() != Device::DEFAULT_OSC_CNT) {
			field.set(XmlObject::getSymbol(OSC_OFF), dev->oszOff());
			addContent(field.getField());
		}
	 
		field.set(XmlObject::getSymbol(OSC_STEP), dev->getCurrOszStep());
		addContent(field.getField());
	}

	// tristate flag
	if (dev->isTristate()) {
		field.set(XmlObject::getSymbol(TRI), dev->getTristateControlPos());
		addContent(field.getField());
	}

	// named output
	if ((dev->type() == DeviceType::fEQU || dev->type() == DeviceType::fNET) && dev->hasNamedOutput()) {
		KlogicList<opStack> *lo = dev->getNamedORef()->First()->Next();
		while (lo) {
			if (dev->type() == DeviceType::fEQU) {
				text = lo->Get()->getEquation();
				field.set(XmlObject::getSymbol(OUT_EQU), XmlField::quote(text));
				field.setAttribute(XmlObject::getSymbol(POS), lo->Get()->getPosition());
				field.addAttribute(XmlObject::getSymbol(INTERN), lo->Get()->isInternal());
				text = lo->getText();
				field.addAttribute(XmlObject::getSymbol(NAME), XmlField::quote(text));
			} else {
				field.set(XmlObject::getSymbol(OUT_ID), lo->getID1());
				field.setAttribute(XmlObject::getSymbol(POS), lo->Get()->getPosition());
			}

			addContent(field.getField());
			lo = lo->Next();
		}
	}

	// named input 
	if ((dev->type() == DeviceType::fEQU || dev->type() == DeviceType::fNET) && dev->hasNamedInput()) {
		KlogicList<Value> *li = dev->getNamedIRef()->First();
		while(li) {
			if (dev->type() == DeviceType::fEQU) {
				text = li->getText();
				field.set(XmlObject::getSymbol(IN_NAME), XmlField::quote(text));
				field.setAttribute(XmlObject::getSymbol(POS), li->Get()->getPosition());
				field.addAttribute(XmlObject::getSymbol(INTERN), li->Get()->isInternal());
			} else {
				field.set(XmlObject::getSymbol(IN_ID), li->getID1());
				field.setAttribute(XmlObject::getSymbol(POS), li->Get()->getPosition());
			}

			addContent(field.getField());
			li = li->Next();
		}
	}

	// ram content
	if (dev->type() == DeviceType::fRAM) {
		field.set(XmlObject::getSymbol(RAM), dev->RAMSize(), dev->getRAMStorageCopy());
		addContent(field.getField());
	}

	// bus type
	if (dev->type() == DeviceType::fBUS) {
		field.set(XmlObject::getSymbol(BUSTYPE), dev->isMux());
		addContent(field.getField());
	}
}

