/*************************************************/
/* member methods for class SymbolOpStack        */
/*                                               */
/* symbolic equation analyzer                    */
/*                                               */
/* Andreas Rostin                                */
/* 09.04.01                                      */
/*************************************************/
#include <qstring.h>
#include <device.h>
#include <array.h>
#include <tool.h>

#include <symbolOpStack.h>

SymbolOpStack::SymbolOpStack()
	:opStack()
{
	input_list = (char **)NULL;
	input_cnt = 0;

	pattern_idx = (int *)NULL;
	patterns = (int **)NULL;
	daa_patterns = (char *)NULL;
	pattern = (int *)NULL;
	pattern_cnt = 0;
	pattern_mode = 0;

	daa_group = (char *)NULL;
	group_size = (int *)NULL;

	tmp_equ = (char *)NULL;
}

SymbolOpStack::~SymbolOpStack()
{
	if (pattern_idx) free(pattern_idx);
	if (daa_patterns) free(daa_patterns);
	if (input_list) {
		int i;
		for (i = 0; i < input_cnt; i++)
			free(input_list[i]);
		free (input_list);
	}
	if (daa_group) free(daa_group);
	if (group_size) free(group_size);
	if (tmp_equ) free(tmp_equ);
}

// symbolic stack calculation
// returns the equation
char * SymbolOpStack::calculateSymbolic()
{	int i;
	char **cstack = (char **)malloc(sizeof(char *) * maxitem);
	char *item;
	int lpar, rpar;

	if (!total) {
		free(cstack);
		return (char *)NULL;	// nothing to do
	}

	plane = -1;
	for (i = 0; i < total; i++) {
		switch (queue[i][0]) {
			case OP_AND:
				lpar = needsParenthesis(cstack[plane - 1], OP_AND);
				rpar = needsParenthesis(cstack[plane], OP_AND);
				item = (char *)malloc(sizeof(char) * (strlen(cstack[plane - 1]) + strlen(cstack[plane]) + lpar + rpar + 4));
				if (!lpar && !rpar) sprintf(item, "%s %c %s", cstack[plane - 1], OP_AND, cstack[plane]);
				if (lpar && !rpar) sprintf(item, "(%s) %c %s", cstack[plane - 1], OP_AND, cstack[plane]);
				if (!lpar && rpar) sprintf(item, "%s %c (%s)", cstack[plane - 1], OP_AND, cstack[plane]);
				if (lpar && rpar) sprintf(item, "(%s) %c (%s)", cstack[plane - 1], OP_AND, cstack[plane]);
				free(cstack[plane - 1]);
				free(cstack[plane]);
				cstack[plane - 1] = item;
				plane--;
				break;
			case OP_OR:
				lpar = needsParenthesis(cstack[plane - 1], OP_OR);
				rpar = needsParenthesis(cstack[plane], OP_OR);
				item = (char *)malloc(sizeof(char) * (strlen(cstack[plane - 1]) + strlen(cstack[plane]) + lpar + rpar + 4));
				if (!lpar && !rpar) sprintf(item, "%s %c %s", cstack[plane - 1], OP_OR, cstack[plane]);
				if (lpar && !rpar) sprintf(item, "(%s) %c %s", cstack[plane - 1], OP_OR, cstack[plane]);
				if (!lpar && rpar) sprintf(item, "%s %c (%s)", cstack[plane - 1], OP_OR, cstack[plane]);
				if (lpar && rpar) sprintf(item, "(%s) %c (%s)", cstack[plane - 1], OP_OR, cstack[plane]);
				free(cstack[plane - 1]);
				free(cstack[plane]);
				cstack[plane - 1] = item;
				plane--;
				break;
			case OP_XOR:
				lpar = needsParenthesis(cstack[plane - 1], OP_XOR);
				rpar = needsParenthesis(cstack[plane], OP_XOR);
				item = (char *)malloc(sizeof(char) * (strlen(cstack[plane - 1]) + strlen(cstack[plane]) + lpar + rpar + 4));
				if (!lpar && !rpar) sprintf(item, "%s %c %s", cstack[plane - 1], OP_XOR, cstack[plane]);
				if (lpar && !rpar) sprintf(item, "(%s) %c %s", cstack[plane - 1], OP_XOR, cstack[plane]);
				if (!lpar && rpar) sprintf(item, "%s %c (%s)", cstack[plane - 1], OP_XOR, cstack[plane]);
				if (lpar && rpar) sprintf(item, "(%s) %c (%s)", cstack[plane - 1], OP_XOR, cstack[plane]);
				free(cstack[plane - 1]);
				free(cstack[plane]);
				cstack[plane - 1] = item;
				plane--;
				break;
			case OP_BIN:
				lpar = needsParenthesis(cstack[plane - 1], OP_BIN);
				rpar = needsParenthesis(cstack[plane], OP_BIN);
				item = (char *)malloc(sizeof(char) * (strlen(cstack[plane - 1]) + strlen(cstack[plane]) + lpar + rpar + 4));
				if (!lpar && !rpar) sprintf(item, "%s %c %s", cstack[plane - 1], OP_BIN, cstack[plane]);
				if (lpar && !rpar) sprintf(item, "(%s) %c %s", cstack[plane - 1], OP_BIN, cstack[plane]);
				if (!lpar && rpar) sprintf(item, "%s %c (%s)", cstack[plane - 1], OP_BIN, cstack[plane]);
				if (lpar && rpar) sprintf(item, "(%s) %c (%s)", cstack[plane - 1], OP_BIN, cstack[plane]);
				free(cstack[plane - 1]);
				free(cstack[plane]);
				cstack[plane - 1] = item;
				plane--;
				break;
			case OP_BIT:
				lpar = needsParenthesis(cstack[plane - 1], OP_BIT);
				rpar = needsParenthesis(cstack[plane], OP_BIT);
				item = (char *)malloc(sizeof(char) * (strlen(cstack[plane - 1]) + strlen(cstack[plane]) + lpar + rpar + 4));
				if (!lpar && !rpar) sprintf(item, "%s %c %s", cstack[plane - 1], OP_BIT, cstack[plane]);
				if (lpar && !rpar) sprintf(item, "(%s) %c %s", cstack[plane - 1], OP_BIT, cstack[plane]);
				if (!lpar && rpar) sprintf(item, "%s %c (%s)", cstack[plane - 1], OP_BIT, cstack[plane]);
				if (lpar && rpar) sprintf(item, "(%s) %c (%s)", cstack[plane - 1], OP_BIT, cstack[plane]);
				free(cstack[plane - 1]);
				free(cstack[plane]);
				cstack[plane - 1] = item;
				plane--;
				break;
			case OP_NOT:
				rpar = needsParenthesis(cstack[plane], OP_NOT);
				item = (char *)malloc(sizeof(char) * (strlen(cstack[plane]) + rpar + 2));
				if (!rpar) sprintf(item, "%c%s", OP_NOT, cstack[plane]);
				if (rpar) sprintf(item, "%c(%s)", OP_NOT, cstack[plane]);
				free(cstack[plane]);
				cstack[plane] = item;
				break;
			case OP_NONE:
				break;
			case OP_INTERFACE:
				break;
			default:
				plane++;
				if (plane > maxitem) {
					fprintf(stderr, "opstack: cstack overflow??\n");
					exit(-1);
				}
				cstack[plane] = strdup(queue[i]);
				break;
		}
	}
	if (plane != 0) {
		fprintf(stderr, "cstack error during symbolic calculation\n");
		exit(-1);
	}
	plane = -1;
	if (tmp_equ) free(tmp_equ);
	tmp_equ = cstack[0];
	free(cstack);
	return tmp_equ;
}

// return a minimized and normalized equation
// derived from the "karnough" graphical analyzing method, but up to 15 input variables
// algorithm:
// (1) find as big as possible rectangular pattern groups in the equation, where each neighbor pattern toggles only one bit
// (2) remove patterns contained in other patterns
// (3) create a minimized subjunctive/disjunctive normalized equation from the pattern-groups found
//  - advantage: againgst graphical method: it is not restricted to a maximum of 6 input variables
//  - disadvantages: needs some MB of memory resources and calculation power for really big equations!
//                 : the resulting minimized and normalized equation must not be shorter than the original one! so check the result!
//
// Arguments:
// (1) 1 : returns a subjunctive normalized equation
//     0 : returns a disjunctive normalized equation
// (2) 1 : remove not as many pattern groups as 0, but cannot produce glitches!
//     0 : remove as many pattern groups as possible fulfilling requirements, may produce glitches!
char * SymbolOpStack::calculateSymbolicNormalized(int subjunctive, int secure)
{
	int **table;				// an array of patterns with some exchanged columns
	char *daa_table;			// the pointer to use with free()
	unsigned daa_table_dimensions[2];	// the dimensions of the table array
	unsigned daa_ret;			// not really used .. the return value of the array allocation function
	int start_arry[2] = {0,0};		// initialisation values of the arrays

	int i, j, k, found, all_found, single_found, bit_found;
	int ii, ij, ik, jj, kk;
	int bit;

	if (!total) {
		return (char *)NULL;	// nothing to do
	}

	// create all result patterns possible
	// daa_patterns		the patterns
	// pattern_cnt		the number of patterns stored
	if (!patterns) calculatePattern();
	if (!patterns) return NULL;

	// ****** find groups greater than 1 pattern (remember the patterns of the group as integer value) ******
	// group: a set of patterns with an equal output and not toggling more than one bit to the neighbors patterns
	// group_size: the number of patterns of a specific groups
	// group_cnt: the total number of groups
	//
	// further expanations (for a group-size of 2):
	//     Two patterns where the first pattern has the same result as the following pattern builds a group of two patterns
	//     if the group does not lies inside a binary jump of bits greater than or equals the group-size.
	//     So a valid group for a group size=2 and 3 input bits is (000 == 1 --> 001 == 1) or (100 == 1 --> 110 == 1) or ...,
	//     and an impossibe group is (011 == 1 --> 100 == 1) or (000 == 0 --> 001 = 1) ...

	int max_groups = pattern_cnt * input_cnt;
	int max_patterns = pattern_cnt;

	// allocate arrays on the heap (up to xxxx MBytes)
	daa_group_dimensions[0] = max_groups;
	daa_group_dimensions[1] = max_patterns;
	if (daa_group) free(daa_group);
	group = (int **)daa(sizeof(int), 2, daa_group_dimensions, start_arry, &daa_ret, &daa_group, (char *)NULL);	// groups to be found in patterns
	if (daa_ret != 0) {
		if (daa_ret == 4) {
			fprintf(stderr, "SymbolOpStack::calculateSymbolicNormalized: unable to allocate group memory\n");
			if (daa_patterns) free(daa_patterns);
			daa_patterns = (char *)NULL;
			if (pattern_idx) free(pattern_idx);
			pattern_idx = (int *)NULL;
			pattern = (int *)NULL;
			return NULL;
		}
	}

	if (group_size) free(group_size);
	group_size = (int *)malloc(sizeof(int) * max_groups);	// no of patterns in the groups
	group_cnt = 0;						// current group_cnt / total number of groups

	daa_table_dimensions[0] = max_patterns;
	daa_table_dimensions[1] = 2;
	table = (int **)daa(sizeof(int), 2, daa_table_dimensions, start_arry, &daa_ret, &daa_table, (char *)NULL);	// pattern with exchanged columns [][value,result]
	if (daa_ret != 0) {
		if (daa_ret == 4) {
			fprintf(stderr, "SymbolOpStack::calculateSymbolicNormalized: unable to allocate table memory\n");
			if (daa_patterns) free(daa_patterns);
			daa_patterns = (char *)NULL;
			if (pattern_idx) free(pattern_idx);
			pattern_idx = (int *)NULL;
			pattern = (int *)NULL;
			return NULL;
		}
	}

	int current_group_size;			// group field size (1, 2, 4, ..)
	int group_start;			// group start at table position
	int curr_pos;				// current group position while scanning

	int bin_a, bin_na, bin_b, bin_nb;	// column exchange
	int oldval, newval;			// pattern value before and after exchange

	for(i=0; i < input_cnt; i++) {
		for (j = i; j < input_cnt; j++) {
			// create tables with exchanged columns: exchange column i with column j
			bin_a = Tool::pow2(i);
			bin_na = Tool::pow2complement(input_cnt - 1, i);
			bin_b = Tool::pow2(j);
			bin_nb = Tool::pow2complement(input_cnt - 1, j);
			for (oldval = 0; oldval < pattern_cnt; oldval++) {
				newval = oldval;
				if (i != j) {
					if (oldval & bin_a) newval |= bin_b;	// set bit
					else newval &= bin_nb;			// clear bit
					if (oldval & bin_b) newval |= bin_a;
					else newval &= bin_na;
				}

				table[newval][0] = oldval;			// remember the original position in patterns
				table[newval][1] = patterns[oldval][input_cnt];	// the result value
			}

			// try to find groups in the new table (leaving single groups)
			for (k = input_cnt - 1; k > 0; k--) {
				current_group_size = Tool::pow2(k);				// beginning with the biggest possible current_group_size
				group_start = 0;
				while (group_start < pattern_cnt) {
					curr_pos = group_start;
					group_size[group_cnt] = 0;
					group[group_cnt][0] = 0;
					found = 1;

					// try find a new group
					while (found && curr_pos < pattern_cnt && group_size[group_cnt] < current_group_size) {
						if (table[curr_pos][1] == subjunctive) {
							group[group_cnt][group_size[group_cnt]++] = table[curr_pos][0];
							curr_pos++;
						}
						else found = 0;
					}

					// validate new group
					if (group_size[group_cnt] < current_group_size) {
						group_size[group_cnt] = 0;		// not enough patterns
					} else {
						// look if group already exists completely in another group
						// the currently found group has a smaller or equal size than all others!
						found = 0;
						ii = 0;
						while (!found && ii < group_cnt) {				// for all old groups..
							all_found = 1;
							ij = 0;
							while (all_found && ij < group_size[group_cnt]) {		// the new group
								single_found = 0;
								ik = 0;
								while (!single_found && ik < group_size[ii]) {	// the old group
									if (group[group_cnt][ij] == group[ii][ik])
										single_found = 1;
									ik++;
								}
								if (!single_found) all_found = 0;
								ij++;
							}
							if (all_found) found = 1;
							ii++;
						}
						if (found) group_size[group_cnt] = 0;
						else group_cnt++;
					}

					// continue search on the next possible starting point
					group_start += current_group_size;

				} // increment group_start
			} // decrement group size
		} // exchange table column
	} // exchange table column

	// ****** remove groups contained in other groups ******
    	if (secure) {
    		// secure method: remove only groups contained in one other group (gliches are surely excluded)
    		i = 0;
    		while (i < group_cnt) {
    			// look if group already exists completely in another group
    			// the currently found group has a smaller or equal size than all others!
    			found = 0;
    			j = 0;
    			while (!found && j < group_cnt) {				// for all other groups..
    				if (i == j || group_size[i] > group_size[j]) all_found = 0;
    				else all_found = 1;
    				ii = 0;
    				while (all_found && ii < group_size[i]) {		// the current group
    					single_found = 0;
    					jj = 0;
    					while (!single_found && jj < group_size[j]) {	// the other group
    						if (group[i][ii] == group[j][jj])
    							single_found = 1;
    						jj++;
    					}
    					if (!single_found) all_found = 0;
    					ii++;
    				}
    				if (all_found) found = 1;
    				j++;
    			}
    			if (found) {
    				for (j = i + 1; j < group_cnt; j++) {
					for (jj=0; jj < group_size[j]; jj++)
						group[j - 1][jj] = group[j][jj];
    					group_size[j - 1] = group_size[j];
    				}
    				group_cnt--;
    			} else i++;
    		}
    	} else {
		// insecure method: could cause glitches in cause of different delays of the gates
		// remove all groups contained in n other groups
		// beginning with the smallest groups
		int all_grp[pattern_cnt * group_cnt];
		int all_grp_cnt = 0;
		for (i = 1; i < input_cnt; i++) {
			current_group_size = Tool::pow2(i);
			j = 0;
			while (j < group_cnt) {
				if (group_size[j] == current_group_size) {
					// create list with all available values
					all_grp_cnt = 0;
					for (k = 0; k < group_cnt; k++) {
						if (k != j) {
							for (kk=0; kk < group_size[k]; kk ++) {
								all_grp[all_grp_cnt++] = group[k][kk];
							}
						}
					}
					// look if all values of the current group are contained in all_grp
					all_found = 1;
					jj = 0;
					while (all_found && jj < current_group_size) {
						single_found = 0;
						k = 0;
						while (!single_found && k < all_grp_cnt) {
							if (all_grp[k] == group[j][jj]) single_found = 1;
							k++;
						}
						if (!single_found) all_found = 0;
						jj++;
					}
					// remove group if all pattern were found
					if (all_found) {
						for (k = j + 1; k < group_cnt; k++) {
							for (kk=0; kk < group_size[k]; kk++)
								group[k - 1][kk] = group[k][kk];
							group_size[k - 1] = group_size[k];
						}
						group_cnt--;
					} else j++;
				} else j++;
			}
		}
	}

	// ****** at least add single groups, hopefully some lost bits ******
	for (i = 0; i < pattern_cnt; i++) {
		if (patterns[i][input_cnt] == subjunctive) {
			found = 0;
			j = 0;
			while (!found && j < group_cnt) {
				jj = 0;
				while (!found && jj < group_size[j]) {
					if (i == group[j][jj]) found = 1;
					jj++;
				}
				j++;
			}
		} else found = 1;
		if (!found) {
			group[group_cnt][0] = i;
			group_size[group_cnt] = 1;
			group_cnt++;
		}
	}

	// ****** create an equation from the groups found ******
	char newpart[1024];
	char *result;
	unsigned int current_size = 16384;
	if (tmp_equ) free(tmp_equ);
	tmp_equ = (char *)malloc(sizeof(char) * current_size + 1);
	tmp_equ[0] = 0;

	int bit_value;
	int first_part = 1;
	for (i = 0; i < group_cnt; i++) {
		// first group: open parenthesis
		if (i == 0 && !subjunctive) strcat(tmp_equ, "(");

		// in between groups
		if (i != 0) {
			if (subjunctive) strcat(tmp_equ, " + ");
			else strcat(tmp_equ, ") & (");
		}

		first_part = 1;
		// figure out the bits needed for the equation
		for (bit = 0; bit < input_cnt; bit++) {
			// scan the whole group if the bit toggles
			// if it does not toggle, the bit is needed, otherwise not
			bit_found = 1;
			bit_value = patterns[group[i][0]][bit];		// the value of the current bit of the first pattern
			ii = 0;
			while (bit_found && group_size[i] > 1 && ii < group_size[i]) {
				if (bit_value != patterns[group[i][ii]][bit]) bit_found = 0;
				ii++;
			}
			if (bit_found) {
				// create the equation part related to the current bit
				if (first_part) {
					if (subjunctive) {
						if (bit_value) sprintf(newpart, "%s", input_list[bit]);
						else sprintf(newpart, "/%s", input_list[bit]);
					} else {
						if (bit_value) sprintf(newpart, "/%s", input_list[bit]);
						else sprintf(newpart, "%s", input_list[bit]);
					}
					first_part = 0;
				} else {
					if (subjunctive) {
						if (bit_value) sprintf(newpart, " & %s", input_list[bit]);
						else sprintf(newpart, " & /%s", input_list[bit]);
					} else {
						if (bit_value) sprintf(newpart, " + /%s", input_list[bit]);
						else sprintf(newpart, " + %s", input_list[bit]);
					}
					
				}

				// get enough memory
				if (current_size < (strlen(newpart) + strlen(tmp_equ) + 1024)) {
					current_size += 16384;
					result = (char *)malloc(sizeof(char) * current_size + 1);
					strcpy(result, tmp_equ);
					free(tmp_equ);
					tmp_equ = result;
				}

				// add new part
				strcat(tmp_equ, newpart);
			} // bit found
		} // next bit

		// last group: close parenthesis
		if (i == group_cnt - 1 && !subjunctive) strcat(tmp_equ, ")");

	} // next group

	free(daa_table);
	return tmp_equ;

	// ******** debugging part ************

	// print original table
	printf("\n******************************\n");
	printf("\noriginal: %s\n", equation);
	printf("******************************\n");
	for (i=input_cnt - 1; i >=0;i--) {
		if (i < input_cnt - 1) printf(" ");
		printf("%s", input_list[i]);
		if (i == 0) printf("\n");
	}
	for (i=0; i < pattern_cnt;i++) {
		for (j=input_cnt - 1; j >= 0;j--)
			printf("%d", patterns[i][j]);
		printf (" %d\n", patterns[i][input_cnt]);
	}

	// print found groups
	printf("%d groups found:\n", group_cnt);
	for (i = 0; i < group_cnt; i++) {
		printf("group contains %d pattern\n", group_size[i]);
		for (j=0; j < group_size[i]; j++) {
			for (bit = input_cnt - 1; bit >= 0; bit--) {
				if (group[i][j] & Tool::pow2(bit)) printf("1");
				else printf("0");
			}
			printf("\n");
		}
		printf("\n");
	}

	// print new table
	printf("******************************\n");
	printf("new: %s\n", tmp_equ);
	printf("******************************\n");
	setEquation(tmp_equ);
	parse();
	calculatePattern();

	for (i=input_cnt - 1; i >=0;i--) {
		if (i < input_cnt - 1) printf(" ");
		printf("%s", input_list[i]);
		if (i == 0) printf("\n");
	}
	for (i=0; i < pattern_cnt;i++) {
		for (j=input_cnt - 1; j >= 0;j--)
			printf("%d", patterns[i][j]);
		printf (" %d\n", patterns[i][input_cnt]);
	}

	free(daa_table);
	free(daa_group);
	free(group_size);
	return tmp_equ;
}

// return the first/next symbol of the queue
const char * SymbolOpStack::getSymbol(int no)
{
	if (!no) scanInputList();

	if (!input_list || !input_cnt) return (const char *)NULL;
	if (no >= 0 && no < input_cnt) return input_list[no];

	return (const char *)NULL;
}

// allocate a list with all input strings
void SymbolOpStack::scanInputList()
{	int i, j, found, plane;

	if (input_list) {
		for (i = 0; i < input_cnt; i++)
			free(input_list[i]);
		free (input_list);
	}

	// create unique list with all inputs
	input_list = (char **)malloc(sizeof(char *) * maxitem);
	pattern_idx = (int *)malloc(sizeof(int) * total);
	plane = -1;
	input_cnt = 0;
	for (i = 0; i < total; i++) {
		switch (queue[i][0]) {
			case OP_AND:
			case OP_OR:
			case OP_XOR:
			case OP_NOT:
			case OP_BIN:
			case OP_BIT:
			case OP_NONE:
			case OP_INTERFACE:
				break;
			default:
				j = 0;
				found = 0;
				plane++;
				while (j < input_cnt && !found) {
					if (!strcmp(input_list[j], queue[i])) {
						found = 1;
						pattern_idx[i] = j;
					}
					j++;
				}
				if (!found) {
					if (input_cnt > maxitem) {
						fprintf(stderr, "opstack: analyze: input overflow??\n");
						exit(-1);
					}
					input_list[input_cnt] = strdup(queue[i]);
					pattern_idx[i] = input_cnt;
					input_cnt++;
				}
				break;
		}
	}
}

// stack calculation for the whole number of possible patterns
void SymbolOpStack::calculatePattern()
{	int i, j;
	unsigned daa_ret;
	int start_arry[2] = {0,0};

	if (daa_patterns) free(daa_patterns);
	daa_patterns = (char *)NULL;
	if (pattern_idx) free(pattern_idx);
	pattern_idx = (int *)NULL;
	pattern = (int *)NULL;

	scanInputList();
	if (!input_list || !input_cnt) return;

	// ****** number of possible pattern ******
	pattern_cnt = Tool::pow2(input_cnt);
	if (!pattern_cnt) return;

	daa_patterns_dimensions[0] = pattern_cnt;
	daa_patterns_dimensions[1] = input_cnt + 1;
	patterns = (int **)daa(sizeof(int), 2, daa_patterns_dimensions, start_arry, &daa_ret, &daa_patterns, (char *)NULL);
	if (daa_ret == 4) {
		fprintf(stderr, "SymbolOpStack::calculatePattern: unable to allocate memory\n");
		if (daa_patterns) free(daa_patterns);
		daa_patterns = (char *)NULL;
		if (pattern_idx) free(pattern_idx);
		pattern_idx = (int *)NULL;
		pattern = (int *)NULL;
		return;
	}

	// ****** calculate patterns ******
	pattern_mode = 1;
	for (i = 0; i < pattern_cnt; i++) {
		// set input
		pattern = patterns[i];
		for (j = 0; j < input_cnt; j++) {
			if (i & Tool::pow2(j)) pattern[j] = 1;
			else pattern[j] = 0;
		}
		// calculate output
		pattern[input_cnt] = calculate(true);
	}
	pattern_mode = 0;
}

// figure out wether string needs an parenthesis
// assumes that there are no parenthesis errors!
int SymbolOpStack::needsParenthesis(char * eq, char op)
{
	char *par = strchr(eq, '(');
	if (!par) {
		switch (op) {
			case OP_NOT:
				if (strchr(eq, OP_XOR)) return 2;
				if (strchr(eq, OP_AND)) return 2;
				if (strchr(eq, OP_OR)) return 2;
				return 0;
				break;
			case OP_XOR:
				if (strchr(eq, OP_AND)) return 2;
				if (strchr(eq, OP_OR)) return 2;
				return 0;
				break;
			case OP_AND:
				if (strchr(eq, OP_OR)) return 2;
				return 0;
				break;
			default:
				return 0;
				break;
		}
	}
	char * deq = strrchr(eq, ')');
	if (!deq) fatal("fatal parenthesis error in class opStack!\n");
	*par = 0;
	switch (op) {
		case OP_NOT:
			if (strchr(deq, OP_XOR) || strchr(eq, OP_XOR)) {
				*par = '(';
				return 2;
			}
			if (strchr(deq, OP_AND) || strchr(eq, OP_AND)) {
				*par = '(';
				return 2;
			}
			if (strchr(deq, OP_OR) || strchr(eq, OP_OR)) {
				*par = '(';
				return 2;
			}
			break;
		case OP_XOR:
			if (strchr(deq, OP_AND) || strchr(eq, OP_AND)) {
				*par = '(';
				return 2;
			}
			if (strchr(deq, OP_OR) || strchr(eq, OP_OR)) {
				*par = '(';
				return 2;
			}
			break;
		case OP_AND:
			if (strchr(deq, OP_OR) || strchr(eq, OP_OR)) {
				*par = '(';
				return 2;
			}
			break;
	}
	*par = '(';
	return 0;
}

// virtual
// retrieve pattern by opStack
int SymbolOpStack::getPattern(int idx)
{
	return pattern[pattern_idx[idx]];
}

// returns a set of all karnaugh groups found in the patterns of the current equation
//    the list of groups in the "_group_list" argument
//    the size of the groups in the "_group_size" argument
//    the number of groups in the "_group_cnt" argument
// the pointers may become invalid on further operations with the same SymbolOpStack instance!!!
// the pointers will become invalid on destroying this instance!
void SymbolOpStack::getGroups(int ***_group_list, int **_group_size, int *_group_cnt)
{
	if (!daa_group) calculateSymbolicNormalized(1,0);
	*_group_list = group;
	*_group_size = group_size;
	*_group_cnt = group_cnt;
}

// returns a set of all inputs, and all patterns which are possible,
//    the list of patterns in the "_pattern_list" argument
//    the number of patterns in the "_pattern_cnt" argument
//    the list of inputs in the "_input_list" argument
//    the number of inputs in the "_input_cnt" argument
// the pointers may become invalid on further operations with the same SymbolOpStack instance!!!
// the pointers will become invalid on destroying this instance!
void SymbolOpStack::getPatterns(char ***_input_list, int *_input_cnt, int ***_pattern_list, int *_pattern_cnt)
{
	if (!daa_patterns) calculatePattern();
	*_pattern_cnt = pattern_cnt;
	*_pattern_list = patterns;
	*_input_list = input_list;
	*_input_cnt = input_cnt;
}

