// ------------------------------------------------
// JK flipflop device implementation
//
// Andreas Rostin
// 21.02.2003
// ------------------------------------------------

#ifndef __DEVICE_JKFF__
#define __DEVICE_JKFF__

#include <qpainter.h>
#include "xdevice.h"

class JKFFDevice : public XDevice
{
public:
	JKFFDevice(int x, int y, int size = -1, int = -1, int undef = -1, int clock = -1)
		: XDevice(DeviceType::fJK, x, y, size, 0, undef, clock)
	{
		init();

	}

	virtual ~JKFFDevice()
	{
	}

	// initialize device
	virtual void init()
	{
		deviceOutputInverted = false;
		addInputName(INPUT_FF_1J, 0);
		addInputName(INPUT_FF_S, 1);
		addInputName(INPUT_FF_C1, 2);
		addInputName(INPUT_FF_R, 3);
		addInputName(INPUT_FF_1K, 4);
		mq_id = addInternalName(INTERNAL_FF_MASTER_Q);
		m_c1_1_id = addInternalName(INTERNAL_C1_PREVIOUS);

		setEquation();
		parseEquation();
		setImage();
	}

	virtual void * getInstance()
	{
		return (void *)this;
		
	}

	virtual void setEquation()
	{
		if (master) {
			// 17.02.2001: asynchroneous RS inputs added
			// 21.02.2003: single edge usual JK added
			switch(clock_type) {
				case CLK_RISING_2EDGE:
					Device::setEquation("(/(C1 & /C1-1) & MQ + /1J & /1K & MQ + C1 & /C1-1 & 1J & /1K + C1 & /C1-1 & 1J & 1K & /MQ) & /R + S", mq_id);
					Device::setEquation("C1", m_c1_1_id);                                           // C1-1 gets the old value of C1: detect edge
					Device::setEquation("(/(/C1 & C1-1) & _this_ + /C1 & C1-1 & MQ) & /R + S");     // equation for output without name assignment
					break;
				case CLK_FALLING_2EDGE:
					Device::setEquation("(/(/C1 & C1-1) & MQ + /1J & /1K & MQ + /C1 & C1-1 & 1J & /1K + /C1 & C1-1 & 1J & 1K & /MQ) & /R + S", mq_id);
					Device::setEquation("C1", m_c1_1_id);
					Device::setEquation("(/(C1 & /C1-1) & _this_ + C1 & /C1-1 & MQ) & /R + S");
					break;
				case CLK_RISING_1EDGE:
					Device::setEquation("C1 & MQ + (1J & /_this_ + /1K & _this_) & /C1", mq_id);    // "MQ" is mq_id output value in clk+1
					Device::setEquation("(/C1 & _this_ + C1 & MQ) & /R + S");       // equation for output without name assignment, also called "_this_"
					break;
				case CLK_FALLING_1EDGE:
					Device::setEquation("/C1 & MQ + (1J & /_this_ + /1K & _this_) & C1", mq_id);    // "MQ" is mq_id output value in clk+1
					Device::setEquation("(C1 & _this_ + /C1 & MQ) & /R + S");       // equation for output without name assignment, also called "_this_"
					break;
				case CLK_HIGH_VALUE:
					Device::setEquation("(/C1 & MQ + /1J & /1K & MQ + C1 & 1J & /1K + C1 & 1J & 1K & /MQ) & /R + S", mq_id);
					Device::setEquation((char *)NULL, m_c1_1_id);
					Device::setEquation("(C1 & _this_ + /C1 & MQ) & /R + S");
					break;
				case CLK_LOW_VALUE:
					Device::setEquation("(C1 & MQ + /1J & /1K & MQ + /C1 & 1J & /1K + /C1 & 1J & 1K & /MQ) & /R + S", mq_id);
					Device::setEquation((char *)NULL, m_c1_1_id);
					Device::setEquation("(/C1 & _this_ + C1 & MQ) & /R + S");
					break;
			}
		} else {
			fatal("Device::setEquation: JK without master??");
			exit(-1);
		}
	}

	virtual bool hasClock()
	{
		return true;
	}

	virtual bool hasNamedInput()
	{
		return true;
	}

	virtual bool hasNamedOutput()
	{
		return false;
	}

	virtual bool sizeChangeable()
	{
		return false;
	}

	virtual void setImage()
	{	QPixmap *pic = getImage();
		QPainter p;
		QPoint pt(0,0);
		QPixmap *part = (QPixmap *)NULL;

		if (clock() == CLK_RISING_1EDGE || clock() == CLK_FALLING_1EDGE ||  clock() == CLK_FALLING_2EDGE || clock() == CLK_RISING_2EDGE) part = pxJKF;
		else part = pxJKF;
		pic->resize(part->width(), part->height());

		p.begin(pic);
		if (!part) fatal("no pixmap??");
		p.drawPixmap(pt, *part);
		p.drawLine(pic->width() - 9, 3, pic->width() - 9, 7);
		switch (clock()) {
			case CLK_FALLING_1EDGE:
			case CLK_FALLING_2EDGE:
			case CLK_LOW_VALUE:
				p.drawLine(pic->width() - 15, 3, pic->width() - 9, 3);
				break;
			case CLK_RISING_1EDGE:
			case CLK_RISING_2EDGE:
			case CLK_HIGH_VALUE:
				p.drawLine(pic->width() - 15, 7, pic->width() - 9, 7);
				break;
			default:
				break;
		}

		// draw connection lines of the new image
		drawConnectionLines(&p);

		// set image dimension, clear position
		XObject::setImage();

		p.end();

		if (text_dev) text_dev->setImage();
	}

	virtual void printImage(QPainter *p, const QRect &r)
	{	KlogicList<Value> *lv;
		int pos;
		int a,b,c;
		int picwidth = getImage()->width();

		p->setFont(STATfont);
		lv = named_input.First();
		while(lv) {
			pos = lv->Get()->getPosition() * GRID;
			if (strcmp(lv->getText(), "C1") == 0 && hasMaster()) {
				a = r.x();
				b = r.x() + 6;
				c = r.y() + pos + 2;
				p->drawLine(a, c - 3, b, c);
				p->drawLine(a, c + 3, b, c);
				p->drawText(r.x() + 6, r.y() + pos + 5, lv->getText());
			}
			lv = lv->Next();
		}
		p->drawText(r.x() + (picwidth / 2), r.y() - 2, picwidth / 2 - 7, 10, Qt::AlignRight, "Q");
	}

	virtual bool outputIsActive()
	{
		return true;
	}

private:
	static const QString INPUT_FF_1J;
	static const QString INPUT_FF_1K;
};

const QString JKFFDevice::INPUT_FF_1J = "1J";
const QString JKFFDevice::INPUT_FF_1K = "1K";

#endif
