/**********************************************************************************************
    Copyright (C) 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

  Garmin and MapSource are registered trademarks or trademarks of Garmin Ltd.
  or one of its subsidiaries.

**********************************************************************************************/
#include "CGarminTrack.h"
#include "IProjection.h"

#include <QtCore>

// ------------------------------------------------------------
// ---------------- CGarminTrack::TrkPt_t -------------------
// ------------------------------------------------------------

CGarminTrack::TrkPt_t::TrkPt_t()
: idx(0xffffffff)
, speed(-1)
, delta(0)
, azimuth(0)
, distance(0)
, flags(0)
{
}


CGarminTrack::TrkPt_t::TrkPt_t(const Garmin::TrkPt_t& pt, quint32 idx)
: Garmin::TrkPt_t(pt)
, idx(idx)
, speed(-1)
, delta(0)
, azimuth(0)
, distance(0)
, flags(0)
{
}


// ------------------------------------------------------------
// --------------------- CGarminTrack -------------------------
// ------------------------------------------------------------
const QColor CGarminTrack::colors[] =
{
    Qt::black                    // 0
    ,Qt::darkRed                 // 1
    ,Qt::darkGreen               // 2
    ,Qt::darkYellow              // 3
    ,Qt::darkBlue                // 4
    ,Qt::darkMagenta             // 5
    ,Qt::darkCyan                // 6
    ,Qt::gray                    // 7
    ,Qt::darkGray                // 8
    ,Qt::red                     // 9
    ,Qt::green                   // 10
    ,Qt::yellow                  // 11
    ,Qt::blue                    // 12
    ,Qt::magenta                 // 13
    ,Qt::cyan                    // 14
    ,Qt::white                   // 15
    ,Qt::transparent             // 16
};

CGarminTrack::CGarminTrack(QObject * parent)
: QObject(parent)
, highlight(false)
, cntSibling(0)
, totalTime(0)
, totalDistance(0)
{
    color = QColor(Garmin::Track_t::color > 16 ? Qt::darkBlue : CGarminTrack::colors[Garmin::Track_t::color]);
}


CGarminTrack::CGarminTrack(QObject * parent,Garmin::Track_t trk)
: QObject(parent)
, Garmin::Track_t(trk)
, highlight(false)
, cntSibling(0)
, totalTime(0)
, totalDistance(0)
{
    ident = trk.ident.c_str();
    color = QColor(Garmin::Track_t::color > 16 ? Qt::darkBlue : CGarminTrack::colors[Garmin::Track_t::color]);
    std::vector<Garmin::TrkPt_t>::const_iterator pt = trk.track.begin();
    while(pt != trk.track.end()) {
        track << TrkPt_t(*pt,track.size());
        ++pt;
    }
    update();
}


CGarminTrack::~CGarminTrack()
{
    qDebug() << "~CGarminTrack()";
}


void CGarminTrack::push_back(TrkPt_t& pt)
{
    track.push_back(pt);
    track.last().idx     = track.size() - 1;
    track.last().flags  &= ~eCursor;
    track.last().flags  &= ~eFocus;
    track.last().flags  &= ~eSelected;
}


void CGarminTrack::setColor(unsigned i)
{
    if(i>16) i = 4;
    color = colors[i];
    Garmin::Track_t::color = i;
}


void CGarminTrack::update(bool reindex)
{
    quint32 t1 = 0, t2 = 0;
    QVector<TrkPt_t>::iterator pt1 = track.begin();
    QVector<TrkPt_t>::iterator pt2 = track.begin();

    totalTime       = 0;
    totalDistance   = 0;

    // reindex track if desired
    if(reindex) {
        quint32 idx = 0;
        while(pt1 != track.end()) {
            pt1->idx = idx++;
            ++pt1;
        }
        pt1 = track.begin();
    }

    // skip leading deleted points
    while((pt1->flags & eDeleted) && (pt1 != track.end())) {
        pt1->azimuth    = 0;
        pt1->delta      = 0;
        pt1->speed      = -1;
        pt1->distance   = 0;
        ++pt1; ++pt2;
    }

    // no points at all?
    if(pt1 == track.end()) {
        emit sigChanged();
        return;
    }
    // reset first valid point
    pt1->azimuth    = 0;
    pt1->delta      = 0;
    pt1->speed      = -1;
    pt1->distance   = 0;
    t1              = pt1->time;

    // process track
    while(++pt2 != track.end()) {

        // reset deleted points
        if(pt2->flags & eDeleted) {
            pt2->azimuth    = 0;
            pt2->delta      = 0;
            pt2->speed      = -1;
            pt2->distance   = 0;
            continue;
        }

        int dt = -1;
        if(pt1->time != 0x00000000 && pt1->time != 0xFFFFFFFF) {
            dt = pt2->time - pt1->time;
        }

        XY p1,p2;
        p1.u = DEG_TO_RAD * pt1->lon;
        p1.v = DEG_TO_RAD * pt1->lat;
        p2.u = DEG_TO_RAD * pt2->lon;
        p2.v = DEG_TO_RAD * pt2->lat;
        double a2 = 0;

        pt2->delta      = gpProj->distance(p1,p2,pt1->azimuth,a2);
        pt2->distance   = pt1->distance + pt2->delta;
        pt2->speed      = (dt > 0) ? pt2->delta / dt * 3.6 : 0;

        t2              = pt2->time;
        totalDistance   = pt2->distance;

        pt1 = pt2;
    }

    totalTime = t2 - t1;

    emit sigChanged();
}


void  CGarminTrack::setPointOfFocus(qint32 idx)
{
    QVector<TrkPt_t>::iterator trkpt = track.begin();
    while(trkpt != track.end()) {
        trkpt->flags &= ~eFocus;
        ++trkpt;
    }

    if(idx < track.size()) {
        track[idx].flags |= eFocus;
    }

    emit sigChanged();
}


QVector<CGarminTrack::TrkPt_t>::iterator CGarminTrack::getPointOfFocus()
{
    QVector<TrkPt_t>::iterator trkpt = track.begin();
    while(trkpt != track.end()) {
        if(trkpt->flags & eFocus) break;
        ++trkpt;
    }
    return trkpt;
}


CGarminTrack& CGarminTrack::operator+=(CGarminTrack& t)
{
    ident += " " + t.ident;
    track += t.track;
    update(true);
    return *this;
}


const QString& CGarminTrack::getSiblingName()
{
    nameSibling = ident + QString(" %1").arg(++cntSibling);
    return nameSibling;
}


void CGarminTrack::addTrkPt(TrkPt_t& pt)
{
    pt.idx = track.size();
    track.push_back(pt);
};
