#include <X11/X.h>
#include <X11/Xlib.h>
#include <math.h>
#include "trippy.h"

typedef struct vt {  /* Structure for individual vertices */
  int lx,ly,lz;    /* Local coordinates of vertex*/
  int wx,wy,wz;    /* World coordinates of vertex */
  int sx,sy,sz;    /* Screen coordinates of vertex*/
} vertex_type;

static float matrix[16];         /* Master transformation matrix */
void matmult(float res[16],float mat1[16],float mat2[16]);
void matcopy(float dest[16],float source[16]);

/* Transformation functions: */

void init_matrix()
{
/*  Set the master matrix to |  1  0  0  0 | 
 *                           |  0  1  0  0 |
 *                           |  0  0  1  0 |
 *                           |  0  0  0  1 |
 */

  memset(matrix,0,16*sizeof(float));
  matrix[0]=matrix[5]=matrix[10]=matrix[15]=1.0;
}

void scale(float sf)
{
  /*
   *  Multiply the master matrix by the scaling factor (sf)
   *
   *
   *  Note that assuming that this funciton will be run right after
   *  the init function, we could reduce the code to:
   *     smat[0]=smat[5]=smat[10]=sf;
   *
   * since that is the only change.
   */
  float mat[16];
  float smat[16];

  memset(smat,0,16*sizeof(float));
  smat[0]=smat[5]=smat[10]=sf;
  smat[15]=1;
  
  matmult(mat,smat,matrix);
  matcopy(matrix,mat);
}

void rotate(float ax,float ay,float az)
{
  /*
   * Factor in the rotations in the X,Y, and Z axes
   */
  float zmat[16];
  float xmat[16];
  float ymat[16];

  float mat1[16];
  float mat2[16];

  float tempCos = cos(ax);
  float tempSin = sin(ax);
  memset(xmat,0,16*sizeof(float));
  xmat[0]=xmat[15]=1;
  xmat[5]=xmat[10]=tempCos;
  xmat[6]=tempSin; xmat[9]= -tempSin;
  matmult(mat1,xmat,matrix);

  tempCos = cos(ay);
  tempSin = sin(ay);
  memset(ymat,0,16*sizeof(float));
  ymat[0]=ymat[10]=tempCos;
  ymat[2]=-tempSin; ymat[8]=tempSin;
  ymat[5]=ymat[15]=1;
  matmult(mat2,ymat,mat1);

  tempCos = cos(az);
  tempSin = sin(az);
  memset(zmat,0,16*sizeof(float));
  zmat[0]=zmat[5]=tempCos;
  zmat[1]=tempSin; zmat[4]=-tempSin;
  zmat[10]=zmat[15]=1;
  matmult(matrix,zmat,mat2);
}

void translate(int xt,int yt,int zt)
{
  float tmat[16];
  float mat[16];

  memset(tmat,0,16*sizeof(float));
  tmat[0]=tmat[5]=tmat[10]=tmat[15]=1;
  tmat[12]=xt; tmat[13]=yt; tmat[14]=zt; 

  matmult(mat,matrix,tmat);
  matcopy(matrix,mat);
}

void transform(int numvert, vertex_type *vertices)
{
  int v;
  vertex_type *vptr=vertices;
  for (v=0; v<numvert; v++)
  {
    vptr->wx=(int)(vptr->lx*matrix[0]+vptr->ly*matrix[4]+
		   vptr->lz*matrix[8]+matrix[12]);
    vptr->wy=(int)(vptr->lx*matrix[1]+vptr->ly*matrix[5]+
		   vptr->lz*matrix[9]+matrix[13]);
    vptr->wz=(int)(vptr->lx*matrix[2]+vptr->ly*matrix[6]+
		   vptr->lz*matrix[10]+matrix[14]);
    vptr++;
  }
}

void project(int distance, int numvert, vertex_type* vertices)
{
/* Project shape onto screen */
  int v;
  vertex_type *vptr=vertices;
  
  /* Loop though vertices: */
  for (v=0; v<numvert; v++,vptr++)
  {
    /* Divide world x & y coords by z coords: */
/* if vptr->wz = 0,  something is not right with the world */
    vptr->sx= (distance*vptr->wx);
    vptr->sx /= vptr->wz;
    vptr->sy= (distance*vptr->wy);
    vptr->sy /= vptr->wz;
  }
}

void matmult(float result[16],float mat1[16], float mat2[16])
{
  int i,j,k;
    
/* Multiply matrix MAT1 by matrix MAT2,
 *  returning the result in RESULT
 */
  float *res = result;
  for (i=0; i<4; i++)
  {
    int i4 = 4*i;
    for (j=0; j<4; j++)
    {
      *res= mat1[i4] * mat2[j];
      *res += mat1[i4+1] * mat2[4+j];
      *res += mat1[i4+2] * mat2[8+j];
      *res += mat1[i4+3] * mat2[12+j];
      res++;
    }
  }
  return;
}

void matcopy(float dest[16], float source[16])
{
/* Copy matrix SOURCE to matrix DEST */

  memcpy(dest,source,16*sizeof(float));
}

void drawFace(int winno,vertex_type **vertices,int clr)
{
  int i;
  long x1= vertices[0]->sx;
  long y1= vertices[0]->sy;
  long x2= vertices[3]->sx;
  long y2= vertices[3]->sy;
  long x3,x4,y3,y4;
  long dx1 = ((x3=vertices[1]->sx)-x1)*256 /24;
  long dy1 = ((y3=vertices[1]->sy)-y1)*256 /24; 
  long dx2 = ((x4=vertices[2]->sx)-x2)*256 /24; 
  long dy2 = ((y4=vertices[2]->sy)-y2)*256 /24; 
  long XORIGIN= CX[winno]>>1;
  long YORIGIN= CY[winno]>>1;
  
  x1+=XORIGIN; x2+=XORIGIN; x3+=XORIGIN; x4+=XORIGIN;
  y1+=YORIGIN; y2+=YORIGIN; y3+=YORIGIN; y4+=YORIGIN;

  XDrawLine(display,window[winno],color_gcs[clr],x1,y1,x3,y3);
  XDrawLine(display,window[winno],color_gcs[clr],x2,y2,x4,y4);
  x1<<=8; y1<<=8; x2<<=8; y2<<=8;
  for(i=0;i<=24;i++)
  {
    XDrawLine(display,window[winno],color_gcs[clr],x1>>8,y1>>8,x2>>8,y2>>8);   
    x1+=dx1; y1+=dy1;
    x2+=dx2; y2+=dy2;
  }
/*  int *temp = new int [2*num_edges];
    
   for(i=0;i<num_edges*2;i+=2)
  {
    int ind = i>>1;
    temp[i]= vertices[v_index[ind]].sx+XORIGIN;
    temp[i+1]= vertices[v_index[ind]].sy+YORIGIN;
  }
  GrFilledConvexPolygon(num_edges,(int[][2])temp,color);
  delete temp; 
*/
}

int
backface(vertex_type **vertices)
{
/* 	 Returns 0 if POLYGON is visible, -1 if not.
 *   POLYGON must be part of a convex polyhedron
 */
   int z; 
   vertex_type *v0,*v1,*v2;  /* Pointers to three vertices */
   /* Point to vertices: */
   v0=vertices[0];
   v1=vertices[1];
   v2=vertices[2];
   z=(v1->sx-v0->sx)*(v2->sy-v0->sy)-(v1->sy-v0->sy)*(v2->sx-v0->sx);
   return(z>=0);
}

void draw_shape(int winno, int clr, int max, int numfaces, vertex_type **vert)
{
  int i;
  int j=0;
  for(i=0; i<numfaces; i++)
  {
    if (backface(vert+(i*4))) /* times 4 to do 2-d indexing in the 1-d 
                                 vert array */
    {
      drawFace(winno,(vert+(i*4)),(clr)?(0):(numcolors*(i%3)/6)+2);
      j++;
    }
    if(j>max)
      break;
  } 
}

vertex_type cube_vertices[]=
{  /* Vertices for cube */
  {-10,-10,10,		
   0,0,0,
   0,0,0},
  {10,-10,10,		
   0,0,0,
   0,0,0},
  { 10,10,10,		
    0,0,0,
    0,0,0},
  {-10,10,10,		
   0,0,0,
   0,0,0},
  {-10,-10,-10,		
   0,0,0,
   0,0,0},
  {10,-10,-10,		
   0,0,0,
   0,0,0},
  {10,10,-10,	     
   0,0,0,
   0,0,0},
  {-10,10,-10,
   0,0,0,
   0,0,0
  }
};

vertex_type *cube_faces[24] =
{
  &cube_vertices[0],&cube_vertices[1],&cube_vertices[5],&cube_vertices[4],
  &cube_vertices[5],&cube_vertices[6],&cube_vertices[7],&cube_vertices[4],
  &cube_vertices[2],&cube_vertices[6],&cube_vertices[5],&cube_vertices[1],
  &cube_vertices[6],&cube_vertices[2],&cube_vertices[3],&cube_vertices[7],
  &cube_vertices[2],&cube_vertices[1],&cube_vertices[0],&cube_vertices[3],
  &cube_vertices[4],&cube_vertices[7],&cube_vertices[3],&cube_vertices[0]
};

void
rotate_cube(int winno)
{
  static float xangle = 0.0, yangle= 0.0, zangle = 0.0;
  static float xrot = 0.1, yrot=0.1, zrot=0.1;
  static int xloc=0,  yloc=0, zloc=50;

  winno=0;  /* only deal with one window... for now */
  
  init_matrix();                    
  scale(1.5);
  rotate(xangle,yangle,zangle);   
  xangle+=xrot;                   
  yangle+=yrot;                    
  zangle+=zrot;
  translate(xloc,yloc,50); 
  
  transform(8, cube_vertices);
                                    
  draw_shape(winno,1,3,6,cube_faces);		

  project(400,8,cube_vertices);
  
  draw_shape(winno,0,3,6,cube_faces);              

  if(random()%500==0) xrot*=-1;   
  if(random()%500==0) yrot*=-1;  
  if(random()%500==0) zrot*=-1;
  return;
}

