/* Copyright (c) 2000  Kevin Sullivan <nite@gis.net>
 *
 * Please refer to the COPYRIGHT file for more information.
 */

/* this file contains procedures related to the main window: drawing,
   redrawing, scrolling, user input, etc. */

#include <stdio.h>
#include <termios.h>
#ifndef MCURSES
  #include <ncurses.h>
#endif
#include <string.h>
#include <sys/ioctl.h>
#include <netdb.h>

#include "defines.h"
#include "codes.h"
#include "nap.h"
#include "colors.h"
#include "winio.h"
#include "alias.h"

extern info_t info;
extern unsigned char finp, ircmode, wmode;
extern chans_t *curchan;
extern file_t *up, *down;
extern scount_t scount;
extern char *mnick;
extern unsigned char noprint;
extern int ircsock;
extern void *cbk;
extern alias_t *alhead;
extern out_nap_cmd_t out[];

WINDOW *wchan=NULL, *winput, *sep, *whead, *swin;
int curx;
int tind=0;
scrls_t *mscroll, *scur;
chans_t *recent = NULL;
cmds_t *cmdl, *ccmd;
int curr = 0;
ssearch_t *search = NULL;
unsigned char discard, bspace;

colortab_t colortab[] = {
#ifndef MCURSES
  { RED, CPR, 1 },
  { GREEN, CPG, 1 },
  { WHITE, CPW, 1 },
  { BLUE, CPB, 1 },
  { YELLOW, CPY, 1 },
  { MAGENTA, CPM, 1 },
  { CYAN, CPC, 1, },
  { BOLD, A_BOLD, 0 },
  { DARK, A_DIM, 0 },
  { BLINK, A_BLINK, 0 },
  { UNDERLINE, A_UNDERLINE, 0 },
#endif
  { NULL, -1, 0 }
};

#ifndef MCURSES

void resize()
{
  sock_t *t;
  cmds_t *cur;
  struct winsize ws;

  memset(&ws, 0, sizeof(ws));
  ioctl(fileno(stdin), TIOCGWINSZ, &ws);
  
  resizeterm(ws.ws_row, ws.ws_col);
  winput->_begy = LINES-1;
  winput->_begx = 0;
  sep->_begy = LINES-2;
  sep->_begx = 0;
  if (info.notop)
    wresize(wchan, LINES-2, COLS);
  else
    wresize(wchan, LINES-3, COLS);
  wresize(winput, 1, COLS);
  if (!info.notop)
    wresize(whead, 1, COLS);
  wresize(sep, 1, COLS);
  drw(wchan);
  drw(sep);
  
  dstatus();
  
  /* redraw input screen */
  werase(winput);
  t = findsock("input");
  if (t) {
    cur = (cmds_t *)t->d;
    if (cur->cmd) {
      indraw(cur->cmd, curx, winput);
    }
  }
  dscr(wchan);
  drw(wchan);
  drw(winput);

  /* redraw search result screen, if any */
  t = findsock("sinput");
  if (finp && t) {
    plist(t->d);
  }
    
  return;
}

#endif

void wstats(WINDOW *win)
{
  wp(win, "stdscr to %i - %i\n", stdscr->_maxy, stdscr->_maxx);
  wp(win, "winput to %i - %i\n", winput->_maxy, winput->_maxx);
  wp(win, "sep to %i - %i\n", sep->_maxy, sep->_maxx);
  wp(win, "wchan to %i - %i\n", wchan->_maxy, wchan->_maxx);
  wp(win, "LINES/COLS to %i - %i\n", LINES, COLS);
  drw(win);
}

void drw(WINDOW *win)
{
  if (info.daemon || !win)
    return;
  
  if (finp && win == swin)
    wrefresh(win);
  else if (!finp)
    wrefresh(win);
}

void dstatus()
{
  int j, lg;
  char *t = NULL;
  user_t *cur;
  
  if (info.daemon) {
    return;
  }

#ifdef MCURSES
  if (cloaked)
    mvwprintw(sep, 0, 0, "\e[0;37;44m (%s) [%i songs in %i libraries (%i gigs)] [nap v%s] [U/%i D/%i]", info.user, scount.songs, scount.libraries, scount.gigs, VERSION, fcnt(up), fcnt(down));
  else
    mvwprintw(sep, 0, 0, "\e[0;37;44m [%s] [%i songs in %i libraries (%i gigs)] [nap v%s] [U/%i D/%i]", info.user, scount.songs, scount.libraries, scount.gigs, VERSION, fcnt(up), fcnt(down));
  mvwaddstr(sep, 0, COLS+10, "\e[0m");
  drw(sep);
#else
  werase(sep);
  if (!info.notop)
    werase(whead);
  for (j=0;j<COLS;j++)
  {
    waddch(sep, ' ');
    if (!info.notop)
      waddch(whead, ' ');
  }
  
  if ((!ircmode && !curchan) || (curchan && (!curchan->q || curchan->q == 1)))
  {
    msprintf(&t, "[U/%i D/%i]", fcnt(up), fcnt(down));
    mvwprintw(sep, 0, COLS-(strlen(t)+1), t);
    free(t);
    if (cloaked)
      mvwprintw(sep, 0, 0, " (%s) [%lu songs in %lu libraries (%lu gigs)]", info.user, scount.songs, scount.libraries, scount.gigs);
    else
      mvwprintw(sep, 0, 0, " [%s] [%lu songs in %lu libraries (%lu gigs)]", info.user, scount.songs, scount.libraries, scount.gigs);
  }
  else
  {
    if (!curchan || !curchan->users)
      mvwprintw(sep, 0, 0, " [%s]", mnick);
    else if (curchan->flag)
    {
      t = strdup(curchan->nm);
      if (strlen(curchan->nm) > COLS/4)
        t[COLS/4] = 0;
      cur = finduser(curchan, mnick);
      if (cur->flag & F_OP)
        mvwprintw(sep, 0, 0, " [@%s] [%s +", mnick, t);
      else if (cur->flag & F_VOICE)
        mvwprintw(sep, 0, 0, " [+%s] [%s +", mnick, t);
      else
        mvwprintw(sep, 0, 0, " [%s] [%s +", mnick, t);
      if (curchan->flag & F_I)
        waddch(sep, 'i');
      if (curchan->flag & F_S)
        waddch(sep, 's');
      if (curchan->flag & F_P)
        waddch(sep, 'p');
      if (curchan->flag & F_T)
        waddch(sep, 't');
      if (curchan->flag & F_M)
        waddch(sep, 'm');
      if (curchan->flag & F_N)
        waddch(sep, 'n');
      if (curchan->flag & F_K)
        waddch(sep, 'k');
      if (curchan->flag & F_L)
        waddch(sep, 'l');
      if (curchan->flag & F_K || curchan->flag & F_L)
        waddch(sep, ' ');
      if (curchan->flag & F_K)
        wprintw(sep, "%s", curchan->key);
      if (curchan->flag & F_K && curchan->flag & F_L)
        waddch(sep, ' ');
      if (curchan->flag & F_L)
        wprintw(sep, "%i", (unsigned int)curchan->l);
      waddch(sep, ']');
      free(t);
      t = NULL;
    }
    else
    {
      t = strdup(curchan->nm);
      if (strlen(curchan->nm) > COLS/4)
        t[COLS/4] = 0;
      cur = finduser(curchan, mnick);
      if (cur->flag & F_OP)
        mvwprintw(sep, 0, 0, " [@%s] [%s]", mnick, t);
      else if (cur->flag & F_VOICE)
        mvwprintw(sep, 0, 0, " [+%s] [%s]", mnick, t);
      else
        mvwprintw(sep, 0, 0, " [%s] [%s]", mnick, t);
      free(t);
      t = NULL;
    }
  }
  
  if (!info.notop)
  {
    lg = COLS-(strlen(VERSION)+8)-4;
    mvwprintw(whead, 0, COLS-(strlen(VERSION)+8), "[nap v%s]", VERSION);
    if (curchan && curchan->q != 1 && curchan->topic)
    {
      t = (char *)malloc(lg+1);
      memset(t, 0, lg+1);
      if (strlen(curchan->topic) <= lg)
        tind = 0;
      if (tind && (((int)strlen(curchan->topic+tind)-lg) < 0 || tind > strlen(curchan->topic)))
      {
        drw(wchan);
        tind = strlen(curchan->topic)-lg;
      }
      if (strlen(curchan->topic) > lg && tind != (strlen(curchan->topic)-lg))
      {
        strncpy(t, curchan->topic+tind, lg-3);
        mvwprintw(whead, 0, 0, " [%s...]", t);
      }
      else
      {
        strncpy(t, curchan->topic+tind, lg);
        mvwprintw(whead, 0, 0, " [%s]", t);
      }
      free(t);
    }
    else if (curchan && curchan->q == 1)
      mvwprintw(whead, 0, 0, " [Query (%s)]", curchan->nm);
    if (!finp)
      wnoutrefresh(whead);
  }
  if (!finp)
  {
    wnoutrefresh(sep);
    doupdate();
  }
#endif
}

void addlog(char *logfile, char *str, unsigned char rp)
{
  FILE *f;
  char t[1024];
  int i, j, k;
  struct tm *r;
  time_t ct = time(NULL);
  
  memset(t, 0, sizeof(t));
  
  if (rp)
  {
    for (i=0,j=0;str[i];i++)
    {
      if (str[i] == '\e')
      {
        for (k=0;str[i]&&str[i]!='m'&&k<6;k++,i++);
        continue;
      }
      t[j] = str[i];
      j++;
    }
  }
  else
    strcpy(t, str);
  
  r = localtime(&ct);
  
  f = fopen(logfile, "a");
  if (f) {
    fprintf(f, "[%i:%02i] %s", r->tm_hour, r->tm_min, t);
    fclose(f);
  }
}

int wp(WINDOW *win, char *fmt, ...)
{
  va_list args;
  char str[4096]; /* note: don't need malloc */
  int i;
  
  if (noprint == 2)
    return(0);
  
  memset(str, 0, 4096);
  
  va_start(args, fmt);
  vsprintf(str, fmt, args);
  va_end(args);
  
  if (info.logallfile)
    addlog(info.logallfile, str, 1);
    
  if (win && !info.daemon)
  {
    addscroll(win, str);
    if (nvar("noscroll")!=1)
      scur = NULL;           /* be sure to scroll to bottom on output */
    dscr(win);
  }
  if (!win && info.daemon!=2)
    printf(str);
  
  i = strlen(str);
  
  return(i);
}

scrls_t *scrend(scrls_t *t)
{
  scrls_t *cur, *cur1 = NULL;
  
  for (cur=t;cur!=NULL;cur=cur->next)
    cur1 = cur;
  
  return(cur1);
}

/* add a line to the "scroll" list of the main screen */
void addscroll(WINDOW *win, char *buf)
{
  int i, j, k = 0, m = 0, bf = 0;
  scrls_t *cur = scrend(mscroll), *cur1, *par;
  char col[10];
  
  if (cur == NULL)
  {
    mscroll = (scrls_t *)malloc(sizeof(scrls_t));
    mscroll->next = NULL;
    mscroll->prev = NULL;
    mscroll->d = 0;
    mscroll->chan = recent;
    mscroll->own = NULL;
    cur = mscroll;
    memset(cur->ln, 0, sizeof(cur->ln));
  }
  else if (cur->d)
  {
    cur1 = cur;
    cur = (scrls_t *)malloc(sizeof(scrls_t));
    cur->prev = cur1;
    cur->next = NULL;
    cur->d = 0;
    cur->chan = recent;
    cur->own = NULL;
    cur1->next = cur;
    memset(cur->ln, 0, sizeof(cur->ln));
  }
  
  if (!cur->d)
  {
    for (;cur->ln[k];k++)
    {
      if (cur->ln[k] == '\e')
      {
        if (!strncmp(cur->ln+k, BOLD, strlen(BOLD)))
          bf = 1;
        else if (!strncmp(cur->ln+k, WHITE, strlen(WHITE)))
          bf = 0;
        if (cur->ln[k+3] == 'm')
          m+=4;
        else
          m+=5;
      }
    }
  }
  
  par = cur;
  
  memset(col, 0, sizeof(col));
  
  for (i=0,j=strlen(buf);i<j;i++,k++)
  {
    if (buf[i] == '\r')
      i++;
    if (k >= (COLS+m-1) || buf[i] == '\n')
    {
      if (k == (COLS+m-1) && strchr(cur->ln, ' '))
      {
        for (;!isspace(cur->ln[k])&&k>=0;k--,i--);
        i++;
      }
      cur->ln[k] = '\0';
      strcat(cur->ln, WHITE);
      cur->d = 1;
      k = 0;
      m = 0;
      if (buf[i] == '\n')
        i++;
      if (i == j)
        break;
      cur1 = cur;
      cur = (scrls_t *)malloc(sizeof(scrls_t));
      cur->prev = cur1;
      cur->next = NULL;
      cur->d = 0;
      cur->chan = recent;
      cur1->next = cur;
      if (buf[i-1] == '\n')
      {
        cur->own = NULL;
        par = cur;
        bf = 0;
      }
      else
        cur->own = par;
      memset(cur->ln, 0, sizeof(cur->ln));
      if (*col)
      {
        if (bf && strcmp(col, BOLD))
          strcpy(cur->ln, BOLD);
        strcat(cur->ln, col);
        m = k = strlen(cur->ln);
      }
      else if (bf)
      {
        strcpy(cur->ln, BOLD);
        m = k = strlen(cur->ln);
      }
      if (cur->own)
        m-=TABSIZE;
    }
    if (buf[i] == '\e')
    {
      if (buf[i+3] == 'm')
      {
        if (!strncmp(buf+i, BOLD, strlen(BOLD)))
          bf = 1;
        else if (!strncmp(buf+i, WHITE, strlen(WHITE)))
          bf = 0;
        memset(col, 0, sizeof(col));
        strncpy(col, buf+i, 4);
        strncpy(cur->ln+k, buf+i, 4);
        m+=4;
        i+=3;
        k+=3;
      }
      else
      {
        memset(col, 0, sizeof(col));
        strncpy(col, buf+i, 5);
        strncpy(cur->ln+k, buf+i, 5);
        m+=5;
        i+=4;
        k+=4;
      }
    }
    else
      cur->ln[k] = buf[i];
  }
}

void dscr(WINDOW *win)
{
  scrls_t *cur;
  int i, j, k, m, at=0, t, c;
  unsigned char bf = 0, uf = 0, gf = 0;
  char ebuf[6];
  
  werase(win);
  wmove(win, 0, 0);
  
  if (scur == NULL)
    cur = scrend(mscroll);
  else
    cur = scur;
  
  if (cur == NULL)
    return;
  
  i = win->_maxy-1;
  
  if (cur->chan && (wmode && cur->chan != curchan))
    i++;
  
  while (i >= 0)
  {
    if (cur->prev == NULL)
      break;
    cur = cur->prev;
    if (cur->chan && (wmode && cur->chan != curchan))
      continue;
    i--;
  }
  
  c = win->_maxy;
  
  for (j=0,i=0;i<=c;cur=cur->next)
  {
    if (cur == NULL)
      break;
    if (cur->chan && (wmode && cur->chan != curchan))
      continue;
    if (cur->own)
      wmove(win, j, TABSIZE);
    else
    {
      bf = 0;
      uf = 0;
      gf = 0;
      wmove(win, j, 0);
    }
#ifndef MCURSES
    for (k=0;cur->ln[k];k++)
    {
      if (cur->ln[k] == '\e' && cur->ln[k+1] == '[' && isdigit(cur->ln[k+2]))
      {
        for (m=0;cur->ln[k]&&cur->ln[k]!='m';m++,k++)
          ebuf[m] = cur->ln[k];
        ebuf[m] = 'm';
        ebuf[m+1] = '\0';
        t = doesc(win, ebuf);
        if (t == COLOR_PAIR(CPW))
          at = 0;
        else
          at |= t;
      }
      else if (cur->ln[k] == 2)
      {
        if (!bf)
        {
          at |= doesc(win, BOLD);
          bf = 1;
        }
        else
        {
          at &= ~doesc(win, BOLD);
          bf = 0;
        }
      }
      else if (cur->ln[k] == 31)
      {
        if (!uf)
        {
          at |= doesc(win, UNDERLINE);
          uf = 1;
        }
        else
        {
          at &= ~doesc(win, UNDERLINE);
          uf = 0;
        }
      }
      else if (cur->ln[k] == 7)
      {
        if (!gf && i == c)
          beep();
        gf = 1;
      }
      else if (cur->ln[k] == 15)
        uf = bf = at = 0;
      else
        waddch(win, cur->ln[k]|at);
    }
#else
    waddstr(win, cur->ln);
#endif
    i++;
    j++;
  }
}

void dscroll(WINDOW *win, int n)
{
  int i;
  scrls_t *cur;
  
  if (scur == NULL)
    cur = scrend(mscroll);
  else
    cur = scur;
  
  if (!cur)
    return;
  
  if (n > 0)
  {
    i = n;
    if (cur->chan && (wmode && cur->chan != curchan))
      i++;
  
    while (i > 0)
    {
      if (cur->prev == NULL)
        break;
      cur = cur->prev;
      if (cur->chan && (wmode && cur->chan != curchan))
        continue;
      i--;
    }
  }
  else if (n < 0)
  {
    i = n;
    if (cur->chan && (wmode && cur->chan != curchan))
      i--;
  
    while (i < 0)
    {
      if (cur->next == NULL)
      {
        cur = NULL;
        break;
      }
      cur = cur->next;
      if (cur->chan && (wmode && cur->chan != curchan))
        continue;
      i++;
    }
  }
  else
    return;
  
  scur = cur;
  
  dscr(win);
  drw(win);
}

int doesc(WINDOW *win, char *in)
{
  int i;
  
  for (i=0;;i++)
  {
    if (!colortab[i].in)
      return(0);
    if (!strcmp(colortab[i].in, in))
      break;
  }
  
  if (colortab[i].c)
    return(COLOR_PAIR(colortab[i].pair));
  else
    return(colortab[i].pair);
}

int cmdcnt(cmds_t *t)
{
  cmds_t *cur;
  int i;
  
  for (i=0,cur=t;cur!=NULL;cur=cur->next,i++);
  
  return(i);
}

cmds_t *cmdend(cmds_t *t)
{
  cmds_t *cur, *cur1 = NULL;
  
  for (cur=t;cur!=NULL;cur=cur->next)
    cur1 = cur;
  
  return(cur1);
}

int input(WINDOW *win, sock_t *m)
{
  chtype cbuf = 0, ebuf = 0;
  cmds_t *cur = (cmds_t *)m->d, *cur1, *cur2, *cur3;
  sock_t *t;
  int i, j, s, cx, n;
  
  t = findsock("server");
  if (!t)
    s = -1;
  else
    s = t->s;
  
  if (!cmdl)
  {
    cmdl = (cmds_t *)malloc(sizeof(cmds_t));
    cur = cmdl;
    cur->next = NULL;
    cur->prev = NULL;
    memset(cur->cmd, 0, sizeof(cur->cmd));
    curx = 0;
  }
  i = curx;
  cx = i;

  ebuf = wgetch(winput);
  if (ebuf == '\n' || ebuf == '\r')
  {
    curr = 0;
    ebuf = n = 0;
    if (*cur->cmd == '/')
    {
      if (strlen(cur->cmd) > 1)
      {
        n = parseout(s, cur->cmd+1, win);
        dstatus();
      }
    }
    else
    {
      if (!curchan)
      {
        wp(win, "%s* Not on a channel%s\n", RED, WHITE);
        drw(win);
      }
      else if (curchan->q == 1)
      {
        sendpack(s, F_TELL, "%s %s", fxp(curchan->nm), cur->cmd);
        recent = curchan;
        wp(win, "%s* --> (%s%s%s)%s %s\n", GREEN, WHITE, curchan->nm, GREEN, WHITE, cur->cmd);
        drw(win);
        recent = NULL;
      }
      else if (curchan->q == 2 && cur->cmd[0] != '\0')
      {
        ssock(ircsock, "PRIVMSG %s :%s\n", fxp(curchan->nm), cur->cmd);
        recent = curchan;
        wp(win, "%s<%s%s%s>%s %s\n", BRIGHT(MAGENTA), WHITE, mnick, BRIGHT(MAGENTA), WHITE, cur->cmd);
        drw(win);
        recent = NULL;
      }
      else if (cur->cmd[0] != '\0' && sendpack(s, F_SAY, "%s %s", fxp(curchan->nm), cur->cmd) == -1)
        delsock(s);
    }
    
    if (cmdcnt(cmdl) == (BACKLOG+1))
    {
      cur1 = cmdl->next;
      free(cmdl);
      cmdl = cur1;
      cmdl->prev = NULL;
    }

    cur1 = cmdend(cmdl);
    cur2 = cur;
    cur3 = cur1;
    
    if (!strlen(cur1->cmd))
      cur = cur1;
    else
    {
      cur = (cmds_t *)malloc(sizeof(cmds_t));
      cur->next = NULL;
      cur->prev = cur1;
      cur1->next = cur;
    }

    cur->cmd[0] = 1;
    
    if (cur2->cmd[0] != 1 && ccmd && ccmd->cmd[0])
    {
      cur1 = (cmds_t *)malloc(sizeof(cmds_t));
      cur3->prev->next = cur1;
      cur1->next = cur;
      memset(cur1->cmd, 0, sizeof(cur1->cmd));
      strcpy(cur1->cmd, cur2->cmd);
      cur1->prev = cur->prev;
      cur->prev = cur1;
    }
    
    for (cur1=cmdl;cur1;cur1=cur1->next)
    {
      if (cur1->cmd[0] == '\0')
      {
        if (cur1->prev)
          cur1->prev->next = cur1->next;
        if (cur1->next)
          cur1->next->prev = cur1->prev;
        free(cur1);
        cur1 = NULL;
        break;
      }
    }
    
    memset(cur->cmd, 0, sizeof(cur->cmd));
    
    ccmd = NULL;
    i = 0;
    cx = 0;
    curx = 0;
    indraw(cur->cmd, cx, winput);
    
    if (n == -1)
    {
      m->d = cbk = (void *)cur;
      curx = i;
      return(0);
    }
    else if (n == -2)
    {
      m->d = cbk = (void *)cur;
      curx = i;
      return(-1);
    }
  }
  else if (ebuf == '\e')
  {
    cbuf = wgetch(winput);
    if (cbuf == KEY_BACKSPACE || cbuf == 127 || cbuf == 8)
    {
      i--;
      while (i>=0&&cur->cmd[i]==' ')
      {
        strncpy(cur->cmd+i, cur->cmd+i+1, strlen(cur->cmd+i+1));
        cur->cmd[strlen(cur->cmd)-1] = 0;
        i--;
      }
      while (i>=0&&cur->cmd[i]!=' ')
      {
        strncpy(cur->cmd+i, cur->cmd+i+1, strlen(cur->cmd+i+1));
        cur->cmd[strlen(cur->cmd)-1] = 0;
        i--;
      }
      if (i < 0)
      {
        memset(cur->cmd, 0, sizeof(cur->cmd));
        i = 0;
      }
      else
        i++;
      if (!i)
        curr = 0;
      cx = curx = i;
      indraw(cur->cmd, cx, winput);
    }
  }
  else if (ebuf == KEY_BACKSPACE || ebuf == 127 || ebuf == 8)
  {
    if (i)
    {
      for (j=i-1;cur->cmd[j];j++)
        cur->cmd[j] = cur->cmd[j+1];
      i--;
      cx--;
      indraw(cur->cmd, cx, winput);
    }
    ebuf = 0;
  }
  else if (ebuf == 23)
  {
    for (j=i,cur->cmd[j]=0;j>=0;j--)
    {
      if (j == 0)
      {
        cur->cmd[j] = 0;
        i = cx = 0;
        break;
      }
      if (cur->cmd[j] == ' ')
      {
        i = cx = j+1;
        break;
      }
      cur->cmd[j] = 0;
    }
    indraw(cur->cmd, cx, winput);
  }
  else if (ebuf == 24)
  {
    cur->cmd[i] = cbuf;
    upchan(curchan);
    indraw(cur->cmd, cx, winput);
    if (wmode)
    {
      dscr(wchan);
      drw(wchan);
    }
    dstatus();
    ebuf = 0;
  }
  else if (ebuf == discard)
  {
    memset(cur->cmd, 0, strlen(cur->cmd)+1);
    cx = i = 0;
    indraw(cur->cmd, cx, winput);
    ebuf = 0;
  }
  else if (ebuf == 154)
    ebuf = 0;
  else if (ebuf == '\t')
  {
    int g;
    ebuf = 0;
    
    for (g=cx;cur->cmd[g]!=' '&&g>=0;g--);
    
    if (cur->cmd[g] != ' ')
      g = 0;
          
    if (*cur->cmd == '/' && !g)
    {
      alias_t *al;
      
      for (al=alhead;al;al=al->next)
        if (!strncasecmp(cur->cmd+1, al->nm, strlen(cur->cmd+1)))
        {
          memset(cur->cmd, 0, sizeof(cur->cmd));
          cur->cmd[0] = '/';
          strcpy(cur->cmd+1, al->nm);
          strcat(cur->cmd, " ");
          cx = curx = i = strlen(cur->cmd);
          indraw(cur->cmd, cx, winput);
          break;
        }

      for (j=0;out[j].func;j++)
        if (!strncasecmp(cur->cmd+1, out[j].nm, strlen(cur->cmd+1)) && out[j].help)
        {
          memset(cur->cmd, 0, sizeof(cur->cmd));
          cur->cmd[0] = '/';
          strcpy(cur->cmd+1, out[j].nm);
          strcat(cur->cmd, " ");
          cx = curx = i = strlen(cur->cmd);
          indraw(cur->cmd, cx, winput);
          break;
        }
    }
    else if (curchan && curchan->users)
    {
      user_t *usr;
      int k, z;
      
      for (k=cx;cur->cmd[k]!=' '&&k>=0;k--);
      
      if (cur->cmd[k] == ' ')
        z = k+1;
      else
        z = 0;
      
      for (usr=curchan->users;usr;usr=usr->next)
        if (!strncasecmp(cur->cmd+z, usr->nm, strlen(cur->cmd+z)))
        {
          for (k=z;k<=cx;k++)
            cur->cmd[k] = 0;
          strcpy(cur->cmd+z, usr->nm);
          if (!z)
            strcat(cur->cmd, ": ");
          cx = curx = i = strlen(cur->cmd);
          indraw(cur->cmd, cx, winput);
          break;
        }
    }
  }
  else if (ebuf == 20)
  {
    if (tind == (strlen(curchan->topic)-(COLS-(strlen(VERSION)+8)-4)))
      tind = 0;
    else
      tind += 5;
    if (!curchan || !curchan->topic || tind > strlen(curchan->topic))
      tind = 0;
    ebuf = 0;
    dstatus();
  }
  else if (ebuf == 5 || ebuf == KEY_PPAGE)
  {
    dscroll(win, 10);
    indraw(cur->cmd, cx, winput);
    ebuf = 0;
  }
  else if (ebuf == 4 || ebuf == KEY_NPAGE)
  {
    dscroll(win, -10);
    indraw(cur->cmd, cx, winput);
    ebuf = 0;
  }
  else if (ebuf == 12)
  {
    werase(wchan);
    drw(wchan);
    werase(sep);
    drw(sep);
    werase(winput);
    drw(winput);
    if (!info.notop)
    {
      werase(whead);
      drw(whead);
    }
    dstatus();
    dscr(wchan);
    drw(wchan);
    indraw(cur->cmd, cx, winput);
    ebuf = 0;
  }
  else if (ebuf == KEY_F(2))
  {
    sscr(search);
    return(0);
  }
  else if (ebuf == KEY_END || ebuf == KEY_SELECT)
  {
    scur = NULL;
    dscr(win);
    drw(win);
  }
  else if (ebuf == KEY_UP)
  {
    if (ccmd == NULL)
      cur1 = cur;
    else
      cur1 = ccmd;
    if (cur1->prev != NULL)
    {
      ccmd = cur1->prev;
      cx = i = strlen(ccmd->cmd);
      cur = ccmd;
      indraw(cur->cmd, cx, winput);
    }
  }
  else if (ebuf == KEY_DOWN)
  {
    if (ccmd != NULL)
    {
      cur1 = ccmd;
      if (cur1->next != NULL)
      {
        ccmd = cur1->next;
        cx = i = strlen(ccmd->cmd);
        cur = ccmd;
        indraw(cur->cmd, cx, winput);
      }
    }
  }
  else if (ebuf == KEY_RIGHT)
  {
    if (cur->cmd[i])
    {
      cx++;
      i++;
    }
    indraw(cur->cmd, cx, winput);
  }
  else if (ebuf == KEY_LEFT)
  {
    if (i)
    {
      cx--;
      i--;
    }
    indraw(cur->cmd, cx, winput);
  }
  else if (ebuf < 256)
  {
    cbuf = cur->cmd[i];
    for (j=i;cur->cmd[j];j++);
    memmove(cur->cmd+i+1, cur->cmd+i, j);
    cur->cmd[i] = ebuf;
    ebuf = 0;
    i++;
    cx++;
    curx++;
    indraw(cur->cmd, cx, winput);
  }
  m->d = (void *)cur;
  
  curx = i;
  
  return(1);
}

void initwin(unsigned char f)
{
  struct termios ts;
  WINDOW *w;
  
  if (info.daemon) {
    info.daemon = 2;
    return;
  }

  tcgetattr(0, &ts);
  bspace = ts.c_cc[VERASE];
  discard = 21;

#ifdef __CYGWIN32__
  setenv("TERMINFO", "./", 1);
  
  if (!newterm("cygwin", stdout, stdin))
  {
    fprintf(stderr, "Error opening terminal\n");
    exit(-1);
  }
  def_prog_mode();
  w = stdscr;
#else
  if (f)
  {
    if (!newterm("nxterm", stdout, stdin))
    {
      fprintf(stderr, "Error opening terminal\n");
      exit(-1);
    }
    def_prog_mode();
    w = stdscr;
  }
  else
    w = initscr();
#endif

/*  COLS = w->_maxx+1;
  LINES = w->_maxy+1; */
/*  dolc(); */
#ifndef MCURSES
  start_color();
  cbreak();
  noecho();
  
  init_pair(1, COLOR_WHITE, COLOR_BLUE);
  init_pair(CPR, COLOR_RED, COLOR_BLACK);
  init_pair(CPG, COLOR_GREEN, COLOR_BLACK);
  init_pair(CPW, COLOR_WHITE, COLOR_BLACK);
  init_pair(CPB, COLOR_BLUE, COLOR_BLACK);
  init_pair(CPY, COLOR_YELLOW, COLOR_BLACK);
  init_pair(CPM, COLOR_MAGENTA, COLOR_BLACK);
  init_pair(CPC, COLOR_CYAN, COLOR_BLACK);
  init_pair(CPBR, COLOR_WHITE, COLOR_RED);
  winput = newwin("input", 1, 0, LINES-1, 0);
#else
  info.notop = 1;
  winput = newwin("input", 1, COLS-2, LINES-1, 0);
#endif
  sep = newwin("sep", 1, 0, LINES-2, 0);
  if (!info.notop)
  {
    wchan = newwin("chan", LINES-3, 0, 1, 0);
    whead = newwin("head", 1, 0, 0, 0);
  }
  else
    wchan = newwin("chan", LINES-2, 0, 0, 0);

#ifndef MCURSES
  nodelay(winput, TRUE);
  wattrset(sep, COLOR_PAIR(1));
  if (!info.notop)
    wattrset(whead, COLOR_PAIR(1));
  idlok(wchan, FALSE);
  scrollok(wchan, TRUE);
  wbkgdset(winput, COLOR_PAIR(CPW));
  wbkgdset(wchan, COLOR_PAIR(CPW));
/*  bkgd(COLOR_PAIR(1)); */
  keypad(winput, TRUE);
#endif
  wmove(winput, 0, 0);
  drw(winput);
  
/*  tcgetattr(0, &ts);
  printf("%i\n", ts.c_iflag&ISTRIP);
  exit(1); */
  
  dstatus();
  drw(winput);
}

void indraw(char *buf, int s, WINDOW *win)
{
  int b = s, i;
  unsigned char t1[512], t[513]; /* note don't need malloc */
  char *p=NULL;

  strncpy(t, buf, 511);
  t[511]=0; /* make sure it's 0-terminated */
  strcat(t, " ");
  
  if (curchan)
  {
    p = strdup(curchan->nm);
    if (strlen(curchan->nm) > (COLS/4))
      p[COLS/4] = 0;
  }
  else
    p = NULL;
  
  if (!p)
  {
    if (b > (curr+COLS-1))
      curr = b-COLS+1;
    else if (b < curr)
    {
      curr-=COLS/4;
      if (curr < 0)
        curr = 0;
    }
    if ((curr+COLS) > strlen(t) && curr)
      curr = strlen(t)-COLS;
    if (curr < 0)
      curr = 0;
    sprintf(t1, fxp(t+curr));
  }
  else if (*buf)
  {
    i = strlen(p)+3;
    if (b > (curr+(COLS-i)-1))
      curr = b-(COLS-i)+1;
    else if (b < curr)
    {
      curr-=COLS/4;
      if (curr < 0)
        curr = 0;
    }
    if ((curr+(COLS-i)) > strlen(t) && curr)
      curr = strlen(t)-(COLS-i);
    if (curr < 0)
      curr = 0;
    if (curchan->q == 1)
      sprintf(t1, "(%s) %s", p, t+curr);
    else
      sprintf(t1, "[%s] %s", p, t+curr);
  }
  else
  {
    if (curchan->q == 1)
      sprintf(t1, "(%s) ", p);
    else
      sprintf(t1, "[%s] ", p);
  }
  
  werase(win);
  for (i=0;t1[i];i++)
  {
    if (t1[i] >= 1 && t1[i] <= 26)
      waddch(win, (t1[i]+64)|doesc(win, BOLD));
    else
      waddch(win, t1[i]);
  }
  if (p)
    b+=strlen(p)+3;
  b = b-curr;
  if (b < 0)
    b = 0;
  if (b >= COLS)
    b = COLS-1;
  wmove(win, 0, b);
  drw(win);
  /* fixed memory leak: forgot to free t */
  if (p)
    free(p);
}

