#ifndef _NAP_LISTS_H
#define _NAP_LISTS_H

/* Copyright (c) 2000  Kevin Sullivan <nite@gis.net>
 *
 * Please refer to the COPYRIGHT file for more information.
 */

#define REF(x, y) ((x)y->d)
#define findlinkn(a, b, c) _findlink(a, (void *)b, &c, 0)
#define findlinks(a, b, c) _findlink(a, b, c, 1)


struct link_s {
  void *d;
  struct link_s *next;
};
typedef struct link_s link_t;


void addlink(link_t **, void *);
void dellink(link_t **, void *);
void *_findlink(link_t *, void *, void *, unsigned char);
void *linknum(link_t *, int);
void dellist(link_t **);

/* experimental: here we define some general list macros. Because they
   are macros, they should work on any datatype with a ->next
   component. Some of them use a "hook". If elt and list are of type
   t* then hook is of type t**. A hook stands for an insertion point
   in the list, i.e., either before the first element, or between two
   elements, or after the last element. If an operation "sets the
   hook" for an element, then the hook is set to just before the
   element. One can insert something at a hook. One can also unlink at
   a hook: this means, unlink the element just after the hook. By "to
   unlink", we mean the element is removed from the list, but not
   deleted. Thus, it and its components still need to be freed. -PS */

/* we enclose macro definitions whose body consists of more than one
   statement in MACRO_BEGIN and MACRO_END, rather than '{' and '}'.  The
   reason is that we want to be able to use the macro in a context
   such as "if (...) macro(...); else ...". If we didn't use this obscure
   trick, we'd have to omit the ";" in such cases. */

#define MACRO_BEGIN do {
#define MACRO_END   } while (0)

/* traverse list. At the end, elt is set to NULL. */
#define list_forall(elt, list)   for (elt=list; elt!=NULL; elt=elt->next)

/* set elt to the first element of list satisfying boolean condition
   c, or NULL if not found */
#define list_find(elt, list, c) \
  MACRO_BEGIN list_forall(elt, list) if (c) break; MACRO_END

/* like forall, except also set hook for elt. */
#define list_forall2(elt, list, hook) \
  for (elt=list, hook=&list; elt!=NULL; elt=elt->next, hook=&elt->next)

/* same as list_find, except also set hook for elt. */
#define list_find2(elt, list, c, hook) \
  MACRO_BEGIN list_forall2(elt, list, hook) if (c) break; MACRO_END

/* same, except only use hook. */
#define _list_forall_hook(list, hook) \
  for (hook=&list; *hook!=NULL; hook=&(*hook)->next)

/* same, except only use hook. Note: c may only refer to *hook, not elt. */
#define _list_find_hook(list, c, hook) \
  MACRO_BEGIN _list_forall_hook(list, hook) if (c) break; MACRO_END

/* insert element after hook */
#define list_insert_athook(elt, hook) \
  MACRO_BEGIN elt->next = *hook; *hook = elt; MACRO_END

/* unlink element after hook, let elt be unlinked element, or NULL. 
   hook remains. */
#define list_unlink_athook(list, elt, hook) \
  MACRO_BEGIN elt = *hook; if (elt) *hook = elt->next; elt->next = NULL; \
  MACRO_END

/* prepend elt to list */
#define list_prepend(list, elt)   list_insert_athook(elt, &list) 

/* append elt to list. */
#define list_append(listtype, list, elt)     \
  MACRO_BEGIN                                \
  listtype **_hook;                          \
  _list_forall_hook(list, _hook) {}          \
  list_insert_athook(elt, _hook);            \
  MACRO_END

/* unlink first element that meets condition c. elt becomes that
   element, or NULL if not found. Hook is set to where element was or
   end of list. Note: c may refer to *hook (the next element after
   hook), but to elt only as the "constant" elt. */
#define _list_unlink_cond(list, elt, c, hook)    \
  MACRO_BEGIN                                    \
  _list_find_hook(list, c, hook);                \
  list_unlink_athook(list, elt, hook);           \
  MACRO_END

/* if elt occurs in list, unlink it. Note: we don't delete it, so
   elt and its components must still be freed. */
#define list_unlink(list, elt, hook) \
  _list_unlink_cond(list, elt, *hook==elt, hook)

/* let elt be the nth element of the list, starting to count from 0. 
   Return NULL if out of bounds.   */
#define list_n(elt, list, n)                                  \
  MACRO_BEGIN                                                 \
  int _x;  /* only evaluate n once */                         \
  for (_x=(n), elt=list; _x && elt; _x--, elt=elt->next) {}   \
  MACRO_END

/* let elt be each element of the list, unlinked. At the end, set list=NULL. */
#define list_forall_unlink(elt, list) \
  for (elt=list; elt ? (list=elt->next, elt->next=NULL), 1 : 0; elt=list)

/* insert the element ELT just before the first element TMP of the list
   for which COND holds. Here COND must be a condition of ELT and TMP.
   Typical usage is to insert an element into an ordered list: for instance,
   list_insert_ordered(listtype, list, elt, tmp, elt->size <= tmp->size) */
#define list_insert_ordered(listtype, list, elt, tmp, cond) \
  MACRO_BEGIN                                           \
  listtype **_hook;                                     \
  _list_find_hook(list, (tmp=*_hook, (cond)), _hook);     \
  list_insert_athook(elt, _hook);                       \
  MACRO_END

/* sort the given list, according to the comparison condition. 
   Typical usage is list_sort(listtype, list, a, b, a->size <= b->size) */
#define list_sort(listtype, list, a, b, cond)            \
  MACRO_BEGIN                                            \
  listtype *_newlist=NULL;                               \
  list_forall_unlink(a, list)                            \
    list_insert_ordered(listtype, _newlist, a, b, cond); \
  list = _newlist;                                       \
  MACRO_END

#endif

