#ifndef _NAP_CMDS_H
#define _NAP_CMDS_H

/* Copyright (c) 2000  Kevin Sullivan <nite@gis.net>
 *
 * Please refer to the COPYRIGHT file for more information.
 */

#include <sys/time.h>

#include "scheck.h"


struct in_nap_cmd_s 
{
  unsigned short op;
  int (*func)(int, char *, char **, int, WINDOW *);
};
typedef struct in_nap_cmd_s in_nap_cmd_t;

struct out_nap_cmd_s 
{
  char *nm;
  unsigned char a;
  int (*func)(int, char *, char **, int, WINDOW *);
  char *help;
};
typedef struct out_nap_cmd_s out_nap_cmd_t;

struct user_s 
{
  char *nm, *addr;
  unsigned char conn;
  unsigned int scount;
  unsigned char flag;
  struct user_s *next;
};
typedef struct user_s user_t;

struct hotlist_s 
{
  char *nm;
  int conn;
  unsigned char on;
};
typedef struct hotlist_s hotlist_t;

struct chans_s 
{
  char *nm;
  char *topic;
  user_t *users;
  unsigned char q;
  unsigned char flag;
  int l;
  char *key;
  unsigned char p;
  struct chans_s *next;
}; 
typedef struct chans_s chans_t;

/* the data structure for search result items. */
struct ssearch_s 
{
  int num;                  /* item number in search results list */
  char *song;               /* title with pathname componentes stripped */
  char *fn;                 /* title as sent by server */
  char *cmp;                /* nickname of remote user */
  int conn;                 /* connection speed of remote client */
  unsigned int sz;          /* total size */
  int brate;                /* bitrate (e.g. 128) */
  int freq;                 /* sample frequency (e.g. 44100) */
  int time;                 /* length, in seconds */
  unsigned long ip;         /* ip address of remote client */
  struct timeval s, r;      /* time ping sent, received */
  int ping;                 /* ping time */
  struct ssearch_s *next;   /* next item in list */
};
typedef struct ssearch_s ssearch_t;

/* The task list is one of the two "backbone" data structures of nap
   (the other is the connection list, type sock_t, see scheck.h). A
   "task" is an upload or a download. Tasks are kept in two linked
   lists, "up" and "down".

   The following explanations are somewhat a "to do" list - I hope to
   converge to a clean state model eventually.

   A task can be in many different states, such as QUEUED, WAITING,
   etc.  A task is in a state while it is awaiting some external
   event, such as a response from the server. When such an event
   happens, it causes the task to move to another state. For some
   states, there is a time limit; if the task remains in the state for
   too long, it times out. There are separate sets of states for
   uploads and downloads (even though they share the same names). In
   each state, only some of the many fields of the file_s structure
   might be relevant. The following chart summarizes the states. It
   also functions as a roadmap for the napster protocol, at least the
   upload/download part of it. The parts in [brackets] are not yet
   implemented. 

   1. Downloads.

   Creation: A download task is created by a user request (from the
           search results screen or via /get).

         * If the download limit (per user or total) is exceeded, go
           to state QUEUED.

         * Otherwise, send 203 (download request) to server and go to
           state REQUESTED.

   QUEUED: when a slot becomes available, send 203 (download request)
           to server and go to state REQUESTED.
   
   REQUESTED: if we receive from the server for this task:

         * 206 (get error), the remote client was unavailable. Go to 
           FAILED.

	 * 609 (accept failure), the remote client refused to let us
	   download the file. Go to FAILED.

	 * 620 (queue limit), the remote client has indicated that its
	   upload limit was exceeded. Go to FAILED. [?? Or maybe back
	   to QUEUED? How does napster deal with "remotely queued"
	   items?]

	 * 204 (download acknowledgment). Check if remote party is
           firewalled (i.e. if their port is 0). If firewalled, send
           500 (alternative download request) to the server and go to
           WAITING.

           If not firewalled, initiate connection to remote client and
           go to CONNECTING.

   WAITING: if the remote client connects to us, negotiate the terms
           of the download (e.g. the byte range to request). If this
           fails, go to FAIL, else send 218 (downloading file) to
           server and go to ONGOING.

   CONNECTING: when we reach the remote client, negotiate the terms of
           the download. On failure go to FAIL, else send 218
           (downloading file) to server and go to ONGOING.

   ONGOING: receive data. If transfer is interrupted, send 219
           (download complete) to server and go to FAILED. 

	   If transfer is complete, [remove "incomplete" flag from
	   file and] go to COMPLETED.

	   [If we discover during a "resume" that the file don't
	   match, go back to state REQUESTED. Remember to request the
	   whole file next time.]
	   
   In addition, while a task is in states REQUESTED, WAITING, or
   CONNECTING, there is a timer running. If the timer expires before
   a task reaches state ONGOING, the task is moved to state TIMEDOUT.

   FAILED, COMPLETED, and TIMEOUT: remove task, [unless user
	   preference states otherwise].

   2. Uploads.

   Creation: an upload is normally created by a 607 (upload request)
           from the server. 

         * if upload limit (per user or total) is exceeded, send 619
           (queue limit) and go to FAILED. [?? maybe we should QUEUE
           uploads too?]

         * check if we have the file. If not, send 608 (accept upload
           request) to the server and go to FAILED. Note: there seems
           to be no other option than 608, since there is no "reject
           upload request" packet. The remote client will try to
           connect, then fail.

         * otherwise, send 608 (accept upload request) and go to
           WAITING. 

   WAITING: * if the remote client connects to us, negotiate
	   transfer. If this fails, go to FAILED, otherwise send 220
	   (uploading file) to server and go to ONGOING.

         * if we receive 501 ("alt download ack" or push request) from
           the server, initiate connection with remote client, then go
           to CONNECTING.

	   *** note: a look at protocol traces reveals that 501(S) is
	   *** sent after 607(S)+608(C) were already sent. Thus, when
	   *** a 501 is received, the corresponding task should
	   *** already be WAITING, and will then go to CONNECTING.

   CONNECTING: when we reach the remote client, negotiate transfer. If
	   this fails, go to FAILED, otherwise send 220 (uploading
	   file) and go to ONGOING.

   ONGOING: receive data. If transfer is interrupted, send 221 (upload
           complete) and go to FAILED. If transfer is completed, send
           221 (upload complete) and go to COMPLETE. 

   Again, there is a timer in states WAITING and CONNECTING, sending
   us to TIMEOUT if we don't read ONGOING. 

   FAILED, COMPLETED, and TIMEOUT: remove task, [unless user
	   preference states otherwise].

   [Note: some users might prefer finished (failed, completed, or
   timed-out) up- or downloads to remain in the task structure until
   cleared explicitly, so that they can be examined later with /pdown
   and /pup. There should be little harm in allowing this, but it is
   not currently supported. Maybe it would be best to keep such "dead"
   items to the end of the task list, so that they don't cost lookup
   time.]

   Some states may have sub-states, for instance, CONNECTING usually
   involves first sending a GET and then waiting one more time for the
   remote client's response. This is implemented by remaining in the
   same state, but updating the (sock_t) "func" entry. 

   Here is how the different states are actually represented in the
   file_s data structure. The states FAILED, COMPLETED, and TIMEOUT
   are not represented; the task is simply deleted. The state QUEUED
   was not previously supported. The remaining states were represented
   as follows: 

   Downloads, REQUESTED: was not represented. The task was created
   upon 204 from server. 

   CONNECTING: represented by (file_t) with sk!=NULL and st==0. 

   WAITING: represented by (file_t) with sk==NULL and st==0.

   ONGOING: represented by (file_t) with sk!=NULL and st!=0.

   Uploads, .....

*/


struct file_s 
{
  unsigned long addr;          /* remote IP (network byte order?) */
  unsigned short port;         /* remote port (host byte order?) */
  time_t st;                   /* time when ONGOING started */ 
  FILE *f;                     /* local file, open for reading or writing */
  char *fn;                    /* uploads: full name of local file,
				  downloads: stripped filename */
  char *nm;                    /* nick of remote user */
  char *rfn;                   /* uploads: stripped filename(?),
				  downloads: remote file name */ 
  void *timer;                 /* timer for TIMEOUT in some states */
  unsigned long sz, csz, bsz;  /* sz: total bytes to send/get, csz: bytes
				  already sent/received, bsz: first
			          byte requested */
  unsigned char g;             /* ? initially 1 */
  unsigned char e;             /* ? */
  sock_t *sk;                  /* this task's client-client connection */
  struct file_s *next;         /* next item in list */
};
typedef struct file_s file_t;

struct scount_s 
{
  unsigned long libraries, songs, gigs;
};
typedef struct scount_s scount_t;

#define ICMP_ECHO 8
struct icmphdr 
{
  unsigned char type, code;
  unsigned short checksum;
  union
  {
    struct
    {
      unsigned short id, sequence;
    } echo;
    unsigned long gateway;
    struct
    {
      unsigned short blah, mtu;
    } frag;
  } un;
};



#define I_NAP_FUNC(x) int (x)(int s, char *str, char **tok, int num, WINDOW *win)

I_NAP_FUNC(sjoin);
I_NAP_FUNC(spart);
I_NAP_FUNC(stopic);
I_NAP_FUNC(ssay);
I_NAP_FUNC(snchan);
I_NAP_FUNC(sjchan);
I_NAP_FUNC(nothing);
I_NAP_FUNC(snotice);
I_NAP_FUNC(sscount);
I_NAP_FUNC(stell);
I_NAP_FUNC(suser);
I_NAP_FUNC(swhois);
I_NAP_FUNC(soff);
I_NAP_FUNC(sget);
I_NAP_FUNC(snget);
#ifdef QUEUE
I_NAP_FUNC(snacc);
I_NAP_FUNC(sqlimit);
#endif /* QUEUE */
I_NAP_FUNC(sfreq);
I_NAP_FUNC(sry);
I_NAP_FUNC(ssf);
I_NAP_FUNC(smpart);
I_NAP_FUNC(suon);
I_NAP_FUNC(suoff);
I_NAP_FUNC(ssret);
I_NAP_FUNC(ssend);
I_NAP_FUNC(srbrowse);
I_NAP_FUNC(sdbrowse);
I_NAP_FUNC(sop);
I_NAP_FUNC(spchange);
I_NAP_FUNC(sbport);
I_NAP_FUNC(sannounce);
I_NAP_FUNC(sbanlist);
I_NAP_FUNC(scping);
I_NAP_FUNC(scpong);
I_NAP_FUNC(ssping);
I_NAP_FUNC(sredir);
I_NAP_FUNC(scycle);
I_NAP_FUNC(sclist);
I_NAP_FUNC(sclist2);
I_NAP_FUNC(snend);
I_NAP_FUNC(sblocklist);
I_NAP_FUNC(signorelist);
I_NAP_FUNC(signoreend);
I_NAP_FUNC(signoreadd);
I_NAP_FUNC(signoreremove);
I_NAP_FUNC(signoreunknown);
I_NAP_FUNC(signoreexists);
I_NAP_FUNC(signoreclear);
I_NAP_FUNC(signorefail);
I_NAP_FUNC(scbanlist);
I_NAP_FUNC(snerr);
I_NAP_FUNC(snadd);
I_NAP_FUNC(sme);


#define O_NAP_FUNC(x) int (x)(int s, char *str, char **tok, int num, WINDOW *win)

O_NAP_FUNC(ddebug);
O_NAP_FUNC(djoin);
O_NAP_FUNC(dpart);
O_NAP_FUNC(dquit);
O_NAP_FUNC(dtell);
O_NAP_FUNC(dwhois);
O_NAP_FUNC(dget);
O_NAP_FUNC(dpup);
O_NAP_FUNC(dpdown);
O_NAP_FUNC(dsearch);
O_NAP_FUNC(dpvars);
O_NAP_FUNC(dkill);
O_NAP_FUNC(dban);
O_NAP_FUNC(dunban);
O_NAP_FUNC(ddatap);
O_NAP_FUNC(dtopic);
O_NAP_FUNC(dlevel);
O_NAP_FUNC(dg);
#ifdef QUEUE
O_NAP_FUNC(dforceq);
#endif /* QUEUE */
O_NAP_FUNC(dopsay);
O_NAP_FUNC(ddns);
O_NAP_FUNC(dannounce);
O_NAP_FUNC(pchans);
O_NAP_FUNC(dbanlist);
O_NAP_FUNC(dlspeed);
O_NAP_FUNC(ddup);
O_NAP_FUNC(ddown);
O_NAP_FUNC(dpsocks);
O_NAP_FUNC(dmuzzle);
O_NAP_FUNC(dunmuzzle);
O_NAP_FUNC(ddisconnect);
O_NAP_FUNC(dreload);
O_NAP_FUNC(dreconnect);
O_NAP_FUNC(dbrowse);
O_NAP_FUNC(dhelp);
O_NAP_FUNC(dping);
O_NAP_FUNC(duserpass);
O_NAP_FUNC(dreloadconf);
O_NAP_FUNC(dsver);
O_NAP_FUNC(dsetconf);
O_NAP_FUNC(dchanclear);
O_NAP_FUNC(dsetlevel);
O_NAP_FUNC(dme);
O_NAP_FUNC(dusers);
O_NAP_FUNC(dgusers);
O_NAP_FUNC(dclist);
O_NAP_FUNC(dclist2);
O_NAP_FUNC(dclear);
O_NAP_FUNC(dnews);
O_NAP_FUNC(dsraw);
O_NAP_FUNC(dquery);
O_NAP_FUNC(dcloak);
O_NAP_FUNC(dblocklist);
O_NAP_FUNC(dblock);
O_NAP_FUNC(dunblock);
O_NAP_FUNC(dwindow);
O_NAP_FUNC(dignore);
O_NAP_FUNC(dunignore);
O_NAP_FUNC(dignoreclear);
O_NAP_FUNC(dignorelist);
O_NAP_FUNC(dalias);
O_NAP_FUNC(dunalias);
O_NAP_FUNC(daliaslist);
O_NAP_FUNC(dhandler);
O_NAP_FUNC(dunhandler);
O_NAP_FUNC(dhandlerlist);
O_NAP_FUNC(dcbanlist);
O_NAP_FUNC(dcban);
O_NAP_FUNC(dcunban);
O_NAP_FUNC(dcbanclear);
O_NAP_FUNC(dkick);
O_NAP_FUNC(dloadalias);
O_NAP_FUNC(dsavealias);
O_NAP_FUNC(dclearalias);
O_NAP_FUNC(dloadhandler);
O_NAP_FUNC(dsavehandler);
O_NAP_FUNC(dclearhandler);
O_NAP_FUNC(dwstats);
O_NAP_FUNC(dnotify);
O_NAP_FUNC(dunnotify);
O_NAP_FUNC(dhotlist);
O_NAP_FUNC(dsay);
O_NAP_FUNC(dhup);
O_NAP_FUNC(dtquit);
O_NAP_FUNC(dabout);
O_NAP_FUNC(dirc);
O_NAP_FUNC(dkickall);
O_NAP_FUNC(deval);
O_NAP_FUNC(dset);
O_NAP_FUNC(dunset);
O_NAP_FUNC(dsaveconfig);
O_NAP_FUNC(dloadconfig);
O_NAP_FUNC(decho);
O_NAP_FUNC(dresults);
O_NAP_FUNC(dexec);
O_NAP_FUNC(dtimer);
O_NAP_FUNC(dif);
O_NAP_FUNC(ddone);
O_NAP_FUNC(dwhile);
O_NAP_FUNC(dinc);
O_NAP_FUNC(ddec);
O_NAP_FUNC(dbreak);
O_NAP_FUNC(dlastlog);
O_NAP_FUNC(drebuild);
O_NAP_FUNC(dchupload);
O_NAP_FUNC(dtlist);
O_NAP_FUNC(dnoprint);
O_NAP_FUNC(dstop);
O_NAP_FUNC(dserv);
O_NAP_FUNC(dserver);
O_NAP_FUNC(dupdate);



void upchan(chans_t *);
static int in_cksum(u_short *, int);
static void tvsub(register struct timeval *, register struct timeval *);
int icmpin(WINDOW *, sock_t *);
char **form_toks(char *, int *);
char **form_tokso(char *, int *);
int parsein(int, unsigned short, char *, WINDOW *);
char **fxv(char **, int, int *);
int parseout(int, char *, WINDOW *);
int parseoutn(int, char *, WINDOW *);
int gnum(unsigned char);
char *cstr(char *, int);
char *cspstr(char **, int, int);
int fcnt(file_t *);
void dupload(file_t *);
void ddownload(file_t *);
file_t *ffile(file_t *, char *, char *);
void adduser(chans_t *, char *, int, unsigned char);
void deluser(chans_t *, char *);
user_t *finduser(chans_t *, char *);

#ifdef QUEUE
/* stuff added for download queuing */
int enqdown(ssearch_t *x);
void deqdown(ssearch_t *x);
int deqdown2(char *nm, char *fn);
int downqhook(char *nick);
/* end of download queuing stuff */
#endif /* QUEUE */
#endif
