/* Copyright (c) 2000  Kevin Sullivan <nite@gis.net>
1 *
 * Please refer to the COPYRIGHT file for more information.
 */

#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <ncurses.h>
#include <sys/time.h>


#include "defines.h"
#include "cmds.h"
#include "vfuncs.h"
#include "alias.h"
#include "lists.h"
#include "nap.h"

/* the procedures in this file operate on two data structures: a
 * linked list of aliases and a linked list of "sets". Here a "set" is
 * something that is set by the command "set", i.e., a variable.
 *
 * alias_t provides for a simple key-value structure, where the key is
 * a string called "nm" and the value is a string called "args" (the
 * replacement for alias "nm").
 *
 * sets_t provides a similar key-value structure: the key is again a
 * string called "nm", and the value is a pair of a string "d" and an
 * integer "t". Apparently the intention is that t==0 if d represents
 * a number, t==1 otherwise. Actually, this is implemented
 * inconsistently:
 * chset() sets t=1 iff either d contains a letter [a-zA-Z] or d=="".
 * loadsets() sets t=1 iff d contains a non-digit. 
 *
 * There is also the array vars, which is defined in vfuncs.c, and
 * maps certain names to functions.
 * */

/* currently three variables have a special meaning: echosets,
   noresultscreen, savechannels. Each of them controls a user
   preference. The only user commands that access variables directly
   are /set, /unset, /inc, and /dec. They are accessed indirectly in
   certain expressions. */

extern vars_t vars[];
extern WINDOW *wchan; 
extern info_t info;

alias_t *alhead;
sets_t *setl;

/* ---------------------------------------------------------------------- */
/* the first part of this file deals with "sets", or variables that
   can be "/set" */

hardwired_t hw[] = {
  { "user", 1, &info.user, "# your napster username" },
  { "pass", 2, &info.pass, "# your napster password - optional" },
  { "email", 1, &info.email, "# your napster email address" },
  { "up", 1, &info.up, "# list of upload directories, separated by semicolon" },
  { "down", 1, &info.down, "# your download directory" },
  { "serverlist", 1, &info.serverlist, "# list of servers, separated by semicolon" },
  { "logfile", 1, &info.logfile, "# log file for transfer logs" },
  { "logallfile", 1, &info.logallfile, "# log file for logging everything" },
};

int hwsize = sizeof(hw)/sizeof(hardwired_t);

/* append sets from file */
int loadsets(char *fn)
{
  FILE *f;
  char buf[4096], *nm;
  int i, j;
  
  f = fopen(fn, "r");
  if (!f)
    return(-1);
  
  while (!feof(f))
  {
    memset(buf, 0, sizeof(buf));
    fgets(buf, sizeof(buf), f);
    if (!strchr(buf, ' '))
      continue;
    if (buf[strlen(buf)-1] == '\n')
      buf[strlen(buf)-1] = 0;
    for (i=0;buf[i]!=' ';i++);
    nm = (unsigned char *)malloc(i+1);
    strncpy(nm, buf, i);
    nm[i] = 0;
    chset(nm, buf+i+1);
    free(nm);
  }
  
  fclose(f);
  
  return(1);
}

/* save sets to file */
int savesets(char *fn)
{
  FILE *f;
  sets_t *cur;
  int i;
  
  f = fopen(fn, "w");
  if (!f)
    return(-1);
  
  /* first deal with hardwired variables */
  for (i=0; i<hwsize; i++) {
    if (hw[i].comment)
      fprintf(f, "%s\n", hw[i].comment);
    if (*hw[i].strp) {
      fprintf(f, "%s=%s\n", hw[i].nm, hw[i].type==2 ? "?" : *hw[i].strp);
    } else {
      fprintf(f, "#%s=\n", hw[i].nm);
    }
    if (hw[i].comment)
      fprintf(f, "\n");
  }
  
  for (cur=setl;cur;cur=cur->next)
    fprintf(f, "%s=%s\n", cur->nm, cur->d);
  
  fclose(f);
  
  return(1);
}

/* add a set to the list. Note: we do not check for duplicates. */
void addset(char *nm, char *val)
{
  sets_t *elt;
  char *endptr;
  int i;

  /* check if this is a hardwired set */
  for (i=0; i<hwsize; i++) {
    if (!strcmp(nm, hw[i].nm)) {
      free(*hw[i].strp);
      *hw[i].strp = strdup(val);
      return;
    }
  }

  /* make new assignment */
  elt = (sets_t *)malloc(sizeof(sets_t));
  elt->nm = strdup(nm);
  elt->d = strdup(val);
  elt->intval = strtol(elt->d, &endptr, 10);
  if (endptr && *endptr=='\0') { /* string was valid number */
    elt->t = 0;
  } else {
    elt->intval = 0;
    elt->t = 1;
  }

  /* now append it to the list */
  list_append(sets_t, setl, elt);
}

/* add a set to the list, but remove duplicate first. Set t=1 if this
   might be a number. */
void chset(char *nm, char *d)
{
  int i;

  delset(nm);
  if (d)
    addset(nm, d);
}

/* print a list of current variable assignments */
void printsets(WINDOW *win) {
  sets_t *elt;
  int count = 0;
  int i;

  /* first print hardwired sets */
  for (i=0; i<hwsize; i++) {
    if (*hw[i].strp) {
      if (hw[i].type == 2) 
	wp(win, "%s=\"%s\"\n", hw[i].nm, "?");
      else
	wp(win, "%s=\"%s\"\n", hw[i].nm, *hw[i].strp);
      count++;
    }
  }
  
  list_forall(elt, setl) {
    wp(win, "%s=\"%s\"\n", elt->nm, elt->d);
    count++;
  }

  if (!count)
    wp(win, "No variables are set\n");

  drw(win);
} 

/* look up set by name, or NULL if not found */
sets_t *findset(char *nm)
{
  sets_t *cur;
  
  for (cur=setl;cur&&strcasecmp(cur->nm, nm);cur=cur->next);
  
  return(cur);
}

/* get both d and t of a named set. *t=0 and d=NULL if not found. Do
   not set *t if t=NULL */
char *getval(char *nm, int *t)
{
  sets_t *cur;
  char *tmp=NULL;
  int i;
  
  if (t)
    *t = 1;

  /* check for hardwired sets first */
  for (i=0; i<hwsize; i++) {
    if (!strcmp(nm, hw[i].nm)) {
      if (*hw[i].strp) {
	if (hw[i].type == 1)
	  return *hw[i].strp;
	else
	  return "?";
      } else 
	goto null;
    }
  }
  
  cur = findset(nm);
  if (!cur)
    goto null;
  
  if (t)
    *t = cur->t;
  
  return(cur->d);

 null:
  if (t)
    *t = 0;
  return(NULL);
}

/* remove one named set from list */
void delset(char *nm)
{
  sets_t *cur, *cur1=NULL;
  int i;

  /* check for hardwired sets first */
  for (i=0; i<hwsize; i++) {
    if (!strcmp(nm, hw[i].nm)) {
      free(*hw[i].strp);
      *hw[i].strp = NULL;
      return;
    }
  }

  for (cur=setl;cur&&strcasecmp(cur->nm, nm);cur=cur->next)
    cur1 = cur;
  
  if (!cur)
    return;
  
  if (cur1)
    cur1->next = cur->next;
  else if (cur->next)
    setl = cur->next;
  else
    setl = NULL;
  
  free(cur->d);
  free(cur->nm);
  free(cur);
}

/* lookup set's value as a number, or -1 if we decided it isn't 
   a number, or if it isn't in the list */
long nvar(char *nm)
{
  sets_t *cur;
  int i;

  /* check for hardwired sets first */
  for (i=0; i<hwsize; i++) {
    if (!strcmp(nm, hw[i].nm)) {
      return -1;
    }
  }
  
  cur = findset(nm);
  if (!cur)
    return(-1);
  
  if (cur->t)
    return(-1);
  
  return(cur->intval);
}

/* ---------------------------------------------------------------------- */

/* return a copy of m with characters bp..ep-1 replaced by n. n may be
   longer or shorter than ep-bp; in fact even ep<bp is permitted.  The
   only requirement is that ep,bp <= strlen(m). */
char *ins(char *m, char *n, int bp, int ep)
{
  char *st;
  
  st = (char *)malloc(bp+strlen(n)+strlen(m+ep)+1);
  memset(st, 0, bp+strlen(n)+strlen(m+ep)+1);
  
  strncpy(st, m, bp);
  sprintf(st, "%s%s%s", st, n, m+ep);
  
  return(st);
}

/* tok is a NULL-terminated list of tokens. Print them to wchan (the
   main window), enclosed in |..|. Currently unused. */
void prt(char **tok)
{
  int i;
  char buf[1024];
  
  memset(buf, 0, sizeof(buf));

  for (i=0;tok[i];i++)
  {
    strcat(buf, tok[i]);
    strcat(buf, " ");
  }
  
  buf[strlen(buf)-1] = 0;
  
  wp(wchan, "|%s|\n", buf);
  drw(wchan);
}

unsigned char *dovars(unsigned char *nm)
{
  unsigned char *r=NULL, *t;
  char **vtok=NULL, b[512], **tok;
  int i, j, vcnt=0, cnt, x, y;
  
  if (findset(nm))
    r = strdup(getval(nm, NULL));
  
  if (strchr(nm, '('))
  {
    memset(b, 0, sizeof(b));
    strcpy(b, strchr(nm, '(')+1);
    b[strlen(b)-1] = 0;
    vtok = form_vtoks(b, &vcnt);
  }
  
  if (!vtok)
  {
    for (i=0;i<vcnt;i++)
    {
      if (*vtok[i] == '$')
      {
        t = dovars(vtok[i]+1);
        if (t)
        {
          free(vtok[i]);
          vtok[i] = t;
        }
      }
    }
  }
  else
  {
    for (i=0;i<vcnt;i++)
    {
      tok = form_toks(vtok[i], &cnt);
      tok = fxv(tok, cnt, &cnt);
      for (j=0;j<cnt;j++)
      {
        if (*tok[j] == '$' && !isdigit(tok[j][1]) && tok[j][1] != '+')
        {
          t = dovars(tok[j]+1);
          if (t)
          {
            free(tok[j]);
            tok[j] = t;
          }
        }
      }
      for (j=0;tok[j];j++)
      {
        if (*tok[j] == '$' && tok[j][1] == '+' && !tok[j][2])
        {
          if (j > 0 && tok[j+1] && tok[j-1])
          {
            for (x=1;tok[j-x]&&!(*tok[j-x]);x++);
            for (y=1;tok[j+y]&&!(*tok[j+y]);y++);
            if (!tok[j-x])
              x--;
            if (!tok[j+y])
              y--;
            tok[j] = (char *)realloc(tok[j], strlen(tok[j-x])+strlen(tok[j+y])+1);
            if (strcmp(tok[j-x], "$+"))
              strcpy(tok[j], tok[j-x]);
            else
              *tok[j] = 0;
            if (strcmp(tok[j+y], "$+"))
              strcat(tok[j], tok[j+y]);
            *tok[j-x] = 0;
            *tok[j+y] = 0;
          }
          else
            *tok[j] = 0;
        }
      }
      
      vtok[i] = (char *)realloc(vtok[i], 2048);
      memset(vtok[i], 0, 2048);
      for (j=0;tok[j];j++)
      {
        if (*tok[j])
        {
          strcat(vtok[i], tok[j]);
          strcat(vtok[i], " ");
        }
        free(tok[j]);
      }
      free(tok);
      vtok[i][strlen(vtok[i])-1] = 0;
      vtok[i] = (char *)realloc(vtok[i], strlen(vtok[i])+1);
    }
  }
  
  memset(b, 0, sizeof(b));
  for (i=0;nm[i]&&nm[i]!='(';i++)
    b[i] = nm[i];
  
  for (i=0;;i++)
  {
    if (!vars[i].nm)
    {
      if (vtok)
      {
        for (j=0;j<vcnt;j++)
          free(vtok[j]);
        free(vtok);
      }
      break;
    }
    if (!strcasecmp(vars[i].nm, b))
    {
      r = vars[i].func(vtok, vcnt);
      if (vtok)
      {
        for (j=0;j<vcnt;j++)
          free(vtok[j]);
        free(vtok);
      }
      break;
    }
  }
  
  if (!r)
    return(strdup(""));
  else  
    return(r);
}

/* tokenize a string t of the form " x1, x2, ..., xn" into the tokens
   x1 through xn. Here x1...xn are well-balanced strings (with respect
   to '(' and ')'). Notes: x1...xn may contain additional commas
   inside parentheses.  Whitespace is skipped at the beginning and after
   commas, but not before commas. The behavior is undefined (messy) if
   x1...xn are not well-balanced. Some tokens may be the empty string;
   this also happens if t consist of one or more spaces only.
   Returns: NULL-terminated array of tokens and number of tokens in
   n. */

char **form_vtoks(char *t, int *n)
{
  int i, j, k, c=0;
  char **ret;
  
  if (!strlen(t))
  {
    *n = 0;
    return(NULL);
  }
  
  ret = (char **)malloc(4096);
  
  for (i=0,j=0;;i++)  /* i=token number, j=character number in t. */
  {
    ret[i] = (char *)malloc(1024);
    memset(ret[i], 0, 1024);
    while (t[j] && isspace(t[j]))
      j++;
    for (k=0,c=0;t[j];k++,j++) /* k=char. number in ret[i], c=nesting level */
    {
      if (t[j] == ',' && !c)
        break;
      if (c < 0)
        c = 0;
      if (t[j] == '(')
        c++;
      else if (t[j] == ')')
        c--;
      ret[i][k] = t[j];
    }
    ret[i] = (char *)realloc(ret[i], strlen(ret[i])+1);
    if (!t[j])
      break;
    j++;
  }
  
  ret[i+1] = NULL;
  *n = i+1;
  
  return(ret);
}

unsigned char *doalias(alias_t *al, char **tok, int cnt)
{
  char *ret, echr;
  char **atok, p[16], pt[2048], tb[2048], *t;
  int i, j, acnt, n, l, c;
  
  t = strdup(al->args);
  
  for (i=0;al->args[i]&&al->args[i] == ' ';i++);
  
  if (al->args[i] == '{')
  {
    for (i=0,c=0;t[i];i++)
    {
      if (c < 0)
        c = 0;
      if (t[i] == '{' && !c)
      {
        t[i] = ' ';
        c++;
      }
      else if (t[i] == '{')
        c++;
      else if (t[i] == '}' && c == 1)
      {
        t[i] = ' ';
        c--;
      }
      else if (t[i] == '}')
        c--;
    }
  }
  
  if (t[strlen(t)-1] == '|' || t[strlen(t)-1] == '\n')
    t[strlen(t)-1] = 0;
  
  memset(pt, 0, sizeof(pt));
  strcpy(pt, t);
  free(t);
  
  for (i=0;pt[i];i++)
  {
    if (pt[i] == '$' || pt[i] == '?')
    {
      for (j=i,echr=0;;j++)
      {
        if (!pt[j] || pt[j] == ',' || pt[j] == ' ' || pt[j] == '=' || pt[j] == '!' || pt[j] == '>' || pt[j] == '<')
        {
          echr = pt[j];
          break;
        }
      }
      if (i > 0 && (pt[i-1] == '(' || pt[i-1] == ')' || pt[i-1] == ',' || pt[i-1] == '=' || pt[i-1] == '!' || pt[i-1] == '<' || pt[i-1] == '>' || pt[i-1] == ' '))
      {
        if (pt[i-1] != ' ')
        {
          memset(tb, 0, sizeof(tb));
          strcpy(tb, pt+i);
          strcpy(pt+i+1, tb);
          pt[i] = ' ';
        }
        else
          j--;
        if (echr && echr != ' ')
        {
          memset(tb, 0, sizeof(tb));
          strcpy(tb, pt+j+1);
          strcpy(pt+j+2, tb);
          pt[j+1] = ' ';
        }
      }
    }
  }
  
  t = strdup(pt);
  
  atok = form_tokso(t, &acnt);
  free(t);
  
  for (i=0,j=0;i<acnt;i++)
    if (*atok[i] == '$' && atok[i][1])
      if (atoi(atok[i]+1) > j)
        j = atoi(atok[i]+1);

  if ((cnt-1) < j)
    return(NULL);
  
  for (i=0;atok[i];i++)
  {
    if (*atok[i] == '?' && isdigit(atok[i][1]))
    {
      if (atoi(atok[i]+1) <= (cnt-1))
        *atok[i] = '$';
      else
      {
        for (j=1;;j++)
        {
          if (!isdigit(atok[i][j]) && atok[i][j] != '-')
          {
            memset(tb, 0, sizeof(tb));
            strcpy(tb, atok[i]+j);
            strcpy(atok[i], tb);
            break;
          }
        }
      }
    }
  }
  
  for (i=0,l=0;atok[i];i++)
  {
    if (*atok[i] == '$' && isdigit(atok[i][1]) && !strchr(atok[i], '-'))
    {
      n = atoi(atok[i]+1);
      memset(tb, 0, sizeof(tb));
      for (j=1;atok[i][j];j++)
        if (!isdigit(atok[i][j]))
          break;
      strcpy(tb, atok[i]+j);
      atok[i] = (char *)realloc(atok[i], strlen(tok[n])+strlen(tb)+1);
      strcpy(atok[i], tok[n]);
      strcat(atok[i], tb);
    }
    else if (*atok[i] == '$' && isdigit(atok[i][1]))
    {
      memset(p, 0, sizeof(p));
      for (j=1;;j++)
      {
        if (!isdigit(atok[i][j]) && atok[i][j] != '-')
        {
          memset(tb, 0, sizeof(tb));
          strcpy(tb, atok[i]+j);
          strcpy(atok[i], tb);
          break;
        }
        if (atok[i][j] != '-')
          p[j-1] = atok[i][j];
      }
      n = atoi(p);
      memset(pt, 0, sizeof(pt));
      for (j=n;tok[j];j++)
      {
        strcat(pt, tok[j]);
        strcat(pt, " ");
      }
      pt[strlen(pt)-1] = 0;
      atok[i] = (char *)realloc(atok[i], strlen(pt)+strlen(tb)+1);
      strcpy(atok[i], pt);
      strcat(atok[i], tb);
    }
    else if (*atok[i] == '$' && !strcasecmp(atok[i]+1, "num"))
    {
      free(atok[i]);
      atok[i] = NULL;
      msprintf(&atok[i], "%i", cnt-1);
    }
    else if (*atok[i] == '$' && !strcasecmp(atok[i]+1, "str"))
    {
      free(atok[i]);
      atok[i] = (char *)malloc(4096);
      memset(atok[i], 0, 4096);
      for (j=0;tok[j];j++)
      {
        if (*tok[j])
        {
          strcat(atok[i], tok[j]);
          strcat(atok[i], " ");
        }
      }
      atok[i][strlen(atok[i])-1] = 0;
      atok[i] = (char *)realloc(atok[i], strlen(atok[i])+1);
    }
    l+=strlen(atok[i])+1;
  }

  ret = (char *)malloc(4096);
  memset(ret, 0, 4096);
  
  for (i=0;atok[i];i++)
  {
    if (*atok[i])
    {
      strcat(ret, atok[i]);
      strcat(ret, " ");
    }
    free(atok[i]);
  }
  
  free(atok);
  
  ret[strlen(ret)-1] = 0;
  ret = (char *)realloc(ret, strlen(ret)+1);
  
  return((unsigned char *)ret);
}

/* save alias list to named file */
int savealiases(char *fn)
{
  FILE *f;
  alias_t *cur;
  
  f = fopen(fn, "w");
  if (!f)
    return(-1);
  
  for (cur=alhead;cur;cur=cur->next)
    fprintf(f, "%s %s\n", cur->nm, cur->args);
  
  fclose(f);
  
  return(1);
}

/* load aliases from named file and append to alias list. The format
   is ad hoc. */
int loadaliases(char *fn)
{
  FILE *f;
  alias_t *cur, *cur1=NULL;
  char buf[4096], tb[4096];
  int i, c, j;
  
  f = fopen(fn, "r");
  if (!f)
    return(-1);
  
  for (cur=alhead;cur;cur=cur->next)
    cur1 = cur;
  /* cur1 now points to last element, or NULL if list was empty. */
  
  while (!feof(f))
  {
    memset(buf, 0, sizeof(buf));
    fgets(buf, sizeof(buf), f);
    if (*buf == '#')   /* skip comments in column 0 */
      continue;
    if (buf[strlen(buf)-1] == '\n')
      buf[strlen(buf)-1] = 0;
    for (i=0,c=0;buf[i];i++)  /* find the first space, if any */
    {
      if (isspace(buf[i]))
      {
        c = 1;
        break;
      }
    }
    /* no space found; continue on next line. Note: no comment allowed
       here.  */

    if (!c)
    {
      buf[strlen(buf)] = ' ';
      fgets(buf+strlen(buf), sizeof(buf), f);
      if (buf[strlen(buf)-1] == '\n')
        buf[strlen(buf)-1] = 0;
    }
    
    buf[strlen(buf)] = '\n';
    
    for (c=0,i=0;;i++)
    {
      if (c < 0)
      {
        c = 0;
        break;
      }
      if (!buf[i] && c)
      {
        while (1)
        {
          memset(tb, 0, sizeof(tb));
          fgets(tb, sizeof(tb), f);
          for (j=0;tb[j]==' ';j++);
          if (tb[j] != '#' || feof(f))
          {
            strcpy(buf+i+1, tb);
            buf[i] = ' ';
            break;
          }
        }
      }
      else if (!buf[i])
        break;
      if (buf[i] == '{')
        c++;
      else if (buf[i] == '}')
        c--;
    }
    
    for (i=0;buf[i];i++)
    {
      if (buf[i] == '\n')
      {
        for (j=i+1;buf[j]==' '&&buf[j]!='{';j++);
        if (buf[j] == '{')
        {
          buf[i] = ' ';
          strncpy(buf+i+1, buf+j, strlen(buf+j));
          buf[strlen(buf)-(j-i)+1] = 0;
        }
      }
    }
    if (*buf == ' ')
      continue;
    for (i=strlen(buf)-1;buf[i] == ' '&&i>=0;i--);
    while (buf[i] == '\n')
    {
      buf[i] = 0;
      i--;
    }
    cur = (alias_t *)malloc(sizeof(alias_t));
    for (i=0;buf[i]!=' ';i++);
    cur->args = strdup(buf+i+1);
    cur->nm = (char *)malloc(i);
    cur->next = NULL;
    if (!cur1)
      alhead = cur;
    else
      cur1->next = cur;
    cur1 = cur;
    strncpy(cur->nm, buf, i);
    cur->nm[i] = 0;
  }
  
  fclose(f);
  
  return(1);
}

/* evaluate a (presumably boolean) expression. Used by such things as
   "/if" and "/while" */
char *strev(char *str, unsigned char l, unsigned char r, char *(*func)())
{
  char *s1, *s2, *rt, *t, *t2;
  char t3[1024], t4[1024], **tok, echr;
  int i, j, cnt;
  
  memset(t3, 0, sizeof(t3));
  strcpy(t3, str);
  
  for (i=0;t3[i];i++)
  {
    if (t3[i] == '$')
    {
      for (j=i,echr=0;;j++)
      {
        if (!t3[j] || t3[j] == ',' || t3[j] == ' ' || t3[j] == '=' || t3[j] == '!' || t3[j] == '>' || t3[j] == '<')
        {
          echr = t3[j];
          break;
        }
      }
      if (i > 0 && (t3[i-1] == '(' || t3[i-1] == ')' || t3[i-1] == ',' || t3[i-1] == '=' || t3[i-1] == '!' || t3[i-1] == '<' || t3[i-1] == '>' || t3[i-1] == ' '))
      {
        if (t3[i-1] != ' ')
        {
          memset(t4, 0, sizeof(t4));
          strcpy(t4, t3+i);
          strcpy(t3+i+1, t4);
          t3[i] = ' ';
        }
        else
          j--;
        if (echr && echr != ' ')
        {
          memset(t4, 0, sizeof(t4));
          strcpy(t4, t3+j+1);
          strcpy(t3+j+2, t4);
          t3[j+1] = ' ';
        }
      }
    }
  }
  
  if (strlen(t3) > 3 && t3[strlen(t3)-1] == ')' && t3[strlen(t3)-2] == ')')
  {
    t3[strlen(t3)-1] = ' ';
    t3[strlen(t3)] = ')';
  }
  
  t3[strlen(t3)-1] = ' ';
  t3[strlen(t3)] = ')';
  
  tok = form_tokso(t3, &cnt);
  tok = fxv(tok, cnt, &cnt);
  
  for (i=0;tok[i];i++)
  {
    if (*tok[i] == '$' && tok[i][1] && !isdigit(tok[i][1]))
    {
      t = dovars(tok[i]+1);
      if (t)
      {
        free(tok[i]);
        tok[i] = t;
      }
    }
  }
  
  memset(t3, 0, sizeof(t3));
  
  for (i=0;tok[i];i++)
  {
    if (*tok[i])
    {
      strcat(t3, tok[i]);
      strcat(t3, " ");
    }
    free(tok[i]);
  }
  
  free(tok);
  
  if (t3[strlen(t3)-1] == ' ')
    t3[strlen(t3)-1] = 0;
  
  if (t3[strlen(t3)-1] != r)
  {
    t3[strlen(t3)] = ' ';
    t3[strlen(t3)] = r;
    t3[strlen(t3)] = 0;
  }
  
  rt = strdup(t3);
  
  for (;;)
  {
    s1 = strrchr(rt, l);
    if (s1 == NULL)
      break;
    if (strchr(rt, r) == NULL)
    {
      free(rt);
      return(NULL);
    }
    s2 = (char *)malloc(abs(strchr(s1, r)-s1));
    strncpy(s2, s1+1, abs(strchr(s1, r)-s1)-1);
    s2[abs(strchr(s1, r)-s1)-1] = '\0';
    t = func(s2);
    if (t == NULL)
    {
      free(rt);
      free(s2);
      return(NULL);
    }
    t2 = strdup(rt);
    free(rt);
    s1 = strrchr(t2, l);
    rt = ins(t2, t, abs(s1-t2), abs((s1-t2))+(abs((strchr(s1, r)-s1))+1));
    free(t2);
    free(t);
  }
  
  return(rt);
}

char *cmp(char *st)
{
  char b[512], r[512], **tok, *tm=NULL, *rt=NULL;
  int i=0, j, cnt, k, l=0, t;
  
  memset(b, 0, sizeof(b));
  memset(r, 0, sizeof(r));
  
  while (st[i])
  {
    while (st[i] == ' ' && st[i])
      i++;
    for (b[0]='\'',j=1;st[i]&&st[i]!='&'&&st[i]!='^';j++,i++)
    {
      if (st[i] == '=' || st[i] == '!' || st[i] == '>' || st[i] == '<')
      {
        b[j++] = '\'';
        b[j++] = ' ';
        b[j++] = st[i];
        b[j++] = ' ';
        b[j] = '\'';
      }
      else
        b[j] = st[i];
    }
    b[j] = '\'';
    b[j+1] = 0;
    if (st[i] == '&' || st[i] == '^')
      i++;
    tok = form_toks(b, &cnt);
    tok = fxv(tok, cnt, &cnt);
    
    if (!cnt)
    {
      tok[0] = NULL;
      tok[1] = NULL;
      tok[2] = NULL;
    }
    if (cnt == 1)
    {
      tok[1] = NULL;
      tok[2] = NULL;
    }
    else if (cnt == 2)
      tok[2] = NULL;
    
    if (tok[0] && *tok[0] == '$')
    {
      tm = dovars(tok[0]+1);
      if (tm)
      {
        free(tok[0]);
        tok[0] = tm;
      }
    }
    if (tok[1] && *tok[1] == '$')
    {
      tm = dovars(tok[1]+1);
      if (tm)
      {
        free(tok[1]);
        tok[1] = tm;
      }
    }
    if (tok[2] && *tok[2] == '$')
    {
      tm = dovars(tok[2]+1);
      if (tm)
      {
        free(tok[2]);
        tok[2] = tm;
      }
    }
    
    tm = NULL;
    if (!tok[0])
      tm = strdup("");
    else
      msprintf(&tm, "\'%s\' %s \'%s\'", tok[0]?tok[0]:"", tok[1]?tok[1]:"", tok[2]?tok[2]:"");
    for (k=0;k<cnt;k++)
      free(tok[k]);
    free(tok);
    tok = form_toks(tm, &cnt);
    free(tm);
    
    if (!cnt)
    {
      tok[0] = NULL;
      tok[1] = NULL;
      tok[2] = NULL;
    }
    if (cnt == 1)
    {
      tok[1] = NULL;
      tok[2] = NULL;
    }
    else if (cnt == 2)
      tok[2] = NULL;
    
    if (tok[2] && !(*tok[2]))
    {
      free(tok[2]);
      tok[2] = NULL;
      cnt--;
    }
    if (tok[1] && !(*tok[1]))
    {
      free(tok[1]);
      tok[1] = NULL;
      if (tok[2])
      {
        tok[1] = strdup(tok[2]);
        free(tok[2]);
        tok[2] = NULL;
      }
      cnt--;
    }
    if (tok[0] && !(*tok[0]))
    {
      free(tok[0]);
      if (tok[1])
      {
        tok[0] = strdup(tok[1]);
        free(tok[1]);
        tok[1] = NULL;
      }
      if (tok[2])
      {
        tok[1] = strdup(tok[2]);
        free(tok[2]);
        tok[2] = NULL;
      }
      cnt--;
    }
    
    if (cnt == 2)
    {
      if (*tok[0] == '=' || *tok[0] == '!' || *tok[0] == '>' || *tok[0] == '<')
      {
        if (*tok[0] == '!')
        {
          if (strcmp(tok[1], "0"))
          {
            free(tok[0]);
            free(tok[1]);
            tok[3] = NULL;
            tok[2] = strdup("0");
            tok[1] = strdup("!");
            tok[0] = strdup("0");
          }
          else
          {
            free(tok[0]);
            free(tok[1]);
            tok[3] = NULL;
            tok[2] = strdup("0");
            tok[1] = strdup("=");
            tok[0] = strdup("0");
          }
        }
        else
        {
          tok[3] = NULL;
          tok[2] = strdup(tok[1]);
          free(tok[1]);
          tok[1] = strdup(tok[0]);
          free(tok[0]);
          tok[0] = strdup("0");
        }
      }
      else if (*tok[1] == '=' || *tok[1] == '!' || *tok[1] == '>' || *tok[1] == '<')
      {
        tok[3] = NULL;
        tok[2] = strdup(tok[0]);
        if (*tok[1] == '=')
          *tok[1] = '!';
        else if (*tok[1] == '!')
          *tok[1] = '=';
      }
      cnt = 3;
    }
    else if (cnt == 1)
    {
      if (strcmp(tok[0], "0") || *tok[0] == '!')
      {
        free(tok[0]);
        tok[0] = strdup("1");
        tok[1] = strdup("=");
        tok[2] = strdup("1");
        tok[3] = NULL;
      }
      else
      {
        free(tok[0]);
        tok[0] = strdup("1");
        tok[1] = strdup("!");
        tok[2] = strdup("1");
        tok[3] = NULL;
      }
      cnt = 3;
    }
    else if (!cnt)
    {
      tok[0] = strdup("1");
      tok[1] = strdup("!");
      tok[2] = strdup("1");
      tok[3] = NULL;
      cnt = 3;
    }
    
    if (cnt == 3)
    {
      if (*tok[1] == '=')
      {
        if (!strcasecmp(tok[0], tok[2]))
          r[l++] = 1;
        else
          r[l++] = 0;
      }
      else if (*tok[1] == '!')
      {
        if (!strcasecmp(tok[0], tok[2]))
          r[l++] = 0;
        else
          r[l++] = 1;
      }
      else if (*tok[1] == '>')
      {
        if (atoi(tok[0]) > atoi(tok[2]))
          r[l++] = 1;
        else
          r[l++] = 0;
      }
      else if (*tok[1] == '<')
      {
        if (atoi(tok[0]) < atoi(tok[2]))
          r[l++] = 1;
        else
          r[l++] = 0;
      }
    }
    
    for (k=0;tok[k];k++)
      free(tok[k]);
    free(tok);
    
    if (i > 0 && (st[i-1] == '&' || st[i-1] == '^'))
      r[l++] = st[i-1];
  }
  
  r[l] = 0;
  
  for (i=0,t=0;i<l;i++)
  {
    if (r[i] == '&')
      t &= r[++i];
    else if (r[i] == '^')
      t |= r[++i];
    else
      t = r[i];
  }
  
  msprintf(&rt, "%i", t);
  
  return(rt);
}
