#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "xposix.h"

#ifdef O_BINARY
void x_setmode(int fd, int mode)
{
    /* 
     * Take into account the fact that Windows is designed to only
     * pipe text streams, i.e., stdin is opened by default in "text"
     * mode.  We need to put it in "binary" mode without losing data.
     */

    /*
     * Warning: BSD defines setmode() too, but it is for setting file
     * permissions as with the "chmod" command.
     */
    if( setmode(fd, O_BINARY) == -1 ) {
        perror("setmod");
        exit(1);
    }
}
#endif    

int
xopen(const char* path, int flags, mode_t mode)
{
    int rv = -1;
    while( (rv = open(path, flags, mode)) == -1 ) {
        if( errno == EINTR ) {
            continue;
        }
        perror("open");
        exit(1);
    }
    return rv;
}

extern int
xclose(int fd)
{
    int rv = -1;
    while( (rv = close(fd)) == -1 ) {
        if( errno == EINTR ) {
            continue;
        }
        perror("close");
        exit(1);
    }
    return rv;
}

ssize_t
xread(int fd, void* buf, size_t buf_size)
{
    ssize_t rcount = -1;
    ssize_t rcount_total = 0;
    /* Simple cast to make life easier.  Be careful with the names! */
    ssize_t buf_ssize = (ssize_t)buf_size;
    while( rcount_total < buf_ssize ) {
        rcount = read(fd, buf + rcount_total, buf_ssize - rcount_total);
        if( rcount == -1 ) {
            if( errno == EINTR ) {
                continue;
            }
            perror("read");
            exit(1);
        }
        if( rcount == 0 ) {
            /* EOF */
            break;
        }
        rcount_total += rcount;
    }
    return rcount_total;
}

ssize_t
xwrite(int fd, const void* buf, size_t buf_size)
{
    ssize_t wcount = -1;
    ssize_t wcount_total = 0;
    /* Simple cast to make life easier.  Be careful with the names! */
    ssize_t buf_ssize = (ssize_t)buf_size;
    while( wcount_total < buf_ssize ) {
        wcount = write(fd, buf + wcount_total, buf_ssize - wcount_total);
        if( wcount == -1 ) {
            if( errno == EINTR ) {
                continue;
            }
            perror("write");
            exit(1);
        }
        wcount_total += wcount;
    }
    return wcount_total;
}

extern pid_t
xfork(void)
{
    pid_t rv = -1;
    while( (rv = fork()) == -1 ) {
        if( errno == EINTR ) {
            continue;
        }
        perror("fork");
        exit(1);
    }
    return rv;
}

int
xpipe(int fds[2])
{
    int rv = -1;
    while( (rv = pipe(fds)) == -1 ) {
        if( errno == EINTR ) {
            continue;
        }
        perror("pipe");
        exit(1);
    }
    return rv;
}

extern int
xdup2(int src, int dst)
{
    int rv = -1;
    while( (rv = dup2(src, dst)) == -1 ) {
        if( errno == EINTR ) {
            continue;
        }
        perror("dup2");
        exit(1);
    }
    return rv;
}

extern int
xselect(int n, fd_set* rset, fd_set* wset, fd_set* eset, struct timeval* t)
{
    int rv = -1;
    while( (rv = select(n, rset, wset, eset, t)) == -1 ) {
        if( errno == EINTR ) {
            continue;
        }
        perror("select");
        exit(1);
    }
    return rv;
}

int
xsigemptyset(sigset_t* s)
{
    int rv = -1;
    while( (rv = sigemptyset(s)) == -1 ) {
        if( errno == EINTR ) {
            continue;
        }
        perror("sigemptyset");
        exit(1);
    }
    return rv;
}

int
xsigaction(int num, const struct sigaction* act, struct sigaction* oact)
{
    int rv = -1;
    while( (rv = sigaction(num, act, oact)) == -1 ) {
        if( errno == EINTR ) {
            continue;
        }
        perror("sigaction");
        exit(1);
    }
    return rv;
}

int
xsigsuspend(const sigset_t* mask)
{
    /* Just pass it through. */
    return sigsuspend(mask);
}

extern void*
xmalloc(size_t buffer_size)
{
    void* rv = malloc(buffer_size);
    if( rv == NULL ) {
        perror("malloc");
        exit(1);
    }
    return rv;
}

extern void*
xcalloc(size_t count, size_t size_of_each)
{
    void* rv = calloc(count, size_of_each);
    if( rv == NULL ) {
        perror("calloc");
        exit(1);
    }
    return rv;
}

extern void*
xrealloc(void* buffer, size_t buffer_size)
{
    void* rv = realloc(buffer, buffer_size);
    if( (buffer_size != 0) && (rv == NULL) ) {
        perror("realloc");
        exit(1);
    }
    return rv;
}

extern int
xsetenv(const char* name, const char* value, int overwrite)
{
    int rv = setenv(name, value, overwrite);
    if( rv == -1 ) {
        perror("setenv");
        exit(1);
    }
    return rv;
}

const char*
xbasename(const char* path)
{
    const char* rv = strrchr(path, '/');
    if( rv == NULL ) {
        rv = strrchr(path, '\\');
    }
    if( rv ) {
        ++rv;
    } else {
        rv = path;
    }
    return rv;
}

int
xgetchar(void)
{
    int rv = 0;
    while ((rv = getchar()) == EOF) {
        if (errno == EINTR) {
            continue;
        }
        perror("getchar");
        exit(1);
    }
    return rv;
}
