#include <paths.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "znum.h"
#include "../xposix/xposix.h"

/* Keep track of the child processes. */
typedef struct {
    const char* cmd;
    int is_running;
    int is_auto_close;
    int is_auto_restart;
    int fd_stdin;
    int fd_stdout;
    pid_t pid;
    /* Track how often shunt has started this child process.  We pass
     * this value to the child process in the SHUNT_STARTS environment
     * variable. */
    znum_t shunt_starts;
} cprocs_t;

/* If and only if the process that is endowed with the "-c" flag has
 * exited should we start automatically shutting down the pipeline. */
static int is_auto_closing;

/* Track how often shunt has started all child process.  We pass this
 * value to the child process in the SHUNT_ALL_STARTS environment
 * variable. */
static znum_t shunt_all_starts;

/* We add one cprocs_t to the end of "children" as a sentinel to make
 * looping over the array easier: We can always just check to see if
 * children[i].cmd is NULL. */
static cprocs_t* children;

/* Size of "children" which is dynamically determined by inspecting
 * the command-line parameters.  This is an upper bound.  It is not an
 * exact count of the children. */
static unsigned long int size_of_children;

/* This is the exact count of the total number of children. */
static unsigned long int number_of_children;

/* File descriptor for /dev/null. */
static int fd_devnull = -1;

/* Make it easier to deal with asynchronous signals. */
static int async_to_sync_pipe[2];

static const char* progname;

static const char * const usage_msg = "\
\n\
Usage: %s <flags> <child> [+ <flags> <child>] . . .\n\
\n\
    Flags:\n\
\n\
         + : Add another child.\n\
        -c : Automatically close the pipe when child exits.  This should\n\
             start a chain reaction that eventually causes all processes\n\
             that are part of the pipeline to close.\n\
        -h : Print help.\n\
        -r : Automatically restart child.\n\
\n\
    Environment Variables Passed to Children:\n\
\n\
            SHUNT_STARTS: How many times this child has been started.\n\
        SHUNT_ALL_STARTS: How many times all children have been started\n\
\n\
    About:\n\
\n\
        Version: 1.7.2\n\
        License: BSD\n\
         Author: Paul Serice\n\
         E-Mail: paul@serice.net\n\
            URL: http://www.serice.net/shunt/\n\
\n\
";

static void
show_usage(FILE* f)
{
    fprintf(f, usage_msg, progname);
}

static void
decipher_command_line(int argc, char** argv)
{
    int ci = 0; /* index for children */
    int i = 1;

    /* First, calculate an upper bound on how many children the user
     * wants to start.  We start with size_of_children = 1 because
     * there is no '+' that precedes the first child. */
    size_of_children = 1;
    for( i = 1 ; i < argc ; ++i ) {
        const char* arg = argv[i];
        if( (arg[0] == '+') && (arg[1] == '\0') ) {
            ++size_of_children;
        }       
    }
    ++size_of_children;  /* One more for the sentinel. */

    /* Allocate the array of children and initialize to all zero. */
    children = (cprocs_t*)xcalloc(size_of_children, sizeof(cprocs_t));
    for( i = 0 ; i < size_of_children ; ++i ) {
        children[i].shunt_starts = znum_new();;
    }

    /* Substantively iterate over the command-line parameters. */
    for( i = 1 ; i < argc ; ++i ) {
        const char* arg = argv[i];

        /*
         * Ignore empty arguments.
         */
        if( arg[0] == '\0' ) {
            continue;
        }

        /*
         * +
         */
        if( (arg[0] == '+') && (arg[1] == '\0') ) {
            /* If this is a '+' but the previous command is
             * incomplete, alter the user to the error. */
            if( children[ci].cmd == NULL ) {
                fprintf(stderr,
                        "\n%s: Error: Cannot move to command #%d because "
                        "command #%d is not complete.\n",
                        progname,
                        ci + 2,
                        ci + 1);
                show_usage(stderr);
                exit(1);
            }
            /* Otherwise, increment the index into "children". */
            ++ci;
            continue;
        }

        /*
         * -h ==> Help
         */
        if( strcmp(arg, "-h") == 0 ) {
            show_usage(stdout);
            exit(0);
        }
        
        /*
         * -c ==> Auto Exit
         */
        if( strcmp(arg, "-c") == 0 ) {
            children[ci].is_auto_close = 1;
            continue;
        }
        
        /*
         * -r ==> Auto Restart
         */
        if( strcmp(arg, "-r") == 0 ) {
            children[ci].is_auto_restart = 1;
            continue;
        }

        /*
         * Warn if some other flag.
         */
        if( arg[0] == '-' ) {
            fprintf(stderr, "\n%s: Error: Invalid flag: %s\n", progname, arg);
            show_usage(stderr);
            exit(1);
        }

        /*
         * What is left must be the command.
         */
        
        /* Detect if the user is trying to set the command twice which
         * could easily occur if the user forgets to put a '+' between
         * commands. */
        if( children[ci].cmd ) {
            fprintf(stderr,
                    "\n%s: Error: Command already set.  Perhaps you "
                    "are missing a '+' flag?\n",
                    progname);
            show_usage(stderr);
            exit(1);
        }
        children[ci].cmd = arg;
        
    }

    /* Verify that at least one child was specified. */
    if( children[0].cmd == NULL ) {
        fprintf(stderr, "\n%s: Error: No children specified.\n", progname);
        show_usage(stderr);
        exit(1);
    }

    number_of_children = ci + 1;
}

static int
max_int(int a, int b)
{
    return ( a > b ) ? a : b;
}

static void
ignore_line()
{
    int c = EOF;
    for( ;; ) {
        c = getchar();
        if( c == '\n' ) {
            break;
        }
    }
}

static void
reap(int num)
{
    ssize_t wcount = write(async_to_sync_pipe[1], &num, sizeof(num));
    if( wcount != sizeof(num) ) {
        static const char* reap_error_msg =
            "\n***\n*** write() failed in reap().  Exiting!\n***\n\n";
        write(STDOUT_FILENO, reap_error_msg, strlen(reap_error_msg));
        _exit(1);  /* exit() is not async-signal safe. */
    }
}

static void
start_child(cprocs_t* child)
{
    /* Given that xfork() exits on error, we might as well do the
     * calculations here only once (i.e., before the fork). */
    znum_increment(child->shunt_starts);
    znum_increment(shunt_all_starts);
    child->is_running = 1;

    child->pid = xfork();

    if( child->pid  == 0 ) {

        /* Child */
        int i = 0;

        /* Set up stdin and stdout for the child. */
        xdup2(child->fd_stdin, STDIN_FILENO);
        xdup2(child->fd_stdout, STDOUT_FILENO);

        /* This for() loop closes all the fds associated with the
         * pipes that are still open.
         *
         * It is true that if shunt is just starting, the parent
         * process will still have all the fds open, but if the chain
         * reaction that shuts down the pipeline has started, some of
         * the processes in the pipeline will have had their fds
         * permanently disconnected.  Thus, we have to be careful not
         * to attempt to reclose those disconnected fds; thus, we
         * explicitly check to make sure the fd != -1.
         *
         * Also, remember that stdout for the last process in the
         * pipeline is connected to the tty which you want to always
         * leave open; thus, we explicitly check to make sure
         * children[i+1].cmd is set. */
        for( i = 0 ; children[i].cmd ; ++i ) {
            if( children[i].fd_stdin != -1 ) {
                xclose(children[i].fd_stdin);
            }
            if( children[i+1].cmd && (children[i].fd_stdout != -1) ) {
               	xclose(children[i].fd_stdout);
            }
        }

        /* Set up the SHUNT_STARTS and SHUNT_ALL_STARTS environment
         * variables. */
        xsetenv("SHUNT_STARTS", znum_as_string(child->shunt_starts), 1);
        xsetenv("SHUNT_ALL_STARTS", znum_as_string(shunt_all_starts), 1);

        /* Start the child. */
        execlp("/bin/sh", "/bin/sh", "-c", child->cmd, NULL);
        perror("execlp");
        exit(1);
    }
}

static void
start_all_children()
{
    int i = 0;
    for( i = 0 ; children[i].cmd ; ++i ) {
        /* Prompt the user if necessary. */
        if( !children[i].is_auto_restart ) {
            printf("Press <enter> to run ==> %s", children[i].cmd);
            ignore_line();
        }
        start_child(&children[i]);
    }
}

static int
is_left_neighbor_running(int index)
{
    int rv = 0;

    /* Validate index. */
    if ((index < 0) || (index >= number_of_children)) {
        fprintf(stderr,
                "shunt: internal Error: "
                "close_child(): Unexpected index.\n");
        exit(1);
    }

    if (index == 0) {
        /* The first process has no left neighbor. */
        rv = 0;
    } else {
        rv = children[index - 1].is_running;
    }

    return rv;
}

/**
 * This function returns whether the pipe connected the child's stdin
 * is empty.
 *
 * This function cannot generally be called.  It can _only_ be called
 * after the left neighbor (i.e., the process that writes to the stdin
 * of child) has been permanently removed from the pipeline.
 *
 * There are two reasons for this restriction.  The first reason is
 * that we want to avoid a potential race whereby the pipe appears
 * empty when we check but, because the left neighbor is still alive,
 * data might still be added to the pipe later.
 *
 * The second reason is that in order to portably detect whether the
 * pipe is empty we need to read from the pipe (as discussed in the
 * comments inside the function).  In order for this function to
 * receive EOF when we read from the pipe, the pipe must have no
 * writers attached (or attaching), and this includes the left
 * neighbor.
 *
 * WARNING: Calling this function invalidates any cached values the
 * caller may have of child->fd_stdin because child->fd_stdin may be
 * connected to a different pipe on return.
 */
static int
is_fd_stdin_empty(cprocs_t* child)
{
    int read_loop_count = 0;
    const int XREALLOC_BLOCK_SIZE = 16384;
    ssize_t rcount = 0;
    ssize_t rcount_total = 0;
    ssize_t wcount_total = 0;
    char* buf = NULL;
    int p[2];

    /*
     * It would be nice to have a peek() or a putback() operation for
     * file descriptors, but we don't.  It would be nice if
     * ioctl(FIONREAD) _portably_ worked on _all_ types of files, but
     * it doesn't seem to.  So all we are really left with are read()
     * and write().
     *
     * Mercifully, because this function is only called when there are
     * no writers attached (or attaching), we can simulate a peek()
     * and a putback() by simply trying to reading data from the pipe.
     * If we get EOF immediately, we know the pipe was empty.  If we
     * are able to read at least one byte from the pipe, it means the
     * pipe was not empty.
     *
     * But what about the data we read out of the pipe?
     *
     * In order to simulate a putback(), instead of reading just one
     * byte, we read everything out of the pipe and then write it all
     * back to the pipe.  Given that "shunt" is the only possible
     * reader and writer at the time this function is called, this
     * kludge should work if coded properly.
     *
     * It should be noted that one of the main design goals of "shunt"
     * is to avoid the inefficiency of having "shunt" read data from
     * one child and write it to another.  In general, this goal is
     * achieved by the technique of suppressing EOF rather.  Here
     * though, because this chunk of code is not in the critical path
     * and because I think the portable nature of the code outweighs
     * the slight inefficiency introduced, I'm willing to make an
     * exception.
     */

    /*
     * Read data from the pipe.
     */

    for (;;) {

        /*
         * NOTE: This is not your typical read/write loop.  In a
         * typical read/write loop, you read one block and immediate
         * turn around and write one block.  Obviously we cannot do
         * that because we would never receive EOF.  Instead, we read
         * one block and save it.  Then we read the next block and
         * save it.  We continue doing this until the pipe is empty.
         * Only then do we turn around and write the blocks back to
         * the pipe.
         */

        /* Count how many times we have been through the loop.  This
         * is used to coordinate our input buffer with the call to
         * xread(). */
        ++read_loop_count;

        /* Increase the size of buf so that it is large enough to hold
         * everything we have already read plus what we are about to
         * read. */
        buf = xrealloc(buf, XREALLOC_BLOCK_SIZE * read_loop_count);

        /* Read in the new data. */
        rcount = xread(child->fd_stdin,
                       buf + rcount_total,
                       XREALLOC_BLOCK_SIZE);

        /* EOF received. */
        if (rcount == 0) {
            break;
        }

        /* Sanity check. */
        if (rcount < 0) {
            fprintf(stderr, "shunt: Internal Error: Unexpected rcount.\n");
            exit(1);
        }

        /* Keep track of the total number of bytes read. */
        rcount_total += rcount;

    }

    /* If the pipe was empty, skip the call to xwrite(). */
    if (rcount_total == 0) {
        goto out;
    }

    /*
     * Write data back to the pipe.
     */

    /* Writing the data back is a little tricky because you cannot
     * write to child->fd_stdin because it is the read-only end of the
     * pipe, and we have already closed the write-only end when the
     * left neighbor was disconnected.  So, we have to close this pipe
     * altogether, open a new one, attach it to "child", and then
     * write the data back. */
    xclose(child->fd_stdin);
    xpipe(p);
    child->fd_stdin = p[0];

    /* The actual act of writing data to the pipe is much easier than
     * reading it from the pipe because we already know the size of
     * the buffer we are writing and xwrite() impliclitly loops until
     * it is all written. */
    wcount_total = xwrite(p[1], buf, rcount_total);

    /* Close the write-only end of the pipe. */
    xclose(p[1]);

    /* Sanity check. */
    if (wcount_total != rcount_total) {
        fprintf(stderr, "shunt: Internal Error: Unexpected wcount_total.\n");
        exit(1);
    }

 out:

    if (buf) {
        free(buf);
    }

    return (rcount_total == 0);
}

static void
close_child(cprocs_t* child)
{
    /* Do not accidentally close our stdin. */
    if (child->fd_stdin != STDIN_FILENO) {
        xclose(child->fd_stdin);
    }
    /* Do not accidentally close our stdout. */
    if (child->fd_stdout != STDOUT_FILENO) {
        xclose(child->fd_stdout);
    }
    child->fd_stdin = -1;
    child->fd_stdout = -1;
    child->is_running = 0;
}

static void
wait_for_children()
{
    cprocs_t* child = NULL;
    pid_t pid = -1;
    int i = 0;
    int c = EOF;

    for( ;; ) {

        pid = waitpid(-1, NULL, WNOHANG);

        /* No more children to reap. */
        if( (pid == 0) || ((pid == -1) && (errno == ECHILD)) ) {
            break;
        }

        /* Error */
        if( pid == -1 ) {
            if( errno == EINTR ) {
                continue;
            }
            perror("waitpid");
            exit(1);
        }

        /*
         * Valid child was reaped.
         */

        /* Find its entry in "children". */
        child = NULL;
        for( i = 0 ; children[i].cmd ; ++i ) {
            if( pid == children[i].pid ) {
                child = &children[i];
                break;
            }
        }

        /* If you found an entry, deal with it. */
        if( child ) {

            /* If this is the child with the -c flag set, start the
             * chain reaction that will eventually shutdown the
             * pipeline. */
            if( child->is_auto_close ) {
                close_child(child);
                is_auto_closing = 1;
                continue;
            }

            /* When the chain reaction that will eventually shutdown
             * the pipeline causes the left neighbor of "child" to
             * close, the left neighbor may leave data in the pipe
             * that is connected to stdin of "child".  Thus, we want
             * to continue the chain reaction and disconnect "child"
             * only if the left neighbor has stopped running
             * (guaranteeing that it will no longer write to the pipe)
             * and only if "child" has read everything from the
             * pipe. */
            if(     is_auto_closing
                && !is_left_neighbor_running(i)
                &&  is_fd_stdin_empty(child) )
            {
                close_child(child);
                continue;
            }

            /* Auto restart. */
            if( child->is_auto_restart ) {
                start_child(child);
                continue;
            }

            /* Prompt the user for what to do. */
            printf("Command exited ==> %s\n", child->cmd);
            for( ;; ) {
                printf("  Restart or close pipe? [R/c] ");
                fflush(stdout);
                c = xgetchar();
                /* Clear the input buffer. */
                if( c != '\n' ) {
                    ignore_line();
                }
                /* Close Pipe */
                if( (c == 'C') || (c == 'c') ) {
                    close_child(child);
                    break;
                }
                /* Restart */
                if( (c == 'R') || (c == 'r') || (c == '\n') ) {
                    start_child(child);
                    break;
                }
                /* Invalid Response */
                puts("    Invalid response.");
            }

        }
        
    }

}

static void
handle_async_signals()
{
    int num = 0;
    /* Clear the pipe.  You do not want it to fill. */
    xread(async_to_sync_pipe[0], &num, sizeof(num));
    if( num == SIGCHLD ) {
        wait_for_children();
    } else {
        fprintf(stderr, "Internal Error: Unexpected signal.\n");
        exit(1);
    }
}

static void
install_signal_handlers(void)
{
    struct sigaction act;

    /* Install the SIGCHLD handler. */
    memset(&act, 0, sizeof(act));
    act.sa_handler = &reap;
    act.sa_flags = 0;
    xsigemptyset(&act.sa_mask);
    xsigaction(SIGCHLD, &act, NULL);

    /* Ignore SIGPIPE. */
    memset(&act, 0, sizeof(act));
    act.sa_handler = SIG_IGN;
    act.sa_flags = 0;
    xsigemptyset(&act.sa_mask);
    xsigaction(SIGPIPE, &act, NULL);
}

static void
set_up_pipeline_fds()
{
    /* Prime the pump by connecting prev_stdin to /dev/null. */
    int prev_stdin = fd_devnull;
    int p[2];
    int i = 0;

    /* Iterate over each of the child processes the user has requested. */
    for( i = 0 ; children[i].cmd ; ++i ) {

        /* Create communications pipe.  The pipe is to stay open in
         * the parent allowing there to always be a process attached
         * to both ends which keeps the pipe from closing when one or
         * more of the children exit. */
        children[i].fd_stdin = prev_stdin;

        /* For all child processes, except the last one, create a pipe
         * that will be used to pass the output of this child process
         * to the input of the next child process.  We save the
         * read-end of the pipe in prev_stdin where it will be
         * attached to the next process on the next iteration.  We
         * connect the write-end of the pipe immediately to this
         * process. */
        if( children[i+1].cmd ) {
            xpipe(p);
            children[i].fd_stdout = p[1];
            prev_stdin = p[0];
        } else {
            /* For the last child process, simply connect its stdout
             * to the tty. */
            children[i].fd_stdout = STDOUT_FILENO;
        }

    }
}

static int
is_child_running()
{
    int rv = 0;
    int i = 0;
    for( i = 0 ; children[i].cmd ; ++i ) {
        if( children[i].is_running ) {
            rv = 1;
            break;
        }
    }
    return rv;
}

int
main(int argc, char* argv[])
{
    int rv = 0;

    znum_initialize();

    shunt_all_starts = znum_new();

    progname = xbasename(argv[0]);

    decipher_command_line(argc, argv);

    xpipe(async_to_sync_pipe);

    install_signal_handlers();

    /* Open /dev/null. */
#ifdef _PATH_DEVNULL
    fd_devnull = xopen(_PATH_DEVNULL, O_RDWR, 0);
#else
    fd_devnull = xopen("/dev/null", O_RDWR, 0);
#endif

    set_up_pipeline_fds();

    start_all_children();

    /* Prepare for main event loop. */
    {
        /* max fd + 1 for select(). */
        int n = max_int(async_to_sync_pipe[0], async_to_sync_pipe[1]) + 1;

        /* fd sets for select(). */
        fd_set rset;
        fd_set wset;
        fd_set eset;

        /* Enter main event loop. */
        for( ;; ) {

            /* Reset the fd sets. */
            FD_ZERO(&rset);
            FD_ZERO(&wset);
            FD_ZERO(&eset);
            FD_SET(async_to_sync_pipe[0], &rset);

            /* Wait for next event. */
            xselect(n, &rset, &wset, &eset, NULL);

            /* Process event. */
            if( FD_ISSET(async_to_sync_pipe[0], &rset) ) {
                handle_async_signals();
            }

            /* When there are no more children left, it is time to
             * exit. */
            if( !is_child_running() ) {
                break;
            }

        }
    }

    /* Free the znum's. */
    {
        unsigned long int i = 0;
        for( i = 0 ; i < size_of_children ; ++i ) {
            znum_delete(children[i].shunt_starts);
        }
        znum_delete(shunt_all_starts);
    }

    znum_finalize();

    /* Free the children. */
    if (children) {
        free(children);
    }

    return rv;
}
