#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "../xposix/xposix.h"
#include "drs.h"

static const char* progname;

/* Command-Line Inputs */
static unsigned long int block_size = 2048;
static unsigned long int file_block_count = 0;
static unsigned long int media_block_count = 0;
static unsigned long int pad_block_count = 150;
static int verbose = 0;

/* Used Internally */
static unsigned long int curr_block = 0;
static unsigned long int root_dir_extent = 0;
static drs_t drs = NULL;

static const char* const usage_msg = "\
\n\
Usage: %s <flags>\n\
\n\
    Flags:\n\
\n\
                             -h : Print help.\n\
         fbc=<file_block_count> : Set the block count of each file.\n\
        mbc=<media_block_count> : Set the total block count of the media.\n\
          pbc=<pad_block_count> : Set the number of blocks to pad at the\n\
                                  end of the disk.  Defaults to 150.\n\
                             -v : Be verbose.\n\
\n\
    Purpose:\n\
\n\
        The ISO 9660 file system is a 32-bit file system.  Thus, files are\n\
        limited to a maximum size of 4GB.  Current storage media are well\n\
        beyond the 4GB limit.  Thus, if you want to use all the space on\n\
        your media, you will need to store data in more than one file.\n\
        To that end, this program will make a proper ISO 9660 file system\n\
        from data you stream on standard input by creating multiple files\n\
        as needed so that you can use the full capacity of your media.\n\
\n\
        Each block is 2048 bytes.\n\
\n\
    About:\n\
\n\
        Version: 1.7.2\n\
        License: BSD\n\
         Author: Paul Serice\n\
         E-Mail: paul@serice.net\n\
            URL: http://www.serice.net/shunt/\n\
\n\
";

static void
usage(FILE* f)
{
    fprintf(f, usage_msg, progname);
}

/* Convert the string in "s" to an unsigned long storing the converted
 * value in i.  Return 1 on success, and 0 on failure. */
static int
string_to_unsigned_long(const char* s, unsigned long int* i)
{
    int rv = 0;
    unsigned long int tmp = 0;
    char* first_bad_char = NULL;
    const char* r = NULL;  /* Remainder of the string. */

    /* You have to manually check for a leading minus sign or an empty
     * string. */
    for( ; *s != '\0' ; ++s ) {
        if( !isspace((int)*s) ) {
            break;
        }
    }
    if( (*s == '-') || (*s == '\0') ) {
        goto stul_exit;
    }

    /* Do the conversion. */
    errno = 0;
    tmp = strtoul(s, &first_bad_char, 0 /* base */);

    /* Check for errors. */
    if( errno ) {
        goto stul_exit;
    }
    /* Allow trailing space. */
    r = first_bad_char;
    for( ; *r != '\0' ; ++r ) {
        if( !isspace((int)*r) ) {
            break;
        }
    }
    if( *r != '\0' ) {
        goto stul_exit;
    }           

    /* Success. */
    *i = tmp;
    rv = 1;

 stul_exit:
    return rv;
}

/* Check to see if block_size * file_block_count > 2^32 - 1.  This
 * gets a little weird because we have to worry about overflow.  It is
 * simple to understand once you realize that block_size is limited to
 * be exactly a power of 2.  This allows us to multiply by shifting.
 * So, we just check before each shift to make sure we are not losing
 * a significant bit. */
static int
is_file_too_long(unsigned long int block_size,
                 unsigned long int file_block_count)
{
    int rv = 0;
    for( block_size >>= 1 ; block_size > 0 ; block_size >>= 1 ) {
        /* If we are about to shift out a significant bit, then
         * overflow would occur during the multiplication. */
        if( (file_block_count & 0x80000000UL) != 0UL ) {
            rv = 1;
            break;
        }
        file_block_count <<= 1;
    }
    return rv;
}

static void
decipher_command_line(int argc, char** argv)
{
    int i = 1;
    for( i = 1 ; i < argc ; ++i ) {
        const char* arg = argv[i];

#if 0
        /* Block Size */
        if( strncmp(arg, "bs=", 3) == 0 ) {
            if( !string_to_unsigned_long(arg + 3, &block_size) ) {
                fprintf(stderr, "\n");
                fprintf(stderr,
                        "Error: Number invalid or out of range: \"%s\"\n",
                        arg + 3);
                fprintf(stderr, "\n");
                exit(1);                
            }
            /* The block size is strictly limited. */
            if(    (block_size != 2048)
                && (block_size != 4096)
                && (block_size != 8192)
                && (block_size != 16384)
                && (block_size != 32768) )
            {
                fprintf(stderr, "\n");
                fprintf(stderr, "Error: The block size is limited to "
                                "the following values:\n");
                fprintf(stderr, "\n");
                fprintf(stderr, "    2048, 4096, 8192, 16348, 32768\n");
                fprintf(stderr, "\n");
                exit(1);                
            }
            continue;
        }
#endif

        /* File Block Count */
        if( strncmp(arg, "fbc=", 4) == 0 ) {
            if( !string_to_unsigned_long(arg + 4, &file_block_count) ) {
                fprintf(stderr, "\n");
                fprintf(stderr,
                        "Error: Number invalid or out of range: \"%s\"\n",
                        arg + 4);
                fprintf(stderr, "\n");
                exit(1);                
            }
            /* We check below to make sure the file_block_count and the
             * block_size result in a file that has a length that can
             * be represented in 32-bits. */
            continue;
        }

        /* Help */
        if( strcmp(arg, "-h") == 0 ) {
            usage(stdout);
            exit(0);
        }

        /* Media Block Count */
        if( strncmp(arg, "mbc=", 4) == 0 ) {
            if( !string_to_unsigned_long(arg + 4, &media_block_count) ) {
                fprintf(stderr, "\n");
                fprintf(stderr,
                        "Error: Number invalid or out of range: \"%s\"\n",
                        arg + 4);
                fprintf(stderr, "\n");
                exit(1);                
            }
            continue;
        }

        /* Pad Block Count */
        if( strncmp(arg, "pbc=", 4) == 0 ) {
            if( !string_to_unsigned_long(arg + 4, &pad_block_count) ) {
                fprintf(stderr, "\n");
                fprintf(stderr,
                        "Error: Number invalid or out of range: \"%s\"\n",
                        arg + 4);
                fprintf(stderr, "\n");
                exit(1);                
            }
            continue;
        }

        /* Verbose */
        if( strcmp(arg, "-v") == 0 ) {
            verbose = 1;
            continue;
        }

        /* Default */
        fprintf(stderr, "\n");
        fprintf(stderr, "Error: Invalid option: \"%s\"\n", arg);
        usage(stderr);
        exit(1);
    }

    /* For a minimum block size of 512 bytes, make sure that your
     * media is at least 8MB.  This allows the code below to make
     * assumptions that simplify some calculations. */
    if( media_block_count < 16384 ) {
        fprintf(stderr, "\n");
        fprintf(stderr, "Error: mbc must be >= 16384.\n");
        fprintf(stderr, "\n");
        exit(1);
    }

    /* Same with the file_block_count. */
    if( file_block_count < 16384 ) {
        fprintf(stderr, "\n");
        fprintf(stderr, "Error: fbc must be >= 16384.\n");
        fprintf(stderr, "\n");
        exit(1);
    }

    /* Verify that the file_block_count and the block_size result in a
     * file that has a length that can be represented in 32-bits. */
    if( is_file_too_long(block_size, file_block_count) ) {
        fprintf(stderr, "\n");
        fprintf(stderr,
                "Error: Your block size (%lu) and your file_block_count (%lu) "
                "result\nin a file with a maximum length greater than "
                "32-bits.\n",
                block_size,
                file_block_count);
        fprintf(stderr, "\n");
        exit(1);
    }

}

/* By my calculation, the length of the file name (including
 * separators and the version number) can be no longer than 37
 * characters.  The rest of the record is 33 bytes.  Thus the total
 * size of one directory record is 70 bytes.
 *
 * Section 6.8.1.1 of the ISO 9660 standard says, "Each Directory
 * Record shall end in the Logical Sector in which it begins."  So, we
 * have to avoid spanning across blocks.  Thus, the integer division
 * is intentional. */
static unsigned long int
get_min_drs_entries_per_block(void)
{
    return block_size / 70;
}

static unsigned long int
get_files_needed(unsigned long int dir_entries_count)
{
    unsigned long int rv = 0;
    unsigned long int free_blocks = 0;
    unsigned long int blocks_needed = 0;
    unsigned long int min_entries_per_block = 0;

    min_entries_per_block = get_min_drs_entries_per_block();

    blocks_needed = dir_entries_count / min_entries_per_block;
    if( dir_entries_count % min_entries_per_block ) {
        /* There was at least one directory record remaining. */
        ++blocks_needed;
    }

    /* Calculate the number of blocks available for data files. */
    free_blocks = media_block_count
                  - pad_block_count
                  - 20 /* Blocks needed for the rest of the ISO filesystem. */
                  - blocks_needed;

    rv = free_blocks / file_block_count;
    if( free_blocks % file_block_count != 0 ) {
        /* We need another file to fill the remainder. */
        ++rv;
    }

    return rv;
}

static unsigned long int
get_max_files_needed(void)
{
    unsigned long int rv = 0;

    /* For our first estimate, carve out enough space to hold the
     * maximum number of files we allow.  "rv" will then be the number
     * of files of "file_block_count" length that are required to
     * completely fill the remaining area. */
    rv = get_files_needed(drs_get_directory_entries_max());

    /* Get a better estimate (which is usually right) based on the
     * rough estimate by reclaiming the space not used by the root
     * directory.  We do this because the estimate above carves out
     * over 4MB for the root directory in order to hold
     * drs_get_directory_entries_max() entries.  As a practical matter
     * we usually have substantially fewer directory entries.
     *
     * We add 3 to max_files_needed to make sure there is enough room
     * in the root dir if we detect the need for another file to fill
     * the remainder plus there must be enough room for the "." and
     * ".."  directories. */
    rv = get_files_needed(rv + 3);

    return rv;
}

/* Write the 32-bit unsigned integer "i" to "buf" (which must be
 * exactly 4 bytes long) as a big endian integer. */
static void
as_big_endian_32(unsigned char* buf, unsigned long int i)
{
    buf[0] = (unsigned char)((i & 0xff000000UL) >> 24);
    buf[1] = (unsigned char)((i & 0x00ff0000UL) >> 16);
    buf[2] = (unsigned char)((i & 0x0000ff00UL) >> 8);
    buf[3] = (unsigned char)(i & 0x000000ffUL);
}

/* Write the 32-bit unsigned integer "i" to "buf" (which must be
 * exactly 4 bytes long) as a little endian integer. */
static void
as_little_endian_32(unsigned char* buf, unsigned long int i)
{
    buf[0] = (unsigned char)(i & 0x000000ffUL);
    buf[1] = (unsigned char)((i & 0x0000ff00UL) >> 8);
    buf[2] = (unsigned char)((i & 0x00ff0000UL) >> 16);
    buf[3] = (unsigned char)((i & 0xff000000UL) >> 24);
}

/* Write the 16-bit unsigned integer "i" to "buf" (which must be
 * exactly 2 bytes long) as a big endian integer. */
static void
as_big_endian_16(unsigned char* buf, unsigned long int i)
{
    buf[0] = (unsigned char)((i & 0x0000ff00UL) >> 8);
    buf[1] = (unsigned char)(i & 0x000000ffUL);
}

/* Write the 16-bit unsigned integer "i" to "buf" (which must be
 * exactly 2 bytes long) as a little endian integer. */
static void
as_little_endian_16(unsigned char* buf, unsigned long int i)
{
    buf[0] = (unsigned char)(i & 0x000000ffUL);
    buf[1] = (unsigned char)((i & 0x0000ff00UL) >> 8);
}

/* Write the current directory entry of "drs" into the space provided
 * by "buf".  This function makes no attempt to verify that buf is
 * large enough to hold the entire directory entry. */
static void
write_current_directory(unsigned char* buf)
{
    /* Length of Directory Record */
    buf[0] = drs_get_length_of_directory_record(drs);

    /* Extended Attribute Record Length*/
    buf[1] = 0;

    /* Location of Extent */
    as_little_endian_32(&buf[2], drs_get_location_of_extent(drs));
    as_big_endian_32(&buf[6], drs_get_location_of_extent(drs));

    /* Data Length */
    as_little_endian_32(&buf[10], drs_get_data_length(drs));
    as_big_endian_32(&buf[14], drs_get_data_length(drs));
    
    /* Recording Date and Time */
    buf[18] = drs_get_years_since_1900(drs);
    buf[19] = drs_get_month(drs);
    buf[20] = drs_get_day(drs);
    buf[21] = drs_get_hour(drs);
    buf[22] = drs_get_minute(drs);
    buf[23] = drs_get_second(drs);
    buf[24] = drs_get_offset_from_GMT(drs);

    /* File Flags */
    buf[25] = drs_get_file_flags(drs);

    /* File Unit Size */
    buf[26] = drs_get_file_unit_size(drs);

    /* Interleave Gap Size */
    buf[27] = drs_get_interleave_gap_size(drs);

    /* Volume Sequence Number */
    as_little_endian_16(&buf[28], 1);
    as_big_endian_16(&buf[30], 1);

    /* Length of File Identifier */
    buf[32] = drs_get_length_of_file_identifier(drs);

    /* File Identifier */
    memcpy(&buf[33],
           drs_get_file_identifier(drs),
           drs_get_length_of_file_identifier(drs));

    /* Padding Field */
    {
        int i = 33 + drs_get_length_of_file_identifier(drs);
        int i_max = i + drs_get_length_of_padding_field(drs);
        for( ; i < i_max ; ++i ) {
            buf[i] = 0;
        }
    }
}

static void
write_root_directory_entry(unsigned char* buf)
{
    size_t curr = drs_get_current(drs);
    if( !drs_set_current(drs, 1) ) {
        fprintf(stderr, "***\n");
        fprintf(stderr, "*** Internal Error: drs not initialized.\n");
        fprintf(stderr, "***\n");
        exit(1);
    }
    write_current_directory(buf);
    if( !drs_set_current(drs, curr) ) {
        fprintf(stderr, "***\n");
        fprintf(stderr, "*** Internal Error: Unable to reset drs.\n");
        fprintf(stderr, "***\n");
        exit(1);
    }
}

static void
write_system_area(void)
{
    int i = 0;
    char* buf = xcalloc(1, block_size);
    if( verbose ) {
        fprintf(stderr, "%s: write_system_area(): curr_block = %lu\n",
                progname, curr_block);
    }
    for( i = 0 ; i < 16 ; ++i ) {
        xwrite(STDOUT_FILENO, buf, block_size);
        ++curr_block;
    }
    free(buf);
}

static void
write_primary_volume_descriptor(void)
{
    int i = 0;
    unsigned char* buf = xcalloc(1, block_size);

    if( verbose ) {
        fprintf(stderr, "%s: write_primary_volume_descriptor(): "
                "curr_block = %lu\n", progname, curr_block);
    }

    /* Volume Descriptor Type */
    buf[0] = 1;

    /* Standard Identifier */
    buf[1] = 'C';
    buf[2] = 'D';
    buf[3] = '0';
    buf[4] = '0';
    buf[5] = '1';

    /* Volume Descriptor Version */
    buf[6] = 1;

    /* Unused */
    buf[7] = 0;

    /* System Identifier */
    buf[8]  = 'N';
    buf[9]  = 'O';
    buf[10] = 'N';
    buf[11] = 'E';
    for( i = 12 ; i < 40 ; ++i ) {
        buf[i] = ' ';
    }

    /* Volume Identifier */
    buf[40] = 'F';
    buf[41] = 'L';
    buf[42] = 'Y';
    for( i = 43 ; i < 72 ; ++i ) {
        buf[i] = ' ';
    }

    /* Unused */
    for( i = 72 ; i < 80 ; ++i ) {
        buf[i] = 0;
    }

    /* Volume Space Size */
    as_little_endian_32(&buf[80], media_block_count);
    as_big_endian_32(&buf[84], media_block_count);

    /* Unused */
    for( i = 88 ; i < 120 ; ++i ) {
        buf[i] = 0;
    }

    /* Volume Set Size */
    as_little_endian_16(&buf[120], 1);
    as_big_endian_16(&buf[122], 1);

    /* Volume Sequence Number */
    as_little_endian_16(&buf[124], 1);
    as_big_endian_16(&buf[126], 1);

    /* Logical Block Size */
    as_little_endian_16(&buf[128], block_size);
    as_big_endian_16(&buf[130], block_size);

    /* Path Table Size */
    as_little_endian_32(&buf[132], 10);
    as_big_endian_32(&buf[136], 10);

    /* Location of Occurrence of Type L Path Table */
    as_little_endian_32(&buf[140], 18);

    /* Location of Optional Occurrence of Type L Path Table */
    as_little_endian_32(&buf[144], 0);

    /* Location of Occurrence of Type M Path Table */
    as_big_endian_32(&buf[148], 19);

    /* Location of Optional Occurrence of Type M Path Table */
    as_big_endian_32(&buf[152], 0);

    /* Directory Record of Root Directory */
    write_root_directory_entry(&buf[156]);

    /* Volume Set Identifier */
    for( i = 190 ; i < 318 ; ++i ) {
        buf[i] = ' ';
    }

    /* Publisher Identifier */
    for( i = 318 ; i < 446 ; ++i ) {
        buf[i] = ' ';
    }

    /* Data Preparer Identifier */
    for( i = 446 ; i < 574 ; ++i ) {
        buf[i] = ' ';
    }

    /* Application Identifier */
    buf[574] = 'F';
    buf[575] = 'L';
    buf[576] = 'Y';
    buf[577] = 'I';
    buf[578] = 'S';
    buf[579] = 'O';
    for( i = 580 ; i < 702 ; ++i ) {
        buf[i] = ' ';
    }

    /* Copyright File Identifier */
    for( i = 702 ; i < 739 ; ++i ) {
        buf[i] = ' ';
    }

    /* Abstract File Identifier */
    for( i = 739 ; i < 776 ; ++i ) {
        buf[i] = ' ';
    }

    /* Bibliographic File Identifier */
    for( i = 776 ; i < 813 ; ++i ) {
        buf[i] = ' ';
    }

    /* Volume Creation Date */
    for( i = 813 ; i < 830 ; ++i ) {
        buf[i] = 0;
    }   

    /* Volume Modification Date and Time */
    for( i = 830 ; i < 847 ; ++i ) {
        buf[i] = 0;
    }   

    /* Volume Expiration Date and Time */
    for( i = 847 ; i < 864 ; ++i ) {
        buf[i] = 0;
    }   

    /* Volume Effective Date and Time */
    for( i = 864 ; i < 881 ; ++i ) {
        buf[i] = 0;
    }   

    /* File Structure Version */
    buf[881] = 1;

    /* Reserved */
    buf[882] = 0;

    /* Application Use */
    for( i = 883 ; i < 1395 ; ++i ) {
        buf[i] = ' ';
    }

    /* Write Data */
    xwrite(STDOUT_FILENO, buf, block_size);

    ++curr_block;
    free(buf);
}

static void
write_volume_descriptor_set_terminator(void)
{
    unsigned char* buf = xcalloc(1, block_size);

    if( verbose ) {
        fprintf(stderr, "%s: write_volume_descriptor_set_terminator(): "
                "curr_block = %lu\n", progname, curr_block);
    }

    /* Volume Descriptor Type */
    buf[0] = 255;

    /* Standard Identifier */
    buf[1] = 'C';
    buf[2] = 'D';
    buf[3] = '0';
    buf[4] = '0';
    buf[5] = '1';

    /* Volume Descriptor Version */
    buf[6] = 1;

    /* Write Data */
    xwrite(STDOUT_FILENO, buf, block_size);

    ++curr_block;
    free(buf);
}

static void
write_l_path_table(void)
{
    unsigned char* buf = xcalloc(1, block_size);

    if( verbose ) {
        fprintf(stderr, "%s: write_l_path_table(): "
                "curr_block = %lu\n", progname, curr_block);
    }

    /* Length of Directory Identifier */
    buf[0] = 1;

    /* Extended Attribute Record Length */
    buf[1] = 0;

    /* Location of Extent */
    as_little_endian_32(&buf[2], root_dir_extent);

    /* Parent Directory Number */
    as_little_endian_16(&buf[6], 1);

    /* Directory Identifier */
    buf[8] = 0;

    /* Padding Field */
    buf[9] = 0;

    /* Write Data */
    xwrite(STDOUT_FILENO, buf, block_size);

    ++curr_block;
    free(buf);
}

static void
write_m_path_table(void)
{
    unsigned char* buf = xcalloc(1, block_size);

    if( verbose ) {
        fprintf(stderr, "%s: write_m_path_table(): "
                "curr_block = %lu\n", progname, curr_block);
    }

    /* Length of Directory Identifier */
    buf[0] = 1;

    /* Extended Attribute Record Length */
    buf[1] = 0;

    /* Location of Extent */
    as_big_endian_32(&buf[2], root_dir_extent);

    /* Parent Directory Number */
    as_big_endian_16(&buf[6], 1);

    /* Directory Identifier */
    buf[8] = 0;

    /* Padding Field */
    buf[9] = 0;

    /* Write Data */
    xwrite(STDOUT_FILENO, buf, block_size);

    ++curr_block;
    free(buf);
}

/* Write the next file starting at the next block.  Return the number
 * of bytes written. */
static unsigned long int
write_next_file(void)
{
    size_t rcount = 0;
    size_t rcount_total = 0;
    unsigned long int blocks = 0;
    unsigned long int blocks_max = 0;
    unsigned char* buf = NULL;

    buf = xmalloc(block_size);

    if( verbose ) {
        fprintf(stderr, "%s: write_next_file(): "
                "curr_block = %lu\n", progname, curr_block);
    }

    /* Do no write beyond the end of media. */
    if( curr_block + file_block_count <= root_dir_extent ) {
        blocks_max = file_block_count;        
    } else {
        blocks_max = root_dir_extent - curr_block;
    }

    for( blocks = 0 ; blocks < blocks_max ; ++blocks ) {
        /* Read a block.  Do NOT use fread() because it buffers data
         * which has a way of corrupting a pipeline when you use
         * "shunt" to repeatedly restart a program because there is no
         * way to push back an unused block. */
        rcount = xread(STDIN_FILENO, buf, block_size);
        if( rcount == 0 ) {
            break;
        }
        rcount_total += rcount;

        /* Write a block. */
        xwrite(STDOUT_FILENO, buf, rcount);

        /* If xread() returned less than a block.  Pad.  You'll exit
         * the loop on the next cycle when xread returns 0.  It is
         * important to wait until the next iteration so that "blocks"
         * will be incremented. */
        if( rcount < block_size ) {
            unsigned long int remainder = block_size - rcount;
            memset(buf, 0, remainder);
            xwrite(STDOUT_FILENO, buf, remainder);
        }
    }

    /* Update the current block. */
    curr_block += blocks;

    free(buf);

    if( verbose ) {
        fprintf(stderr, "%s: write_next_file(): "
                "rcount_total = %zu\n", progname, rcount_total);
    }

    return rcount_total;
}

static void
write_root_directory(void)
{
    size_t dir_len = 0;
    size_t dir_len_block_total = 0;
    unsigned long int i = 0;
    unsigned char* buf = NULL;

    if( verbose ) {
        fprintf(stderr, "%s: write_root_directory(): "
                "curr_block = %lu\n", progname, curr_block);
    }

    buf = xcalloc(1, block_size);

    for( i = 0 ; drs_set_current(drs, i) ; ++i ) {
        /* Section 6.8.1.1 of the ISO 9660 standard says, "Each
         * Directory Record shall end in the Logical Sector in which
         * it begins."  So, we have to avoid spanning across
         * blocks. */
        dir_len = drs_get_length_of_directory_record(drs);
        if( dir_len_block_total + dir_len > block_size ) {
            /* This entry would span across blocks.  So pad until the
             * end of the block.  We've already done calculations
             * above to make sure we have carved out enough blocks for
             * the entire list of directory records including this
             * spanning requirement. */
            size_t remainder_len = (size_t)(block_size - dir_len_block_total);
            memset(buf, 0, remainder_len);
            xwrite(STDOUT_FILENO, buf, remainder_len);
            dir_len_block_total = 0;
            ++curr_block;
        }
        write_current_directory(buf);
        xwrite(STDOUT_FILENO, buf, dir_len);
        dir_len_block_total += dir_len;
        if( dir_len_block_total == block_size ) {
            ++curr_block;
            dir_len_block_total = 0;
        }
    }

    /* Pad the rest of this block. */
    if( dir_len_block_total < block_size ) {
        unsigned long int remainder = block_size - dir_len_block_total;
        /* Clear the block we have been using. */
        memset(buf, 0, remainder);
        /* Write the remainder. */
        xwrite(STDOUT_FILENO, buf, remainder);
        ++curr_block;
    }

    free(buf);
}

static void
write_padding(void)
{
    unsigned long int i = 0;
    unsigned char* buf = xcalloc(1, block_size);

    if( verbose ) {
        fprintf(stderr, "%s: write_padding(): "
                "curr_block = %lu\n", progname, curr_block);
    }

    for( i = 0 ; i < pad_block_count ; ++i ) {
        xwrite(STDOUT_FILENO, buf, block_size);
        ++curr_block;
    }
    free(buf);
}

int main(int argc, char* argv[])
{
    int rv = 0;
    unsigned long int i = 0;
    unsigned long int max_files_needed = 0;
    unsigned long int blocks_needed_for_root_dir = 0;

    progname = xbasename(argv[0]);

    decipher_command_line(argc, argv);

    if( verbose ) {
        fprintf(stderr, "%s: block_size = %lu\n",
                progname, block_size);
        fprintf(stderr, "%s: file_block_count = %lu\n",
                progname, file_block_count);
        fprintf(stderr, "%s: media_block_count = %lu\n",
                progname, media_block_count);
        fprintf(stderr, "%s: pad_block_count = %lu\n",
                progname, pad_block_count);
        fprintf(stderr, "%s: verbose = %d\n",
                progname, verbose);
    }

    /* Make sure stdin and stdout are open as binary streams. */
    x_setmode(STDIN_FILENO, O_BINARY);
    x_setmode(STDOUT_FILENO, O_BINARY);

    /* Figure out how many files we need to fill the entire medium. */
    max_files_needed = get_max_files_needed();
    if( (max_files_needed + 2) > drs_get_directory_entries_max() ) {
        fprintf(stderr, "***\n");
        fprintf(stderr, "*** Error: The values you have specified "
                        "require too many files.\n");
        fprintf(stderr, "***\n");
        exit(1);
    }

    /* Determine an upper bound on the length (in blocks) of the root
     * directory.  At this point, I don't see any reason to be more
     * precise.  */
    blocks_needed_for_root_dir =
        max_files_needed / get_min_drs_entries_per_block();
    if( max_files_needed % get_min_drs_entries_per_block() ) {
        /* There was at least one directory record remaining. */
        ++blocks_needed_for_root_dir;
    }

    /* Find the block where the root directory will reside. */
    root_dir_extent =
        media_block_count - pad_block_count - blocks_needed_for_root_dir;
    if( verbose ) {
        fprintf(stderr, "%s: max_files_needed = %lu\n",
                progname, max_files_needed);
        fprintf(stderr, "%s: root_dir_extent = %lu\n",
                progname, root_dir_extent);
    }

    /* Create a new directory records instance and fill it with
     * files. */
    drs = drs_new();

    /* Add the "this" directory. */
    drs_push_back(drs);
    drs_set_location_of_extent(drs, root_dir_extent);
    drs_set_data_length(drs, blocks_needed_for_root_dir * block_size);
    drs_set_file_flags(drs, 2);
    drs_set_volume_sequence_number(drs, 1);
    drs_set_file_identifier(drs, ".");

    /* Add the "parent" directory. */
    drs_push_back(drs);
    drs_set_location_of_extent(drs, root_dir_extent);
    drs_set_data_length(drs, blocks_needed_for_root_dir * block_size);
    drs_set_file_flags(drs, 2);
    drs_set_volume_sequence_number(drs, 1);
    drs_set_file_identifier(drs, "..");

    /* Write the basic blocks that comprise the ISO 9660 file system.
     * This must follow the creation of the root directory in the
     * "drs" container because we need to be able to write the root
     * directory to the primary volume descriptor. */
    write_system_area();
    write_primary_volume_descriptor();
    write_volume_descriptor_set_terminator();
    write_l_path_table();
    write_m_path_table();

    /* Add the files. */
    for( i = 1 ; i <= max_files_needed ; ++i ) {
        unsigned long int location_of_extent = curr_block;
        unsigned long int data_length = 0;
        char file_id[128];
        data_length = write_next_file();
        if( data_length == 0 ) {
            break;
        }
        if( verbose ) {
            fprintf(stderr, "%s: Adding file %lu\n", progname, i);
        }
        sprintf(file_id, "%05lu.FLY", i);
        drs_push_back(drs);
        drs_set_location_of_extent(drs, location_of_extent);
        drs_set_data_length(drs, data_length);
        drs_set_file_flags(drs, 0);
        drs_set_volume_sequence_number(drs, 1);
        drs_set_file_identifier(drs, file_id);
    }

    /* Pad with zeros until you reach the extent used for the root
     * directory. */
    if( curr_block < root_dir_extent ) {
        unsigned char* buf = xcalloc(1, block_size);
        for( ; curr_block < root_dir_extent ; ++curr_block ) {
            xwrite(STDOUT_FILENO, buf, block_size);
        }
        free(buf);
    }

    /* Write the final ISO 9660 blocks. */
    write_root_directory();
    write_padding();

    drs_delete(drs);

    return rv;
}
