/*
 *  xfce-icontheme - a themed icon lookup class
 *
 *  Copyright (c) 2004 Brian Tarricone <bjt23@cornell.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <glib.h>
#include <gdk/gdk.h>
#include <gtk/gtksettings.h>

#include <libxfce4util/libxfce4util.h>
#include <libxfcegui4/libxfcegui4.h>

#include "xfce-icontheme.h"

#define DEFAULT_THEME         "Rodent"
#define SIZE_THRESHOLD_BELOW  20  /* tweak this if we get ugly upscaled icons */
#define FALLBACK_INDEX        DATADIR "/xfce4/hicolor-index.theme"

typedef struct
{
	guint refcnt;
	
	GdkScreen *gscreen;
	
	gboolean use_svg;
	GHashTable *icons;
	
	GList *search_paths;
	
	GList *custom_categories;
	gint next_custom_id;
	
	GList *dirs_watch;
	GList *dirs_mtimes;
	
	GHashTable *hash_for_stupid_themes;
} XfceIconThemeSingleton;


typedef struct
{
	XfceIconThemeCategory id;
	gchar **icons;	
} XfceIconThemeCategoryList;

typedef struct XfceIcon
{
	gboolean is_in_theme;
	gchar *path;
	gint size;
} XfceIcon;

struct _XfceIconThemePriv
{
	GdkScreen *gscreen;
	XfceIconThemeSingleton *singleton;
};

enum {
	XFCEIC_SIG_CHANGED,
	XFCEIC_N_SIGNALS
};

static void xfce_icon_theme_class_init(XfceIconThemeClass *klass);
static void xfce_icon_theme_init(XfceIconTheme *icon_theme);
static void xfce_icon_theme_finalize(GObject *object);

static guint icon_theme_signals[XFCEIC_N_SIGNALS] = { 0 };
static GObjectClass *parent_class = NULL;
static gboolean _gdk_pixbuf_has_svg = FALSE;

static const gchar *builtin_icon_categories[][XFCE_N_BUILTIN_ICON_CATEGORIES] =
{
	{ "xfce-unknown", "gnome-fs-executable", "exec", "emblem-generic", NULL, NULL },
	{ "xfce-edit", "gedit-icon", "edit", NULL, NULL, NULL },
	{ "xfce-filemanager", "file-manager", "folder", NULL, NULL, NULL },
	{ "xfce-utils", "gnome-util", "utilities", NULL, NULL, NULL },
	{ "xfce-games", "gnome-joystick", "games", "emblem-fun", NULL, NULL },
	{ "xfce-man", "gnome-help", "help", NULL, NULL, NULL },
	{ "xfce-multimedia", "gnome-multimedia", "multimedia", "emblem-multimedia", NULL, NULL },
	{ "xfce-internet", "gnome-globe", "gnome-fs-network", "web-browser", "emblem-web", NULL },
	{ "xfce-graphics", "gnome-graphics", "graphics", NULL, NULL, NULL },
	{ "xfce-printer", "gnome-dev-printer", "printer", NULL, NULL, NULL },
	{ "xfce-schedule", "gnome-month", "productivity", "emblem-documents", NULL, NULL },
	{ "xfce-sound", "gnome-audio", "sound", NULL, NULL, NULL },
	{ "xfce-terminal", "gnome-terminal", "terminal", NULL, NULL, NULL },
	{ "xfce-devel", "gnome-devel", NULL, NULL, NULL },
	{ "xfce-settings", "gnome-settings", NULL, NULL, NULL },
	{ "xfce-system", "gnome-system", NULL, NULL, NULL },
	{ "xfce-wine", "wine", NULL, NULL, NULL }
};

/* list in order of preference */
static gchar *fallback_themes[] =
{
	"gnome", "default.kde", "hicolor", NULL
};

/* list in order of preference */
static const gchar *icon_ext_without_svg[] =
{
	".png", ".xpm", NULL
};
static const gchar *icon_ext_with_svg[] =
{
	".svg", ".svgz", ".png", ".xpm", NULL
};

static void
xfce_icon_theme_category_list_free(XfceIconThemeCategoryList *catlist)
{
	if(catlist->icons)
		g_strfreev(catlist->icons);
	g_free(catlist);
}

static void
icon_list_free(GList *icon_list)
{
	GList *l;
	
	for(l = icon_list; l; l = l->next) {
		XfceIcon *icon = l->data;
		g_free(icon->path);
		g_free(icon);
	}
	g_list_free(icon_list);
}

static void
icon_lists_free_ht(gpointer key, gpointer value, gpointer user_data)
{
	icon_list_free((GList *)value);
}

static gchar *
theme_index_extract_name(const gchar *index_file)
{
	FILE *fp;
	gchar buf[4096];
	
	fp = fopen(index_file, "r");
	if(!fp)
		return NULL;
	
	while(fgets(buf, 4096, fp)) {
		if(strncmp(buf, "Name=", 5))
			continue;
		
		if(buf[strlen(buf)-1] == '\n')
			buf[strlen(buf)-1] = 0;
		if(buf[strlen(buf)-1] == '\r')
			buf[strlen(buf)-1] = 0;
		
		fclose(fp);
		return g_strdup(buf+5);
	}
	
	fclose(fp);
	
	return NULL;
}

static gchar *
theme_name_to_dir(const gchar *base_path, const gchar *theme_name)
{
	GDir *dir;
	gchar fullpath[PATH_MAX], *idx_theme_name;
	
	dir = g_dir_open(base_path, 0, NULL);
	if(dir) {
		const gchar *file;
		while((file = g_dir_read_name(dir))) {
			g_snprintf(fullpath, PATH_MAX, "%s/%s/", base_path, file);
			if(!g_file_test(fullpath, G_FILE_TEST_IS_DIR))
				continue;
			
			g_strlcat(fullpath, "index.theme", PATH_MAX);
			idx_theme_name = theme_index_extract_name(fullpath);
			if(!idx_theme_name)
				continue;
			
			if(!strcmp(theme_name, idx_theme_name)) {
				g_free(idx_theme_name);
				return g_strdup(file);
			} else
				g_free(idx_theme_name);
		}	
	}
	
	return NULL;
}

static void
theme_add_all(XfceIconThemeSingleton *singleton, const gchar *base_path,
		const gchar *theme_name, gboolean is_in_theme)
{
	XfceRc *themerc;
	gchar buf[PATH_MAX], **theme_dirs, *filepath, *p, *icon_name, **inherits;
	XfceIcon *xicon;
	GDir *dir;
	const gchar *file, *type, **icon_ext;
	gint icon_size, j, k;
	GList *l;
	struct stat st;
	
	g_snprintf(buf, PATH_MAX, "%s/%s/index.theme", base_path, theme_name);
	
	/* infinite loops are bad! */
	if(g_hash_table_lookup(singleton->hash_for_stupid_themes, buf))
		return;
	g_hash_table_insert(singleton->hash_for_stupid_themes, g_strdup(buf),
			(gpointer)0xdeadbeef);
	
	if(singleton->use_svg)
		icon_ext = icon_ext_with_svg;
	else
		icon_ext = icon_ext_without_svg;
	
	themerc = xfce_rc_simple_open(buf, TRUE);
    if(!themerc && !g_ascii_strcasecmp(theme_name, "hicolor")
            && (p = g_strrstr(buf, "/")))
    {
        *p = 0;
        if(g_file_test(buf, G_FILE_TEST_IS_DIR))
            themerc = xfce_rc_simple_open(FALLBACK_INDEX, TRUE);
    }
	if(themerc) {
		if(xfce_rc_has_group(themerc, "Icon Theme")) {
			xfce_rc_set_group(themerc, "Icon Theme");
			
			inherits = xfce_rc_read_list_entry(themerc, "Inherits", ",");
			if(inherits) {
				gchar *inherit_themedir;
				for(j = 0; inherits[j]; j++) {
					if((inherit_themedir = theme_name_to_dir(base_path, inherits[j]))) {
						theme_add_all(singleton, base_path, inherit_themedir, FALSE);
						g_free(inherit_themedir);
					}
				}
				g_strfreev(inherits);
			}
			
			theme_dirs = xfce_rc_read_list_entry(themerc, "Directories", ",");
		} else
			theme_dirs = NULL;
		
		if(theme_dirs) {
			for(j = 0; theme_dirs[j]; j++) {
				if(!xfce_rc_has_group(themerc, theme_dirs[j]))
					continue;
				xfce_rc_set_group(themerc, theme_dirs[j]);
				type = xfce_rc_read_entry(themerc, "Type", "");
				if(g_str_has_prefix(theme_dirs[j], "scalable"))
					icon_size = -1;
				else {
					icon_size = xfce_rc_read_int_entry(themerc, "Size", -1);
					if(icon_size == -1) {
						p = g_strrstr(theme_dirs[j], "x");
						if(p)
							icon_size = atoi(p+1);
						if(icon_size == -1)
							continue;
					}
				}
				
				g_snprintf(buf, PATH_MAX, "%s/%s/%s", base_path, theme_name,
						theme_dirs[j]);
				dir = g_dir_open(buf, 0, NULL);
				if(dir) {
					if(!stat(buf, &st)) {
						singleton->dirs_watch = g_list_append(singleton->dirs_watch,
								g_strdup(buf));
						singleton->dirs_mtimes = g_list_append(singleton->dirs_mtimes,
								GUINT_TO_POINTER(st.st_mtime));
					}
					while((file = g_dir_read_name(dir))) {
						gboolean is_icon_file = FALSE;
						
						for(k = 0; icon_ext[k]; k++) {
							if(g_str_has_suffix(file, icon_ext[k])) {
								is_icon_file = TRUE;
								break;
							}
						}
						if(is_icon_file) {
							filepath = g_build_filename(buf, file, NULL);
							xicon = g_new(XfceIcon, 1);
							xicon->path = filepath;
							xicon->size = icon_size;
							xicon->is_in_theme = is_in_theme;
							
							p = g_strrstr(file, ".");
							icon_name = g_strndup(file, p-file);
							l = g_hash_table_lookup(singleton->icons, icon_name);
							l = g_list_append(l, xicon);
							g_hash_table_replace(singleton->icons, icon_name, l);
						}
					}
					g_dir_close(dir);
				}
			}
		}
		xfce_rc_close(themerc);
	}
}

static void
setup_icons(XfceIconTheme *icon_theme)
{
	XfceIconThemeSingleton *singleton = icon_theme->priv->singleton;
	GtkSettings *settings;
	GList *l;
	gchar *theme_name = NULL;
	gint i;
	
	settings = gtk_settings_get_for_screen(singleton->gscreen);
	
	g_object_get(G_OBJECT(settings), "gtk-icon-theme-name", &theme_name, NULL);
	if(!theme_name)
		theme_name = DEFAULT_THEME;
	
	if(singleton->icons) {
		g_hash_table_foreach(singleton->icons, icon_lists_free_ht, NULL);
		g_hash_table_destroy(singleton->icons);
	}
	singleton->icons = g_hash_table_new_full(g_str_hash, g_str_equal,
			(GDestroyNotify)g_free, NULL);
	
	singleton->hash_for_stupid_themes = g_hash_table_new_full(g_str_hash,
			g_str_equal, g_free, NULL);
	
	/* first get all the in-theme paths */
	for(l = singleton->search_paths; l; l = l->next)
		theme_add_all(singleton, l->data, theme_name, TRUE);
	
	/* then do the fallback paths */
	for(l = singleton->search_paths; l; l = l->next) {
		for(i = 0; fallback_themes[i]; i++)
			theme_add_all(singleton, l->data, fallback_themes[i], FALSE);
	}
	
	if(theme_name != DEFAULT_THEME)
		g_free(theme_name);
	g_hash_table_destroy(singleton->hash_for_stupid_themes);
}

static void
invalidate_theme(XfceIconTheme *icon_theme, XfceIconThemeSingleton *singleton)
{
	GList *l;
	
	if(singleton->icons) {
		g_hash_table_foreach(singleton->icons, icon_lists_free_ht, NULL);
		g_hash_table_destroy(singleton->icons);
		singleton->icons = NULL;
	}
	
	if(singleton->dirs_watch) {
		for(l = singleton->dirs_watch; l; l = l->next)
			g_free(l->data);
		g_list_free(singleton->dirs_watch);
		singleton->dirs_watch = NULL;
	}
	
	if(singleton->dirs_mtimes) {
		g_list_free(singleton->dirs_mtimes);
		singleton->dirs_mtimes = NULL;
	}
	
	if(icon_theme) {
		g_signal_emit(G_OBJECT(icon_theme),
				icon_theme_signals[XFCEIC_SIG_CHANGED], 0);
	}
}

static void
icon_theme_changed_cb(GtkSettings *settings, GParamSpec *pspec,
		gpointer user_data)
{
	XfceIconTheme *icon_theme = user_data;
	
	invalidate_theme(icon_theme, icon_theme->priv->singleton);
}

GType
xfce_icon_theme_get_type()
{
	static GType icon_theme_type = 0;
	
	if(!icon_theme_type) {
		static const GTypeInfo icon_theme_info = {
			sizeof(XfceIconThemeClass),
			NULL,  /* base_init */
			NULL,  /* base_finalize */
			(GClassInitFunc)xfce_icon_theme_class_init,
			NULL,  /* class_finalize */
			NULL,  /* class_data */
			sizeof(XfceIconTheme),
			0,  /* n_preallocs */
			(GInstanceInitFunc)xfce_icon_theme_init,
		};
		
		icon_theme_type = g_type_register_static(G_TYPE_OBJECT,
				"XfceIconTheme", &icon_theme_info, 0);
	}
	
	return icon_theme_type;
}

static void
xfce_icon_theme_class_init(XfceIconThemeClass *klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
	GdkPixbuf *svg_test;
		
	parent_class = g_type_class_peek_parent(klass);
	
	gobject_class->finalize = xfce_icon_theme_finalize;
	
/**
 * XfceIconTheme::changed
 * @icon_theme: the icon theme
 * 
 * Emitted when the current icon theme is switched or GTK+ detects that a change
 * has occurred in the contents of the current icon theme.
 *
 * Since 4.1
 **/
	icon_theme_signals[XFCEIC_SIG_CHANGED] = g_signal_new("changed",
			G_TYPE_FROM_CLASS(klass), G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET(XfceIconThemeClass, changed), NULL, NULL,
			g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
	
	/* gtk 2.2 and below don't have gtk-icon-theme-name.  the param spec strings
	 * should probably be translated, but a) we're in string freeze, b) it
	 * doesn't really matter anyway. */
	if(gtk_major_version == 2 && gtk_minor_version < 3) {
		gtk_settings_install_property(g_param_spec_string("gtk-icon-theme-name",
				"Icon theme name",
				"The name of the currently selected icon theme",
				DEFAULT_THEME, G_PARAM_READABLE|G_PARAM_WRITABLE));
	}
	
	/* check SVG format support */
	svg_test = gdk_pixbuf_new_from_file(DATADIR "/xfce4/xfce-svg-test.svg",
			NULL);
	if(svg_test) {
		_gdk_pixbuf_has_svg = TRUE;
		g_object_unref(G_OBJECT(svg_test));
		DBG("SVG support enabled!");
	} else {
		_gdk_pixbuf_has_svg = FALSE;
		DBG("SVG support disabled!");
	}
}

static void
xfce_icon_theme_init(XfceIconTheme *icon_theme)
{
	icon_theme->priv = g_new0(XfceIconThemePriv, 1);
}

static void
free_singleton(XfceIconThemeSingleton *singleton)
{
	GList *l;
	
	g_object_set_data(G_OBJECT(singleton->gscreen), "xfce-icon-theme", NULL);
	
	invalidate_theme(NULL, singleton);
	
	if(singleton->search_paths) {
		for(l = singleton->search_paths; l; l = l->next)
			g_free(l->data);
		g_list_free(singleton->search_paths);
	}
	
	if(singleton->custom_categories) {
		for(l = singleton->custom_categories; l; l = l->next) {
			if(l->data)
				xfce_icon_theme_category_list_free(l->data);
		}
		g_list_free(singleton->custom_categories);
	}
}

static void
xfce_icon_theme_finalize(GObject *object)
{
	XfceIconTheme *icon_theme = XFCE_ICON_THEME(object);
	GtkSettings *settings;
	
	settings = gtk_settings_get_for_screen(icon_theme->priv->gscreen);
	g_signal_handlers_disconnect_by_func(G_OBJECT(settings),
			icon_theme_changed_cb, icon_theme);
	
	if(--icon_theme->priv->singleton->refcnt == 0)
		free_singleton(icon_theme->priv->singleton);
	
	g_free(icon_theme->priv);
	
	G_OBJECT_CLASS(parent_class)->finalize(object);
}

/**
 * xfce_icon_theme_get_for_screen:
 * @screen: A #GdkScreen, or %NULL.
 *
 * Retrieves the icon theme for the specified GdkScreen (or %NULL for the
 * default screen), creating it if necessary.
 *
 * Since 4.1
 *
 * Return value: A new #XfceIconTheme.
 **/
XfceIconTheme *
xfce_icon_theme_get_for_screen(GdkScreen *screen)
{
	XfceIconTheme *icon_theme;
	GtkSettings *settings;
	XfceIconThemeSingleton *singleton;
	
	if(!screen)
		screen = gdk_display_get_default_screen(gdk_display_get_default());
	
	/* create this first so we've definitely inited _gdk_pixbuf_has_svg */
	icon_theme = g_object_new(XFCE_TYPE_ICON_THEME, NULL);
	
	singleton = g_object_get_data(G_OBJECT(screen), "xfce-icon-theme");
	if(!singleton) {
		gchar **paths, *kde_icon_dir = NULL, *homeicons;
		const gchar *kdedir = g_getenv("KDEDIR");
		gint i, datadir_len = strlen(DATADIR);
		gboolean need_datadir = TRUE;
		
		singleton = g_new0(XfceIconThemeSingleton, 1);
		singleton->refcnt = 1;
		singleton->gscreen = screen;
		singleton->next_custom_id = XFCE_N_BUILTIN_ICON_CATEGORIES;
		singleton->use_svg = _gdk_pixbuf_has_svg;
		
		paths = xfce_resource_lookup_all(XFCE_RESOURCE_DATA, "icons/");
		for(i = 0; paths[i]; i++) {
			singleton->search_paths = g_list_append(singleton->search_paths, paths[i]);
			if(strstr(paths[i], DATADIR) == paths[i]
					&& (strlen(paths[i]) - datadir_len) <= 7)
			{
				need_datadir = FALSE;
			}
		}
		
		if(need_datadir) {
			singleton->search_paths = g_list_prepend(singleton->search_paths,
					g_build_path(G_DIR_SEPARATOR_S, DATADIR, "icons/", NULL));
		}
		
		if(kdedir && *kdedir && strcmp(kdedir, "/usr") && strcmp(kdedir, PREFIX)) {
			kde_icon_dir = g_strconcat(kdedir, "/share/icons/", NULL);
			singleton->search_paths = g_list_append(singleton->search_paths, kde_icon_dir);
		}
		
		homeicons = g_strconcat(xfce_get_homedir(), "/.icons/", NULL);
		singleton->search_paths = g_list_prepend(singleton->search_paths, homeicons);
		
		g_object_set_data(G_OBJECT(screen), "xfce-icon-theme", singleton);
	} else
		singleton->refcnt++;
	
	icon_theme->priv->gscreen = screen;
	icon_theme->priv->singleton = singleton;
	
	settings = gtk_settings_get_for_screen(screen);
	g_signal_connect(G_OBJECT(settings), "notify::gtk-icon-theme-name",
			G_CALLBACK(icon_theme_changed_cb), icon_theme);
	
	return icon_theme;
}

static void
check_icon_theme(XfceIconTheme *icon_theme)
{
	XfceIconThemeSingleton *singleton = icon_theme->priv->singleton;
	gint i, ndirs;
	struct stat st;
	
	if(!singleton->icons) {
		setup_icons(icon_theme);
		return;
	}
	
	ndirs = g_list_length(singleton->dirs_watch);
	for(i = 0; i < ndirs; i++) {
		if(!stat((const char *)g_list_nth_data(singleton->dirs_watch, i), &st)) {
			if(st.st_mtime > GPOINTER_TO_UINT(g_list_nth_data(singleton->dirs_mtimes, i))) {
				invalidate_theme(icon_theme, icon_theme->priv->singleton);
				setup_icons(icon_theme);
				break;
			}
		} else {
			invalidate_theme(icon_theme, icon_theme->priv->singleton);
			setup_icons(icon_theme);
		}
	}
}

static gchar *
search_fallback_paths(XfceIconThemeSingleton *singleton, const gchar *icon_name,
		gint icon_size)
{
	gchar *filename = NULL, buf[PATH_MAX];
	const gchar *default_fallback_path = "/usr/share/pixmaps/";
	const gchar *xfce_prefixed_fallback_path = DATADIR "/pixmaps/";
	const gchar *user_fallback_path = "/.local/share/pixmaps/";
	gint i, path_len;
	const gchar **icon_ext;
	
	g_strlcpy(buf, xfce_prefixed_fallback_path, PATH_MAX);
	g_strlcat(buf, icon_name, PATH_MAX);
	path_len = strlen(buf);
	
	if(singleton->use_svg)
		icon_ext = icon_ext_with_svg;
	else
		icon_ext = icon_ext_without_svg;
	
	for(i = 0; icon_ext[i]; i++) {
		g_strlcat(buf, icon_ext[i], PATH_MAX);
		if(g_file_test(buf, G_FILE_TEST_IS_REGULAR|G_FILE_TEST_IS_SYMLINK)) {
			filename = g_strdup(buf);
			break;
		}
		buf[path_len] = 0;
	}
	
	/* check /usr/share/pixmaps only if DATADIR != /usr/share */
	if(!filename && strcmp(default_fallback_path, xfce_prefixed_fallback_path)) {
		g_strlcpy(buf, default_fallback_path, PATH_MAX);
		g_strlcat(buf, icon_name, PATH_MAX);
		path_len = strlen(buf);
		
		for(i = 0; icon_ext[i]; i++) {
			g_strlcat(buf, icon_ext[i], PATH_MAX);
			if(g_file_test(buf, G_FILE_TEST_IS_REGULAR|G_FILE_TEST_IS_SYMLINK)) {
				filename = g_strdup(buf);
				break;
			}
			buf[path_len] = 0;
		}
	}
	
	/* check ~/.local/share/pixmaps */
	if(!filename) {
		g_strlcpy(buf, xfce_get_homedir(), PATH_MAX);
		g_strlcat(buf, user_fallback_path, PATH_MAX);
		g_strlcat(buf, icon_name, PATH_MAX);
		path_len = strlen(buf);
		
		for(i = 0; icon_ext[i]; i++) {
			g_strlcat(buf, icon_ext[i], PATH_MAX);
			if(g_file_test(buf, G_FILE_TEST_IS_REGULAR|G_FILE_TEST_IS_SYMLINK)) {
				filename = g_strdup(buf);
				break;
			}
			buf[path_len] = 0;
		}
	}
	
	return filename;
}

/* besides the filename this function also returns a 'quality' indicator
 * that depends on whether the icon is in the current theme, in an 
 * inherited theme or in a fallback location */
enum { 
	XFCE_ICON_FALLBACK, 
	XFCE_ICON_INHERITED,
	XFCE_ICON_IN_THEME
};

static gchar *
lookup_icon(XfceIconTheme *icon_theme, const gchar *icon_name, 
		gint icon_size, gint *quality)
{
	XfceIconThemeSingleton *singleton;
	GList *icons, *l;
	XfceIcon *xicon, *closest_xicon = NULL;
	gchar *filename = NULL, *p, *new_icon_name = NULL;
	
	g_return_val_if_fail(XFCE_IS_ICON_THEME(icon_theme) && icon_name, NULL);
	singleton = icon_theme->priv->singleton;
	
	/* check if it's an absolute pathname, and return it if so */
	if(*icon_name == '/' && g_file_test(icon_name, G_FILE_TEST_IS_REGULAR))
		return g_strdup(icon_name);
	
	/* chop off file extension */
	p = g_strrstr(icon_name, ".");
	if(p && strlen(p) < 6)
		new_icon_name = g_strndup(icon_name, strlen(icon_name)-strlen(p));
	if(new_icon_name)
		icon_name = (const gchar *)new_icon_name;
	
	check_icon_theme(icon_theme);
		
	icons = g_hash_table_lookup(singleton->icons, icon_name);
	for(l = icons; l; l = l->next) {
		xicon = l->data;
		
		if(xicon->size == icon_size && xicon->is_in_theme) {
			/* non-scalable icon of the right size, prefer in-theme */
			closest_xicon = xicon;
		} else if(xicon->size == -1 && (!closest_xicon 
				|| (xicon->is_in_theme && !closest_xicon->is_in_theme)
				|| (closest_xicon->size != icon_size
					&& closest_xicon->size != -1)))
		{
			/* svg icon in the theme, and we don't already have a non-scalable
			 * non-svg in the proper size */
			closest_xicon = xicon;
		} else {
			/* try to find a "close enough" match */
			gint cur_diff = closest_xicon ? icon_size - closest_xicon->size : 1000;
			gint new_diff = icon_size - xicon->size;
			if((xicon->is_in_theme || !closest_xicon || !closest_xicon->is_in_theme)
					&& new_diff <= SIZE_THRESHOLD_BELOW
					&& new_diff < cur_diff)
			{
				closest_xicon = xicon;
			}
		}
	}
	
	if(closest_xicon) {
		if(quality)
			*quality = closest_xicon->is_in_theme ? XFCE_ICON_IN_THEME :
					XFCE_ICON_INHERITED;
		filename = g_strdup(closest_xicon->path);
	} else {
		if(quality)
			*quality = XFCE_ICON_FALLBACK;
		filename = search_fallback_paths(singleton, icon_name, icon_size);
	}
	
	if(new_icon_name)
		g_free(new_icon_name);
	
	return filename;
}

/**
 * xfce_icon_theme_lookup:
 * @icon_theme: An #XfceIconTheme.
 * @icon_name: The name of an icon to look up.
 * @icon_size: The preferred icon size.
 *
 * Tries to find @icon_name in the current theme.  This function may return
 * a filename of a size different than that requested in @icon_size, if a
 * scalable icon is found.
 *
 * Since 4.1
 *
 * Return value: A filename (free with g_free() when done), or %NULL if no
 *               suitable icon could be found.
 **/
gchar *
xfce_icon_theme_lookup(XfceIconTheme *icon_theme, const gchar *icon_name,
		gint icon_size)
{
	return lookup_icon(icon_theme, icon_name, icon_size, NULL);
}

/**
 * xfce_icon_theme_lookup_list:
 * @icon_theme: An #XfceIconTheme.
 * @icon_names: A list of names of icons to look up, in order of preference.
 * @icon_size: The preferred icon size.
 *
 * Tries to find one of @icon_names in the current theme.  This function may
 * return a filename of a size different than that requested in @icon_size, if a
 * scalable icon is found.
 *
 * Since 4.1
 *
 * Return value: A filename (free with g_free() when done), or %NULL if no
 *               suitable icon could be found.
 **/
gchar *
xfce_icon_theme_lookup_list(XfceIconTheme *icon_theme, GList *icon_names,
		gint icon_size)
{
	XfceIconThemePriv *priv;
	GList *l;
	gchar *filename = NULL, *lookup = NULL;
	gint quality = XFCE_ICON_FALLBACK - 1, lookup_quality;
	
	g_return_val_if_fail(XFCE_IS_ICON_THEME(icon_theme) && icon_names, NULL);
	priv = icon_theme->priv;
	
	for(l = icon_names; l; l = l->next) {
		lookup = lookup_icon(icon_theme, (gchar *)l->data, icon_size, 
				&lookup_quality);

		if(lookup) {
			if(lookup_quality == XFCE_ICON_IN_THEME) {
				g_free(filename);
				filename = lookup;
				break;
			} else if(lookup_quality > quality) {
				g_free(filename);
				filename = lookup;
				quality = lookup_quality;
			} else
				g_free(lookup);
		}
	}
	
	return filename;
}

static gchar *
xfce_icon_theme_lookup_list_gchar_p(XfceIconTheme *icon_theme,
		const gchar **icon_names, gint icon_size)
{
	GList *l = NULL;
	gint i;
	gchar *filename;
	
	for(i = 0; icon_names[i]; i++)
		l = g_list_append(l, (gpointer)icon_names[i]);
	filename = xfce_icon_theme_lookup_list(icon_theme, l, icon_size);
	g_list_free(l);
	
	return filename;
}

/**
 * xfce_icon_theme_lookup_category:
 * @icon_theme: An #XfceIconTheme.
 * @category: The type of icon to look up.
 * @icon_size: The preferred icon size.
 *
 * Tries to find an icon of type @category in the current theme.  This function
 * may return a filename of a size different than that requested in @icon_size,
 * if a scalable icon is found.  The @category can be a builtin
 * #XfceIconThemeCategory, or a custom category registered with
 * xfce_icon_theme_register_category().
 *
 * Since 4.1
 *
 * Return value: A filename (free with g_free() when done), or %NULL if no
 *               suitable icon could be found.
 **/
gchar *
xfce_icon_theme_lookup_category(XfceIconTheme *icon_theme,
		XfceIconThemeCategory category, gint icon_size)
{
	XfceIconThemeSingleton *singleton;
	gchar *filename = NULL;
	XfceIconThemeCategoryList *custom_cat;
	
	g_return_val_if_fail(XFCE_IS_ICON_THEME(icon_theme), NULL);
	singleton = icon_theme->priv->singleton;
	
	if(category < XFCE_N_BUILTIN_ICON_CATEGORIES)
		filename = xfce_icon_theme_lookup_list_gchar_p(icon_theme,
				builtin_icon_categories[category], icon_size);
	else if(category < singleton->next_custom_id) {
		custom_cat = g_list_nth_data(icon_theme->priv->singleton->custom_categories,
				category-XFCE_N_BUILTIN_ICON_CATEGORIES);
		filename = xfce_icon_theme_lookup_list_gchar_p(icon_theme,
				(const gchar **)custom_cat->icons, icon_size);
	}
	
	return filename;
}

/**
 * xfce_icon_theme_load:
 * @icon_theme: An #XfceIconTheme.
 * @icon_name: The name of an icon to look up.
 * @icon_size: The preferred icon size.
 *
 * Tries to find @icon_name in the current theme.  If a suitable icon can be
 * found, this function will always return an icon of the specified @icon_size,
 * even if it requires scaling to do so.
 *
 * Since 4.1
 *
 * Return value: A new #GdkPixbuf (free with g_object_unref() when done), or
 *               %NULL if no suitable icon could be found.
 **/
GdkPixbuf *
xfce_icon_theme_load(XfceIconTheme *icon_theme, const gchar *icon_name,
		gint icon_size)
{
	gchar *filename;
	GdkPixbuf *pix = NULL;
	
	g_return_val_if_fail(XFCE_IS_ICON_THEME(icon_theme) && icon_name, NULL);
	
	filename = xfce_icon_theme_lookup(icon_theme, icon_name, icon_size);
	if(filename) {
		pix = xfce_pixbuf_new_from_file_at_size(filename, icon_size,
				icon_size, NULL);
		g_free(filename);
	}
	
	return pix;
}

/**
 * xfce_icon_theme_load_list:
 * @icon_theme: An #XfceIconTheme.
 * @icon_names: The names of icons to look up.
 * @icon_size: The preferred icon size.
 *
 * Tries to find one of @icon_names in the current theme.  If a suitable icon
 * can be found, this function will always return an icon of the specified
 * @icon_size, even if it requires scaling to do so.
 *
 * Since 4.1
 *
 * Return value: A new #GdkPixbuf (free with g_object_unref() when done), or
 *               %NULL if no suitable icon could be found.
 **/
GdkPixbuf *
xfce_icon_theme_load_list(XfceIconTheme *icon_theme, GList *icon_names,
		gint icon_size)
{
	gchar *filename;
	GdkPixbuf *pix = NULL;
	
	g_return_val_if_fail(XFCE_IS_ICON_THEME(icon_theme) && icon_names, NULL);
	
	filename = xfce_icon_theme_lookup_list(icon_theme, icon_names, icon_size);
	if(filename) {
		pix = xfce_pixbuf_new_from_file_at_size(filename, icon_size,
				icon_size, NULL);
		g_free(filename);
	}
	
	return pix;
}

/**
 * xfce_icon_theme_load_category:
 * @icon_theme: An #XfceIconTheme.
 * @category: The type of icon to look up.
 * @icon_size: The preferred icon size.
 *
 * Tries to find an icon of type @category in the current theme.  If a suitable
 * icon can be found, this function will always return an icon of the specified
 * @icon_size, even if it requires scaling to do so.  The @category can be a
 * builtin #XfceIconThemeCategory, or a custom category registered with
 * xfce_icon_theme_register_category().
 *
 * Since 4.1
 *
 * Return value: A new #GdkPixbuf (free with g_object_unref() when done), or
 *               %NULL if no suitable icon could be found.
 **/
GdkPixbuf *
xfce_icon_theme_load_category(XfceIconTheme *icon_theme,
		XfceIconThemeCategory category, gint icon_size)
{
	gchar *filename;
	GdkPixbuf *pix = NULL;
	
	g_return_val_if_fail(XFCE_IS_ICON_THEME(icon_theme), NULL);
	
	filename = xfce_icon_theme_lookup_category(icon_theme, category, icon_size);
	if(filename) {
		pix = xfce_pixbuf_new_from_file_at_size(filename, icon_size,
				icon_size, NULL);
		g_free(filename);
	}
	
	return pix;
}

/**
 * xfce_icon_theme_get_search_path:
 * @icon_theme: An #XfceIconTheme.
 *
 * Retrieves the current icon search path.
 *
 * Since 4.1
 *
 * Return value: A GList of search paths.  Free with g_list_free() when done.
 **/
GList *
xfce_icon_theme_get_search_path(XfceIconTheme *icon_theme)
{
	GList *new_list = NULL, *l;
	XfceIconThemeSingleton *singleton;
	
	g_return_val_if_fail(XFCE_IS_ICON_THEME(icon_theme), NULL);
	singleton = icon_theme->priv->singleton;
	
	for(l = singleton->search_paths; l; l = l->next)
		new_list = g_list_append(new_list, g_strdup(l->data));
	
	return new_list;
}

/**
 * xfce_icon_theme_set_search_path:
 * @icon_theme: An #XfceIconTheme.
 * @search_paths: A GList of path names, each one terminated with '/'.
 *
 * Sets the current search path to the list of @search_paths.  Note that this
 * will discard any currently-set search path.
 *
 * Since 4.1
 **/
void
xfce_icon_theme_set_search_path(XfceIconTheme *icon_theme, GList *search_paths)
{
	XfceIconThemeSingleton *singleton;
	GList *l;
	
	g_return_if_fail(XFCE_IS_ICON_THEME(icon_theme));
	
	singleton = icon_theme->priv->singleton;
	
	if(singleton->search_paths) {
		for(l = singleton->search_paths; l; l = l->next)
			g_free(l->data);
		g_list_free(singleton->search_paths);
		singleton->search_paths = NULL;
	}
	
	for(l = (GList *)search_paths; l; l = l->next)
		singleton->search_paths = g_list_append(singleton->search_paths, l->data);
	
	invalidate_theme(icon_theme, icon_theme->priv->singleton);
}

/**
 * xfce_icon_theme_prepend_search_path:
 * @icon_theme: An #XfceIconTheme.
 * @search_path: A pathname, terminated with '/'.
 *
 * Prepends @search_path to the current icon theme search path.
 *
 * Since 4.1
 **/
void
xfce_icon_theme_prepend_search_path(XfceIconTheme *icon_theme,
		const gchar *search_path)
{
	XfceIconThemeSingleton *singleton;
	
	g_return_if_fail(XFCE_IS_ICON_THEME(icon_theme) && search_path);
	singleton = icon_theme->priv->singleton;
	
	if(g_list_find_custom(singleton->search_paths, search_path, (GCompareFunc)strcmp))
		return;
	
	singleton->search_paths = g_list_prepend(singleton->search_paths, g_strdup(search_path));
	invalidate_theme(icon_theme, icon_theme->priv->singleton);
}

/**
 * xfce_icon_theme_append_search_path:
 * @icon_theme: An #XfceIconTheme.
 * @search_path: A pathname, terminated with '/'.
 *
 * Appends @search_path to the current icon theme search path.
 *
 * Since 4.1
 **/
void
xfce_icon_theme_append_search_path(XfceIconTheme *icon_theme,
		const gchar *search_path)
{
	XfceIconThemeSingleton *singleton;
	
	g_return_if_fail(XFCE_IS_ICON_THEME(icon_theme) && search_path);
	singleton = icon_theme->priv->singleton;
	
	if(g_list_find_custom(singleton->search_paths, search_path, (GCompareFunc)strcmp))
		return;
	
	singleton->search_paths = g_list_append(singleton->search_paths, g_strdup(search_path));
	invalidate_theme(icon_theme, icon_theme->priv->singleton);
}

/**
 * xfce_icon_theme_register_category:
 * @icon_theme: An #XfceIconTheme.
 * @icon_names: A GList of icon names.
 *
 * Registers a new #XfceIconThemeCategory for use with
 * #xfce_icon_theme_lookup_category() and xfce_icon_theme_load_category().
 *
 * Since 4.1
 *
 * Return value: An identifier that can be used for future category lookups.
 **/
XfceIconThemeCategory
xfce_icon_theme_register_category(XfceIconTheme *icon_theme, GList *icon_names)
{
	XfceIconThemeSingleton *singleton;
	XfceIconThemeCategoryList *new_list;
	gint i, nicons;;
	
	g_return_val_if_fail(XFCE_IS_ICON_THEME(icon_theme) && icon_names, -1);
	singleton = icon_theme->priv->singleton;
	
	new_list = g_new(XfceIconThemeCategoryList, 1);
	new_list->id = singleton->next_custom_id++;
	
	nicons = g_list_length(icon_names);
	new_list->icons = g_new(gchar *, nicons+1);
	for(i = 0; i < nicons; i++)
		new_list->icons[i] = g_strdup((const gchar *)g_list_nth_data(icon_names, i));
	new_list->icons[nicons] = NULL;
	
	singleton->custom_categories = g_list_append(singleton->custom_categories, new_list);
	
	return new_list->id;
}

/**
 * xfce_icon_theme_unregister_category:
 * @icon_theme: An #XfceIconTheme.
 * @category: An #XfceIconThemeCategory registered previously with
 *            xfce_icon_theme_register_category().
 *
 * Unregisters @category, which must have been registered previously with
 * xfce_icon_theme_register_category().  After this call, the category may no
 * longer be used for icon lookups.
 *
 * Since 4.1
 **/
void
xfce_icon_theme_unregister_category(XfceIconTheme *icon_theme,
		XfceIconThemeCategory category)
{
	XfceIconThemeSingleton *singleton;
	GList *old_list;
	
	g_return_if_fail(XFCE_IS_ICON_THEME(icon_theme)
			&& category >= XFCE_N_BUILTIN_ICON_CATEGORIES);
	singleton = icon_theme->priv->singleton;
	
	old_list = g_list_nth(singleton->custom_categories,
			category-XFCE_N_BUILTIN_ICON_CATEGORIES);
	if(old_list->data) {
		xfce_icon_theme_category_list_free((XfceIconThemeCategoryList *)old_list->data);
		old_list->data = NULL;
	}
}

/**
 * xfce_icon_theme_set_use_svg:
 * @icon_theme: An #XfceIconTheme.
 * @use_svg: Whether or not to use SVG icons.
 *
 * Sets whether or not to include SVG icons in icon lookups and loads.  This
 * option defaults to %TRUE.
 *
 * Since 4.1.91
 **/
void
xfce_icon_theme_set_use_svg(XfceIconTheme *icon_theme, gboolean use_svg)
{
	g_return_if_fail(XFCE_IS_ICON_THEME(icon_theme)
			&& icon_theme->priv->singleton);
	
	if(use_svg && !_gdk_pixbuf_has_svg)
		return;
	
	if(use_svg != icon_theme->priv->singleton->use_svg) {
		icon_theme->priv->singleton->use_svg = use_svg;
		invalidate_theme(icon_theme, icon_theme->priv->singleton);
	}
}

/**
 * xfce_icon_theme_get_use_svg:
 * @icon_theme: An #XfceIconTheme.
 *
 * Reports whether or not the icon theme is set to return SVG icons.
 *
 * Since 4.1.91
 **/
gboolean
xfce_icon_theme_get_use_svg(XfceIconTheme *icon_theme)
{
	g_return_val_if_fail(XFCE_IS_ICON_THEME(icon_theme)
			&& icon_theme->priv->singleton, TRUE);
	
	return icon_theme->priv->singleton->use_svg;
}
