/**
 * \file xforms_helpers.C
 * This file is part of LyX, the document processor.
 * Licence details can be found in the file COPYING.
 *
 * \author Angus Leeming
 *
 * Full author contact details are available in file CREDITS.
 */

#include <config.h>

#include "xforms_helpers.h"

#include "debug.h"
#include "gettext.h"
#include "lengthcommon.h"
#include "lyxgluelength.h"
#include "lyxlex.h"

#include "support/filetools.h"
#include "support/lstrings.h" // frontStrip, strip
#include "support/convert.h"
#include "support/fs_extras.h"

#include "lyx_forms.h"
#include "combox.h"

#include <boost/assert.hpp>
#include <boost/filesystem/operations.hpp>

#include <fstream>

using std::make_pair;

using std::ofstream;
using std::vector;
using std::string;

namespace fs = boost::filesystem;

namespace lyx {

using support::AbsolutePath;
using support::isStrDbl;
using support::OnlyPath;
using support::subst;
using support::trim;

namespace frontend {

std::string const buildChoiceLengthString()
{
	string data;
	for (int i = 0; i != num_units; ++i) {
		if (i != 0)
			data += "|";
		data += subst(unit_name_gui[i], "%", "%%");
	}
	return data;
}


bool isActive(FL_OBJECT * ob)
{
	return ob && ob->active > 0;
}


std::pair<string, string> parse_shortcut(string const & str)
{
	string::size_type i = str.find_first_of("&");
	if (i == string::npos || i == str.length() - 1)
		return make_pair(str, string());

	// FIXME: handle &&

	string::value_type c = str[i + 1];
	return make_pair(str.substr(0, i) + str.substr(i + 1),
			 string("#") + c);
}


// A wrapper for the xforms routine, but this one accepts uint args
unsigned long fl_getmcolor(int i,
			   unsigned int * r, unsigned int * g, unsigned int * b)
{
	int r2, g2, b2;
	unsigned long ret_val = ::fl_getmcolor(i, &r2, &g2, &b2);
	*r = r2;
	*g = g2;
	*b = b2;
	return ret_val;
}


// Set an FL_OBJECT to activated or deactivated
void setEnabled(FL_OBJECT * ob, bool enable)
{
	if (enable) {
		fl_activate_object(ob);
		fl_set_object_lcol(ob, FL_LCOL);
	} else {
		fl_deactivate_object(ob);
		fl_set_object_lcol(ob, FL_INACTIVE);
	}
}


// Given an fl_choice or an fl_browser, create a vector of its entries
vector<string> const getVector(FL_OBJECT * ob)
{
	vector <string> vec;

	switch (ob->objclass) {
	case FL_CHOICE:
		for(int i = 0; i < fl_get_choice_maxitems(ob); ++i) {
			string const text = fl_get_choice_item_text(ob, i+1);
			vec.push_back(trim(text));
		}
		break;
	case FL_BROWSER:
		for(int i = 0; i < fl_get_browser_maxline(ob); ++i) {
			string const text = fl_get_browser_line(ob, i+1);
			vec.push_back(trim(text));
		}
		break;
	default:
		BOOST_ASSERT(false);
	}

	return vec;
}


///
string const getString(FL_OBJECT * ob, int line)
{
	// Negative line value does not make sense.
	BOOST_ASSERT(line >= 0);

	char const * tmp = 0;
	switch (ob->objclass) {
	case FL_INPUT:
		tmp = fl_get_input(ob);
		break;

	case FL_BROWSER:
		if (line == 0)
			line = fl_get_browser(ob);

		if (line >= 1 && line <= fl_get_browser_maxline(ob))
			tmp = fl_get_browser_line(ob, line);
		break;

	case FL_CHOICE:
		if (line == 0)
			line = fl_get_choice(ob);

		if (line >= 1 && line <= fl_get_choice_maxitems(ob))
			tmp = fl_get_choice_item_text(ob, line);
		break;

	case FL_COMBOX:
		tmp = fl_get_combox_text(ob);
		break;

	default:
		BOOST_ASSERT(false);
	}

	return tmp ? trim(tmp) : string();
}

string getLengthFromWidgets(FL_OBJECT * input, FL_OBJECT * choice)
{
	// Paranoia check
	BOOST_ASSERT(input  && input->objclass  == FL_INPUT &&
		    choice && choice->objclass == FL_CHOICE);

	string const length = trim(fl_get_input(input));
	if (length.empty())
		return string();

	// don't return unit-from-choice if the input(field) contains a unit
	if (isValidGlueLength(length))
		return length;

	string unit = trim(fl_get_choice_text(choice));
	unit = subst(unit, "%%", "%");

	return length + unit;
}


void updateWidgetsFromLengthString(FL_OBJECT * input, FL_OBJECT * choice,
				   string const & str,
				   string const & default_unit)
{
	// Paranoia check
	BOOST_ASSERT(input  && input->objclass  == FL_INPUT &&
		     choice && choice->objclass == FL_CHOICE);

	// use input field only for gluelengths
	if (!isValidLength(str) && !isStrDbl(str)) {
		fl_set_input(input, str.c_str());
		// we assume that "default_unit" is in the choice as "we"
		// have control over that!
		// No need to check for its presence in the choice, therefore.
		fl_set_choice_text(choice, default_unit.c_str());
	} else {
		updateWidgetsFromLength(input, choice,
					LyXLength(str), default_unit);
	}
}


void updateWidgetsFromLength(FL_OBJECT * input, FL_OBJECT * choice,
			     LyXLength const & len,
			     string const & default_unit)
{
	// Paranoia check
	BOOST_ASSERT(input  && input->objclass  == FL_INPUT &&
		     choice && choice->objclass == FL_CHOICE);

	if (len.empty()) {
		fl_set_input(input, "");
		fl_set_choice_text(choice, default_unit.c_str());
	} else {
		fl_set_input(input, convert<string>(len.value()).c_str());

		// Set the choice to the desired unit, if present in the choice.
		// Else set the choice to the default unit.
		string const unit = subst(stringFromUnit(len.unit()),"%","%%");

		vector<string> const vec = getVector(choice);
		vector<string>::const_iterator it =
			std::find(vec.begin(), vec.end(), unit);
		if (it != vec.end()) {
			fl_set_choice_text(choice, unit.c_str());
		} else {
			fl_set_choice_text(choice, default_unit.c_str());
		}
	}
}


// Take a string and add breaks so that it fits into a desired label width, w
string formatted(string const & sin, int w, int size, int style)
{
	string sout;
	if (sin.empty())
		return sout;

	string::size_type curpos = 0;
	string line;
	for(;;) {
		string::size_type const nxtpos1 = sin.find(' ',  curpos);
		string::size_type const nxtpos2 = sin.find('\n', curpos);
		string::size_type const nxtpos = std::min(nxtpos1, nxtpos2);

		string const word = nxtpos == string::npos ?
			sin.substr(curpos) : sin.substr(curpos, nxtpos-curpos);

		bool const newline = (nxtpos2 != string::npos &&
				      nxtpos2 < nxtpos1);

		string const line_plus_word =
			line.empty() ? word : line + ' ' + word;

		int const length =
			fl_get_string_width(style, size,
					    line_plus_word.c_str(),
					    int(line_plus_word.length()));

		if (length >= w) {
			sout += line + '\n';
			if (newline) {
				sout += word + '\n';
				line.erase();
			} else {
				line = word;
			}

		} else if (newline) {
			sout += line_plus_word + '\n';
			line.erase();

		} else {
			if (!line.empty())
				line += ' ';
			line += word;
		}

		if (nxtpos == string::npos) {
			if (!line.empty())
				sout += line;
			break;
		}

		curpos = nxtpos+1;
	}

	return sout;
}


void setCursorColor(int color)
{
	fl_set_cursor_color(FL_DEFAULT_CURSOR, color, FL_WHITE);
	fl_set_cursor_color(XC_xterm,          color, FL_WHITE);
	fl_set_cursor_color(XC_watch,          color, FL_WHITE);
	fl_set_cursor_color(XC_sb_right_arrow, color, FL_WHITE);
}


namespace {

// sorted by hand to prevent LyXLex from complaining on read().
keyword_item xformTags[] = {
	{ "\\gui_background",   FL_COL1 },
	{ "\\gui_buttonbottom", FL_BOTTOM_BCOL },
	{ "\\gui_buttonleft",   FL_LEFT_BCOL },
	{ "\\gui_buttonright",  FL_RIGHT_BCOL },
	{ "\\gui_buttontop",    FL_TOP_BCOL },
	{ "\\gui_inactive",     FL_INACTIVE },
	{ "\\gui_pointer",      FL_FREE_COL16 },
	{ "\\gui_push_button",  FL_YELLOW },
	{ "\\gui_selected",     FL_MCOL },
	{ "\\gui_text",         FL_BLACK }
};


const int xformCount = sizeof(xformTags) / sizeof(keyword_item);

} // namespace anon


bool XformsColor::read(string const & filename)
{
	LyXLex lexrc(xformTags, xformCount);
	if (fs::is_readable(filename) && !lexrc.setFile(filename)) {
		lyxerr << "XformsColor::read(" << filename << ")\n"
		       << _("Failed to open file.") << std::endl;
		return false;
	}

	while (lexrc.isOK()) {
		int const le = lexrc.lex();

		switch (le) {
		case LyXLex::LEX_UNDEF:
			lexrc.printError("Unknown tag `$$Token'");
			continue;
		case LyXLex::LEX_FEOF:
			continue;
		default: break;
		}

		string const tag = lexrc.getString();

		RGBColor col;

		if (!lexrc.next()) break;
		col.r = lexrc.getInteger();

		if (!lexrc.next()) break;
		col.g = lexrc.getInteger();

		if (!lexrc.next()) break;
		col.b = lexrc.getInteger();

		fl_mapcolor(le, col.r, col.g, col.b);

		if (tag == "\\gui_pointer") {
			setCursorColor(FL_FREE_COL16);
		}
	}

	return true;
}


bool XformsColor::write(string const & filename)
{
	ofstream os(filename.c_str());
	if (!os) {
		lyxerr << "XformsColor::write(" << filename << ")\n"
		       << _("Failed to open file.") << std::endl;
		return false;
	}

	os << "###"
	   << "### file " << filename << "\n\n"
	   << "### This file is written by LyX, if you want to make your own\n"
	   << "### modifications you should do them from inside LyX and save\n"
	   << '\n';

	for (int i = 0; i < xformCount; ++i) {
		string const tag  = xformTags[i].tag;
		int const colorID = xformTags[i].code;
		RGBColor color;

		fl_getmcolor(colorID, &color.r, &color.g, &color.b);

		os << tag << ' '
		   << color.r << ' ' << color.g << ' ' << color.b << '\n';
	}

	return true;
}


string  RWInfo::error_message;

bool RWInfo::WriteableDir(string const & name)
{
	error_message.erase();

	if (!AbsolutePath(name)) {
		error_message = _("The absolute path is required.");
		return false;
	}

	if (!fs::exists(name) || !fs::is_directory(name)) {
		error_message = _("Directory does not exist.");
		return false;
	}

	if (!fs::is_writable(name)) {
		error_message = _("Cannot write to this directory.");
		return false;
	}

	return true;
}


bool RWInfo::ReadableDir(string const & name)
{
	error_message.erase();

	if (!AbsolutePath(name)) {
		error_message = _("The absolute path is required.");
		return false;
	}

	if (!fs::exists(name) || !fs::is_directory(name)) {
		error_message = _("Directory does not exist.");
		return false;
	}

	if (!fs::is_readable(name)) {
		error_message = _("Cannot read this directory.");
		return false;
	}

	return true;
}


bool RWInfo::WriteableFile(string const & name)
{
	// A writeable file is either:
	// * An existing file to which we have write access, or
	// * A file that doesn't yet exist but that would exist in a writeable
	//   directory.

	error_message.erase();

	if (name.empty()) {
		error_message = _("No file input.");
		return false;
	}

	string const dir = OnlyPath(name);
	if (!AbsolutePath(dir)) {
		error_message = _("The absolute path is required.");
		return false;
	}

	string checkFile = name;

	if (!fs::exists(checkFile) || !fs::is_directory(checkFile)) {
		checkFile = dir;
	}

	if (!fs::exists(checkFile) || !fs::is_directory(checkFile)) {
		error_message = _("Directory does not exists.");
		return false;
	}

	if (!fs::is_writable(checkFile)) {
		error_message = _("Cannot write to this directory.");
		return false;
	}

	if (dir == name || (fs::exists(name) && fs::is_directory(name))) {
		error_message = _("A file is required, not a directory.");
		return false;
	}

	if (fs::exists(name) && !fs::is_writable(name)) {
		error_message = _("Cannot write to this file.");
		return false;
	}

	return true;
}


bool RWInfo::ReadableFile(string const & name)
{
	error_message.erase();

	if (name.empty()) {
		error_message = _("No file input.");
		return false;
	}

	string const dir = OnlyPath(name);
	if (!AbsolutePath(dir)) {
		error_message = _("The absolute path is required.");
		return false;
	}

	string checkFile = name;

	if (!fs::exists(checkFile) && !fs::is_directory(checkFile)) {
		checkFile = dir;
	}

	if (!fs::exists(checkFile) || !fs::is_directory(checkFile)) {
		error_message = _("Directory does not exist.");
		return false;
	}

	if (!fs::is_readable(checkFile)) {
		error_message = _("Cannot read from this directory.");
		return false;
	}

	if (dir == name || (fs::exists(name) && fs::is_directory(name))) {
		error_message = _("A file is required, not a directory.");
		return false;
	}

	if (!fs::exists(name)) {
		error_message = _("File does not exist.");
		return false;
	}

	if (!fs::is_readable(name)) {
		error_message = _("Cannot read from this file.");
		return false;
	}

	return true;
}

} // namespace frontend
} // namespace lyx
