/* 
 * dumpquery.c
 *
 * (c) NLnet Labs, 2004
 *
 * See the file LICENSE for the license
 *
 */

#include "common.h"

/**
 * Dumps the given packet to a ;-commented ascii file containing hex values
 */
int
packet2file(struct t_dpacket *packet, char *filename)
{
	uint8_t *wire;// = xmalloc((packet->udppacketsize)*21);
	size_t size;
	FILE *fp;
	size_t bufsize = (packet->udppacketsize) * 2;
	wire = xmalloc(bufsize);
	
	fp = fopen(filename, "w");
	
	if (fp == NULL) {
		perror("Unable to open file for writing");
		return RET_FAIL;
	}
	
	size = packet2wire(packet, wire, bufsize);
	
	fprint_bytes_hex(fp, wire, 0, (int) size, 20);
	
	verbose("saved: %d bytes to %s\n", (int) size, filename);
	
	fclose(fp);
		
	return RET_SUC;
}

/**
 * Reads a packet from the given file name (;-commented hex ascii) and
 * sends it to the nameserver specified in ns
 */
int
sendfromfile(char *filename, struct t_rr *ns)
{
	struct t_dpacket *packet;
	uint8_t *wirebuf = xmalloc(MAX_PACKET);
	size_t wirebuflen = 0;

	uint8_t *reply;
	size_t replysize;
	
	int result;

	wirebuflen = packetbuffromfile(filename, wirebuf);
	

	if (have_drill_opt && drill_opt->verbose) {
		struct t_dpacket *sendp = dpacket_create();
		wire2packet(wirebuf, sendp, wirebuflen);
		print_packet(sendp);
	}

	reply = xmalloc(MAX_PACKET);
	/* TODO: don't just grab first nameserver in opts */
	
	if (is_ipv6_addr_rdata(ns->rdata[0])) {
		result = send_udp_raw6(wirebuf, wirebuflen, ns->rdata[0], &reply, &replysize);
	} else {
		result = send_udp_raw(wirebuf, wirebuflen, ns->rdata[0], &reply, &replysize);
	}
	if (result == RET_SUC) {
		packet = dpacket_create();

		if (wire2packet(reply, packet, replysize) == RET_SUC) {
			/* fclose(fp); */
			packet->serverip = rdata2str(ns->rdata[0]);
			print_packet(packet);
			return RET_SUC;
		} else {
			error("format err\n");
			return RET_FAIL;
		}
	} else {
		warning("No answer\n");
		return RET_FAIL;
	}
}


size_t
packetbuffromfile(char *filename, uint8_t *wire)
{
	FILE *fp = NULL;
	char c;
	
	/* stat hack
	 * 0 = normal
	 * 1 = comment (skip to end of line)
	 * 2 = unprintable character found, read binary data directly
	 */
	int state = 0;
	uint8_t *hexbuf = xmalloc(MAX_PACKET);
	int hexbufpos = 0;
	size_t wirelen;
	
	if (strncmp(filename, "-", 2) == 0) {
		fp = stdin;
	} else {
		fp = fopen(filename, "r");
	}
	if (fp == NULL) {
		perror("Unable to open file for reading");
		return RET_FAIL;
	}

	verbose("Opened %s\n", filename);
	
	c = fgetc(fp);
	while (c != EOF && hexbufpos < MAX_PACKET) {
		if (state < 2 && !isascii(c)) {
			verbose("non ascii character found in file: (%d) switching to raw mode\n", c);
			state = 2;
		}
		switch (state) {
			case 0:
				if (	(c >= '0' && c <= '9') ||
					(c >= 'a' && c <= 'f') ||
					(c >= 'A' && c <= 'F') )
				{
					hexbuf[hexbufpos] = (uint8_t) c;
					hexbufpos++;
				} else if (c == ';') {
					state = 1;
				} else if (c == ' ' || c == '\t' || c == '\n') {
					/* skip whitespace */
				} 
				break;
			case 1:
				if (c == '\n' || c == EOF) {
					state = 0;
				}
				break;
			case 2:
				hexbuf[hexbufpos] = (uint8_t) c;
				hexbufpos++;
				break;
			default:
				error("unknown state while reading file\n");
				return RET_FAIL;
				break;
		}
		c = fgetc(fp);
	}

	if (c == EOF) {
		if (have_drill_opt && drill_opt->verbose) {
			verbose("END OF FILE REACHED\n");
			if (state < 2) {
				verbose("read:\n");
				verbose("%s\n", hexbuf);
			} else {
				verbose("Not printing wire because it contains non ascii data\n");
			}
		}
	}
	if (hexbufpos >= MAX_PACKET) {
		verbose("packet size reached\n");
	}
	
	/* lenient mode: length must be multiple of 2 */
	if (hexbufpos % 2 != 0) {
		hexbuf[hexbufpos] = (uint8_t) '0';
		hexbufpos++;
	}

	if (state < 2) {
		wirelen = hexstr2bin((char *) hexbuf, hexbufpos, wire, 0, MAX_PACKET);
	} else {
		memcpy(wire, hexbuf, (size_t) hexbufpos);
		wirelen = (size_t) hexbufpos;
	}
	return wirelen;
}	
		
