/*
 * drill.c
 * the main file of drill
 * (c) 2004 NLnet Labs
 *
 * See the file LICENSE for the license
 *
 */

#include "common.h"

/* define the args we support */
#ifdef HAVE_GETOPT_H
static struct option long_options[] =
{
	{"port", required_argument, 0, 'p'},
	{"bufsize", required_argument, 0, 'b'},
	{"trace", no_argument, 0, 'T'},
	{"sigchase", no_argument, 0, 'S'},
	{"dnssec", no_argument, 0,'D'}, /* -d as shortcut?? */
	{"ds", no_argument, 0,'s'}, /* -d as shortcut?? */
	{"tcp", no_argument, 0, 'c'},	/* -c -> connected */
	{"udp", no_argument, 0, 'u'},	/* -c -> connected */
	{"version", no_argument, 0, 'v'},
	{"help", no_argument, 0, 'h'},
	{"key", required_argument, 0, 'k'},
	{"nsec", no_argument, 0, 'N'}, 	/* nsec walk */
	{"verbose", no_argument, 0, 'V'},
	{"dumpquery", required_argument, 0, 'q'},
	{"fromfile", required_argument, 0, 'f'},
	{"reverse", no_argument, 0, 'x'},
	{"ip6", no_argument, 0, '6'},
	{"ip4", no_argument, 0, '4'},
	{"answerinfile", required_argument, 0, 'i'},
	{"answertofile", required_argument, 0, 'w'},
	{"fail", no_argument, 0, 'a'},
	{"rd", no_argument, 0, 'r'},
	{NULL, 0, 0, 0}
};
#endif

/**
 * Prints short usage msg to stderr
 */
static void 
usage(void)
{	        
	fprintf(stderr, "drill [options] [type] name [class]\n\n");
	fprintf(stderr, "\t[type] defaults to \'A\'\n");
	fprintf(stderr, "\t[class] defaults to \'IN\'\n");
	fprintf(stderr, "\tThe @server argument made be placed anywhere\n");
	fprintf(stderr, "\t[type] and [class] always need to be specified in that order\n\n");

	fprintf(stderr, "\t@server \tuse server as nameserver\n");
	fprintf(stderr, "\t-T, --trace\ttrace from the root down to \'name\'\n");
	fprintf(stderr, "\t-S, --sigchase\tchase signature from \'name\'\n");
	fprintf(stderr, "\t-D, --dnssec\tenable dnssec\n");
	fprintf(stderr, "\t-I \t\treserved for backwards compatibility\n");
	fprintf(stderr, "\t-V, --verbose\tVerbose mode (give twice for more verbosity [hexdump])\n");
	fprintf(stderr, "\t-4, --ip4\tStay on IPv4\n");
	fprintf(stderr, "\t-6, --ip6\tStay on IPv6\n");
	fprintf(stderr, "\n");
	fprintf(stderr, "\t-b size, --bufsize size\t use size is buffer size\n");
	fprintf(stderr, "\t-f file, --fromfile file\tread packet from file and send that\n");
	fprintf(stderr, "\t-i file, --answerinfile file\tread packet from file and print it\n");
	fprintf(stderr, "\t-p port, --port port\t use port as port number\n");
	fprintf(stderr, "\t-q file, --dumpquery file\tmake a hexdump of the query to file\n");
	fprintf(stderr, "\t-w file, --answertofile file\twrite (first) answer to file\n");
	fprintf(stderr, "\n");
	fprintf(stderr, "\t-a, --fail\tdon't try the next nameserver on SERVFAIL (default is yes)\n");
	fprintf(stderr, "\t-c, --tcp\tonly query in tcp mode (connected)\n");
	fprintf(stderr, "\t-k, --key file\tuse public key from file file as trusted key\n");
	fprintf(stderr, "\t-r, --rd\tdon't set the RD bit in queries (default is on)\n");
	fprintf(stderr, "\t-s, --ds\tprint DS after each DNSKEY\n");
	fprintf(stderr, "\t-u, --udp\tonly query in udp mode (unconnected)\n");
	fprintf(stderr, "\t-v, --version\tshow version\n");
	fprintf(stderr, "\t-x, --reverse\tdo a reverse (PTR) lookup\n");
	fprintf(stderr, "\n\nInfo: drill@nlnetlabs.nl\n");
}

/**
 * Prints the drill version to stderr
 */
static void 
version(void)
{	        
	fprintf(stderr, "drill version %s\n", DRILL_VERSION);
	fprintf(stderr, "Written by NLnet Labs.\n");
	fprintf(stderr, "\nCopyright (c) 2004 NLnet Labs.\n");
	fprintf(stderr, "Licensed under the GPL 2.\n");
	fprintf(stderr, "There is NO warranty; not even for MERCHANTABILITY or FITNESS\n");
	fprintf(stderr, "FOR A PARTICULAR PURPOSE.\n");
}

/**
 * Creates a drill options structure (which keeps track of global settings)
 */
static void
drill_opt_create(void)
{
        drill_opt = xmalloc(sizeof(struct t_opt));
        drill_opt->type = TYPE_A;
	drill_opt->class = CLASS_IN;
        drill_opt->port = DNS_PORT;
        drill_opt->dnssec = 0;
        drill_opt->bufsize = BUFSIZE;
        drill_opt->owner = NULL;
        drill_opt->server = NULL;
        drill_opt->debug = 0;
        drill_opt->reverse = 0;
        drill_opt->verbose = 0;
        drill_opt->transport = 0;
        drill_opt->answerfile = NULL;
        drill_opt->purpose = 0;
	drill_opt->ds = 0;
	drill_opt->fail = 0;
	drill_opt->recurse = 1;
        have_drill_opt = 1;
}

/**
 * Frees the memory allocated for the drill options structure
 */
static void
drill_opt_destroy(void)
{
        /*rr_destroy(drill_opt->server, FOLLOW);*/
        if (drill_opt->server) {
                xfree(drill_opt->server);
        }
        if (drill_opt->owner) {
                xfree(drill_opt->owner);
        }
        xfree(drill_opt);
}

/**
 * Main function of drill
 * parse the arguments and send a query
 */
int
main(int argc, char *argv[])
{
	int protocol = PROTO_ANY;
	struct t_rr *q;
	struct t_rdata *q_rdata;
	struct t_rr *nameserver = NULL;
	struct t_rr *trusted_keys = NULL;
	struct t_rr *r = NULL;
	struct timeval randtime;
	int result; int c;
	struct zone_list *secure_zones = NULL;
	struct zone_list *search_zones = NULL;
	char *tmpstr;
	
	gettimeofday(&randtime, NULL);
	drill_opt_create();
	
#ifdef HAVE_GETOPT_H
	while ((c = getopt_long(argc, argv, "i:w:I46Sk:TNp:b:DsvhVcuaq:f:xr", long_options, 0)) != -1) {
#else
	while ( (c = getopt(argc, argv, "i:w:I46Sk:TNp:b:DsvhVcuaq:f:xr")) != -1) {
#endif
		switch(c) {
			case 'i':
				/* don't send anything, just try to 
				 * read 'answer' from file
				 */
				drill_opt->purpose = ANSWERFROMFILE;
				drill_opt->answerfile = (char*)optarg;
				break;
			case 'a':
				drill_opt->fail = 1;
				break;
			case 'r':
				drill_opt->recurse = 0;
				break;
			case 'w':
				drill_opt->answerfile = (char*)optarg;
				break;
			case 'I':
				/* ignore */
				break;
			case '4':
				drill_opt->transport = 4;
				break;
			case '6':
				drill_opt->transport = 6;
				break;
			case 'S':
				drill_opt->purpose = CHASE;
				drill_opt->dnssec = 1;
				break;
			case 'c':
				protocol = PROTO_TCP;
				break;
			case 'u':
				protocol = PROTO_UDP;
				break;
			case 'T':
				drill_opt->purpose = TRACE;
				break;
			case 'N':
				drill_opt->purpose = NSEC_WALK;
				break;
			case 'p':
				drill_opt->port = (unsigned) atoi(optarg);
				if (drill_opt->port == 0) 
					drill_opt->port = DNS_PORT;
				break; 
			case 'b':
				drill_opt->bufsize = (unsigned) atoi(optarg);
				if (drill_opt->bufsize == 0 || drill_opt->bufsize > 4096)
					drill_opt->bufsize = BUFSIZE;
				break;
			case 'D':
				drill_opt->dnssec = 1;
				if (secure_zones == NULL) {
					secure_zones = zone_list_create();
				}
				break;
			case 's':
				drill_opt->ds = 1;
				break;
			case 'x':
				drill_opt->reverse = 1;
				break;
			case 'h':
				usage();
				drill_opt_destroy();
				exit(EXIT_SUCCESS);
			case 'v':
				version();
				drill_opt_destroy();
				exit(EXIT_SUCCESS);
			case 'k':
				r = pubkey_fromfile(optarg);
				trusted_keys =
					keyset_add_key(trusted_keys, r);
				if (drill_opt->verbose)
					prettyprint_rr(r, NO_FOLLOW, COMMENT, 0);
				rr_destroy(r, FOLLOW);
				break;
			case 'V':
				drill_opt->verbose++;
				break;
			case 'q':
				drill_opt->purpose = DUMPQUERY;
				drill_opt->filename = optarg;
				break;
			case 'f':
				drill_opt->purpose = FROMFILE;
				drill_opt->filename = optarg;
				break;
			default:
				usage();
				drill_opt_destroy();
				exit(EXIT_FAILURE);
		}
	}
	argc -= optind;
	argv += optind;

	if (argc < 1 && !drill_opt->answerfile) {
		usage();
		exit(EXIT_FAILURE);
	}

	/* error checking ? */
	search_zones = read_resolv_conf_search("/etc/resolv.conf");

	secure_zones = read_resolv_conf_must_be_secure("/etc/resolv.conf");
	
	if (trusted_keys == NULL) {
		trusted_keys = read_resolv_conf_dnskey("/etc/resolv.conf");
	} else {
		(void) rr_add_rr(trusted_keys, read_resolv_conf_dnskey("/etc/resolv.conf"));
	}

	/* do the arguments */
	switch(argc) {
		case 1:
			if (argv[0][0] == '@' && drill_opt->purpose == FROMFILE) {
				drill_opt->server = xstrdup(argv[0] + 1);
			} else if (argv[0][0] == '@') {
				error("%s", "no @nameserver allowed");
			}
			drill_opt->owner = xstrdup(argv[0]);
			break;
		case 2:
			if (argv[0][0] == '@') {
				drill_opt->server = xstrdup(argv[0] + 1);
				drill_opt->owner = xstrdup(argv[1]);
				break;
			}
			if (argv[1][0] == '@') {
				/* do nameserver thingy */
				drill_opt->server = xstrdup(argv[1] + 1);
				drill_opt->owner = xstrdup(argv[0]);
				break;
			}
			/* no nameserver given, standard sequence */
			drill_opt->type = typebyname(argv[0], ztypes);
			drill_opt->owner = xstrdup(argv[1]);
			break;
		case 3:
			if (argv[0][0] == '@') {
				drill_opt->server = xstrdup(argv[0] + 1);
				drill_opt->type = typebyname(argv[1], ztypes);
				drill_opt->owner = xstrdup(argv[2]);
				break;
			}
			if (argv[1][0] == '@') {
				drill_opt->server = xstrdup(argv[1] + 1);
				drill_opt->type = typebyname(argv[0], ztypes);
				drill_opt->owner = xstrdup(argv[2]);
				break;
			}
			if (argv[2][0] == '@') {
				drill_opt->server = xstrdup(argv[2] + 1);
				drill_opt->type = typebyname(argv[0], ztypes);
				drill_opt->owner = xstrdup(argv[1]);
				break;
			}
			error("parsing arguments\n");
			usage();
			exit(EXIT_FAILURE);
		case 4:
			if (argv[0][0] == '@') {
				drill_opt->server = xstrdup(argv[0] + 1);
				drill_opt->type = typebyname(argv[1], ztypes);
				drill_opt->owner = xstrdup(argv[2]);
				drill_opt->class = intbyname(argv[3], zclasses);
				break;
			}
			if (argv[1][0] == '@') {
				drill_opt->server = xstrdup(argv[1] + 1);
				drill_opt->type = typebyname(argv[0], ztypes);
				drill_opt->owner = xstrdup(argv[2]);
				drill_opt->class = intbyname(argv[3], zclasses);
				break;
			}
			if (argv[2][0] == '@') {
				drill_opt->server = xstrdup(argv[2] + 1);
				drill_opt->type = typebyname(argv[0], ztypes);
				drill_opt->owner = xstrdup(argv[1]);
				drill_opt->class = intbyname(argv[3], zclasses);
				break;
			}
			if (argv[3][0] == '@') {
				drill_opt->server = xstrdup(argv[3] + 1);
				drill_opt->type = typebyname(argv[0], ztypes);
				drill_opt->owner = xstrdup(argv[1]);
				drill_opt->class = intbyname(argv[2], zclasses);
				break;
			}
			break;
		default:
			if (drill_opt->purpose != ANSWERFROMFILE)
				error("%s", "Please enter options before owner name and server name");
	}

	if (drill_opt->type > 255)
		error("%s", "Unknown type > 255 not supported");

	if (drill_opt->type == 0) {
		error("%s", "no valid type parsed");
	} else {
		/* check for DNSSEC types, if so set dnssec on */
		switch(drill_opt->type) {
			case TYPE_RRSIG:
			case TYPE_NSEC:
			case TYPE_DNSKEY:
			case TYPE_DS:
				drill_opt->dnssec = 1;
				break;
			}
	}
	
	/* CLASS CHECKS??? XXX MIEK */
	
	/* resolve nameserver if needed */
	if (!drill_opt->server) {
		/*verbose("%s", "Found nameservers in /etc/resolv.conf");*/
		
		if (nameserver)
			rr_destroy(nameserver, FOLLOW);
		nameserver = read_resolv_conf_ns("/etc/resolv.conf");
	} else  {
		if (!is_ip_address(drill_opt->server)) {
			nameserver = read_resolv_conf_ns("/etc/resolv.conf");
			verbose("%s", "Found nameservers:");
			if (drill_opt->verbose)
				print_rr(nameserver, FOLLOW);

			/* Jelte, return value? */
			(void)resolve2rr(drill_opt->server, nameserver, trusted_keys, search_zones, 
					secure_zones, protocol, 0, &r);

			if (!r) {
				error("Could not get address for %s\n", drill_opt->server);
			} else {
				rr_destroy(nameserver, FOLLOW);
				nameserver = r;
			}
		} else 
			nameserver = ns_create(NULL, (uint8_t*) drill_opt->server, TYPE_A);	
	}

	/* all three values need to be filled in now */
	/* now we can fiddle with them */

	if (drill_opt->reverse == 1) {
		drill_opt->type = TYPE_PTR;
		/* aaarggggh - reverse lookup, modify the owner name */
		if (!is_ip_address(drill_opt->owner))
			error("%s", "For reverse lookups a IP address is needed");
		else {
			tmpstr = xmalloc(strlen(drill_opt->owner)+1);
			strncpy(tmpstr, drill_opt->owner, strlen(drill_opt->owner) + 1);
			xfree(drill_opt->owner);
			drill_opt->owner = in_addr(tmpstr);
			xfree(tmpstr);
		}
	} else if (drill_opt->purpose != ANSWERFROMFILE) {
		/* a normal lookup, add a final dot if is isn't there */
		if (drill_opt->owner[strlen(drill_opt->owner) - 1] != '.') {
			size_t l = strlen(drill_opt->owner);
			if (drill_opt->owner == NULL) {
				drill_opt->owner = xmalloc(l + 2);
			} else {
				drill_opt->owner = xrealloc(drill_opt->owner,l + 2);
			}
			drill_opt->owner[l] = '.';
			drill_opt->owner[l + 1] = '\0';
		}
	}

	/* setup stuff */
	(void)srandom(randtime.tv_usec); 
	(void)fill_root();

	if (drill_opt->owner) {
		q_rdata = rdata_create((uint8_t *)drill_opt->owner, strlen(drill_opt->owner));
		q = rr_create(q_rdata, drill_opt->type, DEF_TTL, SEC_QUESTION); 
		rdata_destroy(q_rdata);
	}
	
	rr_set_class(drill_opt->class, q);

	switch(drill_opt->purpose) {
	case FROMFILE:
		result = sendfromfile(drill_opt->filename, nameserver);
		return result;
	case ANSWERFROMFILE:
		result = query(NULL, nameserver, protocol);
		return result;
	case TRACE:
		/* nameserver is discarded here */

		/* prime the root-servers - dig does this too */
		root_servers = prime_root_ns(PRINT);

		printf("\n");
		if (drill_opt->dnssec == 0)  {
			verbose("%s", "TRACE");
			q = do_trace(q, protocol, PRINT);
		} else {
			verbose("%s", "SECURE TRACE");
			q = do_trace_secure(q, protocol, PRINT);
		}
		
		if (q) {
			verbose("%s", "Answer received:");
			printf("\n");
			print_rr(q, FOLLOW);
			result = EXIT_SUCCESS;
		} else {
			verbose("%s", "Not found:");
			result = EXIT_FAILURE;
		}
		break;
	case CHASE:
		result = do_chase_name(drill_opt->owner, (uint16_t) drill_opt->type,
				nameserver, trusted_keys, protocol);
		if (result == RET_SUC) {
			verbose("%s", "Chase successful");
			result = EXIT_SUCCESS;
		} else {
			verbose("%s", "Chase failed");
			result = EXIT_FAILURE;
		}
		break;
	case NSEC_WALK:
		result = nsec_walk(drill_opt->owner, nameserver, protocol);
		break;
	default:
		/* MIEK TESTING - prime the root-server - just for fun */
		/* 
		root_servers = prime_root_ns(PRINT);
		*/
		
		result = query(q, nameserver,  protocol);
		break;
	}
	
	if (q)
		rr_destroy(q, FOLLOW);

	if (nameserver) {
		rr_destroy(nameserver, FOLLOW);
	}
	rr_destroy(root_servers, FOLLOW);
	drill_opt_destroy();
	zone_list_destroy(secure_zones);
	zone_list_destroy(search_zones);
	if (trusted_keys) {
		rr_destroy(trusted_keys, FOLLOW);
	}
	show_mem();
	return result;
}
