/**
 * dnstypes.c
 * conversion and lookup function
 * for text type codes to numeric. etc
 *
 * (c) 2004 NLnet Labs
 *
 * See the file LICENSE for the license
 */

#include "common.h"

int have_drill_opt = 0;
/**
 * options descr
 */
 struct t_opt *drill_opt;
/**
 * ztypes descr
 */
struct gtab ztypes[] = Z_TYPES;
struct gtab zclasses[] = Z_CLASSES;
struct gtab rcodes[] = R_CODES;

/* XXX 256!! */
#ifndef S_SPLINT_S
int rr_rdata_types[65535][MAX_RD_COUNT] = RR_RDATA_TYPES;
#endif

/**
 * Returns up the numeric value of the symbol, returns 0 if not found
 */
uint16_t
intbyname(const char *a, struct gtab *tab)
{
        while (tab->name != NULL) {
                if (strcasecmp(a, tab->name) == 0)
                        return tab->sym;
                tab++;
        }
        return 0;
}

/**
 * Returns up the numeric value of the symbol, returns 0 if not found
 * return XXXX for TYPEXXXX unknown stuff
 */
uint16_t
typebyname(const char *a, struct gtab *tab)
{
	uint16_t type = 0;
        while (tab->name != NULL) {
                if (strcasecmp(a, tab->name) == 0)
                        return tab->sym;
                tab++;
        }
	if (strncmp(a, "TYPE", 4) != 0)
		type = (uint16_t) atoi(a + 4);
        return type;
}

/**
 *  Returns up the string value of the symbol, returns NULL if not found.
 */
const char *
namebyint(uint16_t n, struct gtab *tab)
{
	while (tab->sym != 0) {
		if (tab->sym == n)
			return tab->name;
		tab++;
	}
	return "<unknown>";
}

/**
 * Returns the type. This is a wrapper around namebyint, which handles
 * unknown RRs
 * Don't forget this malloc's data, which you have to free
 */
char *
typebyint(uint16_t n, struct gtab *tab)
{
	char *typestr = xmalloc(10);
	
	memset(typestr, 0, 10);
	while (tab->sym != 0) {
		if (tab->sym == n) {
			strncpy(typestr, tab->name, 10);
			return typestr;
		}
		tab++;
	}
	/* we haven't found it in our type db - assume it's unknown */
	snprintf(typestr, 9, "TYPE%d", (int) n);
	return typestr;
}

/**
 * Returns the class. This is a wrapper around namebyint, which handles
 * unknown classes
 */
const char *
classbyint(uint16_t n, struct gtab *tab)
{
	char *unknown;
	while (tab->sym != 0) {
		if (tab->sym == n)
			return tab->name;
		tab++;
	}
	/* we haven't found it in our type db - assume it's unknown */
	unknown = xmalloc(11); /* CLASS XXXXX */
	snprintf(unknown, 10, "CLASS%d", (int) n);
	return (const char*)unknown;
}

/**
 * Determines the type of a packet. A packet can be
 * delegation
 * answer (in there we have nodata)
 * nxdomain
 */
uint8_t 
dpacket_type(struct t_dpacket *p)
{
	/* look inside a packet to determine what kind of an
	 * answer we get
	 */

	/* no assert here, return an error code */
	if (!p) 
		return PKT_ERROR;

	/* check for NXDOMAIN opcode */
	
	/* from here on the opcode needs to be noerror */

	/* nodata respons 
	 * auth is filled with SOA, rest is zero 
	 */
	if (p->count[SEC_ANSWER] == 0 && p->count[SEC_ADD] == 0
			&& p->count[SEC_AUTH] == 1) {
		if (dpacket_get_rrset(NULL, intbyname("soa", ztypes), p, SEC_AUTH)) {
			/* there is a SOA present */
			return PKT_NODATA;
		}
	}

	/* this should be a referral
	 * auth is filled with NS
	 * add has some (unimportant) data
	 */
	if (p->count[SEC_ANSWER] == 0 && p->count[SEC_AUTH] > 0) {
		if (dpacket_get_rrset(NULL, TYPE_NS, p, SEC_AUTH)) {
			/* there are nameserver records in there */
			return PKT_REFERRAL;
		}
	}

	/* if we are still alive here, it's an answer */
	return PKT_ANSWER;
}	
