/* 
 * dnssec.h 
 *
 * contains defines useful in a dnssec world
 *
 * (c) NLnet Labs
 *
 * See the file LICENSE for the license
 *
 */

#include "common.h"

struct gtab dnssec_algos[] = ALG_TYPES;
struct gtab cert_types[] = CERT_TYPES;

/**
 * Creates a zone_list structure
 */
struct zone_list *
zone_list_create()
{
	struct zone_list *sz = xmalloc(sizeof(struct zone_list));
	sz->size = 0;
	sz->zones = xmalloc(sizeof(char *)*MAX_SEC_ZONES);
	return sz;
}

/**
 * Frees the allocated memory for the zone_list structure
 */
void
zone_list_destroy(struct zone_list *sz)
{
	int i;
	for (i = 0; i < sz->size; i++) {
		xfree(sz->zones[i]);
        }
	
	xfree(sz->zones);
	xfree(sz);
}

/**
 * Adds a zone to the zone_list structure
 */
int 
add_zone_list(struct zone_list *sz, const char *zonename)
{
	if (sz->size < MAX_SEC_ZONES) {
		sz->zones[sz->size] = xstrdup(zonename);
		sz->size++;
	}
	
	return RET_SUC;
}

/* TODO: test this */
/**
 * Removes a zone from a zone_list structure
 */
int
remove_zone_list(struct zone_list *sz, const char *zonename)
{
	int i;
	
	for (i = 0; i < sz->size; i++) {
		/* 256 max length of zonename? */
		if (strncmp(sz->zones[i], zonename, 256) == 0) {
			for (; i<sz->size-1; i++) {
				xfree(sz->zones[i]);
				sz->zones[i] = sz->zones[i+1];
			}
			sz->zones[i+1] = NULL;
			sz->size--;
			return RET_SUC;
		}
	}
	
	return RET_FAIL;
}

/**
 * Returns true if the given zone is present in the zone_list structure
 */
int
contains_zone_list(struct zone_list *sz, const char *zonename)
{
	int i;
	size_t j;
	if (sz != NULL) {
		for (i = 0; i < sz->size; i++) {
			/* 256 max length of zonename? */
			for (j = 0; j < strlen(zonename) && j < 256; j++) {
				if (strncmp(sz->zones[i], &zonename[j], 256) == 0) {
					return 1;
				}
			}
		}
	}
	return 0;
}

/**
 * Prints the zones in the zone_list structure to stdout
 */
void
print_zone_list(struct zone_list *sz)
{
	int i;
	if (sz != NULL) {
		for (i=0; i<sz->size; i++) {
			printf("%s\n", sz->zones[i]);
		}
	}

}

/* RRSIG helper functions */

/**
 * Returns the signer's name of the signature
 */
struct t_rdata *
get_signer_name(struct t_rr *sig) 
{
	assert(sig != NULL);
	assert(sig->type == TYPE_RRSIG);
	return (sig->rdata[7]);
}

/**
 * Returns the keytag of the key that created the signature
 */
uint16_t 
get_keyid(struct t_rr *sig)
{
	assert(sig != NULL);
	assert(sig->type == TYPE_RRSIG);
	return (rdata2uint16(sig->rdata[6]));
}

/**
 * Returns the RR type the rrsig covers
 */
uint16_t 
get_typecovered(struct t_rr *sig)
{
	assert(sig != NULL);
	assert(sig->type == TYPE_RRSIG);
	return (rdata2uint16(sig->rdata[0]));
}

/**
 * Returns the signature from a packet belonging to the rrs
 * returns NULL if nothing is found
 */
struct t_rr *
dpacket_get_rrsig(struct t_rr *rr, struct t_dpacket *p)
{
        struct t_rr *tmprr;
        struct t_rr *signature;
        struct t_rr *orig = NULL; 
        int firstpass = 1; /* yech */
        char *rr_name;
        char *tmp_name;

        /* section is in the rrs 
         * type is in there too
         * loop through the packet and pick out the right one
         */
	assert(rr != NULL); 
	assert(p != NULL);

	rr_name = rdata2str(rr->name);
        
        tmprr = p->rrs;
        while(tmprr) { 
                tmp_name = rdata2str(tmprr->name);
                /* is section is SEC_QUESTION the rr does NOT have rdata */
                if (tmprr->type == TYPE_RRSIG &&  /* it's a SIG */
                                tmprr->section != SEC_QUESTION &&  /* it has rdata */
                                strcmp(rr_name, tmp_name) == 0 &&
                                get_typecovered(tmprr) == rr->type
                ) { /* it covers the right type */
                        if (firstpass) {
                                signature = rr_clone(tmprr, NO_FOLLOW);
                                orig = signature;
                                firstpass = 0;
                        } else {
                                signature->next = rr_clone(tmprr, NO_FOLLOW);
                                signature = signature->next;
                        }
                }
                xfree(tmp_name);
                tmprr = tmprr->next;
        }
        xfree(rr_name);
        return orig;
}
