/* GDA library
 * Copyright (C) 1998 - 2007 The GNOME Foundation.
 *
 * AUTHORS:
 *      Michael Lausch <michael@lausch.at>
 *	Rodrigo Moya <rodrigo@gnome-db.org>
 *	Bas Driessen <bas.driessen@xobas.com>
 *      Vivien Malerba <malerba@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <libgda/gda-client.h>
#include <libgda/gda-config.h>
#include <libgda/gda-connection.h>
#include <libgda/gda-connection-private.h>
#include <libgda/gda-connection-event.h>
#include <glib/gi18n-lib.h>
#include <libgda/gda-dict.h>
#include <libgda/gda-log.h>
#include <libgda/gda-server-provider.h>
#include <libgda/gda-parameter-list.h>
#include "gda-marshal.h"
#include <libgda/gda-transaction-status-private.h>
#include <string.h>
#include <libgda/sql-transaction/gda-sql-transaction-parser.h>
#include <libgda/sql-transaction/gda-sql-transaction-tree.h> /* For gda_sql_transaction_destroy(). */
#include <libgda/gda-enum-types.h>

#define PARENT_TYPE G_TYPE_OBJECT

struct _GdaConnectionPrivate {
	GdaClient            *client;
	GdaServerProvider    *provider_obj;
	GdaConnectionOptions  options; /* ORed flags */
	gchar                *dsn;
	gchar                *cnc_string;
	gchar                *username;
	gchar                *password;
	gboolean              is_open;
	GList                *events_list;
	GList                *recset_list;

	GdaTransactionStatus *trans_status;
	GHashTable           *prepared_stmts;
};

static void gda_connection_class_init (GdaConnectionClass *klass);
static void gda_connection_init       (GdaConnection *cnc, GdaConnectionClass *klass);
static void gda_connection_dispose    (GObject *object);
static void gda_connection_finalize   (GObject *object);
static void gda_connection_set_property (GObject *object,
					 guint param_id,
					 const GValue *value,
					 GParamSpec *pspec);
static void gda_connection_get_property (GObject *object,
					 guint param_id,
					 GValue *value,
					 GParamSpec *pspec);

enum {
	ERROR,
	CONN_OPENED,
        CONN_TO_CLOSE,
        CONN_CLOSED,
	DSN_CHANGED,
	TRANSACTION_STATUS_CHANGED,
	LAST_SIGNAL
};

static gint gda_connection_signals[LAST_SIGNAL] = { 0, 0, 0, 0, 0, 0 };

/* properties */
enum
{
        PROP_0,
	PROP_CLIENT,
        PROP_DSN,
        PROP_CNC_STRING,
        PROP_PROVIDER_OBJ,
        PROP_USERNAME,
        PROP_PASSWORD,
        PROP_OPTIONS,
};

static GObjectClass *parent_class = NULL;

/*
 * GdaConnection class implementation
 * @klass:
 */
static void
gda_connection_class_init (GdaConnectionClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	gda_connection_signals[ERROR] =
		g_signal_new ("error",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GdaConnectionClass, error),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__OBJECT,
			      G_TYPE_NONE, 1, GDA_TYPE_CONNECTION_EVENT);
	gda_connection_signals[CONN_OPENED] =
                g_signal_new ("conn_opened",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GdaConnectionClass, conn_opened),
                              NULL, NULL,
                              gda_marshal_VOID__VOID,
                              G_TYPE_NONE, 0);
        gda_connection_signals[CONN_TO_CLOSE] =
                g_signal_new ("conn_to_close",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GdaConnectionClass, conn_to_close),
                              NULL, NULL,
                              gda_marshal_VOID__VOID,
                              G_TYPE_NONE, 0);
        gda_connection_signals[CONN_CLOSED] =    /* runs after user handlers */
                g_signal_new ("conn_closed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GdaConnectionClass, conn_closed),
                              NULL, NULL,
                              gda_marshal_VOID__VOID,
                              G_TYPE_NONE, 0);
	gda_connection_signals[DSN_CHANGED] =
		g_signal_new ("dsn_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GdaConnectionClass, dsn_changed),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);
	gda_connection_signals[TRANSACTION_STATUS_CHANGED] =
		g_signal_new ("transaction_status_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GdaConnectionClass, transaction_status_changed),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);

	/* Properties */
        object_class->set_property = gda_connection_set_property;
        object_class->get_property = gda_connection_get_property;

	g_object_class_install_property (object_class, PROP_CLIENT,
                                         g_param_spec_object ("client", _("GdaClient to use"), NULL,
                                                               GDA_TYPE_CLIENT,
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property (object_class, PROP_DSN,
                                         g_param_spec_string ("dsn", _("DSN to use"), NULL, NULL,
							      (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property (object_class, PROP_CNC_STRING,
                                         g_param_spec_string ("cnc_string", _("Connection string to use"), NULL, NULL,
							      (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property (object_class, PROP_PROVIDER_OBJ,
                                         g_param_spec_object ("provider_obj", _("Provider to use"), NULL,
                                                               GDA_TYPE_SERVER_PROVIDER,
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));

        g_object_class_install_property (object_class, PROP_USERNAME,
                                         g_param_spec_string ("username", _("Username to use"),
                                                              NULL, NULL,
                                                              (G_PARAM_READABLE | G_PARAM_WRITABLE)));
        g_object_class_install_property (object_class, PROP_PASSWORD,
                                         g_param_spec_string ("password", _("Password to use"),
                                                              NULL, NULL,
                                                              (G_PARAM_READABLE | G_PARAM_WRITABLE)));
        g_object_class_install_property (object_class, PROP_OPTIONS,
                                         g_param_spec_flags ("options", _("Options (connection sharing)"),
							    NULL, GDA_TYPE_CONNECTION_OPTIONS, GDA_CONNECTION_OPTIONS_NONE,
							    (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	
	object_class->dispose = gda_connection_dispose;
	object_class->finalize = gda_connection_finalize;
}

static void
gda_connection_init (GdaConnection *cnc, GdaConnectionClass *klass)
{
	g_return_if_fail (GDA_IS_CONNECTION (cnc));

	cnc->priv = g_new0 (GdaConnectionPrivate, 1);
	cnc->priv->client = NULL;
	cnc->priv->provider_obj = NULL;
	cnc->priv->dsn = NULL;
	cnc->priv->cnc_string = NULL;
	cnc->priv->username = NULL;
	cnc->priv->password = NULL;
	cnc->priv->is_open = FALSE;
	cnc->priv->events_list = NULL;
	cnc->priv->recset_list = NULL;
	cnc->priv->trans_status = NULL; /* no transaction yet */
}

static void
gda_connection_dispose (GObject *object)
{
	GdaConnection *cnc = (GdaConnection *) object;

	g_return_if_fail (GDA_IS_CONNECTION (cnc));

	/* free memory */
	gda_connection_close_no_warning (cnc);

	gda_connection_destroy_prepared_statement_hash (cnc);

	if (cnc->priv->provider_obj) {
		g_object_unref (G_OBJECT (cnc->priv->provider_obj));
		cnc->priv->provider_obj = NULL;
	}

	if (cnc->priv->events_list)
		gda_connection_event_list_free (cnc->priv->events_list);

	if (cnc->priv->recset_list)
		g_list_foreach (cnc->priv->recset_list, (GFunc) g_object_unref, NULL);
	
	if (cnc->priv->trans_status) {
		g_object_unref (cnc->priv->trans_status);
		cnc->priv->trans_status = NULL;
	}

        if (cnc->priv->client) {
		g_object_unref (cnc->priv->client);
		cnc->priv->client = NULL;
	}

	/* chain to parent class */
	parent_class->dispose (object);
}

static void
gda_connection_finalize (GObject *object)
{
	GdaConnection *cnc = (GdaConnection *) object;

	g_return_if_fail (GDA_IS_CONNECTION (cnc));

	/* free memory */
	g_free (cnc->priv->dsn);
	g_free (cnc->priv->cnc_string);
	g_free (cnc->priv->username);
	g_free (cnc->priv->password);

	g_free (cnc->priv);
	cnc->priv = NULL;

	/* chain to parent class */
	parent_class->finalize (object);
}

/* module error */
GQuark gda_connection_error_quark (void)
{
        static GQuark quark;
        if (!quark)
                quark = g_quark_from_static_string ("gda_connection_error");
        return quark;
}

/**
 * gda_connection_get_type
 * 
 * Registers the #GdaConnection class on the GLib type system.
 * 
 * Returns: the GType identifying the class.
 */
GType
gda_connection_get_type (void)
{
	static GType type = 0;

	if (G_UNLIKELY (type == 0)) {
		static GTypeInfo info = {
			sizeof (GdaConnectionClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gda_connection_class_init,
			NULL, NULL,
			sizeof (GdaConnection),
			0,
			(GInstanceInitFunc) gda_connection_init
		};
		type = g_type_register_static (PARENT_TYPE, "GdaConnection", &info, 0);
	}

	return type;
}

static void
gda_connection_set_property (GObject *object,
			     guint param_id,
			     const GValue *value,
			     GParamSpec *pspec)
{
	GdaConnection *cnc;

        cnc = GDA_CONNECTION (object);
        if (cnc->priv) {
                switch (param_id) {
                case PROP_CLIENT:
                        if (cnc->priv->client)
				g_object_unref(cnc->priv->client);

			cnc->priv->client = g_value_get_object (value);
			if (cnc->priv->client)
				g_object_ref (cnc->priv->client);
			break;
                case PROP_DSN:
			gda_connection_set_dsn (cnc, g_value_get_string (value));
                        break;
                case PROP_CNC_STRING:
			g_free (cnc->priv->cnc_string);
			cnc->priv->cnc_string = NULL;
			if (g_value_get_string (value)) 
				cnc->priv->cnc_string = g_strdup (g_value_get_string (value));
                        break;
                case PROP_PROVIDER_OBJ:
                        if (cnc->priv->provider_obj)
				g_object_unref(cnc->priv->provider_obj);

			cnc->priv->provider_obj = g_value_get_object (value);
			g_object_ref (G_OBJECT (cnc->priv->provider_obj));
                        break;
                case PROP_USERNAME:
			gda_connection_set_username (cnc, g_value_get_string (value));
                        break;
                case PROP_PASSWORD:
			gda_connection_set_password (cnc, g_value_get_string (value));
                        break;
                case PROP_OPTIONS:
			cnc->priv->options = g_value_get_flags (value);
			break;
                }
        }	
}

static void
gda_connection_get_property (GObject *object,
			     guint param_id,
			     GValue *value,
			     GParamSpec *pspec)
{
	GdaConnection *cnc;

        cnc = GDA_CONNECTION (object);
        if (cnc->priv) {
                switch (param_id) {
                case PROP_CLIENT:
			g_value_set_object (value, G_OBJECT (cnc->priv->client));
			break;
                case PROP_DSN:
			g_value_set_string (value, cnc->priv->dsn);
                        break;
                case PROP_CNC_STRING:
			g_value_set_string (value, cnc->priv->cnc_string);
			break;
                case PROP_PROVIDER_OBJ:
			g_value_set_object (value, G_OBJECT (cnc->priv->provider_obj));
                        break;
                case PROP_USERNAME:
			g_value_set_string (value, cnc->priv->username);
                        break;
                case PROP_PASSWORD:
			g_value_set_string (value, cnc->priv->password);
                        break;
                case PROP_OPTIONS:
			g_value_set_flags (value, cnc->priv->options);
			break;
                }
        }	
}


/**
 * gda_connection_new
 * @client: a #GdaClient object.
 * @provider: a #GdaServerProvider object.
 * @dsn: GDA data source to connect to.
 * @username: user name to use to connect.
 * @password: password for @username.
 * @options: options for the connection.
 *
 * This function creates a new #GdaConnection object. It is not
 * intended to be used directly by applications (use
 * #gda_client_open_connection instead).
 *
 * The connection is not opened at this stage; use 
 * gda_connection_open() to open the connection.
 *
 * Returns: a newly allocated #GdaConnection object.
 */
GdaConnection *
gda_connection_new (GdaClient *client,
		    GdaServerProvider *provider,
		    const gchar *dsn,
		    const gchar *username,
		    const gchar *password,
		    GdaConnectionOptions options)
{
	GdaConnection *cnc;

	g_return_val_if_fail (GDA_IS_CLIENT (client), NULL);
	g_return_val_if_fail (GDA_IS_SERVER_PROVIDER (provider), NULL);

	cnc = g_object_new (GDA_TYPE_CONNECTION, "client", client, "provider_obj", provider, 
			    "dsn", dsn, 
			    "username", username, 
			    "password", password, 
			    "options", options, NULL);
	return cnc;
}

/**
 * gda_connection_open
 * @cnc: a #GdaConnection object
 * @error: a place to store errors, or %NULL
 *
 * Tries to open the connection.
 *
 * Returns: TRUE if the connection is opened, and FALSE otherwise.
 */
gboolean
gda_connection_open (GdaConnection *cnc, GError **error)
{
	GdaDataSourceInfo *dsn_info = NULL;
	GdaQuarkList *params;
	char *real_username = NULL;
	char *real_password = NULL;

	g_return_val_if_fail (cnc && GDA_IS_CONNECTION (cnc), FALSE);
	g_return_val_if_fail (cnc->priv, FALSE);

	/* don't do anything if connection is already opened */
	if (cnc->priv->is_open)
		return TRUE;

	/* connection string */
	if (cnc->priv->dsn) {
		/* get the data source info */
		dsn_info = gda_config_find_data_source (cnc->priv->dsn);
		if (!dsn_info) {
			gda_log_error (_("Data source %s not found in configuration"), cnc->priv->dsn);
			g_set_error (error, GDA_CONNECTION_ERROR, GDA_CONNECTION_NONEXIST_DSN_ERROR,
				     _("Data source %s not found in configuration"), cnc->priv->dsn);
			return FALSE;
		}

		g_free (cnc->priv->cnc_string);
		cnc->priv->cnc_string = g_strdup (dsn_info->cnc_string);
	}
	else {
		if (!cnc->priv->cnc_string) {
			gda_log_error (_("No DSN or connection string specified"));
			g_set_error (error, GDA_CONNECTION_ERROR, GDA_CONNECTION_NO_CNC_SPEC_ERROR,
				     _("No DSN or connection string specified"));
			return FALSE;
		}
		/* try to see if connection string has the <provider>://<rest of the string> format */
	}

	/* provider test */
	if (!cnc->priv->provider_obj) {
		gda_log_error (_("No provider specified"));
		g_set_error (error, GDA_CONNECTION_ERROR, GDA_CONNECTION_NO_PROVIDER_SPEC_ERROR,
			     _("No provider specified"));
		return FALSE;
	}

	params = gda_quark_list_new_from_string (cnc->priv->cnc_string);

	/* retrieve correct username/password */
	if (cnc->priv->username)
		real_username = g_strdup (cnc->priv->username);
	else {
		if (dsn_info && dsn_info->username)
			real_username = g_strdup (dsn_info->username);
		else {
			const gchar *s;
			s = gda_quark_list_find (params, "USER");
			if (s) {
				real_username = g_strdup (s);
				gda_quark_list_remove (params, "USER");
			}
		}
	}

	if (cnc->priv->password)
		real_password = g_strdup (cnc->priv->password);
	else {
		if (dsn_info && dsn_info->password)
			real_password = g_strdup (dsn_info->password);
		else {
			const gchar *s;
			s = gda_quark_list_find (params, "PASSWORD");
			if (s) {
				real_password = g_strdup (s);
				gda_quark_list_remove (params, "PASSWORD");
			}
		}
	}

	/* try to open the connection */
	if (gda_server_provider_open_connection (cnc->priv->provider_obj, cnc, params,
						 real_username, real_password)) {
		cnc->priv->is_open = TRUE;
		gda_client_notify_connection_opened_event (cnc->priv->client, cnc);
	}
	else {
		const GList *events;
		
		events = gda_connection_get_events (cnc);
		if (events) {
			GList *l;

			for (l = (GList *) events; l != NULL; l = l->next) {
				GdaConnectionEvent *event;

				event = GDA_CONNECTION_EVENT (l->data);
				if (gda_connection_event_get_event_type (event) == GDA_CONNECTION_EVENT_ERROR) {
					if (error && !(*error))
						g_set_error (error, GDA_CONNECTION_ERROR, GDA_CONNECTION_OPEN_ERROR,
							     gda_connection_event_get_description (event));
					gda_client_notify_error_event (cnc->priv->client, cnc, 
								       GDA_CONNECTION_EVENT (l->data));
				}
			}
		}

		cnc->priv->is_open = FALSE;
	}

	/* free memory */
	if (dsn_info)
		gda_data_source_info_free (dsn_info);
	gda_quark_list_free (params);
	g_free (real_username);
	g_free (real_password);

	if (cnc->priv->is_open) {
#ifdef GDA_DEBUG_signal
        g_print (">> 'CONN_OPENED' from %s\n", __FUNCTION__);
#endif
        g_signal_emit (G_OBJECT (cnc), gda_connection_signals[CONN_OPENED], 0);
#ifdef GDA_DEBUG_signal
        g_print ("<< 'CONN_OPENED' from %s\n", __FUNCTION__);
#endif
	}

	return cnc->priv->is_open;
}


/**
 * gda_connection_close
 * @cnc: a #GdaConnection object.
 *
 * Closes the connection to the underlying data source, but first emits the 
 * "conn_to_close" signal.
 */
void
gda_connection_close (GdaConnection *cnc)
{
	g_return_if_fail (GDA_IS_CONNECTION (cnc));
	g_return_if_fail (cnc->priv);

	if (! cnc->priv->is_open)
		return;

#ifdef GDA_DEBUG_signal
        g_print (">> 'CONN_TO_CLOSE' from %s\n", __FUNCTION__);
#endif
        g_signal_emit (G_OBJECT (cnc), gda_connection_signals[CONN_TO_CLOSE], 0);
#ifdef GDA_DEBUG_signal
        g_print ("<< 'CONN_TO_CLOSE' from %s\n", __FUNCTION__);
#endif

        gda_connection_close_no_warning (cnc);
}

/**
 * gda_connection_close_no_warning
 * @cnc: a #GdaConnection object.
 *
 * Closes the connection to the underlying data source, without emiting any warning signal.
 */
void
gda_connection_close_no_warning (GdaConnection *cnc)
{
	g_return_if_fail (GDA_IS_CONNECTION (cnc));
	g_return_if_fail (cnc->priv);

	if (! cnc->priv->is_open)
		return;

	gda_server_provider_close_connection (cnc->priv->provider_obj, cnc);
	gda_client_notify_connection_closed_event (cnc->priv->client, cnc);
	cnc->priv->is_open = FALSE;

#ifdef GDA_DEBUG_signal
        g_print (">> 'CONN_CLOSED' from %s\n", __FUNCTION__);
#endif
        g_signal_emit (G_OBJECT (cnc), gda_connection_signals[CONN_CLOSED], 0);
#ifdef GDA_DEBUG_signal
        g_print ("<< 'CONN_CLOSED' from %s\n", __FUNCTION__);
#endif
}


/**
 * gda_connection_is_opened
 * @cnc: a #GdaConnection object.
 *
 * Checks whether a connection is open or not.
 *
 * Returns: %TRUE if the connection is open, %FALSE if it's not.
 */
gboolean
gda_connection_is_opened (GdaConnection *cnc)
{
        g_return_val_if_fail (GDA_IS_CONNECTION (cnc), FALSE);

	return cnc->priv->is_open;
}

/**
 * gda_connection_get_client
 * @cnc: a #GdaConnection object.
 *
 * Gets the #GdaClient object associated with a connection. This
 * is always the client that created the connection, as returned
 * by #gda_client_open_connection.
 *
 * Returns: the client to which the connection belongs to.
 */
GdaClient *
gda_connection_get_client (GdaConnection *cnc)
{
	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), NULL);
	g_return_val_if_fail (cnc->priv, NULL);

	return cnc->priv->client;
}

/**
 * gda_connection_get_options
 * @cnc: a #GdaConnection object.
 *
 * Gets the #GdaConnectionOptions used to open this connection.
 *
 * Returns: the connection options.
 */
GdaConnectionOptions
gda_connection_get_options (GdaConnection *cnc)
{
	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), -1);
	g_return_val_if_fail (cnc->priv, -1);

	return cnc->priv->options;
}

/**
 * gda_connection_get_provider_obj
 * @cnc: a #GdaConnection object
 *
 * Get a pointer to the #GdaServerProvider object used to access the database
 *
 * Returns: the #GdaServerProvider (NEVER NULL)
 */
GdaServerProvider *
gda_connection_get_provider_obj (GdaConnection *cnc)
{
	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), NULL);
	g_return_val_if_fail (cnc->priv, NULL);

	return cnc->priv->provider_obj;
}

/**
 * gda_connection_get_infos
 * @cnc: a #GdaConnection object
 *
 * Get a pointer to a #GdaServerProviderInfo structure (which must not be modified)
 * to retreive specific information about the provider used by @cnc.
 */
GdaServerProviderInfo *
gda_connection_get_infos (GdaConnection *cnc)
{
	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), NULL);
	g_return_val_if_fail (cnc->priv, NULL);
	if (!cnc->priv->provider_obj)
		return NULL;

	return gda_server_provider_get_info (cnc->priv->provider_obj, cnc);
}

/**
 * gda_connection_get_server_version
 * @cnc: a #GdaConnection object.
 *
 * Gets the version string of the underlying database server.
 *
 * Returns: the server version string.
 */
const gchar *
gda_connection_get_server_version (GdaConnection *cnc)
{
	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), NULL);
	g_return_val_if_fail (cnc->priv, NULL);
	if (!cnc->priv->provider_obj)
		return NULL;

	return gda_server_provider_get_server_version (cnc->priv->provider_obj, cnc);
}

/**
 * gda_connection_get_database
 * @cnc: A #GdaConnection object.
 *
 * Gets the name of the currently active database in the given
 * @GdaConnection.
 *
 * Returns: the name of the current database.
 */
const gchar *
gda_connection_get_database (GdaConnection *cnc)
{
	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), NULL);
	g_return_val_if_fail (cnc->priv, NULL);
	if (!cnc->priv->provider_obj)
		return NULL;

	return gda_server_provider_get_database (cnc->priv->provider_obj, cnc);
}

/**
 * gda_connection_set_dsn
 * @cnc: a #GdaConnection object
 * @datasource: a gda datasource
 *
 * Sets the data source of the connection. If the connection is already opened,
 * then no action is performed at all and FALSE is returned.
 *
 * If the requested datasource does not exist, then nothing is done and FALSE
 * is returned.
 *
 * Returns: TRUE on success
 */
gboolean
gda_connection_set_dsn (GdaConnection *cnc, const gchar *datasource)
{
	GdaDataSourceInfo *dsn;

        g_return_val_if_fail (cnc && GDA_IS_CONNECTION (cnc), FALSE);
        g_return_val_if_fail (cnc->priv, FALSE);
        g_return_val_if_fail (datasource && *datasource, FALSE);

        if (cnc->priv->is_open)
                return FALSE;

        dsn = gda_config_find_data_source (datasource);
        if (!dsn)
                return FALSE;

	g_free (cnc->priv->dsn);
	cnc->priv->dsn = g_strdup (datasource);
#ifdef GDA_DEBUG_signal
        g_print (">> 'DSN_CHANGED' from %s\n", __FUNCTION__);
#endif
        g_signal_emit (G_OBJECT (cnc), gda_connection_signals[DSN_CHANGED], 0);
#ifdef GDA_DEBUG_signal
        g_print ("<< 'DSN_CHANGED' from %s\n", __FUNCTION__);
#endif

	return TRUE;
}

/**
 * gda_connection_get_dsn
 * @cnc: a #GdaConnection object
 *
 * Returns the data source name the connection object is connected
 * to.
 */
const gchar *
gda_connection_get_dsn (GdaConnection *cnc)
{
	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), NULL);
	g_return_val_if_fail (cnc->priv, NULL);

	return (const gchar *) cnc->priv->dsn;
}

/**
 * gda_connection_get_cnc_string
 * @cnc: a #GdaConnection object.
 *
 * Gets the connection string used to open this connection.
 *
 * The connection string is the string sent over to the underlying
 * database provider, which describes the parameters to be used
 * to open a connection on the underlying data source.
 *
 * Returns: the connection string used when opening the connection.
 */
const gchar *
gda_connection_get_cnc_string (GdaConnection *cnc)
{
	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), NULL);
	g_return_val_if_fail (cnc->priv, NULL);

	return (const gchar *) cnc->priv->cnc_string;
}

/**
 * gda_connection_get_provider
 * @cnc: a #GdaConnection object.
 *
 * Gets the provider id that this connection is connected to.
 *
 * Returns: the provider ID used to open this connection.
 */
const gchar *
gda_connection_get_provider (GdaConnection *cnc)
{
	GdaServerProviderInfo *pinfo = NULL;

	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), NULL);
	g_return_val_if_fail (cnc->priv, NULL);

	if (cnc->priv->provider_obj)
		pinfo = gda_server_provider_get_info (cnc->priv->provider_obj, NULL);
	if (pinfo)
		return(const gchar *) pinfo-> provider_name;
	else
		return NULL;
}

/**
 * gda_connection_set_username
 * @cnc: a #GdaConnection object
 * @username:
 *
 * Sets the user name for the connection. If the connection is already opened,
 * then no action is performed at all and FALSE is returned.
 *
 * Returns: TRUE on success
 */
gboolean
gda_connection_set_username (GdaConnection *cnc, const gchar *username)
{
	g_return_val_if_fail (cnc && GDA_IS_CONNECTION (cnc), FALSE);
        g_return_val_if_fail (cnc->priv, FALSE);

        if (cnc->priv->is_open)
                return FALSE;

        g_free (cnc->priv->username);
	if (username)
		cnc->priv->username = g_strdup (username);
	else
		cnc->priv->username = NULL;
        return TRUE;
}

/**
 * gda_connection_get_username
 * @cnc: a #GdaConnection object.
 *
 * Gets the user name used to open this connection.
 *
 * Returns: the user name.
 */
const gchar *
gda_connection_get_username (GdaConnection *cnc)
{
	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), NULL);
	g_return_val_if_fail (cnc->priv, NULL);

	return (const gchar *) cnc->priv->username;
}

/**
 * gda_connection_set_password
 * @cnc: a #GdaConnection object
 * @password:
 *
 * Sets the user password for the connection to the server. If the connection is already opened,
 * then no action is performed at all and FALSE is returned.
 *
 * Returns: TRUE on success
 */
gboolean
gda_connection_set_password (GdaConnection *cnc, const gchar *password)
{
	g_return_val_if_fail (cnc && GDA_IS_CONNECTION (cnc), FALSE);
        g_return_val_if_fail (cnc->priv, FALSE);

        if (cnc->priv->is_open)
                return FALSE;

        g_free (cnc->priv->password);
	if (password)
		cnc->priv->password = g_strdup (password);
	else
		cnc->priv->password = NULL;

        return TRUE;
}

/**
 * gda_connection_get_password
 * @cnc: a #GdaConnection object.
 *
 * Gets the password used to open this connection.
 *
 * Returns: the password.
 */
const gchar *
gda_connection_get_password (GdaConnection *cnc)
{
	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), NULL);
	g_return_val_if_fail (cnc->priv, NULL);

	return (const gchar *) cnc->priv->password;
}

/**
 * gda_connection_add_event
 * @cnc: a #GdaConnection object.
 * @event: is stored internally, so you don't need to unref it.
 *
 * Adds an event to the given connection. This function is usually
 * called by providers, to inform clients of events that happened
 * during some operation.
 *
 * As soon as a provider (or a client, it does not matter) calls this
 * function with an @event object which is an error,
 * the connection object (and the associated #GdaClient object)
 * emits the "error" signal, to which clients can connect to be
 * informed of events.
 *
 * WARNING: the reference to the @event object is stolen by this function!
 */
void
gda_connection_add_event (GdaConnection *cnc, GdaConnectionEvent *event)
{
	static gint debug = -1;
	g_return_if_fail (GDA_IS_CONNECTION (cnc));
	g_return_if_fail (cnc->priv);
	g_return_if_fail (GDA_IS_CONNECTION_EVENT (event));

	if (debug == -1) {
		const gchar *str;
		debug = 0;
		str = getenv ("GDA_CONNECTION_EVENTS_SHOW");
		if (str) {
			gchar **array;
			gint i;
			array = g_strsplit_set (str, " ,/;:", 0);
			for (i = 0; i < g_strv_length (array); i++) {
				if (!g_ascii_strcasecmp (array[i], "notice"))
					debug += 1;
				else if (!g_ascii_strcasecmp (array[i], "warning"))
					debug += 2;
				else if (!g_ascii_strcasecmp (array[i], "error"))
					debug += 4;
				else if (!g_ascii_strcasecmp (array[i], "command"))
					debug += 8;
			}
			g_strfreev (array);
		}
	}

	cnc->priv->events_list = g_list_append (cnc->priv->events_list, event);

	if (debug > 0) {
		const gchar *str = NULL;
		switch (gda_connection_event_get_event_type (event)) {
		case GDA_CONNECTION_EVENT_NOTICE:
			if (debug & 1) str = "NOTICE";
			break;
		case GDA_CONNECTION_EVENT_WARNING:
			if (debug & 2) str = "WARNING";
			break;
		case GDA_CONNECTION_EVENT_ERROR:
			if (debug & 4) str = "ERROR";
			break;
		case GDA_CONNECTION_EVENT_COMMAND:
			if (debug & 8) str = "COMMAND";
			break;
		default:
			break;
		}
		if (str)
			g_print ("EVENT> %s: %s (on cnx %p, %s)\n", str, 
				 gda_connection_event_get_description (event), cnc,
				 gda_connection_event_get_sqlstate (event));
	}

	if (gda_connection_event_get_event_type (event) == GDA_CONNECTION_EVENT_ERROR)
		g_signal_emit (G_OBJECT (cnc), gda_connection_signals[ERROR], 0, event);
}

/**
 * gda_connection_add_event_string
 * @cnc: a #GdaConnection object.
 * @str: a format string (see the printf(3) documentation).
 * @...: the arguments to insert in the error message.
 *
 * Adds a new error to the given connection object. This is just a convenience
 * function that simply creates a #GdaConnectionEvent and then calls
 * #gda_server_connection_add_error.
 *
 * Returns: a new #GdaConnectionEvent object, however the caller does not hold a reference to the returned
 * object, and if need be the caller must call g_object_ref() on it.
 */
GdaConnectionEvent *
gda_connection_add_event_string (GdaConnection *cnc, const gchar *str, ...)
{
	GdaConnectionEvent *error;

	va_list args;
	gchar sz[2048];

	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), NULL);
	g_return_val_if_fail (cnc->priv, NULL);
	g_return_val_if_fail (str != NULL, NULL);

	/* build the message string */
	va_start (args, str);
	vsprintf (sz, str, args);
	va_end (args);
	
	error = gda_connection_event_new (GDA_CONNECTION_EVENT_ERROR);
	gda_connection_event_set_description (error, sz);
	gda_connection_event_set_code (error, -1);
	gda_connection_event_set_source (error, gda_connection_get_provider (cnc));
	gda_connection_event_set_sqlstate (error, "-1");
	
	gda_connection_add_event (cnc, error);

	return error;
}

/** TODO: This actually frees the input GList. That's very very unusual. murrayc. */

/**
 * gda_connection_add_events_list
 * @cnc: a #GdaConnection object.
 * @events_list: a list of #GdaConnectionEvent.
 *
 * This is just another convenience function which lets you add
 * a list of #GdaConnectionEvent's to the given connection.*
 * As with
 * #gda_connection_add_event and #gda_connection_add_event_string,
 * this function makes the connection object emit the "error"
 * signal for each error event.
 *
 * @events_list is copied to an internal list and freed.
 */
void
gda_connection_add_events_list (GdaConnection *cnc, GList *events_list)
{
	GList *l;

	g_return_if_fail (GDA_IS_CONNECTION (cnc));
	g_return_if_fail (cnc->priv);
	g_return_if_fail (events_list != NULL);

	cnc->priv->events_list = g_list_concat (cnc->priv->events_list, events_list);

	/* notify errors */
	for (l = events_list; l ; l = g_list_next (l))
		if (gda_connection_event_get_event_type (GDA_CONNECTION_EVENT (l->data)) ==
		    GDA_CONNECTION_EVENT_ERROR)
			g_signal_emit (G_OBJECT (cnc), gda_connection_signals[ERROR], 0, l->data);

	g_list_free (events_list);
}

/**
 * gda_connection_clear_events_list
 * @cnc: a #GdaConnection object.
 *
 * This function lets you clear the list of #GdaConnectionEvent's of the
 * given connection. 
 */
void
gda_connection_clear_events_list (GdaConnection *cnc)
{
	g_return_if_fail (GDA_IS_CONNECTION (cnc));
	g_return_if_fail (cnc->priv);
	
	if (cnc->priv->events_list != NULL) {
		gda_connection_event_list_free (cnc->priv->events_list);
		cnc->priv->events_list =  NULL;
	}
}


/**
 * gda_connection_change_database
 * @cnc: a #GdaConnection object.
 * @name: name of database to switch to.
 *
 * Changes the current database for the given connection. This operation
 * is not available in all providers.
 *
 * Returns: %TRUE if successful, %FALSE otherwise.
 */
gboolean
gda_connection_change_database (GdaConnection *cnc, const gchar *name)
{
	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), FALSE);
	g_return_val_if_fail (name != NULL, FALSE);
	if (!cnc->priv->provider_obj)
		return FALSE;

	return gda_server_provider_change_database (cnc->priv->provider_obj, cnc, name);
}

/**
 * gda_connection_execute_command
 * @cnc: a #GdaConnection object.
 * @cmd: a #GdaCommand.
 * @params: parameter list for the commands
 * @error: a place to store an error, or %NULL
 *
 * If you know what to expect from @command (ie if you know it contains a query which will return
 * a data set or a query which will not return a data set) and if @command contains only one query,
 * then you should use
 * gda_connection_execute_select_command() and gda_connection_execute_non_select_command() which are easier
 * to use.
 *
 * This function provides the way to send several commands
 * at once to the data source being accessed by the given
 * #GdaConnection object. The #GdaCommand specified can contain
 * a list of commands in its "text" property (usually a set
 * of SQL commands separated by ';').
 *
 * The return value is a GList of #GdaDataModel's, and #GdaParameterList which you
 * are responsible to free when not needed anymore (and unref the
 * data models and parameter lists when they are not used anymore). See the documentation
 * of gda_server_provider_execute_command() for more information about the returned list.
 *
 * The @params can contain the following parameters:
 * <itemizedlist>
 *   <listitem><para>a "ITER_MODEL_ONLY" parameter of type #G_TYPE_BOOLEAN which, if set to TRUE
 *             will preferably return a data model which can be accessed only using an iterator.</para></listitem>
 *   <listitem><para>a "ITER_CHUNCK_SIZE" parameter of type #G_TYPE_INT which specifies, if "ITER_MODEL_ONLY"
 *             is set to TRUE, how many rows are fetched (and cached) from the database everytime the iterator needs
 *             to access a row for which the data must be fetched from the database. For the providers which support this
 *             setting this parameter to a value greater than one will increase mamory usage but reduce the time spent
 *             to transfer data from the database.</para></listitem>
 * </itemizedlist>
 *
 * Returns: a list of #GdaDataModel and #GdaParameterList or %NULL, as returned by the underlying
 * provider, or %NULL if an error occurred.
 */
GList *
gda_connection_execute_command (GdaConnection *cnc, GdaCommand *cmd,
				GdaParameterList *params, GError **error)
{
	GList *retval, *events;
	gboolean has_error = FALSE;

	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), NULL);
	g_return_val_if_fail (cnc->priv, NULL);
	g_return_val_if_fail (cmd != NULL, NULL);
	g_return_val_if_fail (cnc->priv->provider_obj, NULL);

	/* execute the command on the provider */
	retval = gda_server_provider_execute_command (cnc->priv->provider_obj,
						      cnc, cmd, params);

	/* make an error if necessary */
	events = cnc->priv->events_list;
	while (events && !has_error) {
		if (gda_connection_event_get_event_type (GDA_CONNECTION_EVENT (events->data)) == 
		    GDA_CONNECTION_EVENT_ERROR) {
			g_set_error (error, GDA_CONNECTION_ERROR, GDA_CONNECTION_EXECUTE_COMMAND_ERROR,
				     gda_connection_event_get_description (GDA_CONNECTION_EVENT (events->data)));
			has_error = TRUE;
		}
		events = g_list_next (events);
	}
	if (has_error) {
		GList *list;

		for (list = retval; list; list = list->next)
			if (list->data)
				g_object_unref ((GObject*) list->data);
		g_list_free (retval);
		retval = NULL;
	}

	return retval;
}

/**
 * gda_connection_get_last_insert_id
 * @cnc: a #GdaConnection object.
 * @recset: recordset.
 *
 * Retrieve from the given #GdaConnection the ID of the last inserted row.
 * A connection must be specified, and, optionally, a result set. If not NULL,
 * the underlying provider should try to get the last insert ID for the given result set.
 *
 * Beware however that the interpretation and usage of this value depends on the
 * DBMS type being used, , see the <link linkend="limitations">limitations</link> 
 * of each DBMS for more information.
 *
 * Returns: a string representing the ID of the last inserted row, or NULL
 * if an error occurred or no row has been inserted. It is the caller's
 * reponsibility to free the returned string.
 */
gchar *
gda_connection_get_last_insert_id (GdaConnection *cnc, GdaDataModel *recset)
{
	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), NULL);
	g_return_val_if_fail (cnc->priv, NULL);
	g_return_val_if_fail (cnc->priv->provider_obj, NULL);

	return gda_server_provider_get_last_insert_id (cnc->priv->provider_obj, cnc, recset);
}

/**
 * gda_connection_execute_select_command
 * @cnc: a #GdaConnection object.
 * @cmd: a #GdaCommand.
 * @params: parameter list for the command
 * @error: a place to store an error, or %NULL
 *
 * Executes a selection command on the given connection.
 *
 * This function returns a #GdaDataModel resulting from the SELECT statement, or %NULL
 * if an error occurred.
 *
 * Note that no check is made regarding the actual number of statements in @cmd or if it really contains a SELECT
 * statement. This function is just a convenience function around the gda_connection_execute_command()
 * function. If @cmd contains several statements, the last #GdaDataModel is returned.
 *
 * See the documentation of the gda_connection_execute_command() for information
 * about the @params list of parameters.
 *
 * Returns: a #GdaDataModel containing the data returned by the
 * data source, or %NULL if an error occurred
 */
GdaDataModel *
gda_connection_execute_select_command (GdaConnection *cnc, GdaCommand *cmd,
				       GdaParameterList *params, GError **error)
{
	GList *reclist, *list;
	GdaDataModel *model = NULL;

	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), NULL);
	g_return_val_if_fail (cnc->priv, NULL);
	g_return_val_if_fail (cmd != NULL, NULL);

	reclist = gda_connection_execute_command (cnc, cmd, params, error);
	if (!reclist)
		return NULL;

	for (list = g_list_last (reclist); list && !model; list = list->prev) {
		model = (GdaDataModel *) (g_list_last (reclist)->data);
		if (!GDA_IS_DATA_MODEL (model))
			model = NULL;
	}
	if (model) {
		GdaConnectionEvent *event;
		gchar *str;
		gint nb = gda_data_model_get_n_rows (model);

		event = gda_connection_event_new (GDA_CONNECTION_EVENT_NOTICE);
		if (nb > 1)
			str = g_strdup_printf (_("(%d rows)"), nb);
		else if (nb >= 0)
			str = g_strdup_printf (_("(%d row)"), nb);
		else
			str = g_strdup_printf (_("(unknown number of rows)"));
			
		gda_connection_event_set_description (event, str);
		g_free (str);
		gda_connection_add_event (cnc, event);

		g_object_ref (G_OBJECT (model));
	}

	list = reclist;
	for (list = reclist; list; list = g_list_next (list))
		if (list->data)
			g_object_unref (list->data);
	g_list_free (reclist);

	return model;
}

/**
 * gda_connection_execute_non_select_command
 * @cnc: a #GdaConnection object.
 * @cmd: a #GdaCommand.
 * @params: parameter list for the command
 * @error: a place to store an error, or %NULL
 *
 * Executes a non-selection command on the given connection.
 *
 * This function returns the number of rows affected by the execution of @cmd, or -1
 * if an error occurred, or -2 if the provider does not return the number of rows affected.
 *
 * Note that no check is made regarding the actual number of statements in @cmd or if it really contains a non SELECT
 * statement. This function is just a convenience function around the gda_connection_execute_command()
 * function. If @cmd contains several statements, the last #GdaParameterList is returned.
 *
 * See the documentation of the gda_connection_execute_command() for information
 * about the @params list of parameters.
 *
 * Returns: the number of rows affected (&gt;=0) or -1 or -2 
 */
gint
gda_connection_execute_non_select_command (GdaConnection *cnc, GdaCommand *cmd,
					   GdaParameterList *params, GError **error)
{
	GList *reclist, *list;
	GdaParameterList *plist = NULL;
	gint retval = 0;

	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), -1);
	g_return_val_if_fail (cnc->priv, -1);
	g_return_val_if_fail (cmd != NULL, -1);

	reclist = gda_connection_execute_command (cnc, cmd, params, error);
	if (!reclist)
		return -1;

	for (list = g_list_last (reclist); list && !plist; list = list->prev) {
		plist = (GdaParameterList *) (g_list_last (reclist)->data);
		if (!GDA_IS_PARAMETER_LIST (plist))
			plist = NULL;
	}
	if (plist) {
		GdaParameter *param;

		param = gda_parameter_list_find_param (plist, "IMPACTED_ROWS");
		if (param) {
			const GValue *value;

			value = gda_parameter_get_value (param);
			if (G_VALUE_TYPE (value) == G_TYPE_INT)
				retval = g_value_get_int (value);
			else
				retval = -2;
		} 
		else
			retval = -2;
	}

	list = reclist;
	for (list = reclist; list; list = g_list_next (list))
		if (list->data)
			g_object_unref (list->data);
	g_list_free (reclist);

	return retval;
}

/**
 * gda_connection_begin_transaction
 * @cnc: a #GdaConnection object.
 * @name: the name of the transation to start
 * @level:
 * @error: a place to store errors, or %NULL
 *
 * Starts a transaction on the data source, identified by the
 * @xaction parameter.
 *
 * Before starting a transaction, you can check whether the underlying
 * provider does support transactions or not by using the
 * #gda_connection_supports_feature() function.
 *
 * Returns: %TRUE if the transaction was started successfully, %FALSE
 * otherwise.
 */
gboolean
gda_connection_begin_transaction (GdaConnection *cnc, const gchar *name, GdaTransactionIsolation level,
				  GError **error)
{
	gboolean retval;

	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), FALSE);
	g_return_val_if_fail (cnc->priv, FALSE);
	g_return_val_if_fail (cnc->priv->provider_obj, FALSE);

	retval = gda_server_provider_begin_transaction (cnc->priv->provider_obj, cnc, name, level, error);
	if (retval)
		gda_client_notify_event (cnc->priv->client, cnc, GDA_CLIENT_EVENT_TRANSACTION_STARTED, NULL);

	return retval;
}

/**
 * gda_connection_commit_transaction
 * @cnc: a #GdaConnection object.
 * @name: the name of the transation to commit
 * @error: a place to store errors, or %NULL
 *
 * Commits the given transaction to the backend database. You need to call
 * gda_connection_begin_transaction() first.
 *
 * Returns: %TRUE if the transaction was finished successfully,
 * %FALSE otherwise.
 */
gboolean
gda_connection_commit_transaction (GdaConnection *cnc, const gchar *name, GError **error)
{
	gboolean retval;

	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), FALSE);
	g_return_val_if_fail (cnc->priv, FALSE);
	g_return_val_if_fail (cnc->priv->provider_obj, FALSE);

	retval = gda_server_provider_commit_transaction (cnc->priv->provider_obj, cnc, name, error);
	if (retval)
		gda_client_notify_event (cnc->priv->client, cnc, GDA_CLIENT_EVENT_TRANSACTION_COMMITTED, NULL);

	return retval;
}

/**
 * gda_connection_rollback_transaction
 * @cnc: a #GdaConnection object.
 * @name: the name of the transation to commit
 * @error: a place to store errors, or %NULL
 *
 * Rollbacks the given transaction. This means that all changes
 * made to the underlying data source since the last call to
 * #gda_connection_begin_transaction() or #gda_connection_commit_transaction()
 * will be discarded.
 *
 * Returns: %TRUE if the operation was successful, %FALSE otherwise.
 */
gboolean
gda_connection_rollback_transaction (GdaConnection *cnc, const gchar *name, GError **error)
{
	gboolean retval;

	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), FALSE);
	g_return_val_if_fail (cnc->priv, FALSE);
	g_return_val_if_fail (cnc->priv->provider_obj, FALSE);

	retval = gda_server_provider_rollback_transaction (cnc->priv->provider_obj, cnc, name, error);
	if (retval)
		gda_client_notify_event (cnc->priv->client, cnc, GDA_CLIENT_EVENT_TRANSACTION_CANCELLED, NULL);

	return retval;
}

/**
 * gda_connection_add_savepoint
 * @cnc: a #GdaConnection object
 * @name: name of the savepoint to add
 * @error: a place to store errors or %NULL
 *
 * Adds a SAVEPOINT named @name.
 *
 * Returns: TRUE if no error occurred
 */
gboolean
gda_connection_add_savepoint (GdaConnection *cnc, const gchar *name, GError **error)
{
	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), FALSE);
	g_return_val_if_fail (cnc->priv, FALSE);
	g_return_val_if_fail (cnc->priv->provider_obj, FALSE);
	
	return gda_server_provider_add_savepoint (cnc->priv->provider_obj, cnc, name, error);
}

/**
 * gda_connection_rollback_savepoint
 * @cnc: a #GdaConnection object
 * @name: name of the savepoint to rollback to
 * @error: a place to store errors or %NULL
 *
 * Rollback all the modifications made after the SAVEPOINT named @name.
 *
 * Returns: TRUE if no error occurred
 */
gboolean
gda_connection_rollback_savepoint (GdaConnection *cnc, const gchar *name, GError **error)
{
	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), FALSE);
	g_return_val_if_fail (cnc->priv, FALSE);
	g_return_val_if_fail (cnc->priv->provider_obj, FALSE);	

	return gda_server_provider_rollback_savepoint (cnc->priv->provider_obj, cnc, name, error);
}

/**
 * gda_connection_delete_savepoint
 * @cnc: a #GdaConnection object
 * @name: name of the savepoint to delete
 * @error: a place to store errors or %NULL
 *
 * Delete the SAVEPOINT named @name when not used anymore.
 *
 * Returns: TRUE if no error occurred
 */
gboolean
gda_connection_delete_savepoint (GdaConnection *cnc, const gchar *name, GError **error)
{
	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), FALSE);
	g_return_val_if_fail (cnc->priv, FALSE);
	g_return_val_if_fail (cnc->priv->provider_obj, FALSE);

	return gda_server_provider_delete_savepoint (cnc->priv->provider_obj, cnc, name, error);
}

/**
 * gda_connection_get_transaction_status
 * @cnc: a #GdaConnection object
 *
 * Get the status of @cnc regarding transactions. The returned object should not be modified
 * or destroyed; however it may be modified or destroyed by the connection itself.
 *
 * If %NULL is returned, then no transaction has been associated with @cnc
 *
 * Returns: a #GdaTransactionStatus object, or %NULL
 */
GdaTransactionStatus *
gda_connection_get_transaction_status (GdaConnection *cnc)
{
	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), NULL);
	g_return_val_if_fail (cnc->priv, NULL);

	return cnc->priv->trans_status;
}

/**
 * gda_connection_supports_feature
 * @cnc: a #GdaConnection object.
 * @feature: feature to ask for.
 *
 * Asks the underlying provider for if a specific feature is supported.
 *
 * Returns: %TRUE if the provider supports it, %FALSE if not.
 */
gboolean
gda_connection_supports_feature (GdaConnection *cnc, GdaConnectionFeature feature)
{
	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), FALSE);
	g_return_val_if_fail (cnc->priv, FALSE);
	g_return_val_if_fail (cnc->priv->provider_obj, FALSE);

	return gda_server_provider_supports_feature (cnc->priv->provider_obj, cnc, feature);
}

/**
 * gda_connection_get_schema
 * @cnc: a #GdaConnection object.
 * @schema: database schema to get.
 * @params: parameter list.
 * @error: a place to store errors, or %NULL
 *
 * Asks the underlying data source for a list of database objects.
 *
 * This is the function that lets applications ask the different
 * providers about all their database objects (tables, views, procedures,
 * etc). The set of database objects that are retrieved are given by the
 * 2 parameters of this function: @schema, which specifies the specific
 * schema required, and @params, which is a list of parameters that can
 * be used to give more detail about the objects to be returned.
 *
 * The list of parameters is specific to each schema type, see the
 * <link linkend="libgda-provider-get-schema">get_schema() virtual method for providers</link> for more details.
 *
 * Returns: a #GdaDataModel containing the data required. The caller is responsible
 * for freeing the returned model using g_object_unref().
 */
GdaDataModel *
gda_connection_get_schema (GdaConnection *cnc,
			   GdaConnectionSchema schema,
			   GdaParameterList *params,
			   GError **error)
{
	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), NULL);
	g_return_val_if_fail (cnc->priv->provider_obj, NULL);

	return gda_server_provider_get_schema (cnc->priv->provider_obj, cnc, schema, params, error);
}

/**
 * gda_connection_get_events
 * @cnc: a #GdaConnection.
 *
 * Retrieves a list of the last errors occurred during the connection.
 * You can make a copy of the list using #gda_connection_event_list_copy.
 * 
 * Returns: a GList of #GdaConnectionEvent.
 *
 */
const GList *
gda_connection_get_events (GdaConnection *cnc)
{
	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), NULL);
	g_return_val_if_fail (cnc->priv, FALSE);

	return cnc->priv->events_list;
}

/**
 * gda_connection_value_to_sql_string
 * @cnc: a #GdaConnection object.
 * @from: #GValue to convert from
 *
 * Produces a fully quoted and escaped string from a GValue
 *
 * Returns: escaped and quoted value or NULL if not supported.
 */
gchar *
gda_connection_value_to_sql_string (GdaConnection *cnc, GValue *from)
{
	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), FALSE);
	g_return_val_if_fail (cnc->priv, FALSE);
	g_return_val_if_fail (from != NULL, FALSE);
	g_return_val_if_fail (cnc->priv->provider_obj, FALSE);

	/* execute the command on the provider */
	return gda_server_provider_value_to_sql_string (cnc->priv->provider_obj, cnc, from);
}

/*
 * Internal functions to keep track
 * of the transactional status of the connection
 */
void
gda_connection_internal_transaction_started (GdaConnection *cnc, const gchar *parent_trans, const gchar *trans_name, 
					     GdaTransactionIsolation isol_level)
{
	GdaTransactionStatus *parent, *st;

	st = gda_transaction_status_new (trans_name);
	st->isolation_level = isol_level;
	parent = gda_transaction_status_find (cnc->priv->trans_status, parent_trans, NULL);
	if (!parent)
		cnc->priv->trans_status = st;
	else {
		gda_transaction_status_add_event_sub (parent, st);
		g_object_unref (st);
	}
#ifdef GDA_DEBUG_signal
        g_print (">> 'TRANSACTION_STATUS_CHANGED' from %s\n", __FUNCTION__);
#endif
        g_signal_emit (G_OBJECT (cnc), gda_connection_signals[TRANSACTION_STATUS_CHANGED], 0);
#ifdef GDA_DEBUG_signal
        g_print ("<< 'TRANSACTION_STATUS_CHANGED' from %s\n", __FUNCTION__);
#endif

#ifdef GDA_DEBUG_NO
	if (cnc->priv->trans_status)
		gda_transaction_status_dump (cnc->priv->trans_status, 5);
#endif
}

void 
gda_connection_internal_transaction_rolledback (GdaConnection *cnc, const gchar *trans_name)
{
	GdaTransactionStatus *st = NULL;
	GdaTransactionStatusEvent *ev = NULL;

	if (cnc->priv->trans_status)
		st = gda_transaction_status_find (cnc->priv->trans_status, trans_name, &ev);
	if (st) {
		if (ev) {
			/* there is a parent transaction */
			gda_transaction_status_free_events (ev->trans, ev, TRUE);
		}
		else {
			/* no parent transaction */
			g_object_unref (cnc->priv->trans_status);
			cnc->priv->trans_status = NULL;
		}
#ifdef GDA_DEBUG_signal
		g_print (">> 'TRANSACTION_STATUS_CHANGED' from %s\n", __FUNCTION__);
#endif
		g_signal_emit (G_OBJECT (cnc), gda_connection_signals[TRANSACTION_STATUS_CHANGED], 0);
#ifdef GDA_DEBUG_signal
		g_print ("<< 'TRANSACTION_STATUS_CHANGED' from %s\n", __FUNCTION__);
#endif
	}
	else
		g_warning (_("Connection transaction status tracking: no transaction exists for ROLLBACK"));
#ifdef GDA_DEBUG_NO
	if (cnc->priv->trans_status)
		gda_transaction_status_dump (cnc->priv->trans_status, 5);
#endif
}

void 
gda_connection_internal_transaction_committed (GdaConnection *cnc, const gchar *trans_name)
{
	GdaTransactionStatus *st = NULL;
	GdaTransactionStatusEvent *ev = NULL;

	if (cnc->priv->trans_status)
		st = gda_transaction_status_find (cnc->priv->trans_status, trans_name, &ev);
	if (st) {
		if (ev) {
			/* there is a parent transaction */
			gda_transaction_status_free_events (ev->trans, ev, TRUE);
		}
		else {
			/* no parent transaction */
			g_object_unref (cnc->priv->trans_status);
			cnc->priv->trans_status = NULL;
		}
#ifdef GDA_DEBUG_signal
		g_print (">> 'TRANSACTION_STATUS_CHANGED' from %s\n", __FUNCTION__);
#endif
		g_signal_emit (G_OBJECT (cnc), gda_connection_signals[TRANSACTION_STATUS_CHANGED], 0);
#ifdef GDA_DEBUG_signal
		g_print ("<< 'TRANSACTION_STATUS_CHANGED' from %s\n", __FUNCTION__);
#endif
	}
	else
		g_warning (_("Connection transaction status tracking: no transaction exists for COMMIT"));
#ifdef GDA_DEBUG_NO
	if (cnc->priv->trans_status)
		gda_transaction_status_dump (cnc->priv->trans_status, 5);
#endif
}

void
gda_connection_internal_sql_executed (GdaConnection *cnc, const gchar *sql, GdaConnectionEvent *error)
{
	GdaTransactionStatus *st = NULL;
	
	if (cnc->priv->trans_status)
		st = gda_transaction_status_find_current (cnc->priv->trans_status, NULL, FALSE);
	if (st)
		gda_transaction_status_add_event_sql (st, sql, error);
#ifdef GDA_DEBUG_signal
		g_print (">> 'TRANSACTION_STATUS_CHANGED' from %s\n", __FUNCTION__);
#endif
		g_signal_emit (G_OBJECT (cnc), gda_connection_signals[TRANSACTION_STATUS_CHANGED], 0);
#ifdef GDA_DEBUG_signal
		g_print ("<< 'TRANSACTION_STATUS_CHANGED' from %s\n", __FUNCTION__);
#endif
#ifdef GDA_DEBUG_NO
	if (cnc->priv->trans_status)
		gda_transaction_status_dump (cnc->priv->trans_status, 5);
#endif
}

void
gda_connection_internal_savepoint_added (GdaConnection *cnc, const gchar *parent_trans, const gchar *svp_name)
{
	GdaTransactionStatus *st;

	st = gda_transaction_status_find (cnc->priv->trans_status, parent_trans, NULL);
	if (st) {
		gda_transaction_status_add_event_svp (st, svp_name);
#ifdef GDA_DEBUG_signal
		g_print (">> 'TRANSACTION_STATUS_CHANGED' from %s\n", __FUNCTION__);
#endif
		g_signal_emit (G_OBJECT (cnc), gda_connection_signals[TRANSACTION_STATUS_CHANGED], 0);
#ifdef GDA_DEBUG_signal
		g_print ("<< 'TRANSACTION_STATUS_CHANGED' from %s\n", __FUNCTION__);
#endif
	}
	else
		g_warning (_("Connection transaction status tracking: no transaction exists for ADD SAVEPOINT"));
#ifdef GDA_DEBUG_NO
	if (cnc->priv->trans_status)
		gda_transaction_status_dump (cnc->priv->trans_status, 5);
#endif
}

void
gda_connection_internal_savepoint_rolledback (GdaConnection *cnc, const gchar *svp_name)
{
	GdaTransactionStatus *st;
	GdaTransactionStatusEvent *ev = NULL;

	st = gda_transaction_status_find (cnc->priv->trans_status, svp_name, &ev);
	if (st) {
		gda_transaction_status_free_events (st, ev, TRUE);
#ifdef GDA_DEBUG_signal
		g_print (">> 'TRANSACTION_STATUS_CHANGED' from %s\n", __FUNCTION__);
#endif
		g_signal_emit (G_OBJECT (cnc), gda_connection_signals[TRANSACTION_STATUS_CHANGED], 0);
#ifdef GDA_DEBUG_signal
		g_print ("<< 'TRANSACTION_STATUS_CHANGED' from %s\n", __FUNCTION__);
#endif
	}
	else
		g_warning (_("Connection transaction status tracking: no transaction exists for ROLLBACK SAVEPOINT"));
#ifdef GDA_DEBUG_NO
	if (cnc->priv->trans_status)
		gda_transaction_status_dump (cnc->priv->trans_status, 5);
#endif	
}

void
gda_connection_internal_savepoint_removed (GdaConnection *cnc, const gchar *svp_name)
{
	GdaTransactionStatus *st;
	GdaTransactionStatusEvent *ev = NULL;

	st = gda_transaction_status_find (cnc->priv->trans_status, svp_name, &ev);
	if (st) {
		gda_transaction_status_free_events (st, ev, FALSE);
#ifdef GDA_DEBUG_signal
		g_print (">> 'TRANSACTION_STATUS_CHANGED' from %s\n", __FUNCTION__);
#endif
		g_signal_emit (G_OBJECT (cnc), gda_connection_signals[TRANSACTION_STATUS_CHANGED], 0);
#ifdef GDA_DEBUG_signal
		g_print ("<< 'TRANSACTION_STATUS_CHANGED' from %s\n", __FUNCTION__);
#endif
	}
	else
		g_warning (_("Connection transaction status tracking: no transaction exists for REMOVE SAVEPOINT"));
#ifdef GDA_DEBUG_NO
	if (cnc->priv->trans_status)
		gda_transaction_status_dump (cnc->priv->trans_status, 5);
#endif		
}


void
gda_connection_internal_treat_sql (GdaConnection *cnc, const gchar *sql, GdaConnectionEvent *error)
{
	gboolean done = FALSE;

	if (!error || (error && (gda_connection_event_get_event_type (error) != GDA_CONNECTION_EVENT_ERROR))) {
		GdaSqlTransaction *trans;

		trans = gda_sql_transaction_parse_with_error (sql, NULL);
		if (trans) {
			switch (trans->trans_type) {
			case GDA_SQL_TRANSACTION_BEGIN:
				gda_connection_internal_transaction_started (cnc, NULL, trans->trans_name, 
									     GDA_TRANSACTION_ISOLATION_UNKNOWN);
				break;
			case GDA_SQL_TRANSACTION_COMMIT:
				gda_connection_internal_transaction_committed (cnc, trans->trans_name);
				break;
			case GDA_SQL_TRANSACTION_ROLLBACK:
				gda_connection_internal_transaction_rolledback (cnc, trans->trans_name);
				break;
			case GDA_SQL_TRANSACTION_SAVEPOINT_ADD:
				gda_connection_internal_savepoint_added (cnc, NULL, trans->trans_name);
				break;
			case GDA_SQL_TRANSACTION_SAVEPOINT_REMOVE:
				gda_connection_internal_savepoint_removed (cnc, trans->trans_name);
				break;
			case GDA_SQL_TRANSACTION_SAVEPOINT_ROLLBACK:
				gda_connection_internal_savepoint_rolledback (cnc, trans->trans_name);
				break;
			default:
				g_assert_not_reached ();
			}
			gda_sql_transaction_destroy (trans);
			done = TRUE;
		}
	}
	
	if (!done)
		gda_connection_internal_sql_executed (cnc, sql, error);
}

void
gda_connection_internal_change_transaction_state (GdaConnection *cnc,
						  GdaTransactionStatusState newstate)
{
	g_return_if_fail (cnc->priv->trans_status);

	if (cnc->priv->trans_status->state == newstate)
		return;

	cnc->priv->trans_status->state = newstate;
#ifdef GDA_DEBUG_signal
	g_print (">> 'TRANSACTION_STATUS_CHANGED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit (G_OBJECT (cnc), gda_connection_signals[TRANSACTION_STATUS_CHANGED], 0);
#ifdef GDA_DEBUG_signal
	g_print ("<< 'TRANSACTION_STATUS_CHANGED' from %s\n", __FUNCTION__);
#endif
}

void
gda_connection_force_status (GdaConnection *cnc, gboolean opened)
{
	g_return_if_fail (GDA_IS_CONNECTION (cnc));

	if (opened && !cnc->priv->is_open) {
		cnc->priv->is_open = TRUE;
#ifdef GDA_DEBUG_signal
		g_print (">> 'CONN_OPENED' from %s\n", __FUNCTION__);
#endif
		g_signal_emit (G_OBJECT (cnc), gda_connection_signals[CONN_OPENED], 0);
#ifdef GDA_DEBUG_signal
		g_print ("<< 'CONN_OPENED' from %s\n", __FUNCTION__);
#endif
		if (cnc->priv->client)
			gda_client_notify_connection_opened_event (cnc->priv->client, cnc);
		return;
	}

	if (!opened && cnc->priv->is_open) {
		cnc->priv->is_open = FALSE;

#ifdef GDA_DEBUG_signal
		g_print (">> 'CONN_TO_CLOSE' from %s\n", __FUNCTION__);
#endif
		g_signal_emit (G_OBJECT (cnc), gda_connection_signals[CONN_TO_CLOSE], 0);
#ifdef GDA_DEBUG_signal
		g_print ("<< 'CONN_TO_CLOSE' from %s\n", __FUNCTION__);
#endif
#ifdef GDA_DEBUG_signal
		g_print (">> 'CONN_CLOSED' from %s\n", __FUNCTION__);
#endif
		g_signal_emit (G_OBJECT (cnc), gda_connection_signals[CONN_CLOSED], 0);
#ifdef GDA_DEBUG_signal
		g_print ("<< 'CONN_CLOSED' from %s\n", __FUNCTION__);
#endif

		if (cnc->priv->client)
			gda_client_notify_connection_closed_event (cnc->priv->client, cnc);
	}
}

/*
 * Prepared statements handling
 */

static void prepared_stms_query_destroyed_cb (GdaQuery *query, GdaConnection *cnc);
static void
prepared_stms_foreach_func (GdaQuery *query, gpointer prepared_stmt, GdaConnection *cnc)
{
	g_signal_handlers_disconnect_by_func (query, G_CALLBACK (prepared_stms_query_destroyed_cb), cnc);
}

void 
gda_connection_destroy_prepared_statement_hash (GdaConnection *cnc)
{
	if (!cnc->priv->prepared_stmts)
		return;

	g_hash_table_foreach (cnc->priv->prepared_stmts, (GHFunc) prepared_stms_foreach_func, cnc);
	g_hash_table_destroy (cnc->priv->prepared_stmts);
	cnc->priv->prepared_stmts = NULL;
}

void
gda_connection_init_prepared_statement_hash (GdaConnection *cnc, GDestroyNotify stmt_destroy_func)
{
	g_return_if_fail (GDA_IS_CONNECTION (cnc));
	g_return_if_fail (cnc->priv);

	if (!cnc->priv->prepared_stmts)
		cnc->priv->prepared_stmts = g_hash_table_new_full (g_direct_hash, g_direct_equal, NULL, stmt_destroy_func);
}

static void 
prepared_stms_query_destroyed_cb (GdaQuery *query, GdaConnection *cnc)
{
	g_signal_handlers_disconnect_by_func (query, G_CALLBACK (prepared_stms_query_destroyed_cb), cnc);
	g_hash_table_remove (cnc->priv->prepared_stmts, query);
}

void 
gda_connection_add_prepared_statement (GdaConnection *cnc, GdaQuery *query, gpointer prepared_stmt)
{
	g_return_if_fail (GDA_IS_CONNECTION (cnc));
	g_return_if_fail (cnc->priv);

	if (!cnc->priv->prepared_stmts) {
		g_warning (_("Prepared statements hash not initialized, "
			     "call gda_connection_init_prepared_statement_hash() first"));
		return;
	}
	g_hash_table_remove (cnc->priv->prepared_stmts, query);
	g_hash_table_insert (cnc->priv->prepared_stmts, query, prepared_stmt);
	
	/* destroy the prepared statement if query is destroyed, or changes */
	gda_object_connect_destroy (GDA_OBJECT (query), G_CALLBACK (prepared_stms_query_destroyed_cb), cnc);
	g_signal_connect (G_OBJECT (query), "changed", G_CALLBACK (prepared_stms_query_destroyed_cb), cnc);

}

gpointer
gda_connection_get_prepared_statement (GdaConnection *cnc, GdaQuery *query)
{
	g_return_val_if_fail (GDA_IS_CONNECTION (cnc), NULL);
	g_return_val_if_fail (cnc->priv, NULL);

	if (!cnc->priv->prepared_stmts) {
		g_warning (_("Prepared statements hash not initialized, "
			     "call gda_connection_init_prepared_statement_hash() first"));
		return NULL;
	}
	return g_hash_table_lookup (cnc->priv->prepared_stmts, query);
}

void
gda_connection_del_prepared_statement (GdaConnection *cnc, GdaQuery *query)
{
	g_return_if_fail (GDA_IS_CONNECTION (cnc));
	prepared_stms_query_destroyed_cb (query, cnc);
}
