/*
 * Copyright (c) 2001-2003 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 *
 * $Id: hf77.c,v 6.31.2.6 2003/01/26 19:15:54 jsquyres Exp $
 *
 *	Function:	- wrapper for FORTRAN program compilation
 */

#include <lam_config.h>

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <ctype.h>

#include <args.h>
#include <boot.h>
#include <portable.h>

#ifndef DEFTF77
#define DEFTF77 LAM_F77
#endif

/*
 * external functions
 */
extern int		cnfexec();

/*
 * local functions
 */
static void strip_white(char* str);
static int check_file(char *dir, char *file);

/*
 * local variables
 */
static char		ph[4][LAM_PATH_MAX];
static char		plib[3][LAM_PATH_MAX];
static char		extra[LAM_PATH_MAX];

int
main(int argc, char *argv[])
{
	int		ac;		/* f77 # args */
	int		status;		/* f77 return code */
	int		fl_libs;	/* add libs to command line */
	int		fl_show;	/* show what would be done */
	int             fl_building;    /* are we building LAM right now? */
	int             fl_profile;     /* want profile lib? */
	int		i;		/* favourite index */
	int		num_incs;	/* number of -I's */
	int		num_libs;	/* number of -L's */
	char		**av;		/* f77 arg vector */
	char		**sys;		/* system libraries arg vector */
	char		*inchome;	/* include base dir */
	char		*libhome;	/* library base dir */
	char		*tf77;		/* f77 tool */
	char            *syslibs = strdup(LAM_SYSLIBS);
	char            *extra_flags = strdup(EXTRA_MPIF77_FLAGS);
#if LAM_WANT_ROMIO || LAM_WANT_MPI2CPP
	int             found;
#endif
#if LAM_WANT_PROFILE
	int             added_lpmpi = 0;
#endif

	if (strcmp(DEFTF77, "/bin/false") == 0) {
	  fprintf(stderr, "Warning: it appears that LAM was configured with the --without-fc option.\n");
	  fprintf(stderr, "Warning: LAM/MPI compilation of Fortran programs may fail!\n");
	}

/*
 * LAM_TOP_BUILDDIR and LAM_TOP_SRCDIR should only be used internally
 * -- it is *not* a user-level flag (and as such, is not documented
 * :-)
 */
	libhome = getenv("LAM_TOP_BUILDDIR");
	if (libhome != 0) {
	  inchome = getenv("LAM_TOP_SRCDIR");
	  if (inchome == 0) {
	    show_help("compile", "internal_build", NULL);
	    exit(1);
	  }
	} else {
	  libhome = getenv("LAMHOME");
	  if (libhome == 0) libhome = getenv("TROLLIUSHOME");
	  if (libhome == 0) libhome = LAM_PREFIX;

	  inchome = libhome;
	}

	/* Clean up inchome and libhome -- trim any extra whitespace
           or any trailing /'s so that we can do some easy strcmp's
           below.  This also catches the corner case where $prefix is
           simply "/". */

	for (i = strlen(inchome) - 1; i >= 0; ++i) {
	  if (isspace(inchome[i]) || inchome[i] == '/')
	    inchome[i] = '\0';
	  else
	    break;
	}
	for (i = strlen(libhome) - 1; i >= 0; ++i) {
	  if (isspace(libhome[i]) || libhome[i] == '/')
	    libhome[i] = '\0';
	  else
	    break;
	}

/*
 * Command line options
 */
	fl_show = 0;
	fl_libs = 1;
	fl_building = 0;
	fl_profile = 0;

	for (i = 1; i < argc; ++i) {
		if (strcmp(argv[i], "-showme") == 0) {
			fl_show = 1;
		} else if (strcmp(argv[i], "-c") == 0) {
			fl_libs = 0;
		} else if (strcmp(argv[i], "-lam-building") == 0) {
			fl_building = 1;
		} else if (strcmp(argv[i], "-lpmpi") == 0) {
			fl_profile = 1;
		}
	}
#if !LAM_WANT_PROFILE
/*
 * Sanity check
 */
	if (fl_profile == 1) {
	  show_help("compile", "no-profiling-support", argv[0], NULL);
	  fl_profile = 0;
	}
#endif

	if ((strlen(inchome) + 32) >= LAM_PATH_MAX ||
	    (strlen(libhome) + 32) >= LAM_PATH_MAX) {
	  char buffer[32];

	  snprintf(buffer, 32, "%d", LAM_PATH_MAX - 32);
	  if (strlen(inchome) + 32 >= LAM_PATH_MAX)
	    show_help("compile", "phome-too-long", inchome, buffer, "hcc", 
		      NULL);
	  else
	    show_help("compile", "phome-too-long", libhome, buffer, "hcc", 
		      NULL);
	  exit(1);
	}
/*
 * Form include directory pathname.
 * If we're building LAM:
 *
 * - if it's a VPATH build, lam_config.h will be in
 * LAM_TOP_BUILDDIR/share/include, and the rest will be in
 * LAM_TOP_SRCDIR/share/include.  
 *
 * - if it's not a VPATH build, everything will be in
 * LAM_TOP_SRCDIR/share/include.  So it doesn't hurt to list both
 * (from above); it simplifies the logic.
 *
 * If we're not building LAM, everything is in $prefix/include 
 *
 * Note that in hcc, if $prefix is /usr and we're not lam_building, we
 * do *not* add -I/usr/include because that may disrupt the compiler's
 * normal sequence of include directories.  This causes a problem for
 * some versions of RedHat (and probably other distros as well), for
 * example.  We do *NOT* do that here -- the fortran compiler *always*
 * need -I, regardless of what $prefix is.
 */
	num_incs = 0;
	if (fl_building) {
	  strcpy(ph[num_incs], "-I");
	  strcat(ph[num_incs], inchome);
	  strcat(ph[num_incs], STRSDIR);
	  strcat(ph[num_incs], "share");
	  strcat(ph[num_incs], STRSDIR);
	  strcat(ph[num_incs], "include");
	  ++num_incs;

	  strcpy(ph[num_incs], "-I");
	  strcat(ph[num_incs], libhome);
	  strcat(ph[num_incs], STRSDIR);
	  strcat(ph[num_incs], "share");
	  strcat(ph[num_incs], STRSDIR);
	  strcat(ph[num_incs], "include");
	  ++num_incs;
	} else {
	  strcpy(ph[0], "-I");
	  strcat(ph[0], inchome);
	  strcat(ph[0], STRSDIR);
	  strcat(ph[0], "include");
	  num_incs = 1;
	}
/*
 * Form library directory pathname.
 *
 * If we're building LAM, use "-LLAM_TOP_BUILDDIR/share
 * -LLAM_TOP_BUILDDIR/share/pmpi".
 *
 * Otherwise, use "-L$prefix/lib".
 *
 * Apply similar logic here as hcc does with -I when the prefix is
 * /usr -- if libhome is /usr, there's no need to explicitly add it,
 * because the linker will already be looking there.  This is
 * different than fortran's handling of -I/usr/include, but that's
 * because the fortran compiler has different default behavior than
 * the linker.
 */
	num_libs = 0;
	if (fl_building) {
#if LAM_WANT_PROFILE
	  strcpy(plib[0], LAM_FLIBPATH);
	  strcat(plib[0], libhome);
	  strcat(plib[0], STRSDIR);
	  strcpy(plib[1], LAM_FLIBPATH);
	  strcat(plib[1], libhome);
	  strcat(plib[1], STRSDIR);
	  strcat(plib[1], "share");
	  strcat(plib[1], STRSDIR);
	  strcat(plib[1], "pmpi");

	  strcpy(plib[2], LAM_FLIBPATH);
	  strcat(plib[2], libhome);
	  strcat(plib[2], STRSDIR);
#endif
	  strcat(plib[2], "share");
	  num_libs = 3;
	} else {

	  /* Ensure that we don't -L/usr/lib, for the reasons listed
             above */

	  strcpy(plib[0], LAM_FLIBPATH);
	  strcat(plib[0], libhome);
	  strcat(plib[0], STRSDIR);
	  strcat(plib[0], "lib");
	  if (strcmp(libhome, "/usr") != 0)
	    num_libs = 1;
	}

	tf77 = getenv("LAMHF77");
	if (tf77 == 0) tf77 = getenv("TROLLIUSHF77");
	if (tf77 == 0) tf77 = DEFTF77;

	if (strcmp(tf77, "false") == 0 || strlen(tf77) == 0) {
	  show_help("hf77", "no-fortran-support", NULL);
	  exit(1);
	}

	if ((av = sfh_argv_break(tf77, ' ')) == 0) {
	  show_help(NULL, "lib-call-fail", "sfh_argv_break", NULL);
	  exit(errno);
	}

	ac = sfh_argv_count(av);
/*
 * If we're building LAM right now, #define LAM_BUILDING to be 1 so
 * that we don't try to include mpi++.h and mpio.h in mpi.h
 */
	if (fl_building) {
	  strcpy(extra, "-DLAM_BUILDING=1");
	  sfh_argv_add(&ac, &av, extra);
	}

	for (i = 0; i < num_incs; i++)
	  sfh_argv_add(&ac, &av, ph[i]);
	if (fl_libs) {
	  for (i = 0; i < num_libs; i++)
	    sfh_argv_add(&ac, &av, plib[i]);
	}

/*
 * Don't copy -showme or -lam-building, and -lpmpi was already insertted
 */
	for (i = 1; i < argc; ++i) {
		if (strcmp(argv[i], "-showme") &&
		    strcmp(argv[i], "-lam-building") &&
		    strcmp(argv[i], "-lpmpi")) {
			sfh_argv_add(&ac, &av, argv[i]);
		}
	}

	if (fl_libs) {
/*
 * Add system dependent link arguments.
 */
		if (strlen(LAM_HBIND) > 0) {
			if ((sys = sfh_argv_break(LAM_HBIND, ' ')) == 0) {
			  show_help(NULL, "lib-call-fail", "sfh_argv_break", 
				    NULL);
			  exit(errno);
			}

			for ( ; *sys; sys++) {
				sfh_argv_add(&ac, &av, *sys);
			}
		}
		if (fl_building == 0) {
/*
 * Do we want ROMIO support?
 * Tricky -- we have to test for presence of libmpio.a first, "just in 
 * case".
 */
#if LAM_WANT_ROMIO
		  found = check_file(plib[0] + 2, "liblammpio.a");
		  if (!found)
		    found = check_file(plib[0] + 2, "liblammpio.so");
		  if (found) {
		    sfh_argv_add(&ac, &av, "-llammpio");
		    /* Don't have to check for LAM_WANT_PROFILE here; we
		       can't have LAM_WANT_ROMIO without it */
		    sfh_argv_add(&ac, &av, "-lpmpi");
		    added_lpmpi = 1;
#if HAVE_LIBAIO
		    sfh_argv_add(&ac, &av, "-laio");
#endif
		  } else if (!fl_building) {
		    fprintf(stderr, 
			    "WARNING: %s expected to find liblammpio.* in %s\n",
			    argv[0], plib[0] + 2);
		    fprintf(stderr, 
			    "WARNING: MPI-2 IO support will be disabled\n");
		  }
#endif
		}
/*
 * Add standard LAM libraries.
 */
#if LAM_WANT_PROFILE
		if (fl_profile && !added_lpmpi)
		  sfh_argv_add(&ac, &av, "-lpmpi");
#endif
#if LAM_WANT_FORTRAN
		  sfh_argv_add(&ac, &av, "-llamf77mpi");
#endif
		sfh_argv_add(&ac, &av, "-lmpi");
		sfh_argv_add(&ac, &av, "-llam");
/*
 * Add system dependent libraries.
 */
		strip_white(syslibs);
		if (strlen(syslibs) > 0) {
			if ((sys = sfh_argv_break(syslibs, ' ')) == 0) {
			  show_help(NULL, "lib-call-fail", "sfh_argv_break", 
				    NULL);
			  exit(errno);
			}

			for ( ; *sys; sys++) {
				sfh_argv_add(&ac, &av, *sys);
			}
		}
/*
 * Add any extra flags
 */
		strip_white(extra_flags);
		if (strlen(extra_flags) > 0) {
			if ((sys = sfh_argv_break(extra_flags, ' ')) == 0) {
			  show_help(NULL, "lib-call-fail", "sfh_argv_break", 
				    NULL);
			  exit(errno);
			}

			for ( ; *sys; sys++) {
				sfh_argv_add(&ac, &av, *sys);
			}
		}

	}

	if (fl_show) {
		for (i = 0; i < ac; i++) {
			printf("%s ", av[i]);
		}
		printf("\n");
	} else {
		status = cnfexec(av);

		if (status) {
		  if (errno != 0)
		    perror("hf77");
		  exit(status);
		}
	}

	return(0);
}


/*
 * Remove leading and trailing white space from a given string.
 * Must be sent a null-terminated string.
 */
static void 
strip_white(char* str)
{
  char *src = str;

  /* Remove leading whitespace */

  for (; src != '\0'; src++) {
    if (!isspace(*src)) {
      strcpy(str, src);
      break;
    }
  }

  /* Remove trailing whitespace */

  for (src = str + strlen(str) + 1; src != str; src--) {
    if (isspace(*src)) 
      *src = '\0';
    else
      break;
  }
}


static int 
check_file(char *dir, char *file)
{
  int ret;
  struct stat buf;
  char *name = malloc(strlen(dir) + strlen(file) + 8);

  sprintf(name, "%s/%s", dir, file);
  ret = stat(name, &buf);
  free(name);

  return (ret == 0);
}
