/*
 * Copyright (c) 2001-2002 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	GDB/RBD
 *
 *	$Id: hboot.c,v 6.15.2.1 2002/10/09 19:49:57 brbarret Exp $
 *
 *	Function:	- boots OTB operating system
 */

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>

#include <all_list.h>
#include <args.h>
#include <boot.h>
#include <debug.h>
#include <kio.h>
#include <portable.h>
#include <proc_schema.h>
#include <sfh.h>
#include <terror.h>

#define MAXCLOSEFD	32			/* max close-on-exec fdecs. */

/*
 * external functions
 */
extern int		_lam_few();
extern int		psc_parse();
extern struct psc	*psc_find();

/*
 * local functions
 */
static void setdefaults(void);
static void setvars(int *ac, char ***av, char *var, char *opt);

/*
 * local variables
 */
static char		*f_psc;			/* process schema file */
static char		**av_psc_path;		/* f_psc pathname prefixes */
static char		*t_tkill;		/* tkill tool */
static char		rtfbuf[32];		/* RTF env. var. */

static int		ac_psc_path;		/* # of p_psc prefixes */
static int		fl_debug;		/* debugging option */
static int		fl_verbose;		/* verbose option */


int
main(int argc, char *argv[], char *env[])
{
	FILE		*fp_psc;	/* process schema file ptr */
	LIST		*list_psc;	/* parsed process schema list */
	struct psc	*p;
	int		i, n;
	int		fd;		/* file descriptor for /dev/null */
	int		pid;		/* child PID */
	int		ac_cmd;		/* # command arguments */
	int		ac_topo;	/* # topology variable overrides */
	char		**av_cmd;	/* command arguments */
	char		**av_topo;	/* topology variable overrides */
	char		buf[32];	/* formatting buffer */
	char		*full;		/* full pathname */
	char		*tail;		/* tail of full pathname */

	/* Ensure that we are not root */

	if (getuid() == 0 || geteuid() == 0) {
	  show_help(NULL, "deny-root", NULL);
	  exit(EACCES);
	}

/*
 * Initialize option parser.
 */
	validopts("dhstvNVn");
	followed("cIRb");

	if ((do_args(&argc, argv) < 0) || (errno = (argc == 1) ? 0 : EUSAGE)) {
	  show_help("hboot", "usage", NULL);
	  exit(errno);
	}

	if (opt_taken('h')) {
	  show_help("hboot", "usage", NULL);
	  exit(0);
	}

	setdefaults();
/*
 * Locate the conf file.
 */
	full = sfh_path_findv(f_psc, av_psc_path, R_OK, env);

	if (! full) {
		fprintf(stderr, "hboot: cannot find process schema %s: ",
				f_psc);
		perror("");
		exit(errno);
	} else {
		f_psc = full;
	}
/*
 * Dump debugging information.
 */
	DBUG("hboot: process schema = \"%s\"\n", f_psc);
	fflush(stdout);
/*
 * Open the conf file.
 */
	fp_psc = fopen(f_psc, "r");

	if (! fp_psc) {
		fprintf(stderr, "hboot: cannot open %s: ", f_psc);
		perror("");
		exit(errno);
	}
/*
 * Handle command line override of variables.
 */
	ac_topo = 0;
	av_topo = 0;

	if (opt_taken('I')) {
	  setvars(&ac_topo, &av_topo, "$inet_topo", getparam('I'));
	}
	if (opt_taken('R')) {
	  setvars(&ac_topo, &av_topo, "$rtr_topo", getparam('R'));
	}
	if (fl_debug) {
	  setvars(&ac_topo, &av_topo, "$debug", "-d");
	} else {
	  setvars(&ac_topo, &av_topo, "$debug", "");
	}
/*
 * Parse the conf file.
 */
	if (psc_parse(fp_psc, &list_psc, av_topo)) {
	  show_help("hboot", "psc-parse", f_psc, NULL);
	  exit(errno);
	}

	argvfree(av_topo);
	fclose(fp_psc);

	if (al_count(list_psc) == 0) {
	  show_help("hboot", "nothing-to-do", f_psc, NULL);
	  exit(0);
	}
/*
 * Find programs with sufficient permissions.
 */
	p = psc_find(list_psc);

	if (p) {
	  show_help("hboot", "cant-find-executables", p->psc_argv[0]);
	  exit(errno);
	}

	if (fl_debug) {
		for (p = al_top(list_psc); p; p = al_next(list_psc, p)) {
			printf("hboot: found %s\n", p->psc_argv[0]);
		}
	}
/*
 * Bail out here, if pretending.
 */
	if (opt_taken('N')) {
	        if (fl_verbose)
		  printf("Fake hboot -- quitting\n");
		exit(0);
	}
/*
 * Tkill if needed.
 */
	if (opt_taken('t')) {
		DBUG("hboot: performing %s\n", t_tkill);

		ac_cmd = 0;
		av_cmd = 0;
		argvadd(&ac_cmd, &av_cmd, t_tkill);

		if (opt_taken('b')) {
		  argvadd(&ac_cmd, &av_cmd, "-b");
		  argvadd(&ac_cmd, &av_cmd, getparam('b'));
		}
		if (fl_debug) {
		  printf("hboot: ");
		  for (i = 0; i < ac_cmd; i++)
		    printf("%s ", av_cmd[i]);
		  printf("\n");
		}

		if (_lam_few(av_cmd)) {
		  show_help("hboot", "tkill-fail", NULL);
		  exit(errno);
		}
	}
/*
 * Boot.
 */
	DBUG("hboot: booting...\n");

	sprintf(rtfbuf, "TROLLIUSRTF=%d", RTF_SYSGEN);

	if (putenv(rtfbuf) < 0) {
	  show_help(NULL, "lib-call-fail", "putenv", NULL);
	  exit(errno);
	}

#if 1
	/* Comment this out to make the TM extensions to PBS work
           nicely -- everything will be in one session, so TM can kill
           it when it dies. */
	setsid();
#endif

	if (opt_taken('s')) {
/*
 * Make any extraneous file descriptors close-on-exec.
 */
		for (i = 3; i < MAXCLOSEFD; ++i) {

			if ((fcntl(i, F_SETFD, 1) != 0) && (errno != EBADF)) {
			  show_help(NULL, "system-call-fail", 
				    "fcnt (set close-on-exec)l", NULL);
			  exit(errno);
			}
		}
	}

	n = 0;
/*
 * Loop through all the programs in the parsed config file.
 */
	for (p = al_top(list_psc); p; p = al_next(list_psc, p)) {
		DBUG("hboot: fork %s\n", p->psc_argv[0]);

		if ((pid = fork()) < 0) {
		  show_help(NULL, "system-call-fail", "fork", NULL);
		  exit(errno);
		}

		else if (pid == 0) {		/* child */

			if (opt_taken('s')) {
/*
 * Safely get rid of the stdio descriptors.
 */
				if ((fd = open("/dev/null", O_RDWR)) < 0) {
				  show_help(NULL, "system-call-fail", 
					    "open(\"/dev/null\"/, O_RDWR)", 
					    NULL);
				  exit(errno);
				}

				if ((dup2(fd, 0) < 0) || (dup2(fd, 1) < 0) ||
						(dup2(fd, 2) < 0)) {
				  show_help(NULL, "system-call-fail", "dup2",
					    NULL);
				  exit(errno);
				}

				close(fd);
			}

			if (fl_debug) {
			  printf("hboot: attempting to execute \n");
			}
			execvp(p->psc_argv[0], p->psc_argv);
			exit(errno);
		}

		else {				/* parent */
			n++;

			if (fl_debug) {
				tail = strrchr(p->psc_argv[0], STRDIR);
				tail = (tail) ? tail + 1 : p->psc_argv[0];
				sprintf(buf, "[%d]", n);
				printf("%-4.4s %5d %s", buf, pid, tail);

				for (i = 1; i < p->psc_argc; i++) {
					printf(" %s", p->psc_argv[i]);
				}

				printf("\n");
			}
		}

		if (p->psc_delay > 0) {
			sleep((unsigned int) p->psc_delay);
		}
	}

	return(0);
}

/*
 *	setdefaults
 *
 *	Function:	- sets default files and paths
 */
static void
setdefaults(void)
{
/*
 * prefix paths
 */
	ac_psc_path = 0;
	av_psc_path = 0;
	argvadd(&ac_psc_path, &av_psc_path, "");
	argvadd(&ac_psc_path, &av_psc_path, "$LAMHOME/etc");
	argvadd(&ac_psc_path, &av_psc_path, "$TROLLIUSHOME/etc");
	argvadd(&ac_psc_path, &av_psc_path, LAM_SYSCONFDIR);
/*
 * tools
 */
	t_tkill = DEFTRESETH;
/*
 * flags
 */
	fl_debug = opt_taken('d');;
	fl_verbose = opt_taken('v');
/*
 * files
 */
	if (opt_taken('c')) {
		f_psc = getparam('c');
	} else {
		f_psc = DEFFCONFIGH;
	}
}

/*
 *	setvars
 *
 *	Function:	- handles command line override of variables
 *			- vars: $inet_topo, $rtr_topo
 */
static void
setvars(int *ac, char ***av, char *var, char *opt)
{
	char		*override;

	override = malloc((unsigned) (strlen(var) + strlen(opt) + 2));
	if (override == 0) {
	  show_help(NULL, "lib-call-fail", "malloc", NULL);
	  exit(errno);
	}

	strcpy(override, var);
	strcat(override, " ");
	strcat(override, opt);

	if (argvadd(ac, av, override)) {
	  show_help(NULL, "lib-call-fail", "argvadd", NULL);
	  exit(errno);
	}

	free(override);
}
