/*
 * Copyright (c) 2001-2002 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD/JRV
 *
 *	$Id: scatter.c,v 6.4.2.1 2002/10/09 19:49:12 brbarret Exp $
 *
 *	Function:	- scatters buffers from root in process rank order
 *	Accepts:	- send buffer
 *			- send count
 *			- send datatype
 *			- recv buffer
 *			- recv count
 *			- recv datatype
 *			- root
 *			- communicator
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <lam_config.h>
#include <app_mgmt.h>
#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <rpisys.h>

/*@

MPI_Scatter - Sends data from one task to all other tasks in a group

Input Parameters:
+ sbuf - address of send buffer (choice, significant 
only at 'root') 
. scount - number of elements sent to each process 
(integer, significant only at 'root') 
. sdtype - data type of send buffer elements (significant only at 'root') 
(handle) 
. rcount - number of elements in receive buffer (integer) 
. rdtype - data type of receive buffer elements (handle) 
. root - rank of sending process (integer) 
- comm - communicator (handle) 

Output Parameter:
. rbuf - address of receive buffer (choice) 

.N IMPI

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_COMM
.N MPI_ERR_INTERCOMM
.N MPI_ERR_IMPI
.N MPI_ERR_COUNT
.N MPI_ERR_TYPE
.N MPI_ERR_BUFFER
.N MPI_ERR_ROOT

.N ACK
@*/
int MPI_Scatter(void *sbuf, int scount, MPI_Datatype sdtype, 
		void *rbuf, int rcount, MPI_Datatype rdtype, 
		int root, MPI_Comm comm)
{
	int		i;			/* favourite index */
	int		rank;			/* my rank */
	int		size;			/* group size */
	int		err;			/* error code */
	char		*ptmp;			/* temp. buffer */
	MPI_Aint	incr;			/* buffer increments */
	MPI_Status	status;			/* recv status */
	struct _gps	*p;			/* favourite pointer */

	lam_initerr();
	lam_setfunc(BLKMPISCATTER);
/*
 * Check for invalid arguments.
 */
	if ((comm == MPI_COMM_NULL) || LAM_IS_INTER(comm)) {
		return(lam_errfunc(comm, BLKMPISCATTER,
				lam_mkerr(MPI_ERR_COMM, 0)));
	}

	MPI_Comm_rank(comm, &rank);
	MPI_Comm_size(comm, &size);

	if ((root >= size) || (root < 0)) {
		return(lam_errfunc(comm, BLKMPISCATTER,
				lam_mkerr(MPI_ERR_ROOT, 0)));
	}

	if ((root == rank && sdtype == MPI_DATATYPE_NULL)
			|| (rdtype == MPI_DATATYPE_NULL)) {
		return(lam_errfunc(comm, BLKMPISCATTER,
				lam_mkerr(MPI_ERR_TYPE, 0)));
	}

	if ((rcount < 0) || (rank == root && scount < 0)) {
		return(lam_errfunc(comm, BLKMPISCATTER,
				lam_mkerr(MPI_ERR_COUNT, 0)));
	}

#if LAM_WANT_IMPI
	/* Remove this when IMPI collectives are implemented */
 
        if (LAM_IS_IMPI(comm)) {
	  return lam_err_comm(comm, MPI_ERR_COMM, 0, 
			      "Collectives not yet implemented on IMPI communicators");
	}
#endif


	LAM_TRACE(lam_tr_cffstart(BLKMPISCATTER));
/*
 * Remember required parameters.
 */
	p = &(comm->c_group->g_procs[root]->p_gps);

	lam_setparam(BLKMPISCATTER, root | (p->gps_grank << 16),
				(p->gps_node << 16) | p->gps_idx);
/*
 * Switch to collective communicator.
 */
	lam_mkcoll(comm);
/*
 * If not root, receive data.
 */
	if (rank != root) {
		err = MPI_Recv(rbuf, rcount, rdtype,
					root, BLKMPISCATTER, comm, &status);
		lam_mkpt(comm);

		if (err != MPI_SUCCESS) {
			return(lam_errfunc(comm, BLKMPISCATTER, err));
		}

		LAM_TRACE(lam_tr_cffend(BLKMPISCATTER,
						root, comm, rdtype, rcount));
		
		lam_resetfunc(BLKMPISCATTER);
		return(MPI_SUCCESS);
	}
/*
 * I am the root, loop sending data.
 */
	MPI_Type_extent(sdtype, &incr);
	incr *= scount;

	for (i = 0, ptmp = (char *) sbuf; i < size; ++i, ptmp += incr) {
/*
 * simple optimization
 */
		if (i == rank) {
			err = lam_dtsndrcv(ptmp, scount, sdtype, rbuf,
					rcount, rdtype, BLKMPISCATTER, comm);
		} else {
			err = MPI_Send(ptmp, scount, sdtype,
					i, BLKMPISCATTER, comm);
		}

		if (err != MPI_SUCCESS) {
			lam_mkpt(comm);
			return(lam_errfunc(comm, BLKMPISCATTER, err));
		}
	}

	lam_mkpt(comm);

	LAM_TRACE(lam_tr_cffend(BLKMPISCATTER, root, comm, rdtype, rcount));

	lam_resetfunc(BLKMPISCATTER);
	return(MPI_SUCCESS);
}
