/*
 * Copyright (c) 2001-2002 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD
 *
 *	$Id: lamcid.c,v 6.7.2.1 2002/10/09 19:49:10 brbarret Exp $
 *
 *	Function:	- manage context identifiers
 *			- trivial version
 */

#include <typical.h>
#include <mpi.h>
#include <mpisys.h>
#include <lam_config.h>

/*
 * local variables
 */
static int first = 1;
static unsigned char map[MPI_MAX_CID / (sizeof(char) * 8)];

/*
 * local functions
 */
static void rmcid(int cid);


/*
 *	lam_getcid
 *
 *	Function:	- get next available context ID
 *	Returns:	- context ID or LAMERROR
 */
int
lam_getcid(void)
{
  int byte, bit, cid;

  /* Ensure that the bitmap is initialized properly */

  if (first == 1) {
    lam_nukecids();
    first = 0;
  }

  /* See if everything is taken right up front */

  if (map[(MPI_MAX_CID / (sizeof(char) * 8)) - 1] >= (1 << 7))
    return LAMERROR;

  /* Note that the case where one communicator takes more than one CID
     is already handled (as long as *all* communicators take the same
     number of CIDs) by the fact that lam_setcid() will mark those
     CIDs unavailable, and lam_rmcid() unmarks them. */

  cid = 0;
  for (byte = (MPI_MAX_CID / (sizeof(char) * 8)) - 1; byte >= 0; byte--) {

    /* Catch the case where at least the MSB of the byte (i.e., the
       highest CID represented by this byte) is taken, so move on to
       the next byte (which must have alredy been zero, 'cause we
       passed it). */

    if (map[byte] >= (1 << 7)) {
      byte++;
      map[byte] = 1;
      cid = 8 * byte;
      break;
    } 

    /* Now the case where we must have some open bits on the MSB side
       of the byte; figure out what the largest one is */

    else if (map[byte] > 0) {
      for (bit = 7; bit >= 0; bit--)
	if ((map[byte] & (1 << bit)) != 0)
	  break;

      /* bit will always be >= 0 'cause of the outside if statement
	 (map[i] > 0), and will always be < 7 'cause of the first if
	 statement (>= (1 << 7) */
      
      bit++;
      map[byte] |= (1 << bit);
      cid = (8 * byte) + bit;
      break;
    }
  }

  /* Special case for MPI_COMM_WORLD */

  if (cid == 0)
    map[0] = 1;

  /* Return the calculated cid */

  return cid;
}

/*
 *	lam_setcid
 *
 *	Function:	- set highest used context ID
 *      Accepts:        - context ID 
 */
void
lam_setcid(int cid)
{
  int byte, bit;
  
  byte = cid / 8;
  bit = (1 << (cid % 8));
  map[byte] |= bit;

#if LAM_WANT_IMPI
  /* Shadow/IMPI collective communicator */
  cid++;
  byte = cid / 8;
  bit = (1 << (cid % 8));
  map[byte] |= bit;

  /* Datasync (positive)/Syancack (negative) communicator */
  cid++;
  byte = cid / 8;
  bit = (1 << (cid % 8));
  map[byte] |= bit;

#endif
}

/*
 *	lam_rmcid
 *
 *	Function:	- deallocate (remove) a context ID
 *	Accepts:	- context ID
 */
void
lam_rmcid(int cid)
{
  rmcid(cid);
#if LAM_WANT_IMPI
  /* Free the next 3 communicators as well */
  rmcid(cid + 1);
  rmcid(cid + 2);
#endif
}


/*
 *	rmcid
 *
 *	Function:	- deallocate (remove) a context ID
 *	Accepts:	- context ID
 */
static void
rmcid(int cid)
{
  int byte, bit;
  
  byte = cid / 8;
  bit = (1 << (cid % 8));

  map[byte] &= ~bit;
}


/*
 *	lam_nukecids
 *
 *	Function:	- deallocate all context IDs (cleanup)
 */
void
lam_nukecids(void)
{
  int i;
  for (i = 0; i < (MPI_MAX_CID / (sizeof(char) * 8)); i++)
    map[i] = 0;
}
