/* 
 * Copyright (C) 1999-2001 Joachim Wieland <joe@mcknight.de>
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA 02111, USA.
 */

#include <stdio.h>
#include <stdarg.h>
#include <sys/time.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/wait.h>
#include "jftpgw.h"

#ifdef HAVE_CRYPT_H
#	include <pwd.h>    /* getpass() */
#	define _XOPEN_SOURCE
#	include <crypt.h>  /* crypt() */
#endif

/* don't care for HAVE_SNPRINTF and HAVE_VSNPRINTF here, these functions are
   declared in jftpgw.h and implemented in sprintf.c which is sourced by
   log.c */


extern int timeout;
extern int sig_chld_occurred;
extern unsigned int number_of_children;
sigset_t chldterm_sigset, chldterm_oldset;
struct uidstruct runasuser;
char* errstr = 0;

void enough_mem(const void* ptr) {
	if (ptr == (void*) 0) {
		log(0, "Not enough memory for malloc. Exiting.");
		exit(1);
	}
}

/* concating snprintf
 *  *
 *  * determines the length of the string pointed to by `os', appending
 *  * formatted string to a maximium length of `len'.
 *  *
 *  */

void scnprintf (char *os, size_t len, const char *str, ...) {
	va_list vl;
	char *ostmp = os + strlen (os);

	va_start (vl, str);
	vsnprintf (ostmp, len - strlen (os) - 1, str, vl);
	va_end (vl);

	return;
}

void set_errstr(const char* s) {
	if (errstr) {
		free(errstr);
	}
	errstr = strdup(s);
}

const char* get_errstr(void) {
	if (errstr) {
		return errstr;
	} else {
		return "No detailed error information available... :-(";
	}
}

void free_errstr(void) {
	if (errstr) {
		free(errstr);
		errstr = (char*) 0;
	}
}


/* writes a char* to an fd and checks the return value */

int say(int fd, const char* phrase) {
	int i;
	struct timeval writetime = { 300, 0 };
	fd_set writeset;
	FD_ZERO(&writeset);
	FD_SET(fd, &writeset);

	i = select(fd + 1, NULL, &writeset, NULL, &writetime);
	if (i == 0) {
	        log(2, "Timeout reached in say()");
		log(2, "Timeout in %s line %d\n", __FILE__ ,__LINE__);
		timeout = 1;
		return -1;
	}
	if (i == -1) {
		log(1, "Error in select() in say(): %s", strerror(errno));
		return -1;
	}

	i = write(fd, phrase, strlen(phrase));
	if (i < 0) {
		log(3, "write (say) failed: %s", strerror(errno));
		log(3, "should say %s to %d", phrase, fd);
	}
	return i;
}


int changeid(const char* who, int what) {
	int i;
	uid_t uid;
	if (!who) {
		log(3, "No effective user name specified");
		return -1;
	}
	if (strcmp(who, "root") == 0) {
		uid = 0;
	} else if (strcmp(who, runasuser.username) == 0) {
		uid = runasuser.uid;
	} else {
		log(2, "Invalid user specified: %s", who);
		uid = getuid();
	}
	if (what == EUID) {
#ifdef HAVE_SETEUID
		i = seteuid(uid);
#else
		/* HP-UX does not know seteuid() */
		i = setreuid(-1, uid);
#endif

	}
	else {
		i = setuid(uid);
	}
	if (i) {
		log(3, "Could not change the %s to %d: %s",
				what == UID ? "UID" : "EUID",
				uid, strerror(errno));
	}
	else {
		log(8, "Changed %s to %d (%s)",
				what == UID ? "UID" : "EUID",
				uid, who);
	}
	return i;
}


/* just get the status of the child so that it can end */
void reap_chld_info (int signo) {
	int err = errno;
	int status;
	/* signal handler but waitpid() is reentrant */
	waitpid (-1, &status, WNOHANG);
	errno = err;
}

/* get the status of the child and unregister it */
void childterm (int signo) {
	int err = errno;
	int status;
	int ret;
	pid_t pid;

	/* block further signals of this type until we have unregistered the
	 * pid */

	sigemptyset(&chldterm_sigset);
	sigemptyset(&chldterm_oldset);
	sigaddset(&chldterm_sigset, SIGCHLD);
	while ((ret = sigprocmask(SIG_BLOCK, &chldterm_sigset,
		&chldterm_oldset)) < 0 && errno == EINTR)  {}
	if (ret < 0) {
		sig_chld_occurred = -errno;
		while ((ret = sigprocmask(SIG_UNBLOCK, &chldterm_sigset,
				&chldterm_oldset)) < 0 && errno == EINTR) {}
		if (ret < 0) {
			/* if an error occurres it is fatal */
			exit(1);
		}
		return;
	}
	pid = waitpid (-1, &status, WNOHANG);
	if (pid < 0) {
		sig_chld_occurred = -errno;
		while ((ret = sigprocmask(SIG_UNBLOCK, &chldterm_sigset,
				&chldterm_oldset)) < 0 && errno == EINTR) {}
		if (ret < 0) {
			/* if an error occurres it is fatal */
			exit(1);
		}
		return;
	}
	if (pid > 0) {
		/* okay, there was really a chlid */
		sig_chld_occurred = pid;
	} else {
		sig_chld_occurred = 0;
		while ((ret = sigprocmask(SIG_UNBLOCK, &chldterm_sigset,
			&chldterm_oldset)) < 0 && errno == EINTR) {}
		sig_chld_occurred = -errno;
	}
	errno = err;
}

int get_chld_pid() {
	int pid = sig_chld_occurred;
	int ret;

	log(9, "A child exited. Pid: %d", pid);
	number_of_children--;
	log(8, "Number of children: %d", number_of_children);
	if (pid < 0) {
		log(1, "waitpid() or sigprocmask() error: %s",
			strerror(-pid));
		return 0;
	}
	if (pid == 0) {
		return 0;
	}
	log(9, "unregistering pid ...");
	ret = unregister_pid(pid);
	if (ret) {
		log(3, "Error unregistering pid");
	} else {
		log(9, "unregistered");
	}
	sig_chld_occurred = 0;
	while ((ret = sigprocmask(SIG_UNBLOCK, &chldterm_sigset,
		&chldterm_oldset)) < 0 && errno == EINTR) {}

	if (ret < 0) {
		log(2, "Error unblocking signal mask: %s", strerror(errno));
		return -1;
	}
	return 0;
}


/* returns the first characters up to a whitespace character */

/* Checks if PATTERN is the beginning of RESPONSE */

int checkbegin(const char* response, const char* pattern) {
    if (strlen(response) < strlen(pattern)) {
        return 0;
    }
    return !strncasecmp(response, pattern, strlen(pattern));

}


/* returns the code in a response */

int getcode(const char* response) {
	char buffer[4];
	strncpy(buffer, response, sizeof(buffer) - 1);
	buffer[3] = '\0';
	return atoi(buffer);
}


/* Checks if the string RESONSE starts with SHOULDBE */

int checkdigits(const char* response, const int shouldbe) {
	return (getcode(response) == shouldbe);
}

/* extracts the numerical code out of an FTP server response. */

int respcode(const char* response) {
	int resp = 0;
	int i;
	const char* respoff = response;

	if (!response) {
		return 0;
	}

	while (respoff[3] != ' ') {
		respoff = strchr(respoff, '\n');
		if (!respoff) {
			return -1;
		} else {
			respoff++;   /* skip over '\n' */
		}
	}

	i = sscanf(respoff, "%d ", &resp);
	if (i != 1 && resp < 100) {
		return -1;
	}
	return resp;
}

char* gethostentip(char* iplist) {
        static char ipbuf[16];
        snprintf(ipbuf, 16, "%d.%d.%d.%d",
                (unsigned char) iplist[0],
                (unsigned char) iplist[1],
                (unsigned char) iplist[2],
                (unsigned char) iplist[3]
        );
        return ipbuf;
}

/* parsesock parses a comma separated list of IP and Port like in
 *  PORT 127,0,0,1,15,216
 * or the PASV answer.
 *
 */

int parsesock(char* buffer, struct sockaddr_in *sin, int mode) {
	int i1, i2, i3, i4, lo, hi, count;
	unsigned long int iaddr;
	unsigned int port;
	char ipbuf[16];

	memset((void*)sin, 0, sizeof(*sin));
	count = sscanf(buffer, "%d,%d,%d,%d,%d,%d",
			&i1, &i2, &i3, &i4, &hi, &lo);
	/* sscanf must have read 6 arguments and all the parameters must be
	 * less than 255 ( 0xff )
	 */
	if (!(count != 6 || i1 > 0xff || i2 > 0xff || i3 > 0xff || i4 > 0xff
			|| hi > 0xff || lo > 0xff)) {
		snprintf(ipbuf, 16, "%d.%d.%d.%d",
				i1, i2, i3, i4);
		iaddr = inet_addr(ipbuf);
		if (iaddr != -1 || !strcmp(ipbuf, BROADCAST)) {
			sin->sin_family = AF_INET;
			sin->sin_addr.s_addr = iaddr;
			port = hi * 256 + lo;
			sin->sin_port = htons(port);
			return 0;
		}
		else {
			log(3, "Invalid address in the PASV or PORT command: %s",
					buffer);
		}
	}
	log(3, "Error parsing IP and port from %s", buffer);
	return -1;
}

/* sets HOWMANY bits on. */

unsigned long int setlastbits(int howmany) {
        int i;
        unsigned long int e = 0;
        unsigned long int f = 0;
        for (i =0; i < (howmany - howmany % 8); i++) {
	        e = e << 1 | 1;
        }
	for (i = 0; i < (howmany % 8); i++) {
		f = f << 1 | 1;
	}
	f = f << (32 - (howmany % 8));
        return e | f;
}

void toupstr(char* s) {
	while (*s) {
		*s = toupper((int)*s);
		s++;
	}
}

char* trim(char * const s) {
	while( s[strlen(s)-1] == '\t' ||
	       s[strlen(s)-1] == ' ' ||
	       s[strlen(s)-1] == '\n' ||
	       s[strlen(s)-1] == '\r' ) {

		s[strlen(s)-1] = '\0';
	}
	return s;
}

void char_squeeze(char *const s, int c) {
	char last = 0;
	int i = 0, j = 0;
	char* tmp = malloc(strlen(s) + 1);
	enough_mem(tmp);

	do {
		if (s[i] == last && s[i] == c) { /* do nothing */ }
		else { tmp[j++] = s[i]; }

		last = s[i];

		i++;
	} while (s[i]);
	tmp[j] = '\0';
	strcpy(s, tmp);
	free(tmp);
}


struct ip_t parse_ip(const char* s) {
	struct ip_t s_ip;
	struct in_addr iaddr;
	int ret;
	const char* slash = strchr(s, '/');
	const char* dot;
	char* ip;
	if (!slash) {
		/* no netmask specified */
		slash = s + strlen(s);
	}
	ip = malloc(slash - s + 1);
	enough_mem(ip);
	strncpy(ip, s, slash - s);
	ip[slash - s] = '\0';

	ret = inet_aton(ip, &iaddr);
	free(ip); ip =0;
	if (!ret) {
		s_ip.ip = -1;
		s_ip.netmask = -1;
		return s_ip;
	}
	s_ip.ip = iaddr.s_addr;

	if (*slash) {
		slash++;
	}
	/* slash points to the netmask now or is 0 if none has been specified */
	if (!*slash) {
		s_ip.netmask = -1;   /* 255.255.255.255 */
		return s_ip;
	}
	dot = strchr(slash, '.');
	if (!dot) {
		/* a decimal number netmask has been specified */
		s_ip.netmask = setlastbits(atoi(slash));
		return s_ip;
	}
	ret = inet_aton(slash, &iaddr);
	s_ip.netmask = iaddr.s_addr;

	return s_ip;
}

int cmp_domains(const char* name, const char* pattern) {
	const char* start;
	/* the hostname may not be shorter than the pattern
	 *
	 * pattern: .foobar.mit.edu
	 * name:        bla.mit.edu
	 *
	 * => won't match
	 */

	if (strlen(name) < strlen(pattern)) {
		return 0;
	}
	start = name + strlen(name) - strlen(pattern);
	return !strcasecmp(start, pattern);
}

void err_time_readline(int fd) {
	log(2, "Timeout reached in readline()");
	say(fd, "500 Connection timed out\r\n");
}

void err_readline(int fd) {
	char* m;
	int e = errno;
	size_t msize;
	const char* err;
	char* s = "An error occured in ftp_readline: %s";

	if (!(err = get_errstr())) {
		err = strerror(e);
	}
	log(2, s, err);
	msize = strlen(s) + strlen(err) + 7;
	m = (char*) malloc(msize);
	enough_mem(m);
	strcpy(m, "500 ");
	snprintf(m + strlen(m),
			msize         /* initial size */
			- strlen(m)   /* "500 " */
			- 3,          /* \r\n\0 */
			s, err);
	strcat(m, "\r\n");
	say(fd, m);
	free(m);
}

/* extracts joe from joe@foo,21 or   foo bar from "foo bar"@bla,21 */

char* extract_username(const char* s) {
	const char* p;
	char* r;
	if (*s == '"') {
		p = strchr(s+1, '"');
		if (!p) {
			return 0;
		}
	} else {
		p = strchr(s, '@');
		if (!p) {
			p = strchr(s, ',');
		}
		if (!p) {
			p = s + strlen(s);
		}
	}
	r = malloc(p - s + 1);
	enough_mem(r);
	strncpy(r, s, p - s);
	r[p - s] = '\0';
	return r;
}

/* extracts joe@host from joe@host,21 */

char* extract_userhost(const char* s) {
	const char *p;
	char *r;
	if (*s == '"') {
		p = strchr(s+1, '"');
		if (!p) {
			return 0;
		}
		p = strchr(p+1, ',');
		if (!p) {
			p = s + strlen(s);
		}
	} else {
		p = strchr(s, ',');
		if (!p) {
			p = s + strlen(s);
		}
	}
	r = (char*)malloc(p - s + 1);
	enough_mem(r);
	strncpy(r, s, p - s);
	r[p - s] = '\0';
	return r;
}


int cryptcmp(const char* encrypted, const char* clear) {
	const char* cmp;
#ifdef HAVE_CRYPT
	char salt[3];
	salt[0] = encrypted[0];
	salt[1] = encrypted[1];
	salt[2] = '\0';
	cmp = crypt(clear, salt);
#else
	cmp = clear;
#endif
	return strcmp(cmp, encrypted);
}

char* cryptpw(const char* clear) {
#ifdef HAVE_CRYPT
	char* crypted = 0;
	char* ret = 0;
	char salt[3];
	char saltposs[] = { 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i',
		'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't',
		'u', 'v', 'w', 'x', 'y', 'z', 'A', 'B', 'C', 'D', 'E',
		'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P',
		'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '0',
		'1', '2', '3', '4', '5', '6', '7', '8', '9', '.', '/' };
	int idx1, idx2;

	time_t tm;
	tm = time(NULL) * getpid();
	srand(tm);

	idx1 = (int)((double)rand() / ((double)RAND_MAX + 1) * sizeof saltposs);
	idx2 = (int)((double)rand() / ((double)RAND_MAX + 1) * sizeof saltposs);
	salt[0] = saltposs[idx1];
	salt[1] = saltposs[idx2];
	salt[2] = '\0';

	crypted = crypt(clear, salt);
	ret = (char*) malloc(strlen(crypted) + 1);
	enough_mem(ret);
	strcpy(ret, crypted);
	printf("%s\n", crypted);
	return ret;
#else
	return "foo";
#endif
}


/* encrypt_password() just reads a password from stdin and outputs the
 * encrypted version */

void encrypt_password() {
#ifdef HAVE_CRYPT
	char* pw = getpass("Password: ");
	char* crypted = cryptpw(pw);
	memset(crypted, 0, strlen(pw));
#else
	char* crypted = "No crypt support compiled in.";
#endif
	printf("%s\n", crypted);
}


int gethostaddr(const char* name, struct in_addr* in) {
	struct hostent* host;
	int is_valid;

	is_valid = inet_aton(name, in);
	if (!is_valid) {
		host = gethostbyname(name);
		if (!host) {
			log(3, "could not look up %s", name);
			in->s_addr = -1;
			return 0;
		} else {
			memcpy((void*)&in->s_addr, (void*)host->h_addr,
					host->h_length);
		}
	}
	return 1;
}


char* to_ascii(char *data, int *len)
{
	int count, len2 = *len;
	char* buffer2 = (char*) malloc(*len * 2);
	int last = 0;
	char *b2ptr = buffer2;

	for (count = 0; count < len2; count++)
	{
		if ((data[count] == 10) && (last != 13))
		{
			*b2ptr = 13;
			b2ptr++;
			(*len)++;
		}
		*b2ptr = data[count];
		last = *b2ptr;
		b2ptr++;
	}
	return buffer2;
}


char* from_ascii(char *data, int *len)
{
	char *dptr = data;
	int count, len2 = *len;

	for (count = 0; count < len2; count++)
	{
		if (data[count] != 13)
		{
			*dptr = data[count];
			dptr++;
		}
		else
			(*len)--;
	}
	return data;
}

