/* upsd.c - watches ups state files and answers queries 

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              
*/

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "shared.h"
#include "shared-tables.h"
#include "upsd.h"
#include "config.h"
#include "version.h"
#include "common.h"
#include "proto.h"

#ifdef HAVE_SHMAT
#include <sys/ipc.h>
#include <sys/shm.h>
#endif

#ifdef HAVE_MSGGET
#include <sys/ipc.h>
#include <sys/msg.h>
#endif

/* structure for the linked list of each UPS that we track */
typedef struct {
	char	*fn;
	int	shmid;
	char	*name;
	itype	*info;
	int	stale;
#ifdef HAVE_SHMAT
	struct	shmid_ds shmbuf;
#endif
	int	numinfo;
	int	numlogins;
	int	fsd;		/* forced shutdown in effect? */
	void	*next;
}	upstype;

typedef struct {
	char	*name;
	unsigned int	addr;
	unsigned int	mask;
	void	*next;
}	acltype;

typedef struct {
	char	*addr;
	int	fd;
	struct sockaddr_in sock;
	char	rq[SMALLBUF];
	int	rqpos;
	char	*loginups;
	char	*password;
	void	*next;
}	ctype;

#define ACTION_GRANT	1
#define ACTION_DENY	2
#define ACTION_DROP	4

				/* 76543210 */
#define LEVEL_BASE	1	/* .......* */
#define LEVEL_MONITOR	3	/* ......** */
#define LEVEL_LOGIN	7	/* .....*** */
#define LEVEL_MASTER	15	/* ....**** */
#define LEVEL_MANAGER	31	/* ...***** */
#define LEVEL_ALL	255	/* ******** */

typedef struct {
	int	action;
	int	level;
	char	*aclname;
	char	*password;
	void	*next;
}	acctype;

	/* pointers to linked lists */
	upstype	*firstups = NULL;
	acltype *firstacl = NULL;
	acctype *firstacc = NULL;
	ctype	*firstclient = NULL, *currclient = NULL;

	/* stock reply when a variable is unknown */
	char	*unkreply = "UNKNOWN";

	/* by default TCP uses the same port - use -t to override */
	int	udpfd, udpport = UDPPORT, upsnum = 0;
	int	listenfd, tcpport = UDPPORT;

	/* default is to listen on all local interfaces */
	struct	in_addr	listenaddr;

	/* default 15 seconds before data is marked stale */
	int	maxage = 15;

/* search for a ups given the name */
upstype *findups (char *upsname)
{
	upstype	*temp;

	temp = firstups;

	/* NULL name -> return first one */
	if (!upsname)
		return (firstups);

	/* return first one if a good name isn't given */
	if ((!strcasecmp(upsname, "default")) || (!strcmp(upsname, "")))
		return (firstups);
	
	while (temp != NULL) {
		if (!strcasecmp(temp->name, upsname))
			return (temp);

		temp = temp->next;
	}

	return (NULL);
}

/* mark the data stale if this is new, otherwise do nothing */
void datastale (upstype *ups, const char* why)
{
	/* don't do anything if it's been stale or if we're just starting */
	if ((ups->stale == 1) || (ups->stale == -1))
		return;

	ups->stale = 1;

	syslog (LOG_NOTICE,
                "Data for UPS [%s] is stale - check support module (%s)\n",
	        ups->name, why);

	ups->numinfo = 0;

	/* if we just dropped a state file, free the buffer */
	if ((ups->info != NULL) && (ups->shmid < 0)) {
		free(ups->info);
		ups->info = NULL;
	}

	return;
}

/* mark the data ok if this is new, otherwise do nothing */
void dataok (upstype *ups)
{
	if (ups->stale == 0)
		return;

	if (ups->stale == -1)		/* first time */
		syslog (LOG_NOTICE, "Read data for UPS [%s] successfully\n",
		        ups->name);
	else				/* subsequent time */
		syslog (LOG_NOTICE, "Data for UPS [%s] is now OK\n", ups->name);

	if (ups->shmid >= 0)
		syslog (LOG_NOTICE, "Data source for UPS [%s]: SHM (%i)\n", 
		        ups->name, ups->shmid);
	else
		syslog (LOG_NOTICE, "Data source for UPS [%s]: %s\n", 
		        ups->name, ups->fn);

	ups->stale = 0;
}

/* load data for a ups called <upsname> */
void updateups (char *upsname)
{
	int	stfd, ret;
	long	tdiff;
	struct	stat fs;
	time_t	tod;
	upstype	*ups;
	itype	tempinfo;
static	int	brokencompile = 0;

	ups = findups (upsname);

	if (ups == NULL)		/* sanity check */
		return;

	if (ups->fn == NULL)		/* sanity check */
		return;

	time (&tod);

#ifdef HAVE_SHMAT
	if (ups->shmid >= 0) {

		/* update information on shared memory segment */
		if (shmctl(ups->shmid, IPC_STAT, &ups->shmbuf) != 0) {
			datastale(ups, "shmctl failed");
			ups->shmid = -1;
			ups->info = NULL;
			return;
		}

		/* check attaches - should be at least 2 - model + upsd */
		if (ups->shmbuf.shm_nattch < 2) {
			datastale(ups, "shm_nattch < 2");

			/* mark for deletion */
			shmctl (ups->shmid, IPC_RMID, NULL); 

			shmdt ((char *)ups->info);	/* detach */
			ups->info = NULL;
			ups->shmid = -1;
			return;
		}

		tdiff = tod - ups->shmbuf.shm_ctime;
		if (tdiff > maxage) {
			datastale(ups, "shm_ctime too old");
			return;
		}

		dataok(ups);

		/* make sure we track how big the array is */
		ups->numinfo = atoi(ups->info[0].value);

		/* in shm mode, we're done at this point */
		return;
	}
#endif	/* HAVE_SHMAT */

	stfd = open (ups->fn, O_RDONLY);
	if (stfd < 0) {
		datastale (ups, "open state file failed");
		return;
	}

	/* bring in first record to see what's up */
	ret = read (stfd, &tempinfo, sizeof(itype));
	if (ret < 1) {
		close (stfd);
		return;
	}

	/* if it's a shm pointer file, dereference it and switch modes */
	if (tempinfo.type == INFO_SHMID) {
		close (stfd);
#ifdef HAVE_SHMAT
		ups->shmid = atoi(tempinfo.value);
		ups->info = (itype *) shmat (ups->shmid, 0, 0);

		/* see if the shmat actually worked */
		if (ups->info == (itype *)(-1)) {
			ups->info = NULL;
			datastale(ups, "shmat failed");
		}

		brokencompile = 0;	/* keep gcc quiet */
#else
		if (brokencompile == 0) {	/* only complain once */
			brokencompile = 1;
			syslog (LOG_NOTICE, "Need shared memory mode but HAVE_SHMAT wasn't defined during compile!\n");
		}
#endif
		return;
	}

	/* otherwise it must be a normal file, so do a sanity check */
	if (tempinfo.type != INFO_MEMBERS) {
		syslog (LOG_NOTICE, "Broken state file - doesn't start with INFO_MEMBERS!\n");
		close (stfd);
		datastale(ups, "type != INFO_MEMBERS");
	}

	/* if the struct changes sizes, realloc it */
	if (atoi(tempinfo.value) != ups->numinfo) {
		ups->numinfo = atoi(tempinfo.value);
		ups->info = realloc (ups->info, ups->numinfo * sizeof(itype));
	}		

	ret = read (stfd, ups->info, sizeof(itype) * atoi(tempinfo.value) - 1);
	if (ret == -1) {
		perror ("read");
		close (stfd);
		return;
	}
	close (stfd);

	/* now see if the state file is stale or not */
	if (stat (ups->fn, &fs)) { /* failed */
		datastale(ups, "stat of state file failed");
		return;
	}

	time (&tod);
	tdiff = tod - fs.st_mtime;
	if (tdiff > maxage) {
		datastale(ups, "state file too old");
		return;
	}

	dataok(ups);
}

/* start monitoring a ups called <name> from a file called <fn> */
void addups (char *fn, char *name)
{
	upstype	*temp, *last;

	temp = last = firstups;

	if (!strcasecmp(name, "default")) {
		printf ("You can't add a UPS called default!\n");
		return;
	}

	/* find end of linked list */
	while (temp != NULL) {
		last = temp;

		if (!strcasecmp(temp->name, name)) {
			printf ("UPS name [%s] is already in use!\n", name);
			return;
		}

		temp = temp->next;
	}

	/* grab some memory and add the info */
	temp = malloc (sizeof(upstype));
	temp->fn = strdup(fn);
	temp->shmid = -1;
	temp->name = strdup(name);
	temp->info = NULL;
	temp->stale = -1;
	temp->numinfo = 0;
	temp->numlogins = 0;
	temp->fsd = 0;
	temp->next = NULL;

	if (last == NULL)	/* first one */
		firstups = temp;
	else			/* not the first */
		last->next = temp;

	updateups (name);
	updateups (name);	/* yes, this is supposed to be called twice */
	upsnum++;

	if (temp->stale)
		printf ("Warning: Data for UPS [%s] is stale at startup\n", name);
}

/* setup a socket for incoming udp requests */
void setupudp()
{
	struct	sockaddr_in	recv;

	if ((udpfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		perror ("Can't create socket");
		exit (1);
	}

	memset (&recv, '\0', sizeof(recv));
	recv.sin_addr = listenaddr;
	recv.sin_family = AF_INET;
	recv.sin_port = htons(udpport);

	if (bind(udpfd, (struct sockaddr *) &recv, sizeof(recv)) < 0) {
		perror ("Can't bind to socket");
		exit (1);
	}

	return;
}

/* create a listening socket for tcp connections */
void setuptcp()
{
	struct	sockaddr_in	server;
	int	res;

	if ((listenfd = socket (AF_INET, SOCK_STREAM, 0)) < 0) {
		perror ("socket");
		exit (1);
	}

	memset (&server, '\0', sizeof(server));
	server.sin_addr = listenaddr;
	server.sin_family = AF_INET;
	server.sin_port = htons(tcpport);

	if (bind (listenfd, (struct sockaddr *) &server, 
	   sizeof (struct sockaddr_in)) == -1) {
		perror ("bind");
		exit (1);
	}

	if ((res = fcntl(listenfd, F_GETFL, 0)) == -1) {
		perror ("fcntl(get)");
		exit (1);
	}

	if (fcntl(listenfd, F_SETFL, res | O_NDELAY) == -1) {
		perror ("fcntl(set)");
		exit (1);
	}

	if (listen (listenfd, 16)) {
		perror ("listen");
		exit (1);
	}

	return;
}

/* send the buffer <sendbuf> of length <sendlen> to host <dest> */
void sendback (struct sockaddr_in *dest, char *sendbuf, int sendlen)
{
	int	res, destfd;

	if (currclient == NULL)		/* udp */
		destfd = udpfd;
	else
		destfd = currclient->fd;

	res = sendto (destfd, sendbuf, sendlen, 0, (struct sockaddr *) dest,
	              sizeof (struct sockaddr));
	return;
}

/* see if <addr> matches the acl <aclname> */
int checkacl (char *aclname, struct sockaddr_in *addr)
{
	acltype	*tmp;
	int	aclchk, addrchk;

	tmp = firstacl;
	while (tmp != NULL) {
		if (!strcmp(tmp->name, aclname)) {
			aclchk = tmp->addr & tmp->mask;
			addrchk = ntohl(addr->sin_addr.s_addr) & tmp->mask;

			if (aclchk == addrchk) 
				return 1;	/* match */
		}

		tmp = tmp->next;
	}

	return 0;	/* not found */
}

int checkaccess (struct sockaddr_in *ip, int level)
{
	acctype	*tmp;
	int	ret, checkpw;
	char	*pw;

	/* only check password for certain levels */
	if (level >= LEVEL_LOGIN)
		checkpw = 1;
	else
		checkpw = 0;

	tmp = firstacc;

	/* only use password if in tcp mode */
	if (currclient == NULL)
		pw = NULL;
	else
		pw = currclient->password;

	while (tmp != NULL) {
		ret = checkacl (tmp->aclname, ip);

		/* if ip matches and access line provides the right level.. */

		if ((ret == 1) && ((tmp->level & level) == level)) {
			/* handle denials first */
			switch (tmp->action) {
				case ACTION_GRANT:
					break;	/* fallthrough to pw checks */
				
				case ACTION_DENY: 
					sendback (ip, "ERR ACCESS-DENIED\n", 18);

				/* silent return, no response */
				case ACTION_DROP:
				default:
					return (0);
			}

			/* ok, must be a grant now */

			if (checkpw == 0)	/* password not required */
				return (1);	/* OK */

			/* no password set on this access line? */
			if ((tmp->password == NULL) || 
			    (strlen(tmp->password) == 0))
				return (1);	/* OK */

			/* must require a password now, so check it */

			/* if this is a UDP client, bail out */
			if (!currclient) {
				sendback (ip, "ERR PASSWORD-REQUIRED\n", 22);
				return (0);	/* failed */
			}			

			/* if the client hasn't given us a password yet */
			if (currclient->password == NULL) {
				sendback (ip, "ERR PASSWORD-REQUIRED\n", 22);
				return (0);	/* failed */
			}

			/* if it matches, they're good */
			if (!strcmp(tmp->password, currclient->password))
				return (1);	/* OK */

			/* only thing left: failed password */

			sendback (ip, "ERR PASSWORD-INCORRECT\n", 23);
			return (0);	/* failed */

		} 	/* if (ip matches) && (level is provided) */

		/* otherwise ip didn't match or the level was inadequate */

		tmp = tmp->next;
	}

	/* default = drop */
	return (0);
}

/* decrement the login counter for this ups */
void declogins (char *upsname)
{
	upstype	*ups;

	ups = findups (upsname);

	if (ups == NULL) {
		syslog (LOG_INFO, "Tried to decrement invalid ups name\n");
		return;
	}

	ups->numlogins--;
}
/* increment the login counter for this ups */
void inclogins (char *upsname)
{
	upstype	*ups;

	ups = findups (upsname);

	if (ups == NULL) {
		syslog (LOG_INFO, "Tried to increment invalid ups name\n");
		return;
	}

	ups->numlogins++;
}

/* remove a client struct from the linked list */
void delclient (ctype *dclient)
{
	ctype	*tmp, *last;

	last = NULL;
	tmp = firstclient;

	while (tmp != NULL) {
		if (tmp == dclient) {		/* found it */
			free (tmp->addr);

			if (tmp->loginups != NULL) {
				declogins (tmp->loginups);
				free (tmp->loginups);
			}

			if (tmp->password != NULL)
				free (tmp->password);

			if (last == NULL)	/* deleting first entry */
				firstclient = tmp->next;
			else
				last->next = tmp->next;

			free (tmp);
			return;
		}
		last = tmp;
		tmp = tmp->next;
	}

	/* not found?! */
	syslog (LOG_INFO, "Tried to delete client struct that doesn't exist!\n");

	return;
}

/* return a name for a command given its identifier (from shared.h) */
char *cmdname (int cmdnum)
{
	int	i;
	char	logbuf[SMALLBUF];

	for (i = 0; netvars[i].name != NULL; i++)
		if (netvars[i].type == cmdnum)
			return (netvars[i].name);

	if (cmdnum != INFO_UNUSED) {
		snprintf (logbuf, sizeof(logbuf), "Unknown variable type 0x%04x", cmdnum);
		syslog (LOG_INFO, logbuf);
	}
	
	return (unkreply);
}

/* return a name for an instant command given its identifier (from shared.h) */
char *instcmdname (int cmdnum)
{
	int	i;
	char	logbuf[SMALLBUF];

	for (i = 0; instcmds[i].name != NULL; i++)
		if (instcmds[i].cmd == cmdnum)
			return (instcmds[i].name);

	if (cmdnum != INFO_UNUSED) {
		snprintf (logbuf, sizeof(logbuf), "Unknown cmd num 0x%04x", cmdnum);
		syslog (LOG_INFO, logbuf);
	}
	
	return (unkreply);
}

/* handler for "REQ NUMLOGINS" */
void numlogins (struct sockaddr_in *dest, upstype *ups, char *varname)
{
	char	ans[128];

	if (!checkaccess (dest, LEVEL_LOGIN))
		return;

	snprintf (ans, sizeof(ans), "ANS %s %i\n", varname, ups->numlogins);
	sendback (dest, ans, strlen(ans));
	return;
}

#ifdef HAVE_MSGGET

/* timeout handler */
void nomsg (int sig)
{
	return;
}

/* make our own struct since some systems don't have struct msgbuf */
typedef struct {
	long	mtype;
	msgtype	msg;
}	mbuf;

/* try to send a message to a ups */
int sendupsmsg (upstype *ups, msgtype *buf)
{
	mbuf	*mb;
	int	i, ret, msgid = -1;

	for (i = 1; i < ups->numinfo; i++) {
		if (ups->info[i].type == INFO_MSGID) {
			msgid = atoi(ups->info[i].value);
			break;
		}
	}

	if (msgid == -1)
		return (0);	/* no msg sent */

	mb = malloc (sizeof(mbuf) + UPSMSG_MAXLEN);

	mb->mtype = UPSMSG_TOMODEL;
	memcpy (&mb->msg, buf, UPSMSG_MAXLEN);
	
	ret = msgsnd (msgid, (struct msgbuf *) mb, UPSMSG_MAXLEN, 0);
	free (mb);

	if (ret == -1)
		return (0);	/* no msg sent */

	return (1);	/* send ok */
}	

/* try to receive a message from a ups module */
int getupsmsg (upstype *ups, msgtype *buf, int buflen, int wait)
{
	mbuf	*mb;
	int	i, ret, msgid = -1;

	for (i = 1; i < ups->numinfo; i++) {
		if (ups->info[i].type == INFO_MSGID) {
			msgid = atoi(ups->info[i].value);
			break;
		}
	}

	if (msgid == -1)
		return (0);	/* no msg received */

	mb = malloc (sizeof(mbuf) + UPSMSG_MAXLEN);

	/* timeout handler */
	signal (SIGALRM, nomsg);
	alarm (wait);

	ret = msgrcv (msgid, (struct msgbuf *) mb, UPSMSG_MAXLEN, UPSMSG_TOUPSD, 0);

	/* cancel */
	alarm (0);
	signal (SIGALRM, SIG_IGN);

	if (ret == -1) {
		free (mb);
		return (0);	/* no msg received */
	}

	memcpy (buf, &mb->msg, buflen);

	free (mb);
	return (1);	/* msg received */
}

#else

/* stubs for systems that don't have MSGGET */
int sendupsmsg (upstype *ups, msgtype *buf) { return 0; }
int getupsmsg (upstype *ups, msgtype *buf, int buflen) { return 0; }

#endif	/* HAVE_MSGGET */

/* parse varname[@upsname] into separate variables */
upstype *parsearg (char *arg, char **varname)
{
	char	*argtmp, *upsname;
	upstype	*ups;

	argtmp = strdup (arg);

	upsname = strchr (argtmp, '@');
	if (upsname) {
		*upsname++ = '\0';
		ups = findups (upsname);
	}
	else
		ups = findups ("default");

	/* varname may be a subset of argtmp, so copy it and free argtmp */
	*varname = strdup(argtmp);
	free (argtmp);

	return (ups);
}

/* handler for "REQ" - send a reply */
void do_sendans (struct sockaddr_in *dest, char *varin, char *rest)
{
	int	i, type = 0;
	char	ans[SMALLBUF], *varname;
	upstype	*ups;

	if (!checkaccess (dest, LEVEL_MONITOR))
		return;

	if (varin == NULL) {
		snprintf (ans, sizeof(ans), "ERR MISSING-ARGUMENT\n");
		sendback (dest, ans, strlen(ans));
		return;
	}

	ups = parsearg (varin, &varname);

	if (ups == NULL) {
		snprintf (ans, sizeof(ans), "ANS %s UNKNOWN-UPS\n", varin);
		sendback (dest, ans, strlen(ans));
		free (varname);
		return;
	}

	/* special case: numlogins is an internal value */
	if (!strcasecmp(varname, "NUMLOGINS")) {
		numlogins(dest, ups, varin);
		free (varname);
		return;
	}

	for (i = 0; netvars[i].name != NULL; i++)
		if (!strcasecmp(netvars[i].name, varname))
			type = netvars[i].type;
	
	/* type wasn't resolved in the netvars[] array */
	if (type == 0) {
		snprintf (ans, sizeof(ans), "ANS %s UNKNOWN\n", varin);
		sendback (dest, ans, strlen(ans));
		free (varname);
		return;
	}

	/* stock reply - overwritten if an answer is found */
	snprintf (ans, sizeof(ans), "ANS %s NOT-SUPPORTED\n", varin);

	if (ups->stale == 1)
		snprintf (ans, sizeof(ans), "ANS %s DATA-STALE\n", varin);
	else
		for (i = 1; i < ups->numinfo; i++) 
			if (ups->info[i].type == type) {

				/* special case - make FSD show up in STATUS */
				if ((type == INFO_STATUS) && (ups->fsd == 1)) 
					snprintf (ans, sizeof(ans), "ANS %s FSD %s\n", 
					          varin, ups->info[i].value);
				else
					snprintf (ans, sizeof(ans), "ANS %s %s\n", 
					          varin, ups->info[i].value);
				break;
			}

	sendback (dest, ans, strlen(ans));
	free (varname);
}

/* handler for "LISTVARS" */
void do_listvars (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	ans[SMALLBUF];
	int	i;
	upstype *ups;

	if (!checkaccess (dest, LEVEL_MONITOR))
		return;

	ups = findups (arg);

	if (ups == NULL) {
		snprintf (ans, sizeof(ans), "ERR UNKNOWN-UPS\n");
		sendback (dest, ans, strlen(ans));
		return;
	}

	snprintf (ans, sizeof(ans), "VARS");

	/* insert "@upsname" if explicitly specified in request */
	if ((arg) && (strcmp(arg, "") != 0))
		snprintf (&ans[strlen(ans)], sizeof(ans)-strlen(ans), " @%s",
		          arg); 

	for (i = 1; i < ups->numinfo; i++)
		if ((ups->info[i].type & INFO_SYSMASK) == 0)
			snprintf (&ans[strlen(ans)], sizeof(ans)-strlen(ans),
			          " %s", cmdname(ups->info[i].type));

	snprintf (&ans[strlen(ans)], sizeof(ans)-strlen(ans), "\n");
	sendback (dest, ans, strlen(ans));
}

/* handler for "VER" */
void do_sendver (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	ans[SMALLBUF];

	if (!checkaccess (dest, LEVEL_BASE))
		return;

	snprintf (ans, sizeof(ans),
	          "Network UPS Tools upsd %s - http://www.exploits.org/nut/\n", UPS_VERSION);
	sendback (dest, ans, strlen(ans));
}

/* handler for "HELP" */
void do_sendhelp (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	ans[SMALLBUF];
	int	i;

	if (!checkaccess (dest, LEVEL_BASE))
		return;

	snprintf (ans, sizeof(ans), "Commands:");

	for (i = 0; netcmds[i].name != NULL; i++) {
		snprintf (&ans[strlen(ans)], sizeof(ans)-strlen(ans), " %s",
		          netcmds[i].name);
	}

	snprintf (&ans[strlen(ans)], sizeof(ans)-strlen(ans), "\n");
	sendback (dest, ans, strlen(ans));
}

/* handler for "LOGOUT" */
void do_logout (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	logbuf[SMALLBUF];

	if (currclient == NULL) {
		sendback (dest, "Can't logout in UDP mode!\n", 26);
		return;
	}

	sendback (dest, "Goodbye...\n", 11);
	shutdown (currclient->fd, 2);
	close (currclient->fd);

	snprintf (logbuf, sizeof(logbuf), "Client on %s logged out\n", 
	          currclient->addr);
	syslog (LOG_INFO, logbuf);

	delclient (currclient);
}		

/* handler for "LOGIN" */
void do_login (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	logbuf[SMALLBUF];
	int	found = 0;
	upstype *ups;

	if (currclient == NULL) {
		sendback (dest, "Can't login in UDP mode!\n", 26);
		return;
	}

	if (!checkaccess (dest, LEVEL_LOGIN))
		return;

	if (currclient->loginups != NULL) {
		sendback (dest, "ERR ALREADY-LOGGED-IN\n", 23);
		snprintf (logbuf, sizeof(logbuf), "Client on %s tried to login twice\n",
		          currclient->addr);
		syslog (LOG_INFO, logbuf);
		return;
	}

	if ((arg == NULL) || (strlen(arg) == 0))	/* "LOGIN" only */
		currclient->loginups = strdup ("default");
	else {
		/* see if <arg> is a valid UPS name */

		for (ups = firstups; ups != NULL; ups = ups->next) {
			if (!strcasecmp(ups->name, arg)) {
				found = 1;
			}
		}

		if (!found) {
			sendback (dest, "ERR UNKNOWN-UPS\n", 17);
			return;
		}

		currclient->loginups = strdup (arg);
	}

	inclogins (currclient->loginups);

	snprintf (logbuf, sizeof(logbuf), "Client %s logged into UPS [%s]",
		  currclient->addr, currclient->loginups);
	syslog (LOG_INFO, logbuf);

	sendback (dest, "OK\n", 3);
}

/* handler for "PASSWORD" */
void do_password (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	logbuf[SMALLBUF];

	if (currclient == NULL) {
		sendback (dest, "Can't set password in UDP mode!\n", 33);
		return;
	}

	if (!checkaccess (dest, LEVEL_BASE))
		return;

	if ((arg == NULL) || (strlen(arg) == 0)) {
		sendback (dest, "ERR INVALID-PASSWORD\n", 22);
		snprintf (logbuf, sizeof(logbuf), "Client on %s sent a NULL password!\n",
		          currclient->addr);
		syslog (LOG_INFO, logbuf);
		return;	
	}

	if (currclient->password != NULL) {
		sendback (dest, "ERR ALREADY-SET-PASSWORD\n", 26);
		snprintf (logbuf, sizeof(logbuf), "Client on %s tried to set a password twice\n",
		          currclient->addr);
		syslog (LOG_INFO, logbuf);
		return;
	}

	currclient->password = strdup(arg);

	sendback (dest, "OK\n", 3);
}

/* handler for "LISTRW" */
void do_listrw (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	ans[SMALLBUF];
	int	i;
	upstype *ups;

	if (!checkaccess (dest, LEVEL_MONITOR))
		return;

	ups = findups (arg);

	if (ups == NULL) {
		snprintf (ans, sizeof(ans), "ERR UNKNOWN-UPS\n");
		sendback (dest, ans, strlen(ans));
		return;
	}

	snprintf (ans, sizeof(ans), "RW");

	/* insert "@upsname" if explicitly specified in request */
	if ((arg) && (strcmp(arg, "") != 0)) {
		snprintf (&ans[strlen(ans)], sizeof(ans)-strlen(ans), " @%s",
		          arg);
	}		

	for (i = 1; i < ups->numinfo; i++) {
		if ((ups->info[i].type & INFO_SYSMASK) == 0) {
			if (ups->info[i].flags && FLAG_RW) {
				snprintf (&ans[strlen(ans)], 
				          sizeof(ans)-strlen(ans), " %s",
				          cmdname(ups->info[i].type));
			}
		}
	}

	snprintf (&ans[strlen(ans)], sizeof(ans)-strlen(ans), "\n");
	sendback (dest, ans, strlen(ans));
}

/* handler for "VARTYPE" */
void do_vartype (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	ans[SMALLBUF], *varname;
	int	i, type;
	upstype *ups;

	if (!checkaccess (dest, LEVEL_MONITOR))
		return;

	if (arg == NULL) {
		snprintf (ans, sizeof(ans), "ERR MISSING-ARGUMENT\n");
		sendback (dest, ans, strlen(ans));
		return;
	}

	ups = parsearg (arg, &varname);

	if (ups == NULL) {
		snprintf (ans, sizeof(ans), "ERR UNKNOWN-UPS\n");
		sendback (dest, ans, strlen(ans));
		free (varname);
		return;
	}

	type = 0;

	/* find the variable type for the name */
	for (i = 0; netvars[i].name != NULL; i++)
		if (!strcasecmp(netvars[i].name, varname))
			type = netvars[i].type;

	if (type == 0) {	/* didn't find it */
		snprintf (ans, sizeof(ans), "ERR UNKNOWN-TYPE\n");
		sendback (dest, ans, strlen(ans));
		free (varname);
		return;
	}

	/* stock reply */
	snprintf (ans, sizeof(ans), "ERR NOT-SUPPORTED\n");

	/* try to find the variable in the info array */
	for (i = 1; i < ups->numinfo; i++)
		if (ups->info[i].type == type) {
			if (ups->info[i].flags & FLAG_ENUM)
				snprintf (ans, sizeof(ans), "TYPE ENUM %i\n",
				          ups->info[i].auxdata);

			if (ups->info[i].flags & FLAG_STRING)
				snprintf (ans, sizeof(ans), "TYPE STRING %i\n",
				          ups->info[i].auxdata);
		}

	sendback (dest, ans, strlen(ans));
	free (varname);
}

/* TODO: clean this up ... lots of identical code in here */

/* handler for "VARDESC" */
void do_vardesc (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	ans[SMALLBUF];
	int	i;

	if (!checkaccess (dest, LEVEL_MONITOR))
		return;

	if (arg == NULL) {
		snprintf (ans, sizeof(ans), "ERR MISSING-ARGUMENT\n");
		sendback (dest, ans, strlen(ans));
		return;
	}

	/* stock reply */
	snprintf (ans, sizeof(ans), "ERR NOT-SUPPORTED\n");

	/* find the variable type for the name */
	for (i = 0; netvars[i].name != NULL; i++)
		if (!strcasecmp(netvars[i].name, arg))
			snprintf (ans, sizeof(ans), "DESC \"%s\"\n", 
			          netvars[i].desc);

	sendback (dest, ans, strlen(ans));
}

/* handler for "ENUM" */
void do_enum (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	ans[SMALLBUF], *varname;
	int	i, type, pos;
	upstype *ups;

	if (!checkaccess (dest, LEVEL_MONITOR))
		return;

	if (arg == NULL) {
		snprintf (ans, sizeof(ans), "ERR MISSING-ARGUMENT\n");
		sendback (dest, ans, strlen(ans));
		return;
	}

	ups = parsearg (arg, &varname);

	if (ups == NULL) {
		snprintf (ans, sizeof(ans), "ERR UNKNOWN-UPS\n");
		sendback (dest, ans, strlen(ans));
		free (varname);
		return;
	}

	type = 0;

	/* find the variable type for the name */
	for (i = 0; netvars[i].name != NULL; i++)
		if (!strcasecmp(netvars[i].name, varname))
			type = netvars[i].type;

	if (type == 0) {	/* didn't find it */
		snprintf (ans, sizeof(ans), "ERR UNKNOWN-TYPE\n");
		sendback (dest, ans, strlen(ans));
		free (varname);
		return;
	}

	pos = 0;

	/* find the variable in the info array first */
	for (i = 1; i < ups->numinfo; i++)
		if (ups->info[i].type == type)
			pos = i;

	if (pos == 0) {
		snprintf (ans, sizeof(ans), "ERR NOT-SUPPORTED\n");
		sendback (dest, ans, strlen(ans));
		free (varname);
		return;
	}

	snprintf (ans, sizeof(ans), "ENUM %s\n", varname);
	sendback (dest, ans, strlen(ans));

	/* try to find the variable in the info array */
	for (i = 1; i < ups->numinfo; i++)
		if ((ups->info[i].type == INFO_ENUM) &&
		    (ups->info[i].auxdata == type)) {

			/* add "SELECTED" if this matches the current value */
			if (!strcmp(ups->info[i].value, ups->info[pos].value))
				snprintf (ans, sizeof(ans), "OPTION \"%s\" SELECTED\n",
				          ups->info[i].value);
			else
				snprintf (ans, sizeof(ans), "OPTION \"%s\"\n",
				          ups->info[i].value);
			sendback (dest, ans, strlen(ans));
		}

	snprintf (ans, sizeof(ans), "END\n");
	sendback (dest, ans, strlen(ans));
	free (varname);
}

/* handler for "SET" */
void do_set (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	ans[SMALLBUF], *newval, *varname, *ptr;
	upstype *ups;
	msgtype	*msg;
	int	i, infopos, type, found;

	if (!checkaccess (dest, LEVEL_MANAGER))
		return;

	if ((arg == NULL) || (rest == NULL)) {
		snprintf (ans, sizeof(ans), "ERR MISSING-ARGUMENT\n");
		sendback (dest, ans, strlen(ans));
		return;
	}

	ups = parsearg (arg, &varname);

	if (ups == NULL) {
		snprintf (ans, sizeof(ans), "ERR UNKNOWN-UPS\n");
		sendback (dest, ans, strlen(ans));
		free (varname);
		return;
	}

	type = 0;
	/* make sure 'varname' is part of the protocol as we know it */
	for (i = 0; netvars[i].name != NULL; i++)
		if (!strcasecmp(netvars[i].name, varname))
			type = netvars[i].type;
	
	/* type wasn't resolved in the netvars[] array */
	if (type == 0) {
		snprintf (ans, sizeof(ans), "ERR UNKNOWN-VAR\n");
		sendback (dest, ans, strlen(ans));
		free (varname);
		return;
	}

	/* jump out if this UPS is stale */
	if (ups->stale == 1) {
		snprintf (ans, sizeof(ans), "ERR DATA-STALE\n");
		sendback (dest, ans, strlen(ans));
		free (varname);
		return;
	}

	infopos = -1;
	/* now see if this UPS supports that variable */
	for (i = 1; i < ups->numinfo; i++) 
		if (ups->info[i].type == type) {
			infopos = i;
			break;
		}

	if (infopos == -1) {
		snprintf (ans, sizeof(ans), "ERR NOT-SUPPORTED\n");
		sendback (dest, ans, strlen(ans));
		free (varname);
		return;
	}

	/* is it even settable (RW) ? */
	if ((ups->info[infopos].flags & FLAG_RW) == 0) {	
		snprintf (ans, sizeof(ans), "ERR READONLY\n");
		sendback (dest, ans, strlen(ans));
		free (varname);
		return;
	}

	/* clean up rest into something usable */
	newval = strdup (rest);

	ptr = strchr (newval, 13);
	if (ptr)
		*ptr = '\0';	/* strip trailing CR */

	ptr = strchr (newval, 10);
	if (ptr)
		*ptr = '\0';	/* strip trailing LF */

	/* finally, see if the new value is allowed for this variable */

	/* check the length if this is for a string */
	if (ups->info[infopos].flags & FLAG_STRING) {
		if (strlen(newval) > ups->info[infopos].auxdata) {
			snprintf (ans, sizeof(ans), "ERR TOO-LONG\n");
			sendback (dest, ans, strlen(ans));
			free (newval);
			free (varname);
			return;
		}
	}

	if (ups->info[infopos].flags & FLAG_ENUM) {
		found = 0;
		/* try to find the variable in the info array */
		for (i = 1; i < ups->numinfo; i++) 
			if ((ups->info[i].type == INFO_ENUM) &&
			    (ups->info[i].auxdata == type) &&
			    (!strcmp(ups->info[i].value, newval)))
				found = 1;

		if (!found) {
			snprintf (ans, sizeof(ans), "ERR INVALID-VALUE\n");
			sendback (dest, ans, strlen(ans));
			free (newval);
			free (varname);
			return;
		}
	}

	msg = malloc (sizeof(msgtype) + strlen(newval));

	msg->cmdtype = UPSMSG_CMDSET;
	msg->auxcmd = type;
	strcpy (&msg->data, newval);
	msg->dlen = strlen(newval);

	/* TODO: lose debug msgs when done */
	if (sendupsmsg (ups, msg)) {
		syslog (LOG_INFO, "Sent set command to model\n");
		snprintf (ans, sizeof(ans), "OK\n");
	}
	else {
		syslog (LOG_INFO, "Set command send failed\n");
		snprintf (ans, sizeof(ans), "ERR SET-FAILED\n");
	}

	free (newval);
	free (msg);

	sendback (dest, ans, strlen(ans));
	free (varname);
}

/* handler for "INSTCMD" */
void do_instcmd (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	ans[SMALLBUF], *varname;
	upstype *ups;
	msgtype	*msg;
	int	i, cmd, found;

	if (!checkaccess (dest, LEVEL_MANAGER))
		return;

	ups = parsearg (arg, &varname);

	if (ups == NULL) {
		snprintf (ans, sizeof(ans), "ERR UNKNOWN-UPS\n");
		sendback (dest, ans, strlen(ans));
		free (varname);
		return;
	}

	cmd = 0;

	/* walk through instcmds[], strcmp */
	for (i = 0; instcmds[i].name != NULL; i++)
		if (!strcasecmp(instcmds[i].name, arg))
			cmd = instcmds[i].cmd;

	if (cmd == 0) {
		snprintf (ans, sizeof(ans), "ERR UNKNOWN-INSTCMD\n");
		sendback (dest, ans, strlen(ans));
		free (varname);
		return;
	}

	/* now make sure the UPS can actually DO this command */

	found = 0;
	for (i = 1; i < ups->numinfo; i++)
		if (ups->info[i].type == INFO_INSTCMD)
			if (ups->info[i].auxdata == cmd) {
				found = 1;
				break;
			}

	if (!found) {
		snprintf (ans, sizeof(ans), "ERR NOT-SUPPORTED\n");
		sendback (dest, ans, strlen(ans));
		free (varname);
		return;
	}

	msg = malloc (sizeof(msgtype) + 32);
	msg->cmdtype = UPSMSG_INSTCMD;
	msg->auxcmd = cmd;
	msg->dlen = 0;

	/* TODO: lose debug msgs when done */
	if (sendupsmsg (ups, msg)) {
		syslog (LOG_INFO, "Sent instant command to model\n");
		snprintf (ans, sizeof(ans), "OK\n");
	}
	else {
		syslog (LOG_INFO, "Instant command send failed\n");
		snprintf (ans, sizeof(ans), "ERR INSTCMD-FAILED\n");
	}

	sendback (dest, ans, strlen(ans));
	free (msg);
	free (varname);
}

/* handler for "LISTINSTCMD" */
void do_listinstcmd (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	ans[SMALLBUF];
	int	i;
	upstype *ups;

	if (!checkaccess (dest, LEVEL_MONITOR))
		return;

	ups = findups (arg);

	if (ups == NULL) {
		snprintf (ans, sizeof(ans), "ERR UNKNOWN-UPS\n");
		sendback (dest, ans, strlen(ans));
		return;
	}

	snprintf (ans, sizeof(ans), "INSTCMDS");

	/* insert "@upsname" if explicitly specified in request */
	if ((arg) && (strcmp(arg, "") != 0))
		snprintf (&ans[strlen(ans)], sizeof(ans)-strlen(ans), " @%s",
		          arg); 

	for (i = 1; i < ups->numinfo; i++)
		if (ups->info[i].type == INFO_INSTCMD)
			snprintf (&ans[strlen(ans)], sizeof(ans)-strlen(ans),
			          " %s", instcmdname(ups->info[i].auxdata));

	snprintf (&ans[strlen(ans)], sizeof(ans)-strlen(ans), "\n");
	sendback (dest, ans, strlen(ans));
}

/* handler for "INSTCMDDESC" */
void do_instcmddesc (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	ans[SMALLBUF];
	int	i;

	if (!checkaccess (dest, LEVEL_MONITOR))
		return;

	if (arg == NULL) {
		snprintf (ans, sizeof(ans), "ERR MISSING-ARGUMENT\n");
		sendback (dest, ans, strlen(ans));
		return;
	}

	/* stock reply */
	snprintf (ans, sizeof(ans), "ERR NOT-SUPPORTED\n");

	/* find the variable type for the name */
	for (i = 0; instcmds[i].name != NULL; i++)
		if (!strcasecmp(instcmds[i].name, arg))
			snprintf (ans, sizeof(ans), "DESC \"%s\"\n", 
			          instcmds[i].desc);

	sendback (dest, ans, strlen(ans));
}

/* handler for "FSD" */
void do_fsd (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	ans[SMALLBUF];
	upstype *ups;

	if (!checkaccess (dest, LEVEL_MASTER))
		return;

	ups = findups (arg);

	if (ups == NULL) {
		snprintf (ans, sizeof(ans), "ERR UNKNOWN-UPS\n");
		sendback (dest, ans, strlen(ans));
		return;
	}

	syslog (LOG_INFO, "Setting FSD on UPS [%s]\n", ups->name);

	ups->fsd = 1;
	snprintf (ans, sizeof(ans), "OK FSD-SET\n");
	sendback (dest, ans, strlen(ans));
}

/* handler for "MASTER" */
void do_master (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	ans[SMALLBUF];
	upstype *ups;

	if (!checkaccess (dest, LEVEL_MASTER))
		return;

	ups = findups (arg);

	if (ups == NULL) {
		snprintf (ans, sizeof(ans), "ERR UNKNOWN-UPS\n");
		sendback (dest, ans, strlen(ans));
		return;
	}

	/* really simple here - checkaccess does the work for us */

	snprintf (ans, sizeof(ans), "OK MASTER-GRANTED\n");
	sendback (dest, ans, strlen(ans));
}

/* parse requests from the network */
void parse (int len, char *buf, struct sockaddr_in *from)
{
	int	i;
	char	*cmd, *arg2, *rest, *ptr, ans[SMALLBUF];

	/* check for base access if this is a UDP client */
	if ((currclient == NULL) && (!checkaccess (from, LEVEL_BASE)))
		return;

	arg2 = rest = NULL;

	/* kill any embedded cr/lf chars */
	for (i = 0; i < len; i++)
		if ((buf[i] == 13) || (buf[i] == 10))
			buf[i] = '\0';

	/* parse from buf: <cmd> [<arg2> [<rest>]] */

	cmd = buf;
	ptr = strchr (cmd, ' ');
	if (ptr) {
		*ptr++ = '\0';
		arg2 = ptr;
		ptr = strchr (arg2, ' ');
		if (ptr) {
			*ptr++ = '\0';
			rest = ptr;
		}
	}

	for (i = 0; netcmds[i].name != NULL; i++) {
		if (!strcasecmp(netcmds[i].name, cmd)) {
			netcmds[i].func(from, arg2, rest);
			return;
		}
	}

	snprintf (ans, sizeof(ans), "ERR UNKNOWN-COMMAND\n");
	sendback (from, ans, strlen(ans));
}

/* update every ups that is being monitored */
void updateall (void)
{
	upstype *ups;

	ups = firstups;

	while (ups != NULL) {
		updateups (ups->name);
		ups = ups->next;
	}

	return;
}

/* add the acl to the linked list */
void addacl (char *aclname, char *ipblock)
{
	acltype	*tmp, *last;
	char	*addr, *mask;

	/* ipblock must be in the form <addr>/<mask>	*/

	mask = strchr (ipblock, '/');

	/* 206.253.95.200/32 : VALID			*/
	/* 206.253.95.200/255.255.255.255 : VALID	*/
	/* 206.253.95.200 : INVALID			*/

	if (!mask) 	/* no slash = broken acl declaration */
		return;

	*mask++ = '\0';
	addr = ipblock;

	tmp = last = firstacl;

	while (tmp != NULL) {	/* find end */
		last = tmp;
		tmp = tmp->next;
	}

	tmp = malloc (sizeof (acltype));
	tmp->name = strdup (aclname);
	tmp->addr = ntohl(inet_addr (addr));
	tmp->next = NULL;

	if (strstr(mask, ".") == NULL) { /* must be a /nn CIDR type block */
		if (atoi(mask) != 32)
			tmp->mask = ((unsigned int) ((1 << atoi(mask)) - 1) << 
			            (32 - atoi(mask)));
		else
			tmp->mask = 0xffffffff;	/* avoid overflow from 2^32 */
	}
	else
		tmp->mask = ntohl(inet_addr (mask));

	if (last == NULL)	/* first */
		firstacl = tmp;
	else
		last->next = tmp;
}

/* add to the access linked list */
void addaccess (char *action, char *level, char *aclname, char *password)
{
	acctype	*tmp, *last;

	tmp = last = firstacc;

	while (tmp != NULL) {	/* find end */
		last = tmp;
		tmp = tmp->next;
	}

	tmp = malloc (sizeof(acctype));
	tmp->action = 0;
	tmp->level = 0;

	if (!strcasecmp(action, "grant"))
		tmp->action = ACTION_GRANT;
	if (!strcasecmp(action, "deny"))
		tmp->action = ACTION_DENY;
	if (!strcasecmp(action, "drop"))
		tmp->action = ACTION_DROP;

	if (!strcasecmp(level, "base"))
		tmp->level = LEVEL_BASE;
	if (!strcasecmp(level, "monitor"))
		tmp->level = LEVEL_MONITOR;
	if (!strcasecmp(level, "login"))
		tmp->level = LEVEL_LOGIN;
	if (!strcasecmp(level, "master"))
		tmp->level = LEVEL_MASTER;
	if (!strcasecmp(level, "manager"))
		tmp->level = LEVEL_MANAGER;
	if (!strcasecmp(level, "all"))
		tmp->level = LEVEL_ALL;

	tmp->aclname = strdup(aclname);
	tmp->next = NULL;

	if ((password == NULL) || (strlen(password) == 0))
		tmp->password = NULL;
	else
		tmp->password = strdup(password);

	if (last == NULL)	/* first */
		firstacc = tmp;
	else
		last->next = tmp;	
}

/* parse the config file */
void readconf (void)
{
	FILE	*conf;
	char	buf[SMALLBUF], fn[SMALLBUF], *ptr, *arg[5], *sp;
	int	i;

	snprintf (fn, sizeof(fn), "%s/upsd.conf", CONFPATH);

	conf = fopen (fn, "r");

	if (!conf) {
		printf ("Can't open %s/upsd.conf: %s\n", CONFPATH, 
		        strerror(errno));
		exit (1);
	}

	while (fgets (buf, sizeof(buf), conf)) {
		buf[strlen(buf) - 1] = '\0';

		ptr = buf;

		for (i = 0; i < 5; i++) {
			arg[i] = ptr;

			if (!ptr)
				continue;

			sp = strchr (ptr, ' ');

			if (sp) {
				*sp = '\0';
				ptr = sp + 1;
			}
			else
				ptr = NULL;
		}

		/* UPS <upsname> <statefile> [ optional ] *
		 * optional data is not used by upsd      */
		if (!strcmp(arg[0], "UPS")) {
			if ((!arg[1]) || (!arg[2])) {
				printf ("upsd.conf: UPS directive needs 2 arguments\n");
				continue;
			}

			addups (arg[2], arg[1]);	/* fn, name */
		}

		/* ACL <aclname> <ip block> */
		if (!strcmp(arg[0], "ACL"))
			addacl (arg[1], arg[2]);

		/* ACCESS <action> <level> <aclname> <password> */
		if (!strcmp(arg[0], "ACCESS"))
			addaccess (arg[1], arg[2], arg[3], arg[4]);

		/* MAXAGE <seconds> */
		if (!strncmp(buf, "MAXAGE", 6))
			maxage = atoi (arg[1]);
	}
		
	fclose (conf);
}

/* answer incoming tcp connections */
void answertcp()
{
	int	acc, clen;
	struct	sockaddr_in client;
	char	logbuf[SMALLBUF];
	ctype	*tmp, *last;

	clen = sizeof (client);
	acc = accept (listenfd, (struct sockaddr *) &client, &clen);

	if (acc < 0)
		return;

	if (!checkaccess (&client, LEVEL_BASE)) {
		snprintf (logbuf, sizeof(logbuf),  "Rejecting TCP connection from %s\n", inet_ntoa (client.sin_addr));
		syslog (LOG_INFO, logbuf);
		close (acc);
		return;
	}

	snprintf (logbuf, sizeof(logbuf), "Connection from %s\n", inet_ntoa(client.sin_addr));
	syslog (LOG_INFO, logbuf);

	last = tmp = firstclient;

	while (tmp != NULL) {
		last = tmp;
		tmp = tmp->next;
	}

	tmp = malloc (sizeof(ctype));
	tmp->addr = strdup(inet_ntoa(client.sin_addr));
	tmp->fd = acc;
	memcpy (&tmp->sock, &client, sizeof(struct sockaddr_in));
	tmp->rqpos = 0;
	memset (tmp->rq, '\0', sizeof(tmp->rq));
	tmp->loginups = NULL;
	tmp->password = NULL;
	tmp->next = NULL;

	if (last == NULL)
 		firstclient = tmp;
	else
		last->next = tmp;
}

/* read tcp messages and handle them */
void readtcp (ctype *client)
{
	char	buf[SMALLBUF];
	int	i, ret;

	memset (buf, '\0', sizeof(buf));
	ret = read (client->fd, buf, sizeof(buf));

	if (ret < 1) {
		syslog (LOG_INFO, "Host %s disconnected\n", client->addr);

		shutdown (client->fd, 2);
		close (client->fd);
		delclient (client);
		return;
	}

	/* if an overflow will happen, then clear the queue */
	if ((ret + client->rqpos) >= sizeof(client->rq)) {
		memset (client->rq, '\0', sizeof(client->rq));
		client->rqpos = 0;
	}

	/* fragment handling code */

	for (i = 0; i < ret; i++) {
		/* add to the receive queue one by one */
		client->rq[client->rqpos++] = buf[i];

		/* parse on linefeed ('blah blahCRLF') */
		if (buf[i] == 10) {	/* LF */
			currclient = client;	/* enable tcp replies */
			parse (client->rqpos, client->rq, &client->sock);
			currclient = NULL;	/* disable */
			
			/* reset queue */
			client->rqpos = 0;
			memset (client->rq, '\0', sizeof(client->rq));
		}
	}

	return;
}

/* service requests and check on new data */
void mainloop (void)
{
	fd_set	rfds;
	struct	timeval	tv;
	struct	sockaddr_in from;
	int	res, fromlen, maxfd;
	char	buf[SMALLBUF];
	ctype	*tmpcli, *tmpnext;

	FD_ZERO (&rfds);
	FD_SET (udpfd, &rfds);
	FD_SET (listenfd, &rfds);
	
	tv.tv_sec = 2;
	tv.tv_usec = 0;

	maxfd = (udpfd > listenfd) ? udpfd : listenfd;

	/* scan through clients and add to FD_SET */
	for (tmpcli = firstclient; tmpcli != NULL; tmpcli = tmpcli->next) {
		FD_SET (tmpcli->fd, &rfds);
		if (tmpcli->fd > maxfd)
			maxfd = tmpcli->fd;
	}
	
	res = select (maxfd + 1, (void *) &rfds, (void *) NULL,
		     (void *) NULL, &tv);

	if (res) {
		if (FD_ISSET (udpfd, &rfds)) {
			fromlen = sizeof(from);
			memset (buf, '\0', sizeof(buf));
			res = recvfrom (udpfd, buf, sizeof(buf), 0, 
		        	       (struct sockaddr *) &from, &fromlen);

			if (res > 0)
				parse (res, buf, &from);
		}

		if (FD_ISSET (listenfd, &rfds))
			answertcp();

		/* scan clients for activity */

		tmpcli = firstclient;

		while (tmpcli != NULL) {

			/* preserve for later since delclient may run */
			tmpnext = tmpcli->next;

			if (FD_ISSET (tmpcli->fd, &rfds))
				readtcp (tmpcli);

			tmpcli = tmpnext;
		}
	}

	updateall();
}

#ifdef HAVE_SHMAT
/* detach from shared memory if necessary */
void detach_shm()
{
	upstype	*temp = firstups;

	while (temp != NULL) {
		if (temp->shmid >= 0) {
			/* mark for deletion */
			shmctl (temp->shmid, IPC_RMID, NULL); 
			shmdt ((char *)temp->info);	/* detach */
		}
		temp = temp->next;
	}	
}
#endif HAVE_SHMAT

void help (char *progname)
{
	printf ("usage: %s [-p portnum] [-t tcpport]\n", progname);
	printf ("\n");
	printf ("-p sets both port numbers (UDP, TCP)\n");
	printf ("-t overrides port number for TCP only\n");
	printf ("-i sets IP address of interface to listen\n");

	exit (1);
}

/* basic signal setup to ignore SIGPIPE */
void setupsignals()
{
	struct sigaction sa;
	sigset_t sigmask;

	/* ignore SIGPIPE */
	sa.sa_handler = SIG_IGN;
	sigemptyset (&sigmask);
	sa.sa_mask = sigmask;
	sa.sa_flags = 0;
	sigaction (SIGPIPE, &sa, NULL);
}

int main (int argc, char **argv)
{
	int	i;

	listenaddr.s_addr = INADDR_ANY;

	printf ("Network UPS Tools upsd %s\n", UPS_VERSION);

	while ((i = getopt(argc, argv, "+hp:t:i:")) != EOF) {
		switch (i) {
			case 'h':
				help(argv[0]);
				break;
			case 'p':
				udpport = atoi(optarg);
				tcpport = atoi(optarg);
				break;
			case 't':
				tcpport = atoi(optarg);
				break;
			case 'i':
				if (!inet_aton(optarg, &listenaddr))
				{
					printf ("Invalid IP address\n");
					exit (1);
				}
				break;
			default:
				help (argv[0]);
				break;
		}
	}

	argc -= optind;
	argv += optind;

	setupsignals();
	setupudp();
	setuptcp();

	/* open the log now since readconf() calls syslog() */
	openlog ("upsd", LOG_PID, LOG_FACILITY);

	readconf();

	droproot();

	if (upsnum == 0) {
		printf ("You need to add some UPS lines to your upsd.conf\n");
		exit (1);
	}

	background();

	for (;;)
		mainloop();

	return (0);	
}
