/* optiups.c - model specific routines for Opti-UPS units

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              
*/

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/termios.h>

#include "config.h"
#include "proto.h"
#include "shared.h"
#include "version.h"
#include "upscommon.h"
#include "common.h"

#define INFOMAX 16
#define UPS_DELAY 150000
#define ENDCHAR 10
#define IGNCHARS ""

#define DRIVER_VERSION "0.11.alpha.1"
#define SYSLOG_DEBUG 1

	int	shmok = 1, maxbattvolts = 0, cap_upstemp = 0;
	char	statefn[256];
	itype	*info;
extern	int	flag_timeoutfailure;
	float	lowvolt = 0, voltrange;
extern	int	upsfd;
	int	infomax = 16;
	int	sddelay = 10;	/* wait 10 seconds for shutdown */
	int	shutdowncmdset = 0; /* The shutdown command set to use */


void printheader(void)
{
	printf ("Network UPS Tools %s - Opti-UPS driver %s\n", UPS_VERSION, DRIVER_VERSION);
	printf ("modified by Ted A. Kuban <teddy@ladograd.ru>");
	openlog ("optiups", LOG_PID, LOG_FACILITY);

	printf ("\n*** EXPERIMENTAL VERSION ***\n");
	printf ("Please send strangeness/success/bug reports so this can become\n");
	printf ("a stable version soon.\n\n");
}

void usage (char *prog)
{
	printf ("usage: %s [-h] [-t <command>][-d <num>] [-s <num>] [-k] <device>\n", prog);
	printf ("Example: %s /dev/cuaa1\n", prog);
	exit (1);
}

void help (char *prog)
{
	printheader();
	printf ("usage: %s [-h] [-t <command>][-d <num>] [-s <num>] [-k] <device>\n", prog);
	printf ("\n");
	printf ("-d <num> - wait <num> seconds after sending shutdown command\n");
	printf ("-h       - display this help\n");
	printf ("-k       - force shutdown\n");
	printf ("-s <num> - use shutdown command set <num> for forced shutdown\n");
	printf ("           0 - combined (default), 1 - simple, 2 - delayed\n");
	printf ("-t       - send custom command to port and print result\n");
	printf ("<device> - /dev entry corresponding to UPS port\n");
	exit (1);
}


/* send command , wait for an answer 
   in the form <data><endchar> and store in buf */
void upsrequest(char *upscmd, char *buf, int buflen)
{
	int	i = -1;
#ifdef SYSLOG_DEBUG
	syslog (LOG_DEBUG, "Sending %s", upscmd);
#endif
	while	(upscmd[++i] != 0) {
		upssendchar (upscmd[i]);
	}
	usleep (UPS_DELAY);
	upsrecv (buf, buflen, ENDCHAR, IGNCHARS);
#ifdef SYSLOG_DEBUG
	syslog (LOG_DEBUG, "Get %s", buf);
#endif
}

void initinfo (void)
{
	int	i;

	info = create_info (INFOMAX, shmok);

	/* clear out everything first */
	for (i = 0; i < INFOMAX; i++)
		info[i].type = INFO_UNUSED;

	/* number of variables - max possible */
	info[0].type = INFO_MEMBERS;
	snprintf (info[0].value, sizeof(info[0].value), "%i", INFOMAX);

	/* now set up room for all future variables that are supported */
	info[1].type = INFO_MFR;
	info[2].type = INFO_MODEL;
	info[3].type = INFO_STATUS;
	info[4].type = INFO_UTILITY;
	info[5].type = INFO_LOADPCT;
	info[6].type = INFO_UPSTEMP;
	info[7].type = INFO_BATTPCT;
	info[8].type = INFO_LOWXFER;
	info[9].type = INFO_HIGHXFER;
	info[10].type = INFO_ACFREQ;
}

char *rtrim (char *in)
{
	int	i;
	char	tmp[256];

	strncpy (tmp, in, sizeof(tmp) - 1);

	for (i = 0; i < strlen(tmp); i++)
		if (tmp [i] == ' ')
			tmp[i] = 0;

	return (strdup(tmp));
}

void openser(void)
{
	struct	termios	tio;
	int	fd, flags;
 
	fd = open("/dev/ttyS0", O_RDWR | O_NONBLOCK);

	if (fd < 0) {
		printf ("Port not available.\n");
		exit (-1);
	}

	tio.c_cflag = 0 | CREAD | CS8 | HUPCL | CLOCAL;
	tio.c_iflag = 0 | IXANY | IMAXBEL | IXOFF;
	tio.c_oflag = 0 | ONLCR;
	tio.c_lflag = 0 | ECHOE | ECHOKE | ECHOCTL | PENDIN;
	tio.c_cc[VMIN] = 1;
	tio.c_cc[VTIME] = 0;

#ifdef HAVE_CFSETISPEED
	cfsetispeed (&tio, B2400);
	cfsetospeed (&tio, B2400);
#endif

	tcsetattr (fd, TCSANOW, &tio);

	flags = 0;
	ioctl (fd, TIOCMSET, &flags);

	upsfd = fd;
}

void getbaseinfo (char *port)
{
	char	temp[256], mch[8], mtemp[8];
	int	flags, sl;

	/* kill DTR and RTS */

	flags = 0;
	ioctl (upsfd, TIOCMSET, &flags);

	/* manufacturer */

	upsrequest ("IM", temp, sizeof(temp));
	snprintf (info[1].value, sizeof(info[1].value), temp);

	/* model parsing */

	upsrequest ("IO", temp, sizeof(temp));
	sl = strlen (temp);
	snprintf (mch, sizeof(mch), "%c%c", temp[sl-2], temp[sl-1]);
	strcpy (mtemp, temp);
	mtemp [sl-2] = 0;

	switch (mch[0]) {
		case 'E': snprintf (info[2].value, sizeof(info[2].value), 
		          "PowerES %ses", mtemp);
			  break;

		case 'P': snprintf (info[2].value, sizeof(info[2].value), 
		          "PowerPS %sps", mtemp);
			  break;

		case 'V': snprintf (info[2].value, sizeof(info[2].value), 
		          "PowerVS %svs", mtemp);
			  break;

		default: snprintf (info[2].value, sizeof(info[2].value), 
		          "Unknown (%s)", temp);
			  break;
	}

/*	printf ("Detected %s on %s\n", getdata(INFO_MODEL), port); */
	printf ("Detected\n");
	printf ("Detected %s %s on %s\n", info[1].value, info[2].value, port);

	/* low transfer */
	upsrequest ("FL", temp, sizeof(temp));
	snprintf (info[8].value, sizeof(info[8].value), "%s", temp);
	syslog (LOG_DEBUG, temp);

	/* high transfer */
	upsrequest ("FH", temp, sizeof(temp));
	snprintf (info[9].value, sizeof(info[8].value), "%s", temp);
	return;
}

/* normal idle loop - keep up with the current state of the UPS */
void updateinfo (void)
{
	char	temp[256];
	int	tval, battv;
	float	battf;

	upsrequest ("AG", temp, sizeof(temp));
	tval = strtol (temp, 0, 16);

	strcpy (info[3].value, "");

	if (tval & 8)
		strcat (info[3].value, "OVER ");	/* overload */
	if (tval & 16)
		strcat (info[3].value, "RB ");		/* replace batt */
	if (tval & 32)
		strcat (info[3].value, "OB ");		/* on battery */
	else
		strcat (info[3].value, "OL ");		/* on line */
	if (tval & 64)
		strcat (info[3].value, "LB ");		/* low battery */

	if (info[3].value[strlen(info[3].value)-1] == ' ')
		info[3].value[strlen(info[3].value)-1] = 0;

	/* input voltage */
	upsrequest ("NV", temp, sizeof(temp));
	snprintf (info[4].value, sizeof(info[4].value), "%s", temp);

	/* ups load percent */
	upsrequest ("OL", temp, sizeof(temp));
	snprintf (info[5].value, sizeof(info[5].value), "%s", temp);

	/* ups temp */
	upsrequest ("BT", temp, sizeof(temp));
	snprintf (info[6].value, sizeof(info[6].value), "%s", temp);

	/* battery charge */
	upsrequest ("BV", temp, sizeof(temp));

	/* battery voltage range: 10.4 - 13.0 VDC */
	battv = atoi(temp);
	battf = (((battv / 10.0) - 10.4) / 2.6) * 100.0;
	if (battf > 100.0)
		battf = 100.0;
	snprintf (info[7].value, sizeof(info[7].value), "%2.1f", battf);

	/* input freq */
	upsrequest ("FF", temp, sizeof(temp));

	snprintf (info[10].value, sizeof(info[10].value), "%2.1f", 
	          atof(temp) / 10.0);	

	writeinfo(info);
}

/* power down the attached load immediately */
void forceshutdown(char *port)
{
	char	temp[256];
	int	tval;

	syslog (LOG_INFO, "Initiating UPS shutdown\n");
	printf ("Initiating forced UPS shutdown on port %s!\n", port);

	open_serial (port, B2400);

	upsrequest ("IM", temp, sizeof(temp));
	printf("UPS Model - %s", temp);
	if (strcmp(temp, "OPTI-UPS") != 0)
		printf ("Detection failed.  Trying a shutdown command anyway.\n");

	/* check the line status */

	upsrequest ("AG", temp, sizeof(temp));
	tval = strtol (temp, 0, 16);

	/* Send a combined set of shutdown commands which can work better */
	/* if the UPS gets power during shutdown process */
	/* Specifically it sends both the soft shutdown 'S' */
	/* and the powerdown after grace period - '@000' commands */
	printf ("UPS - currently %s - sending shutdown/powerdown\n",
	       (tval & 16) ? "on battery" : "on-line");
	upsrequest("BT", temp, sizeof(temp));
	if (sddelay > 0) {
		sleep (sddelay);
		upsrequest("BV", temp, sizeof(temp)); /* delayed shutdown */
		printf ("Hmm, did the shutdown fail?  Oh well...\n");
		upsrequest ("AG", temp, sizeof(temp));
		tval = strtol (temp, 0, 16);
		printf ("UPS - currently %s",
		       (tval & 16) ?	"on battery - waiting for empty battery\n"
					: "on-line - rebooting\n");
		exit (1);
	}
	exit (0);
}

int main (int argc, char **argv)
{
/*	char	*portname; */
	char	*portname, *prog, temp[256];
	char	*testcommand = NULL;
	int	i;
	int	killpower = 0;

	/* TODO: getopt */
	prog = argv[0];

	printheader();

	while ((i = getopt(argc, argv, "+hkd:s:t:")) != EOF) {
		switch (i) {
			case 'd':
				sddelay = atoi(optarg);
				break;
			case 'k':
				printf("This function are in testing!\n");
				killpower = 1;
				/* forceshutdown(optarg); */
				break;
			case 'h':
				help(prog);
				break;
			case 's':
				shutdowncmdset = atoi(optarg);
				break;
			case 't':
				testcommand = optarg;
				break;
			default:
				usage(prog);
				break;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc < 1) {
		printf ("Error: no device specified!\n");
		usage (prog);
	}

	portname = NULL;
	for (i = strlen(argv[0]); i >= 0; i--)
		if (argv[0][i] == '/') {
			portname = &argv[0][i+1];
			break;
		}

	if (portname == NULL) {
		printf ("Unable to abbreviate %s\n", argv[0]);
		exit (1);
	}

	if (killpower == 1) {
		forceshutdown(argv[0]);
		exit(0);
	}


	droproot();

	snprintf (statefn, sizeof(statefn), "%s/optiups-%s", STATEPATH,
	          portname);

	/* FIXME: why is open_serial so stupid in this module? */
	open_serial (argv[0], B2400);
/*	openser(); */

	if (testcommand != NULL) {
		upsrequest (testcommand, temp, sizeof(temp));
		printf("Command %s; UPS return:\n", testcommand);
		printf("%s\n", temp);
		exit(0);
	}

	initinfo();

	createmsgq();		/* try to create IPC message queue */

	getbaseinfo(argv[0]);

 	background();

	for (;;) {
		updateinfo();
		/* wait up to 2 seconds for a message from the upsd */
		if (getupsmsg(2))
			syslog (LOG_INFO, "Received a message from upsd\n");
	}
}
