/*
   newapc.c - model specific routines for APC smart protocol units

	$Id: newapc.c,v 1.1.2.27 2000/12/06 11:03:20 cvs Exp $

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>
             (C) 2000  Nigel Metheringham <Nigel.Metheringham@Intechnology.co.uk>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              
*/

#define APC_DRIVER_VERSION	"$Revision: 1.1.2.27 $"
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <time.h>
#include <unistd.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/termios.h>

#include "shared.h"
#include "shared-tables.h"
#include "newapc.h"
#include "config.h"
#include "version.h"
#include "upscommon.h"
#include "common.h"

	int	shmok = 1;
	char	statefn[256];
	itype	*info;
extern	int	flag_timeoutfailure;
	int	infomax = 128;
	int	upslevel = 0;
	
	int	sddelay = 60;	/* wait 60 seconds for shutdown */
	int	shutdowncmdset = 0; /* The shutdown command set to use */
extern	struct	ups_handler	upsh;

	/* alternate cable handling support */
	char	*cable = NULL;
#define ALT_CABLE_1 "940-0095B"

extern	int	upsfd;
extern	int	upsc_debug;
	int	debug = 0;
/* tables of mappings built on startup */
struct apc_cmdtab_t *chrmap[256];
struct apc_cmdtab_t **infomap;
int max_info_num = 0;
struct apc_cmdtab_t **cmdmap;
int max_cmd_num = 0;
struct apc_cmdtab_t **drvmap;
int max_drv_num = 0;

/* this macro gives you the apc cmd table entry for a cmd char */
#define CHR_LOOKUP(x)	(chrmap[(x) & 0xFF])
#define CMDTABNAME(x)	(((x)->name == NULL) ? "UNKNOWN" : (x)->name)
#define CMDTABDESC(x)	(((x)->desc == NULL) ? "unknown" : (x)->desc)

/* Return the apc_cmdtab entry for an INFO/CMD/DRV value */
static struct apc_cmdtab_t * 
info_lookup(int key) {
	if (key & CMD_MASK) {
		key &= ~CMD_MASK;
		return((key < max_cmd_num) ? cmdmap[key] : NULL);
	} else if (key & DRV_MASK) {
		key &= ~DRV_MASK;
		return((key < max_drv_num) ? drvmap[key] : NULL);
	} else {
		return((key < max_info_num) ? infomap[key] : NULL);
	}
}


void intialise_data_tables()
{
	struct apc_cmdtab_t * cmd_entry;
	struct netvars_t * netvar_entry;
	struct instcmds_t *instcmd_entry;

	/* make sure chrmap zeroed */
	memset(chrmap, 0, sizeof(chrmap));

	/* iterate through the command table setting up cmd mapping */
	for(cmd_entry = &apc_cmdtab[0];
	    (cmd_entry->info_type > 0);
	    cmd_entry++) {
		int ival = cmd_entry->info_type & ~(DRV_MASK|CMD_MASK);
		chrmap[cmd_entry->cmd & 0xFF] = cmd_entry;

		/* Record the highest index in each class */
		if (cmd_entry->info_type & DRV_MASK) {
			if (ival > max_drv_num)
				max_drv_num = ival;
		} else if (cmd_entry->info_type & CMD_MASK) {
			if (ival > max_cmd_num)
				max_cmd_num = ival;
		} else {
			if (ival > max_info_num)
				max_info_num = ival;
		}
	}

	/* Now allocate the infomap and zero that out */
	max_info_num++;
	infomap = calloc(max_info_num, sizeof(cmd_entry));

	/* and for the cmdmap */
	max_cmd_num++;
	cmdmap = calloc(max_cmd_num, sizeof(cmd_entry));

	/* and for the cmdmap */
	max_drv_num++;
	drvmap = calloc(max_drv_num, sizeof(cmd_entry));

	/* calloc(3) man says it is pre-zeroed - trust it for now */

	/* iterate through the command table setting up info, cmd & drv mapping */
	for(cmd_entry = &apc_cmdtab[0];
	    (cmd_entry->info_type > 0);
	    cmd_entry++) {
		int ival = cmd_entry->info_type & ~(DRV_MASK|CMD_MASK);
		if (cmd_entry->info_type & CMD_MASK) 
			cmdmap[ival] = cmd_entry;
		else if (cmd_entry->info_type & DRV_MASK) 
			drvmap[ival] = cmd_entry;
		else
			infomap[ival] = cmd_entry;
	}

	/* Now iterate through the netvars table merging them in */
	for(netvar_entry = &netvars[0];
	    netvar_entry->name;
	    netvar_entry++) {
		cmd_entry = info_lookup(netvar_entry->type);
		if (cmd_entry) {
			cmd_entry->name = netvar_entry->name;
			cmd_entry->desc = netvar_entry->desc;
		}
	}
	/* and the instcmds table */
	for(instcmd_entry = &instcmds[0];
	    instcmd_entry->name;
	    instcmd_entry++) {
		cmd_entry = info_lookup(instcmd_entry->cmd);
		if (cmd_entry) {
			cmd_entry->name = instcmd_entry->name;
			cmd_entry->desc = instcmd_entry->desc;
		}
	}
}


char * convert_ups2info (struct apc_cmdtab_t * cmd_entry, char * upsval)
{
	static char tmp[128];
	int  tval;
	char *ptr;

	/* Do the magic formatting stuff
	   Currently this is inlined - could move to using 
	   function pointers in the command table */
	switch(cmd_entry->driver_flags & APC_FORMATMASK) {
	case APC_F_ST:
		tval = strtol (upsval, 0, 16);
		tval &= 0xFF;

		if (tval > 0) {
			strcpy (tmp, "");
			/* Position the more important ones first */
			if (tval & 8)
				strcat(tmp, "OL ");	/* on line */
			if (tval & 16)
				strcat(tmp, "OB ");	/* on battery */
			if (tval & 32)
				strcat(tmp, "OVER ");	/* overload */
			if (tval & 64)
				strcat(tmp, "LB ");	/* low battery */
			if (tval & 1)
				strcat(tmp, "CAL ");	/* calibration */
			if (tval & 2)
				strcat(tmp, "TRIM ");	/* SmartTrim */
			if (tval & 4)
				strcat(tmp, "BOOST ");	/* SmartBoost */
			if (tval & 128)
				strcat(tmp, "RB ");	/* replace batt */	

				/* lose trailing space if present */
			tmp[strlen(tmp)-1] = 0;
		} else
			strcpy (tmp, "OFF");

		ptr = tmp;
		break;
	case APC_F_PERCENT:
	case APC_F_VOLT:
	case APC_F_AMP:
	case APC_F_CELCIUS:
	case APC_F_HEX:
	case APC_F_DEC:
	case APC_F_SECONDS:
	case APC_F_MINUTES:
	case APC_F_HOURS:
	case APC_F_LEAVE:
				/* All of these just pass through at present */
		ptr = upsval;
		break;
	default:
				/* Moan */
		printf("APC capability: %s (%s) has unknown format\n",
		       CMDTABNAME(cmd_entry), CMDTABNAME(cmd_entry));
		ptr = upsval;
		break;
	}
	return ptr;
}

int send_ups(int capability)
{
	struct apc_cmdtab_t * cmd_entry;

	cmd_entry = info_lookup(capability);
	if (cmd_entry == NULL) {
		if (debug) {
			printf("APC capability: %c 0x%04x UNKNOWN\n",
			       (isprint(capability) ? capability : ' '),
			       capability);
		}
		return(1);
	}

	if (debug)
		printf("APC sending %s\n", CMDTABNAME(cmd_entry));
	upssendchar(cmd_entry->cmd);

	/* Deal with commands that need doubling */
	if (cmd_entry->driver_flags & APC_REPEAT) {
		usleep (1500000);
		upssendchar(cmd_entry->cmd);
	}
	return(0);
}

int query_ups(int capability,
	       int first)
{
	char	temp[256], *ptr;
	int	recvret, retval;
	struct apc_cmdtab_t * cmd_entry;

	cmd_entry = info_lookup(capability);
	if (cmd_entry == NULL) {
		if (debug) {
			printf("APC capability: %c 0x%04x UNKNOWN\n",
			       (isprint(capability) ? capability : ' '),
			       capability);
		}
		return(1);
	}
	
	/* This catches things we don't want to send to the UPS like instant commands */
	if ((cmd_entry->driver_flags & APC_IGNORE) ||
	    (capability & (CMD_MASK | DRV_MASK))) {
		if (first) {
			/* We assume this is present, but cannot actually test */
			cmd_entry->driver_flags |= APC_PRESENT;
			if (cmd_entry->driver_flags & APC_CMD)
				addinfo (INFO_INSTCMD, "", 0, cmd_entry->info_type);
		}
		if (debug)
			printf("APC capability: %s not tested - IGNORED\n",
			       CMDTABNAME(cmd_entry));
		return(1);
	}

	retval = 0;
	if (first || (cmd_entry->driver_flags & APC_PRESENT)) {
		upssendchar(cmd_entry->cmd);
		if (first) {
			flag_timeoutfailure = -1;
			strcpy (temp, "NA");
		}
		recvret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);
		if (first)
			flag_timeoutfailure = 0;

		if ((recvret == -1) || (strcmp(temp, "NA") == 0)) {	/* not supported */
			if (first)
				cmd_entry->driver_flags &= ~(APC_PRESENT);
			if (debug)
				printf("APC capability: %s NOT OK\n",
				       CMDTABNAME(cmd_entry));
			retval = -1;
		} else {
			/* If its the first time through, mark this
			 * capability as present
			 */
			if (first) {
				cmd_entry->driver_flags |= APC_PRESENT;
				if (debug)
					printf("APC capability: %s OK [%s]\n",
					       CMDTABNAME(cmd_entry), temp);
			}
			ptr = convert_ups2info(cmd_entry, temp);
			if (first) {
				addinfo(cmd_entry->info_type,
					ptr,
					cmd_entry->info_flags,
					cmd_entry->info_len);
			} else
				setinfo (cmd_entry->info_type, ptr);
		}
	}
	return retval;
}


void do_capabilities (void)
{
	char	upsloc, temp[512], *ptr, cmd, loc, *entptr, etmp[16], *endtemp;
	int	nument, entlen, i, matrix, recvret;
	struct apc_cmdtab_t * cmd_entry;

	if (debug)
		printf("APC: About to get capabilities string\n");
	/* If we can do caps, then we need the Firmware revision which has
	   the locale descriptor as the last character (ugh)
	*/
	ptr = getdata(INFO_FIRMREV);
	if (ptr)
		upsloc = ptr[strlen(ptr) - 1];
	else
		upsloc = 0;

	/* get capability string */
	send_ups (DRV_CAPABILITY);	/* ^Z */
	strcpy (temp, "NA");
	recvret = upsrecv (temp, sizeof(temp), ENDCHAR, MINIGNCHARS);

	if ((recvret == -1) || (!strcmp(temp, "NA"))) {
		/* Early Smart-UPS, not as smart as later ones */
		/* This should never happen since we only call
		   this if the REQ_CAPABILITIES command is supported
		*/
		syslog (LOG_ERR, "ERROR: APC cannot do capabilites but said it could!\n");
		return;
	}

	/* recv always puts a \0 at the end, so this is safe */
	/* however it assumes a zero byte cannot be embedded */
	endtemp = &temp[0] + strlen(temp);

	if (temp[0] != '#') {
		printf ("Unrecognized capability start char %c\n", temp[0]);
		printf ("Please report this error [%s]\n", temp);
		syslog (LOG_ERR, "ERROR: unknown capability start char %c!\n", temp[0]);
		return;
	}

	if (temp[1] == '#') {		/* Matrix-UPS */
		matrix = 1;
		ptr = &temp[0];
	}
	else {
		ptr = &temp[1];
		matrix = 0;
	}

	/* command char, location, # of entries, entry length */

	while (ptr[0] != '\0') {
		if (matrix)
			ptr += 2;	/* jump over repeating ## */

		/* check for idiocy */
		if (ptr >= endtemp) {
			printf ("Capability string has overflowed\n");
			printf ("Please report this error\n");
			syslog (LOG_ERR, "ERROR: capability overflow!\n");
			exit(1);
		}

		cmd = ptr[0];
		loc = ptr[1];
		nument = ptr[2] - 48;
		entlen = ptr[3] - 48;
		entptr = &ptr[4];
		cmd_entry = CHR_LOOKUP(cmd);

		if (debug)
			printf("CAPABILITY %02x (%c) \"%s\" seen\n",
			       cmd,
			       loc,
			       (cmd_entry ? CMDTABNAME(cmd_entry) : "unrecognised"));
		/* Mark this as being a ENUM entry and writable */
		if (cmd_entry && ((loc == upsloc) || (loc == '4'))) {
			setflag(cmd_entry->info_type, FLAG_RW | FLAG_ENUM);
			cmd_entry->info_flags |= FLAG_RW | FLAG_ENUM;
		}
		for (i = 0; i < nument; i++) {
			snprintf (etmp, entlen + 1, "%s", entptr);
			if (cmd_entry && ((loc == upsloc) || (loc == '4')))
				addenum(cmd_entry->info_type,
					convert_ups2info(cmd_entry, etmp));
			entptr += entlen;
		}
		ptr = entptr;
	}
}


void oldapcsetup (void)
{
	int	ret = 0;

	/* really old models ignore REQ_MODEL, so find them first */
	ret = query_ups(INFO_MODEL, 1);

	if (ret != 0) {
		/* force the model name */
		addinfo (INFO_MODEL, "Smart-UPS", 0, 0);
	}

	/* see if this might be an old Matrix-UPS instead */
	if (query_ups(INFO_CURRENT, 1) == 0)
		setinfo (INFO_MODEL, "Matrix-UPS");

	query_ups(INFO_SERIAL, 1);
	query_ups(INFO_STATUS, 1);
	query_ups(INFO_UTILITY, 1); /* This one may fail... no problem */

	/* If we have come down this path then we dont do capabilites and
	   other shiny features
	*/
}


void getbaseinfo ()
{
	char	temp[512];
	int	recvret = 0;
	char 	*alrts;
	char 	*cmds;
	char	*ptr;
	struct	apc_cmdtab_t * cmd_entry;

	if (debug)
		printf("APC - Attempting to find command set\n");
	/* Initially we ask the UPS what commands it takes
	   If this fails we are going to need an alternate
	   strategy - we can deal with that if it happens
	*/
	send_ups(DRV_CMDSET);
	strcpy (temp, "NA");
	/* disable timeout complaints temporarily */
	flag_timeoutfailure = -1;
	recvret = upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);
	flag_timeoutfailure = 0;

	if ((recvret == -1) || (!strcmp(temp, "NA"))) {
		/* We have an old dumb UPS - go to specific code for old stuff */
		oldapcsetup();
		return;
	}

	if (debug)
		printf("APC - Parsing out command set\n");
	/* We have the version.alert.cmdchars string
	   NB the alert chars are normally in IGNCHARS
	   so will have been pretty much edited out.
	   You will need to change the upsrecv above if
	   you want to check those out too....
	*/
 	alrts = strchr(temp, '.');
	if (alrts == NULL) {
		printf("Unable to split APC version string\n");
		printf("Bailing out\n");
		exit(1);
	}
	*alrts++ = 0;

	cmds = strchr(alrts, '.');
	if (cmds == NULL) {
		printf("Unable to find APC command string\n");
		printf("Bailing out\n");
		exit(1);
	}
	*cmds++ = 0;

	for(ptr = cmds; *ptr; ptr++) {
		cmd_entry = CHR_LOOKUP(*ptr);
		if (cmd_entry) {
			query_ups(cmd_entry->info_type, 1);
		} else if (debug) {
			printf("APC command: %c 0x%02x UNKNOWN - ignored\n",
			       (isprint(*ptr) ? *ptr : ' '),
			       *ptr);
		}
	}
	/* If we can handle capabilities then do them here */
	cmd_entry = info_lookup(DRV_CAPABILITY);
	if (strchr(cmds, cmd_entry->cmd) != NULL)
		do_capabilities();

	if (debug)
		printf("APC - UPS capabilities determined\n");
}


/* TODO: roll this into upscommon ala bestups */
void sendstring(char * string, int len, int delay)
{
	int	i;

	for (i=0; (i<len); i++) {
	        upssendchar(string[i]);
		usleep(delay);
	}
}

/* set a value in the hardware using the <cmdchar> '-' (repeat) approach */
void hw_set (struct apc_cmdtab_t * cmd_entry, char *newval)
{
	char	temp[256], orig[256];
	int	tries;

	send_ups(cmd_entry->info_type);
	upsrecv (orig, sizeof(orig), ENDCHAR, IGNCHARS);

	/* don't change it if already set - save wear and tear on the eeprom */
	if (!strcmp(orig, newval)) {
		syslog (LOG_INFO,
			"SET %s='%s' ignored (unchanged value",
			CMDTABNAME(cmd_entry),
			newval);
		return;
	}

	tries = 0;
	while (tries < 6) {
		send_ups(DRV_NEXT);
		upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

		/* TODO: test for "OK", bail out on "NO", etc */
		send_ups(cmd_entry->info_type);
		upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);

		if (!strcmp(temp, newval)) { 		/* got it */
			syslog (LOG_INFO,
				"SET %s='%s' performed",
				CMDTABNAME(cmd_entry),
				newval);
			return;
		}
		if (!strcmp(temp, orig)) {		/* wrapped around */
			syslog (LOG_ERR, "setvar: variable %c wrapped!\n", 
			        CMDTABNAME(cmd_entry));
			return;
		}

		tries++;
	}

	syslog (LOG_ERR, "setvar: gave up after 6 tries\n");
}

void setstring (struct apc_cmdtab_t * cmd_entry, 
		char *data)
{
	char	temp[256];
	int	i;

	upssendchar(cmd_entry->cmd);
	upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);
	if (debug)
		printf ("Current %s: [%s]\n", CMDTABNAME(cmd_entry), temp);

	/* If this value is the same as existing, then just return */
	if (!strcmp(temp, data)) {
		syslog (LOG_INFO,
			"SET %s='%s' ignored (unchanged value",
			CMDTABNAME(cmd_entry),
			data);
		return;
	}

	send_ups(DRV_NEXT);
	usleep (50000);

	for (i = 0; i < strlen(data); i++) {
		upssendchar (data[i]);
		usleep (50000);
	}

	/* pad to 8 chars with CRs */
	for (i = strlen(data); i < cmd_entry->info_len; i++) {
		upssendchar (13);
		usleep (50000);
	}
	/* Make sure a CR was definitely sent */
	if (strlen(data) >= cmd_entry->info_len)
		upssendchar (13);

	syslog (LOG_INFO,
		"SET %s='%s' performed",
		CMDTABNAME(cmd_entry),
		data);
	usleep (50000);
	upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);
	usleep (50000);
	/* TODO: check for "OK" or "NO" or similar */
}

void setvar (int auxcmd, int dlen, char *data)
{
	struct apc_cmdtab_t * cmd_entry;
	cmd_entry = info_lookup(auxcmd);
	if (cmd_entry == NULL) {
		syslog (LOG_ERR, "ERROR: Capability 0x%04x unknown by driver\n", auxcmd);
		return;
	}
	if ((cmd_entry->driver_flags & APC_PRESENT) == 0) {
		syslog (LOG_ERR, "ERROR: %s setting unsupported by UPS\n", CMDTABNAME(cmd_entry));
		return;
	}
	if ((cmd_entry->info_flags & FLAG_RW) == 0) {
		syslog (LOG_ERR, "ERROR: %s cannot be set - read only\n", CMDTABNAME(cmd_entry));
		return;
	}
	if ((cmd_entry->info_flags & FLAG_ENUM)) {
		hw_set (cmd_entry, data);
	} else if ((cmd_entry->info_flags & FLAG_STRING)) {
		setstring(cmd_entry, data);
	} else {
		syslog (LOG_ERR, "ERROR: %s cannot be set - unknown type\n", CMDTABNAME(cmd_entry));
	}
	/* Force update of stored data */
	query_ups(auxcmd, 0);
}

/* check for calibration status and either start or stop */
void do_cal (int start)
{
	char	temp[256];
	int	tval, ret;

	send_ups(INFO_STATUS);
	ret = upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);

	if (!ret)		/* didn't get status, bail out */
		return;

	tval = strtol (temp, 0, 16);

	if (tval & 1) {		/* calibration currently happening */
		if (start == 0)	{		/* stop requested */
			syslog (LOG_INFO, "Stopping runtime calibration");
			send_ups(CMD_CAL1);
			return;
		}

		/* requested start while calibration still running */
		syslog (LOG_INFO, "Runtime calibration already in progress");
		return;
	}

	/* calibration not happening */

	if (start == 0) {		/* stop requested */
		syslog (LOG_INFO, "Runtime calibration not occurring");
		return;
	}

	syslog (LOG_INFO, "Starting runtime calibration");
	send_ups(CMD_CAL0);
}

void instcmd (int auxcmd, int dlen, char *data)
{
	char	temp[256];
	struct apc_cmdtab_t * cmd_entry;
	static	time_t lastcmd = 0;
	time_t	now, elapsed;

	/* TODO: reply to upsd? */

	switch (auxcmd) {
		/* Special cases */
		case CMD_CAL0:		/* stop runtime calibration */
			do_cal(0);
			break;
		case CMD_CAL1:		/* start runtime calibration */
			do_cal(1);
			break;
		default:
			cmd_entry = info_lookup(auxcmd);
			if ((cmd_entry == NULL) || ((cmd_entry->driver_flags & APC_PRESENT) == 0)) {
				syslog (LOG_INFO,
					"instcmd: unknown or unimplemented type %s 0x%04x\n", 
					CMDTABNAME(cmd_entry),
					auxcmd);
				return;
			}
#ifdef CONFIRM_DANGEROUS_COMMANDS
			/* check for dangerous commands */
			if (cmd_entry->driver_flags & APC_NASTY) {
				time (&now);
				elapsed = now - lastcmd;

				/* reset the timer every call - this means if you call it too      *
				 * early, then you have to wait MINCMDTIME again before sending #2 */
				lastcmd = now;

				if ((elapsed < MINCMDTIME) || (elapsed > MAXCMDTIME)) {

					/* FUTURE: tell the user (via upsd) to try it again */
					/* msgreply (UPSMSG_REPAGAIN); */
					return;
				}
			}
#endif
			send_ups(auxcmd);
			upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);
	}
}

/* get the UPS talking to us in smart mode */
int smartmode (void)
{
	int	tries;
	char	temp[256];

	tries = 0;
	for (;;) {
		tries++;

		send_ups(DRV_SMART);
		upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);

		if (!strcmp(temp, "SM"))
			return 1;	/* success */

		if (tries > 5)
			return 0;	/* failure */

		sleep (1);	/* wait before trying again */
	}

	return 0;	/* notreached */
}

/* power down the attached load immediately */
void forceshutdown(char *port)
{
	char	temp[32];
	int	tval;

	syslog (LOG_INFO, "Initiating UPS shutdown\n");
	printf ("Initiating forced UPS shutdown!\n");

	open_serial (port, B2400);

	if (!smartmode())
		printf ("Detection failed.  Trying a shutdown command anyway.\n");

	/* check the line status */
	send_ups (INFO_STATUS);
	upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);
	tval = strtol (temp, 0, 16);

	switch (shutdowncmdset) {
	case 3:		/* shutdown with grace period */
		printf ("Sending delayed power off command to UPS\n");
		send_ups(CMD_SHUTDOWN);
		break;

	case 2:		/* instant shutdown */
		printf ("Sending power off command to UPS\n");
		send_ups(CMD_OFF);
		break;

	case 1:

		/* Send a combined set of shutdown commands which can work better */
		/* if the UPS gets power during shutdown process */
		/* Specifically it sends both the soft shutdown 'S' */
		/* and the powerdown after grace period - '@000' commands */
		printf ("UPS - currently %s - sending shutdown/powerdown\n",
			(tval & 8) ? "on-line" : "on battery");
		sendstring ("S@000", 4, 50000);
		break;

	default:

		/* @000 - shutdown after 'p' grace period             */
		/*      - returns after 000 minutes (i.e. right away) */

		/* S    - shutdown after 'p' grace period, only on battery */
		/*        returns after 'e' charge % plus 'r' seconds      */

		if (tval & 8) {                 /* on line */
			printf ("On line, sending shutdown+return command...\n");
			sendstring ("@000", 4, 50000);
		}
		else {
			printf ("On battery, sending normal shutdown command...\n");
			send_ups(CMD_SOFTDOWN);upssendchar ('S');
		}
	}

	if (sddelay > 0) {
		sleep (sddelay);
		printf ("Hmm, did the shutdown fail?  Oh well...\n");
		exit (1);
	}
	exit (0);
}

/* 940-0095B support: set DTR, lower RTS */
void init_serial()
{
	int	dtr_bit = TIOCM_DTR;
	int	rts_bit = TIOCM_RTS;

	ioctl(upsfd, TIOCMBIS, &dtr_bit);
	ioctl(upsfd, TIOCMBIC, &rts_bit);
}

/* normal idle loop - keep up with the current state of the UPS */
/* If updateall is non-zero query all known UPS variables */
void updateinfo (int updateall)
{
	struct apc_cmdtab_t * cmd_entry;

	/* try to wake up a dead ups once in awhile */
	if (flag_timeoutfailure == 1) {
		if (!smartmode()) {
			syslog (LOG_ERR, "Communication with UPS lost");
			if (debug)
				printf("APC: Communications lost\n");
		}
	}

	
	for(cmd_entry = &apc_cmdtab[0];
	    (cmd_entry->info_type > 0);
	    cmd_entry++) {
		if (cmd_entry->driver_flags & APC_PRESENT) {
			if (updateall || (cmd_entry->driver_flags & APC_POLL))
				query_ups(cmd_entry->info_type, 0);
		}
	}

	writeinfo(info);
}

/* install pointers to functions for msg handlers called from msgparse */
void setuphandlers()
{
	upsh.setvar = setvar;
	upsh.instcmd = instcmd;
}

void usage (char *prog)
{
	printf ("usage: %s [-D] [-h] [-c <cable>] [-d <num>] [-f <statefile>] [-k] [-s <num>] <device>\n", prog);
	printf ("Example: %s /dev/ttyS0\n", prog);
	exit (1);
}

void help (char *prog)
{
	printf ("usage: %s [-D] [-h] [-c <cable>] [-d <num>] [-f <statefile>] [-k] [-s <num>] <device>\n", prog);
	printf ("\n");
	printf ("-D       - Debug.  Another -D gives you more debug.");
	printf ("-c       - specify '" ALT_CABLE_1 "' if using this cable\n");
	printf ("-d <num> - wait <num> seconds after sending shutdown command\n");
	printf ("-h       - display this help\n");
	printf ("-f <stfl>- use <stfl> as the upsd state file\n");
	printf ("-k       - force shutdown\n");
	printf ("-s <num> - use shutdown command set <num> for forced shutdown\n");
	printf ("           0: soft shutdown or powerdown, depending on battery status\n");
	printf ("           1: soft shutdown followed by powerdown\n");
	printf ("           2: instant power off\n");
	printf ("           3: power off with grace period\n");
	printf ("           Modes 0-1 will make the UPS come back when power returns\n");
	printf ("           Modes 2-3 will make the UPS stay turned off when power returns\n");
	printf ("<device> - /dev entry corresponding to UPS port\n");
	exit (1);
}

int main (int argc, char **argv)
{
	char	*portname;
	char 	*prog;
	char	*upsdev;
	int	shutdown = 0;
	int	i;
	time_t last_full;

	printf ("Network UPS Tools (version %s) - APC Smart protocol driver\n",
		UPS_VERSION);
	printf ("\tDriver %s, command table %s\n",
		APC_DRIVER_VERSION,
		APC_TABLE_VERSION);
	openlog ("apcsmart", LOG_PID, LOG_FACILITY);
	intialise_data_tables();

	prog = argv[0];
	*statefn = 0;

	if (argc == 1)
		usage (prog);

	while ((i = getopt(argc, argv, "+Dhd:f:ks:c:")) != EOF) {
		switch (i) {
			case 'D':
				debug++;
				break;
			case 'd':
				sddelay = atoi(optarg);
				break;
			case 'f':
				strcpy(statefn, optarg);
				break;
			case 'k':
				shutdown = 1;
				break;
			case 'h':
				help(prog);
				break;
			case 's':
				shutdowncmdset = atoi(optarg);
				break;
			case 'c':
				cable = strdup(optarg);
				break;
			default:
				usage(prog);
				break;
		}
	}

	argc -= optind;
	argv += optind;
	if (debug > 1)
		upsc_debug = (debug - 1);

	droproot();

	if (argc < 1) {
		printf ("Error: no device specified!\n");
		usage (prog);
	}

	upsdev = argv[0];
	portname = strrchr(upsdev, '/');
	if (portname == NULL)
		portname = upsdev;
	else
		portname++;

	if (!*statefn)
		snprintf (statefn, sizeof(statefn), "%s/apcsmart-%s", STATEPATH,
			  portname);
	if (debug)
		printf("APC: state file name is %s\n", statefn);

	open_serial (upsdev, B2400);

	if (cable != 0 && strcmp(cable,ALT_CABLE_1) == 0)
		init_serial();

	/* do this first - ALWAYS try a shutdown even if detection failed */
	if (shutdown) {
		forceshutdown(upsdev);
		/* NOT REACHED */
	}

	if (!smartmode()) {
		printf ("Unable to detect an APC Smart protocol UPS on port %s\n", argv[0]);
		printf ("Check the cabling, port name or model name and try again\n");
		exit (1);
	}

	info = create_info (infomax, shmok);

	/* manufacturer ID - hardcoded in this particular module */
	addinfo (INFO_MFR, "APC", 0, 0);

	createmsgq();	/* try to create IPC message queue */

	setuphandlers();

	/* see what's out there */
	getbaseinfo();

	printf ("Detected %s [%s] on %s\n", getdata(INFO_MODEL), 
		getdata(INFO_SERIAL), argv[0]);

	if (!debug)
		background();
	else
		printf("Not going into background in debug mode\n");

	addcleanup();
	time(&last_full);

	for (;;) {
		time_t tval;

		time(&tval);
		if (debug) {
			printf("UPS update loop %s", ctime(&tval));
		}

		/* Every hour force a refresh of all variables */
		/* NB This won't catch (say) a Measure-II card being inserted/removed */
		if ((tval - last_full) > 3600) {
			updateinfo(1);
			last_full = tval;
		} else
			updateinfo(0);

		/* wait up to 2 seconds for a message from the upsd */

		if (getupsmsg(2) && debug)	/* TODO: remove debug scaffolding */
			syslog (LOG_INFO, "Received a message from upsd\n");
	}
}

/*
 * Local variables:
 *  c-indent-level: 8
 *  c-basic-offset: 8
 * End:
 */
