/* genericups.c - support for generic contact-closure UPS models

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              
*/

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/termios.h>

#include "config.h"
#include "proto.h"
#include "shared.h"
#include "version.h"
#include "upscommon.h"
#include "genericups.h"
#include "common.h"

	int	shmok = 1, upsfd;
	char	statefn[256];
	itype	*info;

	int	upstype = -1;
	int	infomax = 8;
	int	sddelay = 90;	/* wait 90 seconds for shutdown */

/* simple open - we don't care about baud rate or any flags */
void open_serial_simple(char *port)
{
	int	ret;

	signal (SIGALRM, openfail);
	alarm (3);

	upsfd = open (port, O_RDWR);

	if (upsfd < 1) {
		printf ("Unable to open %s: %s\n", port, strerror(errno));
		exit (1);
	}

	/* set "normal" flags - cable power, or whatever */
	ret = ioctl (upsfd, TIOCMSET, &upstab[upstype].line_norm);

	if (ret != 0)
		perror ("ioctl");

	alarm (0);

	lockport (upsfd, port);
}

void initinfo (void)
{
	info = create_info (infomax, shmok);

	/* setup the basics */

	addinfo (INFO_MFR, upstab[upstype].mfr, 0, 0);
	addinfo (INFO_MODEL, upstab[upstype].model, 0, 0);
	addinfo (INFO_STATUS, "", 0, 0);
}

/* normal idle loop - keep up with the current state of the UPS */
void updateinfo (void)
{
	int	flags, ol, bl, ret;
	char	temp[VALSIZE];

	ret = ioctl (upsfd, TIOCMGET, &flags);

	if (ret != 0) {
		syslog (LOG_INFO, "ioctl failed: %m\n");
		return;
	}

	ol = ((flags & upstab[upstype].line_ol) == upstab[upstype].val_ol);
	bl = ((flags & upstab[upstype].line_bl) == upstab[upstype].val_bl);

	strcpy (temp, "");

	if (bl)
		strcat (temp, "LB ");	/* low battery */

	if (ol)
		strcat (temp, "OL");	/* on line */
	else
		strcat (temp, "OB");	/* on battery */

	setinfo (INFO_STATUS, temp);

	writeinfo(info);
}

/* show all possible UPS types */
void listtypes (void)
{
	int	i;

	printf ("Valid UPS types:\n\n");

	for (i = 0; upstab[i].mfr != NULL; i++)
		printf ("%i: %s\n", i, upstab[i].desc);
}

/* set the flags for this UPS type */
void settype (int type)
{
	int	i;

	upstype = type;

	printf ("UPS type: ");

	for (i = 0; upstab[i].mfr != NULL; i++)
		if (type == i) {
			printf ("%s\n", upstab[i].desc);
			return;
		}

	printf ("Unknown UPS type number\n");
	listtypes();
	exit (1);
}			

/* power down the attached load immediately */
void forceshutdown(char *port)
{
	int	flags, ret;

	if (upstype == -1) {
		printf ("You need to set a UPS type with -t <num>...\n");
		sleep (5);
		exit (1);
	}

	open_serial_simple (port);

	syslog (LOG_INFO, "Initiating UPS shutdown\n");
	printf ("Initiating forced UPS shutdown!\n");

	flags = upstab[upstype].line_sd;

	if (flags == -1)
		printf ("No shutdown command defined for this model!\n");

	if (flags == TIOCM_ST)
#ifdef HAVE_TCSENDBREAK
		tcsendbreak (upsfd, 4901);
#else
		printf ("Need to send a BREAK, but don't have tcsendbreak!\n");
#endif
	else {
		ret = ioctl (upsfd, TIOCMSET, &flags);
		if (ret != 0)
			perror ("ioctl");
	}

	if (sddelay > 0) {
		printf ("Waiting for poweroff...\n");
		sleep (sddelay);
		printf ("Hmm, did the shutdown fail?  Oh well...\n");
		exit (1);
	}
	exit(0);
}

/* install pointers to functions for msg handlers called from msgparse */
void setuphandlers(void)
{
	/* TODO: future */
}

void usage(char *prog)
{
	printf ("usage: %s [-h] [-d <num>] [-l] -t <num> [-k] <device>\n", prog);
	printf ("Example: %s -t 1 /dev/ttyS0\n", prog);
}

void help(char *prog)
{
	printf ("usage: %s [-h] [-d <num>] [-l] -t <num> [-k] <device>\n", prog);
	printf ("\n");
	printf ("-d <num>   - wait <num> seconds after sending shutdown command\n");
	printf ("-h         - display this help\n");
	printf ("-k         - force shutdown\n");
	printf ("-m <model> - set model name to <model>\n");
	printf ("-M <mfr>   - set manufacturer name to <mfr>\n");
	printf ("-l         - show list of valid ups types\n");
	printf ("-t <num>   - set ups type to <num>\n");
	printf ("<device>   - /dev entry corresponding to UPS port\n");
}

int main (int argc, char **argv)
{
	char	*portname, *prog, *forcemodel, *forcemfr;
	int	i;

	forcemodel = forcemfr = NULL;

	printf ("Network UPS Tools - Generic UPS driver 0.50 (%s)\n", UPS_VERSION);
	openlog ("genericups", LOG_PID, LOG_FACILITY);
	prog = argv[0];
	
	while ((i = getopt(argc, argv, "+d:k:t:lhm:M:")) != EOF) {
		switch (i) {
			case 'd':
				sddelay = atoi(optarg);
				break;
			case 'k':
				forceshutdown (optarg);
				break;
			case 't':
				settype (atoi(optarg));
				break;
			case 'l':
				listtypes();
				exit (1);
				break;
			case 'h':
				help (prog);
				exit (1);
				break;
			case 'm':
				forcemodel = strdup (optarg);
				break;
			case 'M':
				forcemfr = strdup (optarg);
				break;
			default:
				usage (prog);
				break;
		}
	}

	if (upstype == -1) {
		printf ("No UPS type specified (use -t <num>)\n");
		exit (1);
	}

	argc -= optind;
	argv += optind;

	if (argc != 1) {
		help (prog);
		exit (1);
	}

	droproot();

	portname = NULL;
	for (i = strlen(argv[0]); i >= 0; i--)
		if (argv[0][i] == '/') {
			portname = &argv[0][i+1];
			break;
		}

	if (portname == NULL) {
		printf ("Unable to abbreviate %s\n", argv[0]);
		exit (1);
	}

	snprintf (statefn, sizeof(statefn), "%s/genericups-%s", STATEPATH,
	          portname);
	open_serial_simple (argv[0]);

	initinfo();

	/* replace model/mfr info if specified on command line */
	if (forcemfr)
		setinfo (INFO_MFR, forcemfr);

	if (forcemodel)
		setinfo (INFO_MODEL, forcemodel);

	createmsgq();   /* try to create IPC message queue */

	setuphandlers();

	background();

	for (;;) {
		updateinfo();

		/* wait up to 2 seconds for a message from the upsd */

		if (getupsmsg(2))
			syslog (LOG_INFO, "Received a message from upsd\n");
	}
}
