/* bestups.c - model specific routines for Best-UPS Fortress models

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              
*/

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/termios.h>

#include "config.h"
#include "proto.h"
#include "shared.h"
#include "version.h"
#include "upscommon.h"
#include "common.h"

#define INFOMAX  16
#define ENDCHAR  13	/* replies end with CR */
#define UPSDELAY  5
#define MAXTRIES 10

	int	upsfd;
	int	shmok = 1;
	char	statefn[256], *modelname, *va;
	itype	*info;
extern	int	flag_timeoutfailure;
	float	lowvolt = 0, highvolt = 0, voltrange = 0;
	int	infomax = 16, linenorm = 0;
	int	sddelay = 90;	/* wait 90 seconds for shutdown */

extern	struct	ups_handler	upsh;

void initinfo (void)
{
	info = create_info (INFOMAX, shmok);

	/* now set up room for all future variables that are supported */
	addinfo (INFO_MFR, "", 0, 0);
	addinfo (INFO_MODEL, "", 0, 0);
	addinfo (INFO_UTILITY, "", 0, 0);
	addinfo (INFO_BATTPCT, "", 0, 0);
	addinfo (INFO_STATUS, "", 0, 0);
	addinfo (INFO_ACFREQ, "", 0, 0);
	addinfo (INFO_LOADPCT, "", 0, 0);
	addinfo (INFO_UPSTEMP, "", 0, 0);
}

	/* TODO: adapt to the standard upscommon open/send/recv calls */

void setup_serial(void)
{	
	struct	termios	tio;

	if (tcgetattr (upsfd, &tio) == -1) {
		perror ("tcgetattr");
		exit (1);
	}

	tio.c_iflag = IXON | IXOFF;
	tio.c_oflag = 0;
	tio.c_cflag = (B2400 | CS8 | CREAD | HUPCL | CLOCAL);
	tio.c_lflag = 0;
	tio.c_cc[VMIN] = 1;
	tio.c_cc[VTIME] = 0; 

#ifdef HAVE_CFSETISPEED
	cfsetispeed (&tio, B2400);
	cfsetospeed (&tio, B2400);
#endif

	if (tcsetattr (upsfd, TCSANOW, &tio) == -1) {
		perror ("tcsetattr");
		exit (1);
	}
}

char readchar (void)
{
	char	c;
	int	ret;

	ret = read (upsfd, &c, 1);

	if (ret > 0)
		return (c & 0x7f);
	else
		return 0;
}

/* TODO: roll this into upscommon */
int xsend (char *str)
{
	int	ret;

	while (*str) {
		ret = write (upsfd, str++, 1);
		if (ret == -1)
			return (-1);
	}

	return 0;
}

/* TODO: also roll this into upscommon */
int xrecv (char *buf)
{
	char	ch, *p;

	p = buf;
	*p = '\0';

	while ((ch = readchar ())) {
		if (ch == 13) {
			*p = '\0';
			return 0;
		}
	
		*p++ = ch;
	}

	*p = '\0';
	return 0;
}

void ups_sync(void)
{
	char	buf[256];
	int	tries = 0;

	printf ("Syncing with UPS: ");
	fflush (stdout);

	for (;;) {
		tries++;
		if (tries > MAXTRIES) {
			printf ("\nFailed - giving up...\n");
			exit (1);
		}
		xsend ("\rQ1\r");
		printf (".");
		fflush (stdout);
		sleep (UPSDELAY);
		printf (".");
		fflush (stdout);
		xrecv (buf);
		printf (".");
		fflush (stdout);
		if (buf[0] == '(')
			break;			
	}

	printf (" done\n");
}

/* power down the attached load immediately */
void forceshutdown(char *port)
{
	char	temp[256], stat[32];
	int	tries = 0;

	syslog (LOG_INFO, "Initiating UPS shutdown\n");
	printf ("Initiating forced UPS shutdown!\n");

	upsfd = open(port, O_RDWR | O_NDELAY);
	if (upsfd == -1) {
		perror ("open");
		exit (1);
	}

	setup_serial();

	/* basic idea: find out line status and send appropriate command */

	printf ("Checking status: ");
	fflush (stdout);

	for (;;) {
		tries++;
		if (tries > MAXTRIES) {
			printf ("\nFailed - giving up...\n");
			exit (1);
		}
		xsend ("\rQ1\r");
		printf (".");
		fflush (stdout);
		sleep (UPSDELAY);
		printf (".");
		fflush (stdout);
		xrecv (temp);
		printf (".");
		fflush (stdout);
		if ((temp[0] == '(') && (strlen(temp) == 46))
			break;
	}

	printf (" done\n");

	sscanf (temp, "%*s %*s %*s %*s %*s %*s %*s %s", stat);

	/* on battery: send S01<cr>, ups will return by itself on utility */
	/* on line: send S01R0001<cr>, ups will cycle and return soon */

	xsend ("S01");

	if (stat[0] == '0') {			/* on line */
		printf ("On line, sending shutdown+return command...\n");
		xsend ("R0001");
	}
	else
		printf ("On battery, sending normal shutdown command...\n");

	xsend ("\r");	/* end sequence */

	if (sddelay > 0) {
		printf ("Waiting for poweroff...\n");
		sleep (sddelay);
		printf ("Hmm, did the shutdown fail?  Oh well...\n");
		exit (1);                               
	}
	exit (0);
}

void usage (char *prog)
{
	printf ("usage: %s [-h] [-d <num>] [-k] <device>\n", prog);
	printf ("Example: %s /dev/ttyS0\n", prog);
	exit (1);
}

void help (char *prog)
{
	printf ("usage: %s [-h] [-d <num>] [-k] <device>\n", prog);
	printf ("\n");
	printf ("-d <num> - wait <num> seconds after sending shutdown command\n");
	printf ("-h       - display this help\n");
	printf ("-k       - force shutdown\n");
	printf ("<device> - /dev entry corresponding to UPS port\n");
}

void setmodel (char *abbr, char *va)
{
	char	temp[256];

	if (!strcmp(abbr, "FOR")) {
		setinfo (INFO_MFR, "Best Power");
		snprintf (temp, sizeof(temp), "Fortress %s", va);
		setinfo (INFO_MODEL, temp);
		return;
	}

	if (!strcmp(abbr, "FTC")) {
		setinfo (INFO_MFR, "Best Power");
		snprintf (temp, sizeof(temp), "Fortress Telecom %s", va);
		setinfo (INFO_MODEL, temp);
		return;
	}

	if (!strcmp(abbr, "PRO")) {
		setinfo (INFO_MFR, "Best Power");
		snprintf (temp, sizeof(temp), "Patriot Pro %s", va);
		setinfo (INFO_MODEL, temp);
		return;
	}

	if (!strcmp(abbr, "PR2")) {
		setinfo (INFO_MFR, "Best Power");
		snprintf(temp, sizeof(temp), "Patriot Pro II %s", va);
		setinfo (INFO_MODEL, temp);
		return;
	}

	if (!strcmp(abbr, "520")) {
		setinfo (INFO_MFR, "Sola Australia");
		snprintf (temp, sizeof(temp), "Sola 520 %s", va);
		setinfo (INFO_MODEL, temp);
		return;
	}

	setinfo (INFO_MFR, "Unknown");
	snprintf (temp, sizeof(temp), "Unknown %s (%s)", abbr, va);
	setinfo (INFO_MODEL, temp);

	printf ("Unknown model detected - please report this ID: '%s'\n", abbr);

	return;
}

void ups_ident(void)
{
	char	buf[256], *ptr, *com;
	int	i, tries = 0;

	printf ("Identifying UPS: ");
	fflush (stdout);

	for (;;) {
		tries++;
		if (tries > MAXTRIES) {
			printf ("\nFailed - giving up...\n");
			exit (1);
		}
		xsend ("\r\rID\r");
		printf (".");
		fflush (stdout);
		sleep (UPSDELAY);
		printf (".");
		fflush (stdout);
		xrecv (buf);
		printf (".");
		fflush (stdout);
		if ((buf[0] != '\0') && (buf[0] != '(') &&
		    ((strlen(buf) == 25) || (strlen(buf) == 26)))
			break;
	}

	printf (" done\n");

	modelname = va = NULL;

	/* FOR,750,120,120,20.0,27.6 */
	ptr = strdup(buf);

	for (i = 0; i < 6; i++) {
		com = strchr (ptr, ',');

		if (com)
			*com = '\0';

		switch (i) {
			case 0: modelname = strdup (ptr);
				break;
			case 1: va = strdup (ptr);
				break;
			case 2: linenorm = atoi(ptr);
				break;
			case 4: lowvolt = atof (ptr);
				break;
			case 5: highvolt = atof (ptr);
				voltrange = highvolt - lowvolt;
				break;
			default:
				break;
		}
		ptr = com + 1;
	}
}

void ups_update (void)
{
	char	utility[16], loadpct[16], acfreq[16], battvolt[16],
		upstemp[16], stat[16], buf[256], temp[VALSIZE];
	float	bvoltp;

	xsend ("\rQ1\r");     
	sleep (UPSDELAY); 
	xrecv (buf);

	if ((strlen(buf) < 46) || (buf[0] != '('))
		return;

	sscanf (buf, "%*c%s %*s %*s %s %s %s %s %s", utility, loadpct, 
	        acfreq, battvolt, upstemp, stat);
	
	setinfo (INFO_UTILITY, utility);

	bvoltp = ((atof (battvolt) - lowvolt) / voltrange) * 100.0;

	if (bvoltp > 100.0)
		bvoltp = 100.0;

	snprintf (temp, sizeof(temp), "%02.1f", bvoltp);
	setinfo (INFO_BATTPCT, temp);

	strcpy (temp, "");

	if (stat[0] == '0')
		strcat (temp, "OL ");		/* on line */
	else
		strcat (temp, "OB ");		/* on battery */

	if (stat[1] == '1')
		strcat (temp, "LB ");		/* low battery */

	if (stat[2] == '1') {		/* boost or trim in effect */
		if (atoi(utility) < linenorm)
			strcat (temp, "BOOST");

		if (atoi(utility) > linenorm)
			strcat (temp, "TRIM");
	}

	/* lose trailing space if present */
	if (temp[strlen(temp)-1] == ' ')
		temp[strlen(temp)-1] = 0;

	setinfo (INFO_STATUS, temp);
	setinfo (INFO_ACFREQ, acfreq);
	setinfo (INFO_LOADPCT, loadpct);
	setinfo (INFO_UPSTEMP, upstemp);

	writeinfo(info);
}

void instcmd (int auxcmd, int dlen, char *data)
{
	/* TODO: reply to upsd? */

	switch (auxcmd) {
		case CMD_BTEST0:	/* stop battery test */
			upssendchar ('C');
			upssendchar ('T');
			upssendchar (13);
			break;
		case CMD_BTEST1:	/* start battery test */
			upssendchar ('T');
			upssendchar (13);
			break;
		default:
			syslog (LOG_INFO, "instcmd: unknown type 0x%04x\n",
			        auxcmd);
	}
}

void setuphandlers(void)
{
	upsh.instcmd = instcmd;
}

int main (int argc, char **argv)
{
	char	*prog, *portname;
	int	i;

	printf ("Network UPS Tools - Best UPS driver 0.36 (%s)\n", UPS_VERSION);
	openlog ("bestups", LOG_PID, LOG_FACILITY);
	prog = argv[0];

	while ((i = getopt(argc, argv, "+d:hk:")) != EOF) {
		switch (i) {
			case 'd':
				sddelay = atoi(optarg);
				break;
			case 'k':
				forceshutdown(optarg);
				break;
			case 'h':
				help(prog);
				break;
			default:
				usage(prog);
				break;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc != 1) {
		help (prog);
		exit (1);
	}

	droproot();

	portname = NULL;
	for (i = strlen(argv[0]); i >= 0; i--)
		if (argv[0][i] == '/') {
			portname = &argv[0][i+1];
			break;
		}

	if (portname == NULL) {
		printf ("Unable to abbreviate %s\n", argv[0]);
		exit (1);
	}

	snprintf (statefn, sizeof(statefn), "%s/bestups-%s", STATEPATH,
	          portname);

	upsfd = open(argv[0], O_RDWR | O_NDELAY);
	if (upsfd == -1) {
		perror ("open");
		exit (1);
	}

	setup_serial();
	ups_sync();
	ups_ident();

	initinfo();
	createmsgq();	/* try to create IPC message queue */
	setuphandlers();
	setmodel (modelname, va);

        addinfo (INFO_INSTCMD, "", 0, CMD_BTEST0);
        addinfo (INFO_INSTCMD, "", 0, CMD_BTEST1);
	printf ("Detected %s %s on %s\n", getdata(INFO_MFR), 
	        getdata(INFO_MODEL), argv[0]);

	background();

	for (;;) {
		ups_update();

                if (getupsmsg(2))       /* TODO: remove debug scaffolding */
                        syslog (LOG_INFO, "Received a message from upsd\n");

	}

	return 0;
}
