/* belkin.c - model specific routines for Belkin Smart-UPS units.

   Copyright (C) 2000 Marcus Mller <marcus@ebootis.de>
     
   based on:

   apcsmart.c - model specific routines for APC smart protocol units

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              
*/

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/termios.h>
#include <sys/ioctl.h>
#include <time.h>
#include <limits.h>


#include "config.h"
#include "proto.h"
#include "shared.h"
#include "version.h"
#include "upscommon.h"
#include "common.h"
#include "belkin.h"

int	shmok = 1;
char	statefn[256];
itype	*info;
int	infomax = 45;




int init_communication ()
  {
  int i;
  int res;
  char temp[100];
  res = -1;
  for (i=1;i<=10 && res == -1;i++) {
    send_belkin_command (STATUS,MANUFACTURER,"");
    res = get_belkin_reply (temp);
    }
  if (res == -1 || strcmp (temp,"BELKIN")) return (res);
  return (0);
  }

void do_status(void)
  {
  char	temp[256], itemp[256],st[256];
  int	res;

  send_belkin_command (STATUS,STAT_STATUS,"");
  res = get_belkin_reply (temp);
  if (res == -1) return;
  strcpy (itemp, "");
  get_belkin_field (temp,st,6);
  if (*st == '1') strcat (itemp, "OFF ");
  get_belkin_field (temp,st,2);
  if (*st == '1') {
    strcat (itemp, "OB ");		/* on battery */
    send_belkin_command (STATUS,STAT_BATTERY,"");
    res = get_belkin_reply (temp);
    if (res == -1) return;
    get_belkin_field (temp,st,10);
    res = atoi (st);
    get_belkin_field (temp,st,2);
    if (*st == '1' || res < LOW_BAT) strcat (itemp, "LB "); /* low battery */
    }
  else strcat (itemp, "OL ");				/* on line */
  if (itemp[strlen(itemp)-1] == ' ') itemp[strlen(itemp)-1] = 0;
  setinfo (INFO_STATUS, itemp);
  }

int init_ups_data ()
  {
  int res;
  double low,high;
  char temp[100],st[100];
  send_belkin_command (STATUS,MODEL,"");
  res = get_belkin_reply (temp);
  if (res == -1) return (res);
  addinfo (INFO_MODEL, temp, 0, 0);
  send_belkin_command (STATUS,VERSION,"");
  res = get_belkin_reply (temp);
  if (res == -1) return (res);
  addinfo (INFO_SERIAL, temp, 0, 0);
  send_belkin_command (STATUS,RATING,"");
  res = get_belkin_reply (temp);
  get_belkin_field (temp,st,8);
  low = atof (st) / 0.88;
  get_belkin_field (temp,st,9);
  high = atof (st) * 0.88;
  tcflush (upsfd,TCIOFLUSH);
  addinfo (INFO_STATUS, "", 0, 0);
  addinfo (INFO_UTILITY, "", 0, 0);
  addinfo (INFO_BATTPCT, "", 0, 0);
  addinfo (INFO_ACFREQ, "", 0, 0);
  addinfo (INFO_UPSTEMP, "", 0, 0);
  addinfo (INFO_LOADPCT, "", 0, 0);
  addinfo (INFO_BATTVOLT, "", 0, 0);
  addinfo (INFO_OUTVOLT, "", 0, 0);
  sprintf (st,"%03.1f",low);
  addinfo (INFO_LOWXFER, st, 0, 0);
  sprintf (st,"%03.1f",high);
  addinfo (INFO_HIGHXFER, st, 0, 0);
  addinfo (INFO_INSTCMD, "", 0, CMD_OFF);
  addinfo (INFO_INSTCMD, "", 0, CMD_ON);
  updateinfo ();
  return (0);
  }

/* normal idle loop - keep up with the current state of the UPS */
void updateinfo (void)
  {
  int res;
  double val;
  char	temp[256],st[256];

  do_status();

  send_belkin_command (STATUS,STAT_INPUT,"");
  res = get_belkin_reply (temp);
  if (res == -1) return;
  get_belkin_field (temp,st,3);
  val = atof (st) / 10;
  sprintf (st,"%05.1f",val);
  setinfo (INFO_UTILITY, st);

  send_belkin_command (STATUS,STAT_BATTERY,"");
  res = get_belkin_reply (temp);
  if (res == -1) return;
  get_belkin_field (temp,st,10);
  val = atof (st);
  sprintf (st,"%03.0f",val);
  setinfo (INFO_BATTPCT, st);
  get_belkin_field (temp,st,7);
  val = atof (st) / 10;
  sprintf (st,"%4.1f",val);
  setinfo (INFO_BATTVOLT, st);
  get_belkin_field (temp,st,9);
  val = atof (st);
  sprintf (st,"%03.0f",val);
  setinfo (INFO_UPSTEMP, st);
  send_belkin_command (STATUS,STAT_OUTPUT,"");
  res = get_belkin_reply (temp);
  if (res == -1) return;
  get_belkin_field (temp,st,2);
  val = atof (st) / 10;
  sprintf (st,"%.1f",val);
  setinfo (INFO_ACFREQ, st);
  get_belkin_field (temp,st,4);
  val = atof (st) / 10;
  sprintf (st,"%05.1f",val);
  setinfo (INFO_OUTVOLT, st);
  get_belkin_field (temp,st,7);
  val = atof (st);
  sprintf (st,"%03.0f",val);
  setinfo (INFO_LOADPCT, st);
  writeinfo (info);
  }


void get_belkin_field (char *temp,char *data,int n)
  {
  char st[256];
  int i = 0,f = 1;
  strcpy (st,temp);
  while (f < n) {
    while (st[i] && st[i] != ';') i++;
    st[i++] = 0;
    f++;
    }
  f = i;
  while (st[i] && st[i] != ';') i++;
  st[i] = 0;
  strcpy (data,st+f);
  }

int get_belkin_reply (char* buf)
  {
  int res,cnt;
  res = recvbinary (buf,7);
  if (res) return (res);
  buf[7] = 0;
  cnt = atoi (buf+4);
  res = recvbinary (buf,cnt);
  buf[cnt] = 0;
  return (res);
  }

void send_belkin_command (char command,char *subcommand,char *data)
  {
  char st[250];
  tcflush (upsfd,TCIOFLUSH);
  sprintf (st,"~00%c%03d%s%s",command,strlen (data) + 3,subcommand,data);
  send_string (st);
  }



/* power down the attached load immediately */
void forceshutdown(char *port)
{
	int	res;
	char temp[100],st[100];

	syslog (LOG_INFO, "Initiating UPS shutdown\n");
	printf ("Initiating forced UPS shutdown!\n");

	open_serial (port, B2400);
	
	set_serialDTR0RTS1();

	sleep (1);
  	tcflush (upsfd,TCIOFLUSH);

	res = init_communication ();
	if (res == -1) {
	  printf ("Detection failed.  Trying a shutdown command anyway.\n");
          send_belkin_command (CONTROL,POWER_OFF,"1;1");
	  exit (-1);
          }
        send_belkin_command (STATUS,STAT_STATUS,"");
        res = get_belkin_reply (temp);
        get_belkin_field (temp,st,2);
        if (*st == '1') {
          send_belkin_command (CONTROL,POWER_OFF,"1;1");
	  exit (0);
          }
	printf ("Power restored - initating a reboot !\n");
        exit (1);
}

/* handle the CMD_OFF with some paranoia */
void do_off (void)
{
	static	time_t lastcmd = 0;
	time_t	now, elapsed;
#ifdef CONFIRM_DANGEROUS_COMMANDS
	time (&now);
	elapsed = now - lastcmd;

	/* reset the timer every call - this means if you call it too      *
	 * early, then you have to wait MINCMDTIME again before sending #2 */
	lastcmd = now;

	if ((elapsed < MINCMDTIME) || (elapsed > MAXCMDTIME)) {

		/* FUTURE: tell the user (via upsd) to try it again */
		/* msgreply (UPSMSG_REPAGAIN); */
		return;
	}
#endif

	syslog (LOG_INFO, "Sending powerdown command to UPS\n");
        send_belkin_command (CONTROL,POWER_OFF,"1;1");
	usleep (1500000);
        send_belkin_command (CONTROL,POWER_OFF,"1;1");
}

void instcmd (int auxcmd, int dlen, char *data)
  {
  switch (auxcmd) {
	case CMD_OFF:		/* power off the load */
		do_off();
		break;
	case CMD_ON:
	        send_belkin_command (CONTROL,POWER_ON,"1;1");
		break;
	default:
		syslog (LOG_INFO, "instcmd: unknown type 0x%04x\n",auxcmd);
	}
  }

/* install pointers to functions for msg handlers called from msgparse */
void setuphandlers(void)
{
	upsh.instcmd = instcmd;
}

void usage (char *prog)
{
	printf ("usage: %s [-h] [-k] <device>\n", prog);
	printf ("Example: %s /dev/ttyS0\n", prog);
	exit (1);
}

void help (char *prog)
{
	printf ("usage: %s [-h] [-k] <device>\n", prog);
	printf ("\n");
	printf ("-h       - display this help\n");
	printf ("-k       - force shutdown\n");
	printf ("<device> - /dev entry corresponding to UPS port\n");
	exit (1);
}

int main (int argc, char **argv)
{
	char	*portname, *prog;
	int	i,res;

	printf ("Network UPS Tools - Belkin Smart protocol driver 0.10 (%s)\n", UPS_VERSION);
/*
	openlog ("belkin", LOG_PID, LOG_FACILITY);
*/

	prog = argv[0];

	if (argc == 1)
		usage (prog);

	while ((i = getopt(argc, argv, "+hk:")) != EOF) {
		switch (i) {
			case 'k':
				forceshutdown(optarg);
				break;
			case 'h':
				help(prog);
				break;
			default:
				usage(prog);
				break;
		}
	}

	argc -= optind;
	argv += optind;

	droproot();

	portname = NULL;
	for (i = strlen(argv[0]); i >= 0; i--)
		if (argv[0][i] == '/') {
			portname = &argv[0][i+1];
			break;
		}

	if (portname == NULL) {
		printf ("Unable to abbreviate %s\n", argv[0]);
		exit (1);
	}

	snprintf (statefn, sizeof(statefn), "%s/belkin-%s", STATEPATH,
	          portname);
	open_serial (argv[0], B2400);
	
	set_serialDTR0RTS1();

	sleep (1);
  	tcflush (upsfd,TCIOFLUSH);

	res = init_communication ();
	if (res == -1) {
		printf ("Unable to detect an Belkin Smart protocol UPS on port %s\n", argv[0]);
		printf ("Check the cabling, port name or model name and try again\n");
		exit (1);
	}

	info = create_info (infomax, shmok);

	/* manufacturer ID - hardcoded in this particular module */
	addinfo (INFO_MFR, "BELKIN", 0, 0);

	createmsgq();	/* try to create IPC message queue */

	setuphandlers();

	/* see what's out there */
	init_ups_data ();

	printf ("Detected %s [%s] on %s\n", getdata(INFO_MODEL), 
		getdata(INFO_SERIAL), argv[0]);

	background();
	addcleanup();

	for (;;) {
		updateinfo();

		/* wait up to 2 seconds for a message from the upsd */

		if (getupsmsg(2))
			syslog (LOG_INFO, "Received a message from upsd\n");
	}
}

int send_string (char * data)
{
	tcflush (upsfd, TCIFLUSH);
	return (write (upsfd, data, strlen (data)));
}

int recvbinary (char *buf, int buflen)
{
	unsigned char in;
	int ret, counter = 0, retval = 0;
	struct sigaction sa;
	sigset_t sigmask;

	sa.sa_handler = timeout;
	sigemptyset (&sigmask);
	sa.sa_mask = sigmask;
	sa.sa_flags = 0;
	sigaction (SIGALRM, &sa, NULL);

	alarm (3);

        while (counter < buflen) {
    	    ret = read (upsfd, &in, 1);
	    
	    if (ret > 0) {
		buf[counter] = in;
		counter ++;
		nolongertimeout();	   
	    }
	    else {
	        syslog (LOG_DEBUG, "error reading from serial device!");
		retval = -1;
		break;
	    }
	}
	
	alarm (0);
	signal (SIGALRM, SIG_IGN);
        return (retval);
}

void set_serialDTR0RTS1(void)
{
  int dtr_bit = TIOCM_DTR;
  int rts_bit = TIOCM_RTS;
  // set DTR to low and RTS to high
  ioctl(upsfd, TIOCMBIC, &dtr_bit);
  ioctl(upsfd, TIOCMBIS, &rts_bit);
}
