// TTY-Grin Widget Set
// Copyright (C) 2001 Daniel Beer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <string.h>
#include <sys/types.h>
#include <regex.h>
#include "tgws.h"

tgws::ctree_node::ctree_node(ctree *t) : parent(t), prev(0), next(0), first(0),
				      last(0), owner(0), expanded(false),
				      indent(0), tag_collapsed(' '),
					 tag_expanded(' ') {
  text=new string[t->num];
  user_data=new void*[t->num];
}

tgws::ctree_node::~ctree_node(void) {
  delete[] text;
  delete[] user_data;
  if(prev) prev->next=next;
  if(next) next->prev=prev;
  if(parent->cursor==this) parent->cursor=prev;
  if(parent->scr==this) parent->scr=parent->cursor;
  if(parent->first==this) parent->first=next;
  if(parent->last==this) parent->last=prev;
  if(owner) {
    if(owner->first==this) owner->first=next;
    if(owner->last==this) owner->last=prev;
    if(!owner->first) owner->expanded=false;
  }
  while(first) delete first;
}

tgws::ctree_node *
tgws::ctree_node::up(void) const {
  if(prev) {
    ctree_node *x;
    for(x=prev;x->expanded&&x->last;x=x->last);
    return x;
  }
  return owner;
}

tgws::ctree_node *
tgws::ctree_node::down(void) const {
  if(expanded) return first;
  if(next) return next;
  ctree_node *x;
  for(x=owner;x;x=x->owner) if(x->next) return x->next;
  return 0;
}

tgws::ctree_node *
tgws::ctree_node::up_expand(void) const {
  if(prev) {
    ctree_node *x;
    for(x=prev;x->last;x=x->last);
    return x;
  }
  return owner;
}

tgws::ctree_node *
tgws::ctree_node::down_expand(void) const {
  if(first) return first;
  if(next) return next;
  ctree_node *x;
  for(x=owner;x;x=x->owner) if(x->next) return x->next;
  return 0;
}

void
tgws::ctree_node::expand_all(bool e) {
  expanded=e;
  for(ctree_node *n=get_first();n;n=n->get_next())
    if(n->get_first()) n->expand_all(e);
}

tgws::ctree::ctree(int n) : num(n), count(0), indent(2), headings(false),
			  cursor(0), scr(0), first(0), last(0) {
  title=new string[n];
  columns=new int[n];
  memset(columns, 0, sizeof(*columns)*n);
}

tgws::ctree::~ctree(void) {
  visible=false;
  delete[] title;
  delete[] columns;
  while(first) delete first;
}

void
tgws::ctree::draw(bool running=false) {
  wbkgdset(win, look->get_attr(LIST));
  werase(win);
  if(first) {
    if(!cursor) scr=cursor=first;
    while(cursor->owner&&!cursor->owner->expanded) cursor=cursor->owner;
    {
      if(scr->up()==cursor) scr=scr->up();
      else {
	int i=headings?1:0;
	ctree_node *n;
	for(n=scr;n&&i<height&&n!=cursor;n=n->down()) i++;
	if(i==height||!n) {
	  if(n==cursor) scr=scr->down();
	  else {
	    scr=cursor;
	    for(i=headings?2:1;i<height&&scr->up();i++) scr=scr->up();
	  }
	}
      }
    }
    if(headings) {
      wattrset(win, look->get_attr(TEXT));
      draw_line(title);
    }
    
    ctree_node *n=scr;
    int i=headings?1:0;
    while(n&&i<height) {
      wattrset(win, look->get_attr(cursor==n&&running?SELECTION:LIST));
      draw_line(n->text, n->indent, n->first?(n->expanded?'-':'+'):' ',
		n->expanded?n->tag_expanded:n->tag_collapsed);
      n=n->down();
      i++;
    }
  }
  input::draw();
}

int
tgws::ctree::run(void) {
  if(!first) {
    doupdate();
    return wgetch(win);
  }
  for(;;) {
    draw(true);
    doupdate();
    switch(int key=wgetch(win)) {
    case '=':
    case '+': if(cursor->first) cursor->set_expanded(true); break;
    case '-': cursor->set_expanded(false); break;
    case KEY_HOME: cursor=first; break;
    case KEY_LL:
    case KEY_END: for(cursor=last;cursor->expanded;cursor=cursor->last); break;
    case KEY_UP: if(cursor->up()) cursor=cursor->up(); break;
    case KEY_DOWN: if(cursor->down()) cursor=cursor->down(); break;
    case KEY_PPAGE:
      for(int i=headings?1:0;i<height&&cursor->up();i++) cursor=cursor->up();
      break;
    case KEY_NPAGE:
      for(int i=headings?1:0;i<height&&cursor->down();i++)
	cursor=cursor->down();
      break;
    case '/':
    case '?':
      {
	string pattern;
	if(tgws::get_line("Find:", pattern)) {
	  if(pattern.size()) last_pattern=pattern;
	  else pattern=last_pattern;

	  regex_t preg;
	  tgws::ctree_node *n=cursor;
	  int i;

	  regcomp(&preg, pattern.c_str(), REG_EXTENDED|REG_ICASE|REG_NOSUB);
	  do {
	    if((n=(key=='/'?n->down_expand():n->up_expand())))
	      for(i=0;i<num&&regexec(&preg, n->text[i].c_str(), 0, 0, 0);i++);
	  } while(n&&i>=num);

	  if(n) {
	    cursor=n;
	    for(ctree_node *x=cursor->owner;x;x=x->owner)
	      x->set_expanded(true);
	  } else tgws::error_message("Pattern not found.");
	}
      }
      break;
    default:
      draw();
      return key;
    }
  }
}

tgws::ctree_node *
tgws::ctree::insert(ctree_node *owner, ctree_node *sibling) {
  ctree_node *n=new ctree_node(this);

  n->next=sibling;
  n->owner=owner;
  if(sibling) {
    n->prev=sibling->prev;
    sibling->prev->next=n;
    sibling->prev=n;
  }

  if(owner) {
    n->indent=owner->indent+1;
    if(!sibling) {
      n->prev=owner->last;
      if(owner->last) owner->last->next=n;
      owner->last=n;
    }
    if(!owner->first) owner->first=n;
  } else {
    if(!sibling) {
      n->prev=last;
      if(last) last->next=n;
      last=n;
    }
    if(!first) first=n;
  }
  return n;
}

void
tgws::ctree::draw_line(const string *title, int in, char prefix,
		       unsigned char tag) {
  int i, total=in*indent+7;
  for(i=0;i<in*indent;i++) waddch(win, ' ');
  waddch(win, ' ');
  waddch(win, (prefix==' ')?' ':'[');
  waddch(win, (unsigned char)prefix);
  waddch(win, (prefix==' ')?' ':']');
  waddch(win, ' ');
  waddch(win, tag);
  waddch(win, ' ');
  for(i=0;total<width&&i<num;i++) {
    int j;
    for(j=0;total+1<width&&j<(signed)title[i].size()&&
	  j+1<columns[i];j++) {
      waddch(win, (unsigned char)title[i][j]);
      total++;
    }
    waddch(win, ' ');
    total++;
    while(total<width&&j++<columns[i]) {
      waddch(win, ' ');
      total++;
    }
  }
  while(total++<width) waddch(win, ' ');
}

void
tgws::ctree::expand_all(bool e) {
  for(ctree_node *n=get_first();n;n=n->get_next())
    if(n->get_first()) n->expand_all(e);
}
