// TTY-Grin Widget Set
// Copyright (C) 2001 Daniel Beer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <string.h>
#include <sys/types.h>
#include <regex.h>
#include "tgws.h"

tgws::clist_node::clist_node(clist *p) : parent(p), prev(0), next(0) {
  text=new string[p->num];
  user_data=new void*[p->num];
  p->count++;
}

tgws::clist_node::~clist_node(void) {
  if(prev) prev->next=next;
  if(next) next->prev=prev;
  delete[] text;
  delete[] user_data;
  parent->count--;
  if(parent->cursor==this) parent->cursor=prev;
  if(parent->scr==this) parent->scr=parent->cursor;
  if(parent->first==this) parent->first=next;
  if(parent->last==this) parent->last=prev;
  if(parent->visible) parent->draw();
}

tgws::clist::clist(int n) : num(n), count(0), headings(false),
			  first(0), last(0), cursor(0) {
  title=new string[n];
  columns=new int[n];
  memset(columns, 0, sizeof(*columns)*n);
}

tgws::clist::~clist(void) {
  visible=false;
  delete[] title;
  delete[] columns;
  while(first) delete first;
}

void
tgws::clist::clear(void) {
  while(first) delete first;
  first=last=cursor=scr=0;
  count=0;
}

void
tgws::clist::draw(bool running=false) {
  wbkgdset(win, look->get_attr(LIST));
  werase(win);
  if(headings) {
    wattrset(win, look->get_attr(TEXT));
    draw_line(title);
  }
  if(first) {
    if(!cursor) cursor=scr=first;
    {
      if(scr->prev==cursor) scr=scr->prev;
      else {
	int i=headings?1:0;
	clist_node *n;
	for(n=scr;n&&i<height&&n!=cursor;n=n->next) i++;
	if(i==height||!n) {
	  if(n==cursor) scr=scr->next;
	  else {
	    scr=cursor;
	    for(i=headings?2:1;i<height&&scr->prev;i++) scr=scr->prev;
	  }
	}
      }
    }
    clist_node *n=scr;
    for(int i=headings?1:0;i<height&&n;i++) {
      wattrset(win, look->get_attr(n==cursor&&running?SELECTION:LIST));
      draw_line(n->text);
      n=n->next;
    }
  }
  input::draw();
}

int
tgws::clist::run(void) {
  if(!first) {
    doupdate();
    return wgetch(win);
  }
  for(;;) {
    draw(true);
    doupdate();
    switch(int key=wgetch(win)) {
    case KEY_HOME: cursor=first; break;
    case KEY_LL:
    case KEY_END: cursor=last; break;
    case KEY_UP: if(cursor->prev) cursor=cursor->prev; break;
    case KEY_DOWN: if(cursor->next) cursor=cursor->next; break;
    case KEY_PPAGE:
      for(int i=headings?1:0;i<height&&cursor->prev;i++) cursor=cursor->prev;
      break;
    case KEY_NPAGE:
      for(int i=headings?1:0;i<height&&cursor->next;i++) cursor=cursor->next;
      break;
    case '/':
    case '?':
      {
	string pattern;
	if(tgws::get_line("Find:", pattern)) {
	  if(pattern.size()) last_pattern=pattern;
	  else pattern=last_pattern;

	  regex_t preg;
	  tgws::clist_node *n=cursor;
	  int i;

	  regcomp(&preg, pattern.c_str(), REG_EXTENDED|REG_ICASE|REG_NOSUB);
	  do {
	    if((n=(key=='/'?n->next:n->prev)))
	      for(i=0;i<num&&regexec(&preg, n->text[i].c_str(), 0, 0, 0);i++);
	  } while(n&&i>=num);

	  if(n) cursor=n;
	  else tgws::error_message("Pattern not found.");
	}
      }
      break;
    default:
      draw();
      return key;
    }
  }
}

tgws::clist_node *
tgws::clist::insert(clist_node *sibling) {
  clist_node *n=new clist_node(this);

  n->next=sibling;
  if(sibling) {
    n->prev=sibling->prev;
    sibling->prev->next=n;
    sibling->prev=n;
  } else {
    n->prev=last;
    if(last) last->next=n;
    last=n;
  }

  if(!first) first=n;
  return n;
}

void
tgws::clist::draw_line(const string *title) {
  int i, total=0;
  for(i=0;total<width&&i<num;i++) {
    int j;
    for(j=0;total+1<width&&j<(signed)title[i].size()&&
	  j+1<columns[i];j++) {
      waddch(win, (unsigned char)title[i][j]);
      total++;
    }
    while(total<width&&j++<columns[i]) {
      waddch(win, ' ');
      total++;
    }
  }
  while(total++<width) waddch(win, ' ');
}
