//
//  kjots
//
//  Copyright (C) 1997 Christoph Neerfeld
//  email:  Christoph.Neerfeld@home.ivm.de or chris@kde.org
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include <qobjectlist.h>
#include <qclipboard.h>
#include <qtimer.h>
#include <qcursor.h>

#include <kaccel.h>
#include <kaction.h>
#include <kstdaccel.h>
#include <kstdaction.h>
#include <kapplication.h>
#include <kfiledialog.h>
#include <kdebug.h>
#include <kpopupmenu.h>
#include <kkeydialog.h>
#include <klocale.h>
#include <kmenubar.h>
#include <kmessagebox.h>
#include <kopenwith.h>
#include <krun.h>
#include <kstandarddirs.h>
#include <kurl.h>

extern "C" {
#include <unistd.h>
#include <stdlib.h>
};

#include "KJotsMain.h"
#include "SubjList.h"
// #include "ReadListConf.h"
#include "cpopmenu.h"
#include "optiondialog.h"
#include <klocale.h>
#include <qlabel.h>


#define KJOTSVERSION "0.4"

const unsigned int HOT_LIST_SIZE = 8;
const unsigned int BUTTON_WIDTH = 56;


//----------------------------------------------------------------------
// ASKFILENAME
//----------------------------------------------------------------------
AskFileName::AskFileName(QWidget* parent, const char* name)
  : KDialogBase( parent, name, TRUE, i18n("New Book"), Ok|Cancel, Ok )
{
  QFrame *page = makeMainWidget();
  QVBoxLayout *vlay = new QVBoxLayout( page, 0, spacingHint() );

  QLabel *label = new QLabel( i18n("Book name:"), page );
  vlay->addWidget(label);

  i_name = new QLineEdit( page );
  i_name->setMinimumWidth( fontMetrics().maxWidth()*20 );
  i_name->setFocus();
  vlay->addWidget(i_name );
  connect(i_name, SIGNAL(textChanged(const QString &)),
	  this, SLOT(textChanged(const QString &)));

  vlay->addStretch( 10 );

  enableButtonOK( false );
}

void AskFileName::textChanged(const QString &s)
{
  enableButtonOK( s.length() > 0 );
}


//----------------------------------------------------------------------
// MYMULTIEDIT
//----------------------------------------------------------------------
MyMultiEdit::MyMultiEdit (QWidget* parent, const char* name)
  : KEdit(parent, name)
{
  setWordWrap(QTextEdit::WidgetWidth);
  web_menu = new CPopupMenu;
  web_menu->insertItem(i18n("Open URL"), this, SLOT(openUrl()) );
}

void MyMultiEdit::keyPressEvent( QKeyEvent *e )
{
  if( e->key() == Key_Tab )
    {
      int line, col;
      getCursorPosition(&line, &col);
      KEdit::insertAt("\t", line, col);
      return;
    }
  KEdit::keyPressEvent(e);
  return;
}

void MyMultiEdit::mousePressEvent( QMouseEvent *e )
{
  if(e->button() == RightButton)
  {
    if(hasMarkedText())
  	{
      KURL url(markedText());

      if(url.isValid())
      {
  	    web_menu->popup(QCursor::pos());
	    }
      
      return;
  	}
  }

  KEdit::mousePressEvent(e);
}

void MyMultiEdit::openUrl()
{
  if(hasMarkedText())
  {
    KURL url(markedText());
    if(url.isValid())
    {
	     KFileOpenWithHandler fowh;	  
      (void) new KRun(url);
    }
  }
}

//----------------------------------------------------------------------
// MYBUTTONGROUP
//----------------------------------------------------------------------

MyButtonGroup::MyButtonGroup (QWidget* parent, const char* name)
  : QButtonGroup(parent, name)
{
}

void MyButtonGroup::resizeEvent( QResizeEvent * )
{
  int x = width()-2;
  QObjectList  *list = queryList( "QPushButton" );
  QObjectListIt it( *list );
  while ( it.current() ) {
    x -= (BUTTON_WIDTH+4);
    ((QPushButton *) it.current())->move(x, 4);
    ++it;
  }
  delete list;
}

//----------------------------------------------------------------------
// KJOTSMAIN
//----------------------------------------------------------------------

KJotsMain::KJotsMain(const char* name)
  : KMainWindow( 0, name )
{
  mOptionDialog = 0;
  folderOpen    = FALSE;
  current       = 0;
  unique_id     = 0;

  //
  // Main widget
  //
  QFrame* f_main = new QFrame( this, "Frame_0" );
  f_main->setFrameStyle( QFrame::NoFrame );
  setCentralWidget(f_main);

  //
  // Text area
  //
  QFrame* f_text = new QFrame( f_main, "Frame_1" );
  f_text->setFrameStyle( QFrame::NoFrame );

  //
  // Status bar area
  //
  QFrame* f_labels = new QFrame( f_main, "Frame_2" );
  f_labels->setFrameStyle( QFrame::NoFrame );

  //
  // Data widgets.
  //
  s_bar = new QScrollBar( f_main, "ScrollBar_1" );
  s_bar->setOrientation( QScrollBar::Horizontal );
  s_bar->setRange(0,0);
  s_bar->setValue(0);
  s_bar->setSteps(1,1);
  connect( s_bar, SIGNAL(valueChanged(int)), this, SLOT(barMoved(int)) );

  me_text = new MyMultiEdit( f_text, "me_text" );
  me_text->setMinimumWidth( fontMetrics().maxWidth()*40 );
  me_text->setFocusPolicy(QWidget::StrongFocus);
  me_text->setEnabled(FALSE);

  l_folder = new QLabel( f_labels, "Label_4" );
  l_folder->setFrameStyle( QFrame::WinPanel | QFrame::Sunken );
  QFont font_label(l_folder->fontInfo().family());
  font_label.setBold(TRUE);
  l_folder->setFont(font_label);

  le_subject = new QLineEdit( f_labels, "le_subject" );
  le_subject->setFocusPolicy(QWidget::ClickFocus);
  le_subject->setEnabled(FALSE);

  l_folder->setMinimumWidth( fontMetrics().maxWidth()*8 );
  l_folder->setFixedHeight( le_subject->sizeHint().height() );

  bg_top = new MyButtonGroup( f_main, "ButtonGroup_2" );
  QPushButton tp(0, "Test");
  bg_top->setFixedHeight( tp.sizeHint().height()+8 );
  bg_top->setMinimumWidth( HOT_LIST_SIZE * (BUTTON_WIDTH+4) + 6 );
  bg_top->setFrameStyle( QFrame::Box | QFrame::Sunken );
  bg_top->setExclusive(TRUE);

  //
  // Put data widgets into layouts
  //
  QVBoxLayout *l1 = new QVBoxLayout(f_text, 0, 0);
  l1->addWidget(me_text);

  QHBoxLayout *l2 = new QHBoxLayout(f_labels);
  l2->addWidget(l_folder);
  l2->addWidget(le_subject, 1 );

  QVBoxLayout *tl = new QVBoxLayout(f_main, 4);
  tl->addWidget(f_text, 1);
  tl->addWidget(s_bar);
  tl->addWidget(bg_top);
  tl->addWidget(f_labels);



  entrylist.setAutoDelete(TRUE);
  entrylist.append(new TextEntry);
  button_list.setAutoDelete(TRUE);

  subj_list = new SubjList( topLevelWidget(), 0, false );
  connect( this, SIGNAL(folderChanged(QPtrList<TextEntry> *)),
	   subj_list, SLOT(rebuildList( QPtrList<TextEntry> *)) );
  connect( this, SIGNAL(entryMoved(int)),
	   subj_list, SLOT( select(int)) );
  connect( subj_list, SIGNAL(entryMoved(int)),
	   this, SLOT( barMoved(int)) );
  connect( le_subject, SIGNAL(textChanged(const QString &)),
	   subj_list, SLOT(entryChanged(const QString &)) );



  KConfig *config = kapp->config();
  config->setGroup("kjots");

  // read hotlist

  hotlist = config->readListEntry( "Hotlist" );
  while( hotlist.count() > HOT_LIST_SIZE )
    hotlist.remove(hotlist.last());
  // read list of folders
  QDir dir(locateLocal("appdata", ""));
  QString absPath = dir.absPath() + '/';
  QStringList files = dir.entryList(QDir::Files |  QDir::Readable); // config->readListEntry( "Folders" );

  // create hotlist buttons and hotlist menu
  folders_menu = new QPopupMenu;
  QPushButton *temp_button;
  for ( QStringList::Iterator it = files.begin();
        it != files.end();
        ++it)
  {
    QFile folder(absPath + *it);
    if(!folder.open(IO_ReadWrite))
    {
      continue;
    }

    QTextStream st( (QIODevice *) &folder);
    QString buf = st.readLine();
    
    if(buf.left(9) != "\\NewEntry")
    {
      continue;
    }
    
    QString decodedFilename = KURL::decode_string(*it);
    folder_list.push_back(decodedFilename);
    folders_menu->insertItem(decodedFilename, unique_id);
    if(hotlist.contains(decodedFilename))
    {
      temp_button = new QPushButton(decodedFilename, bg_top);
      temp_button->setFocusPolicy(QWidget::ClickFocus);
      temp_button->setToggleButton(TRUE);
      temp_button->setFixedSize(BUTTON_WIDTH, temp_button->sizeHint().height());
      bg_top->insert(temp_button, unique_id);
      button_list.append(temp_button);
    }
 
    ++unique_id;
  }

  connect( folders_menu, SIGNAL(activated(int)), this, SLOT(openFolder(int)) );
  connect( bg_top, SIGNAL(clicked(int)), this, SLOT(openFolder(int)) );

  updateConfiguration();

  // creat keyboard shortcuts
  // CTRL+Key_J := insert date
  // CTRL+Key_J := previous page
  // CTRL+Key_K := next page
  // CTRL+Key_L := show subject list
  // CTRL+Key_A := add new page
  // CTRL+Key_M := move focus

  keys = new KAccel( this );
  keys->insertItem(i18n("Move Focus"),       "MoveFocus",       CTRL+Key_M);
  keys->connectItem( "MoveFocus", this, SLOT(moveFocus()) );

  // create menu
  int id;
  KAction *action;
  QPopupMenu *file_menu = new QPopupMenu;
  
  // New Book
  action = KStdAction::action(KStdAction::New, this, SLOT(createFolder()), actionCollection());
  action->setText(i18n("&New Book..."));
  action->plug(file_menu);

  // added more actionized items and toolbar (antonio)
  action = new KAction(i18n("New P&age"), "filenew", CTRL+Key_A, this,
                       SLOT(newEntry()), actionCollection(), "new_page");
  action->plug(file_menu);
  action->plug(toolBar(0));
  action->plugAccel(keys);

  file_menu->insertSeparator();
  
  // Save Book
  action = KStdAction::action(KStdAction::Save, this, SLOT(saveFolder()), actionCollection());
  action->setText(i18n("Save Current Book"));
  action->plug(file_menu);
  
  id_f_save_book_ascii = file_menu->insertItem(i18n("Save Book to ASCII File"),
					                                     this, SLOT(writeBook()));
  id_f_save_page_ascii = file_menu->insertItem(i18n("Save Page to ASCII File"),
					                                     this, SLOT(writePage()));

  file_menu->insertSeparator();

  action = new KAction(i18n("Delete Current Book"), "editdelete", 0, this,
                       SLOT(deleteFolder()), actionCollection(), "del_folder");
  action->plug(file_menu);
  action->plugAccel(keys);

  action = new KAction(i18n("&Delete Page"), "editdelete", CTRL+Key_D, this,
                       SLOT(deleteEntry()), actionCollection(), "del_page");
  action->plug(file_menu);
  action->plug(toolBar(0));
  action->plugAccel(keys);

  file_menu->insertSeparator();

  // Quit
  action = KStdAction::action(KStdAction::Quit, this, SLOT(slotQuit()), actionCollection());
  action->plug(file_menu);

  QPopupMenu *edit_menu = new QPopupMenu;

  action = KStdAction::action(KStdAction::Cut, me_text, SLOT(cut()), actionCollection());
  action->plug(edit_menu);
  
  action  = KStdAction::action(KStdAction::Copy, me_text, SLOT(copy()), actionCollection());
  action->plug(edit_menu);

  action = new KAction(i18n("Copy &Into Page Title"), "editcopy", CTRL+Key_T, this,
                       SLOT(copySelection()), actionCollection(), "copyIntoTitle");
  action->plug(edit_menu);
  action->plugAccel(keys);

  action  = KStdAction::action(KStdAction::Paste, me_text, SLOT(paste()), actionCollection());
  action->plug(edit_menu);
  
  edit_menu->insertSeparator();

  // added actionized items (pfeiffer)
  action = KStdAction::find( this, SLOT( slotSearch() ), actionCollection() );
  action->plug( edit_menu );
  action = KStdAction::findNext( this, SLOT( slotRepeatSearch() ), actionCollection() );
  action->plug( edit_menu );
  action = KStdAction::replace( this, SLOT( slotReplace() ), actionCollection() );
  action->plug( edit_menu );
  edit_menu->insertSeparator();

  action = new KAction(i18n("Insert Date"), "date", CTRL+Key_I, this,
                       SLOT(insertDate()), actionCollection(), "insert_date");
  action->plug(edit_menu);
  action->plug(toolBar(0));
  action->plugAccel( keys );
  edit_menu->insertSeparator();

  action = new KAction(i18n("Previous Page"), "back", CTRL+Key_J, this,
                       SLOT(prevEntry()), actionCollection(), "prev_page");
  action->plug( edit_menu );
  action->plug( toolBar(0) );
  action->plugAccel( keys );
  action = new KAction(i18n("Next Page"), "forward", CTRL+Key_K, this,
                       SLOT(nextEntry()), actionCollection(), "next_page");
  action->plug( edit_menu );
  action->plug( toolBar(0) );
  action->plugAccel( keys );

  toolBar(0)->insertSeparator();

  QPopupMenu *options_menu = new QPopupMenu;
  action = KStdAction::keyBindings(this, SLOT(configureKeys()), actionCollection());
  action->plug(options_menu);
  /*
   * TODO: configure toolbar? need to use xmlgui
   * action = KStdAction::configureToolbars(this, SLOT(configureToolbars()), actionCollection());
   */
  action = KStdAction::preferences(this, SLOT(configure()), actionCollection());
  action->plug(options_menu);

  QPopupMenu *hotlist_menu = new QPopupMenu;
  id_fav_add = hotlist_menu->insertItem(i18n("Add Current Book to Hotlist"),
				                                this, SLOT(addToHotlist()) );
  id_fav_del = hotlist_menu->insertItem(i18n("Remove Current Book from Hotlist"),
				                                this, SLOT(removeFromHotlist()) );

  action=new KAction(i18n("Subject List"), "view_detailed", CTRL+Key_L, this,
        SLOT(toggleSubjList()), actionCollection(), "subject_list");
  action->plug(hotlist_menu);
  action->plug(toolBar(0));
  action->plugAccel(keys);

  menuBar()->insertItem( i18n("&File"), file_menu );
  menuBar()->insertItem( i18n("&Edit"), edit_menu );
  menuBar()->insertItem( i18n("&Books"), folders_menu );
  menuBar()->insertItem( i18n("Hot&list"), hotlist_menu );
  menuBar()->insertItem( i18n("&Settings"), options_menu );
  menuBar()->insertSeparator();
  menuBar()->insertItem( i18n("&Help"), helpMenu() );

  int nr = folder_list.findIndex(KURL::decode_string(config->readEntry("LastOpenFolder")));
  if( nr >= 0 )
  {
    openFolder(nr);
  }

  //
  // Set minimum size of main widget
  //
  f_main->setMinimumSize( f_main->sizeHint() );

  //
  // Set startup size.
  //
  int w = config->readNumEntry("Width");
  int h = config->readNumEntry("Height");
  w = QMAX( w, sizeHint().width() );
  h = QMAX( h, sizeHint().height() );
  resize(w, h);

  keys->readSettings();
  updateMenu();
}



KJotsMain::~KJotsMain()
{
  delete mOptionDialog;
  saveProperties();
}


void KJotsMain::saveProperties( void )
{
  KConfig &config = *kapp->config();
  config.setGroup("kjots");
  button_list.clear();
  if( folderOpen )
  {
    QFileInfo fi(current_folder_name);
    config.writeEntry("LastOpenFolder", fi.fileName());
  }
  saveFolder();
  config.writeEntry("Width", width());
  config.writeEntry("Height", height());
  config.writeEntry("ToolBarPos", (int)toolBar(0)->barPos() );
  config.sync();
}

int KJotsMain::readFile( QString name )
{
  int pos;
  QString buf, subj;
  TextEntry *entry;

  entrylist.clear();
  QFile folder(name);
  if( !folder.open(IO_ReadWrite) )
    return -1;

  QTextStream st( (QIODevice *) &folder);
  buf = st.readLine();
  if( buf.left(9) != "\\NewEntry" )
    return -1;

  entry = new TextEntry;
  entrylist.append( entry );
  subj = buf.mid(10, buf.length() );
  entry->subject = subj.isNull() ? (QString) "" : (QString) subj;
  while( !st.eof() )
    {
      buf = st.readLine();
      if( buf.left(9) == "\\NewEntry" )
	{
	  entry = new TextEntry;
	  entrylist.append(entry);
	  subj = buf.mid(10, buf.length() );
	  entry->subject = subj.isNull() ? (QString) "" : (QString) subj;
	  buf = st.readLine();
	}
      pos = 0;
      while( (pos = buf.find( '\\', pos )) != -1 )
	if( buf[++pos] == '\\' )
	  buf.remove( pos, 1 );
      entry->text.append( buf );
      entry->text.append("\n");
    }
  folder.close();
  while( entry->text.right(1) == "\n" )
    entry->text.truncate(entry->text.length() - 1);
  entry->text.append("\n");

  updateMenu();
  return 0;
}

int KJotsMain::writeFile( QString name ) 
{
  int pos;
  TextEntry *entry = entrylist.first();
  QString buf;
  QFile folder(name);
  if( !folder.open(IO_WriteOnly | IO_Truncate) )
    return -1;
  QTextStream st( (QIODevice *) &folder);

  if (entry == 0)
  {
    st << "\\NewEntry \n\n";
  }
  else  
  {
    for(; entry != NULL; entry = entrylist.next())
    {
      st << "\\NewEntry ";
      st << entry->subject;
      st << "\n";
      buf = entry->text;

      pos = 0;
      while( (pos = buf.find( '\\', pos )) != -1 )
     	{
    	  buf.insert( ++pos, '\\' );
     	  pos++;
     	}
      st << buf;
      if( buf.right(1) != "\n" )
       	st << '\n';
    }
  }

  folder.close();
  return 0;
}

void KJotsMain::openFolder(int id) 
{
  if (folders_menu->indexOf(id) < 0)
  {
    return;
  }
 
  QPushButton *but;
  for( but = button_list.first(); but != NULL; but = button_list.next() )
    but->setOn(FALSE);
  but = (QPushButton *) bg_top->find(id);
  if( but )
    but->setOn(TRUE);

  QDir dir = QDir( locateLocal("appdata", "") );
  QString file_name = dir.absPath();
  file_name += '/';
  file_name += KURL::encode_string_no_slash(*folder_list.at(folders_menu->indexOf(id)));
  if( current_folder_name == file_name )
    return;

  if( folderOpen )
    saveFolder();
  current_folder_name = file_name;

  if( readFile(current_folder_name) < 0)
    {
      folderOpen = FALSE;
      kdDebug() << "Unable to open folder" << endl;
      return;
    }

  current = 0;
  me_text->deselect();
  me_text->setText(entrylist.first()->text);
  emit folderChanged(&entrylist);
  emit entryMoved(current);
  le_subject->setText(entrylist.first()->subject);
  folderOpen = TRUE;
  l_folder->setText( *folder_list.at(folders_menu->indexOf(id)) );
  me_text->setEnabled(TRUE);
  le_subject->setEnabled(TRUE);
  me_text->setFocus();
  s_bar->setValue(0);
  s_bar->setRange(0,entrylist.count()-1);

  updateMenu();
}


void KJotsMain::createFolder()
{
  AskFileName *ask = new AskFileName(this);
  if( ask->exec() == QDialog::Rejected )
  {
    delete ask;
    return;
  }
  QString name = ask->getName();
  delete ask;
  if( folder_list.contains(name) )
  {
    QString msg = i18n("A book with this name already exists.");
    KMessageBox::sorry( topLevelWidget(), msg );
    return;
  }
  saveFolder();
  entrylist.clear();
  folderOpen = TRUE;
  me_text->setEnabled(TRUE);
  le_subject->setEnabled(TRUE);
  me_text->setFocus();
  me_text->clear();
  me_text->deselect();
  TextEntry *new_entry = new TextEntry;
  entrylist.append(new_entry);
  new_entry->subject = "";
  current = 0;
  s_bar->setValue(0);
  s_bar->setRange(0,0);
  emit folderChanged(&entrylist);
  emit entryMoved(current);
  le_subject->setText(entrylist.first()->subject);
  folder_list.append(name);

  // TODO: this does not work with qt 2.0. Why?
//     if( folders->text(folders->idAt(0)) == 0 )
//      folders->removeItemAt(0);

  folders_menu->insertItem(name, unique_id++);
  QDir dir = QDir( locateLocal("appdata", "") );
  current_folder_name = dir.absPath();
  current_folder_name += '/';
  current_folder_name += KURL::encode_string_no_slash(name);
  l_folder->setText(name);
  QPushButton *but;
  for( but = button_list.first(); but != 0; but = button_list.next() )
    but->setOn(FALSE);

  updateMenu();
}

void KJotsMain::deleteFolder()
{
  if( !folderOpen )
    return;

  QString msg = i18n("Are you sure you want to delete the current book?");
  int result = KMessageBox::warningYesNo( topLevelWidget(), msg );
  if( result != KMessageBox::Yes )
  {
    return;
  }

  QFileInfo fi(current_folder_name);
  QDir dir = fi.dir(TRUE);
  QString name = fi.fileName();
  int index = folder_list.findIndex(name);
  if( index < 0 )
    return;
  dir.remove(current_folder_name);
  folder_list.remove(folder_list.at(index));
  int id = folders_menu->idAt(index);
  folders_menu->removeItemAt(index);
  if( hotlist.contains(name) )
    {
      hotlist.remove(name);
      QButton *but = bg_top->find(id);
      bg_top->remove(but);
      button_list.remove( (QPushButton *) but );
      resize(size());
    }
  KConfig *config = KApplication::kApplication()->config();
  config->setGroup("kjots");
  config->writeEntry( "Folders", folder_list );
  config->writeEntry( "Hotlist", hotlist );
  config->sync();
  entrylist.clear();
  current_folder_name = "";
  folderOpen = FALSE;
  me_text->setEnabled(FALSE);
  me_text->clear();
  me_text->deselect();
  le_subject->setEnabled(FALSE);
  le_subject->clear();
  emit folderChanged(&entrylist);
  s_bar->setValue(0);
  s_bar->setRange(0,0);
  l_folder->clear();
  subj_list->repaint(TRUE);

  openFolder(folders_menu->idAt(0));
  updateMenu();
}

void KJotsMain::saveFolder()
{
  if( !folderOpen )
    return;
  entrylist.at(current)->text = me_text->text();
  entrylist.at(current)->subject = le_subject->text();
  writeFile(current_folder_name);
}

void KJotsMain::nextEntry()
{
  if( !folderOpen )
    return;
  if( current+1 >= (int) entrylist.count() )
    return;
  entrylist.at(current)->text = me_text->text();
  entrylist.at(current)->subject = le_subject->text();
  me_text->setText( entrylist.at(++current)->text );
  me_text->deselect();
  me_text->repaint();
  emit entryMoved(current);
  le_subject->setText( entrylist.at(current)->subject );
  s_bar->setValue(current);

  updateMenu();
}

void KJotsMain::prevEntry()
{
  if( !folderOpen )
    return;
  if( current-1 < 0 )
    return;
  entrylist.at(current)->text = me_text->text();
  entrylist.at(current)->subject = le_subject->text();
  me_text->setText( entrylist.at(--current)->text );
  me_text->deselect();
  me_text->repaint();
  emit entryMoved(current);
  le_subject->setText( entrylist.at(current)->subject );
  s_bar->setValue(current);

  updateMenu();
}

void KJotsMain::newEntry()
{
  if( !folderOpen )
    return;
  entrylist.at(current)->text = me_text->text();
  entrylist.at(current)->subject = le_subject->text();
  me_text->clear();
  le_subject->clear();
  TextEntry *new_entry = new TextEntry;
  entrylist.append(new_entry);
  new_entry->subject = "";
  current = entrylist.count()-1;
  s_bar->setRange(0,current);
  s_bar->setValue(current);
  emit folderChanged(&entrylist);
  emit entryMoved(current);

  updateMenu();
}

void KJotsMain::deleteEntry()
{
  if( !folderOpen )
    return;
  if( entrylist.count() == 0 )
    return;
  else if( entrylist.count() == 1 )
    {
      entrylist.at(0)->text = "";
      entrylist.at(0)->subject = "";
      s_bar->setValue(0);
      me_text->clear();
      le_subject->clear();
      return;
    }
  entrylist.remove(current);
  if( current >= (int) entrylist.count() - 1 )
    {
      if( current )
      current--;
      s_bar->setValue(current);
      s_bar->setRange(0, entrylist.count()-1 );
    }
  me_text->setText( entrylist.at(current)->text );
  emit entryMoved(current);
  le_subject->setText( entrylist.at(current)->subject );
  s_bar->setRange(0, entrylist.count()-1 );
  emit folderChanged(&entrylist);

  updateMenu();
}

void KJotsMain::barMoved( int new_value )
{
  if( !folderOpen )
    return;
  if( current == new_value )
    return;
  entrylist.at(current)->text = me_text->text();
  entrylist.at(current)->subject = le_subject->text();
  current = new_value;
  me_text->setText( entrylist.at(current)->text );
  me_text->deselect();
  me_text->repaint();
  emit entryMoved(current);
  le_subject->setText( entrylist.at(current)->subject );
  s_bar->setValue(new_value);

  updateMenu();
}

void KJotsMain::addToHotlist()
{
  if( hotlist.count() == HOT_LIST_SIZE )
    return;
  QFileInfo fi(current_folder_name);
  QString name = fi.fileName();
  if( hotlist.contains(name) )
    return;
  hotlist.append(name);
  int index = folder_list.findIndex(name);
  if( index < 0 )
    return;
  int id = folders_menu->idAt(index);
  QPushButton *but = new QPushButton(name, bg_top);
  button_list.append(but);
  bg_top->insert(but, id);
  KConfig *config = KApplication::kApplication()->config();
  config->setGroup("kjots");
  config->writeEntry( "Hotlist", hotlist );
  config->sync();
  but->setFocusPolicy(QWidget::ClickFocus);
  but->setToggleButton(TRUE);
  but->setFixedSize(BUTTON_WIDTH, but->sizeHint().height());
  but->show();
  but->setOn(TRUE);
  bg_top->forceResize();
}

void KJotsMain::removeFromHotlist()
{
  QFileInfo fi(current_folder_name);
  QString name = fi.fileName();
  if( !hotlist.contains(name) )
    return;
  hotlist.remove(name);
  int index = folder_list.findIndex(name);
  if( index < 0 )
    return;
  int id = folders_menu->idAt(index);
  QButton *but = bg_top->find(id);
  bg_top->remove(but);
  button_list.remove( (QPushButton *) but );
  KConfig *config = KApplication::kApplication()->config();
  config->setGroup("kjots");
  config->writeEntry( "Hotlist", hotlist );
  config->sync();
  bg_top->forceResize();
}

void KJotsMain::toggleSubjList()
{
  if( subj_list->isVisible() )
  {
    subj_list->hide();
  }
  else
  {
    subj_list->setInitialSize( QSize(width()/2,height()) );
   //subj_list->resize( QSize(width() / 2, height()) );
    subj_list->show();
  }
}

void KJotsMain::configure()
{
  if( mOptionDialog == 0 )
  {
    mOptionDialog = new ConfigureDialog( topLevelWidget(), 0, false );
    if( mOptionDialog == 0 ) { return; }
    connect( mOptionDialog, SIGNAL(hidden()),this,SLOT(configureHide()));
    connect( mOptionDialog, SIGNAL(valueChanged()),
	     this, SLOT(updateConfiguration()) );
  }

  mOptionDialog->show();
}

void KJotsMain::configureHide()
{
  QTimer::singleShot( 0, this, SLOT(configureDestroy()) );
}

void KJotsMain::configureDestroy()
{
  if( mOptionDialog != 0 && mOptionDialog->isVisible() == false )
  {
    delete mOptionDialog; mOptionDialog = 0;
  }
}

void KJotsMain::slotSearch()
{
    me_text->search();
}

void KJotsMain::slotRepeatSearch()
{
    me_text->repeatSearch();
}

void KJotsMain::slotReplace()
{
    me_text->replace();
}

void KJotsMain::updateConfiguration()
{
  KConfig &config = *kapp->config();

  config.setGroup("kjots");
  QFont font( config.readEntry("EFontFamily"),
	      config.readNumEntry("EFontSize"),
	      config.readNumEntry("EFontWeight"),
	      config.readNumEntry("EFontItalic"));
  me_text->setFont(font);
}

void KJotsMain::writeBook()
{
  saveFolder();
  QString name;
  while( name.isNull() )
  {
    KURL url = KFileDialog::getSaveURL();

    if ( url.isEmpty() )
      return;

    if ( !url.isLocalFile() )
    {
      KMessageBox::sorry( 0L, i18n( "Only local files are currently supported." ) );
      return;
    }

    name = url.path();

    QFileInfo f_info(name);
    if( f_info.exists() )
    {
      QString msg = i18n("File already exists.\n"
	                 "Do you want to overwrite it?");
      int result = KMessageBox::warningYesNo( topLevelWidget(), msg );
      if( result != KMessageBox::Yes )
      {
	name = "";
      }
    }
  }
  QFile ascii_file(name);
  if( !ascii_file.open(IO_WriteOnly | IO_Truncate) )
    return;
  QTextStream st( (QIODevice *) &ascii_file);
  TextEntry *entry;
  for( entry = entrylist.first(); entry != NULL; entry = entrylist.next() )
    {
      writeEntry( st, entry );
    }
  ascii_file.close();
}

void KJotsMain::writePage()
{
  saveFolder();
  QString name;
  while( name.isNull() )
  {
    KURL url = KFileDialog::getSaveURL();

    if( url.isEmpty() )
      return;

    if( !url.isLocalFile() )
    {
      KMessageBox::sorry( 0L, i18n( "Only local files are currently supported." ) );
      return;
    }

    name = url.path();

    QFileInfo f_info(name);
    if( f_info.exists() )
    {
      QString msg = i18n("File already exists.\n"
	                 "Do you want to overwrite it?");
      int result = KMessageBox::warningYesNo( topLevelWidget(), msg );
      if( result != KMessageBox::Yes )
      {
	name = "";
      }
    }
  }
  QFile ascii_file(name);
  if( !ascii_file.open(IO_WriteOnly | IO_Truncate) )
    return;
  QTextStream st( (QIODevice *) &ascii_file);
  writeEntry( st, entrylist.at(current) );
  ascii_file.close();
}

void KJotsMain::writeEntry( QTextStream &st, TextEntry *entry )
{
  QString line, buf;
  line.fill('#', entry->subject.length()+2 );
  st << line << '\n';
  st << "# ";
  st << entry->subject << '\n';
  st << line << '\n';
  buf = entry->text;
  st << buf;
  if( buf.right(1) != "\n" )
    st << '\n';
  st << '\n';
}

void KJotsMain::moveFocus()
{
  if( me_text->hasFocus() )
    le_subject->setFocus();
  else
    me_text->setFocus();
}

void KJotsMain::configureKeys()
{
  if (KKeyDialog::configureKeys(keys, true, topLevelWidget()))
  {
    keys->writeSettings();
    kapp->config()->sync();
    keys->readSettings();
  }
}

void KJotsMain::copySelection()
{
  if(me_text->hasMarkedText())
  {
    le_subject->setText(me_text->markedText());
  }
}

void KJotsMain::slotQuit() 
{
    // just save on exit...
    saveFolder();
    close();
}

void KJotsMain::insertDate()
{
  KLocale locale("kjots");
  QString* date = new QString(locale.formatDateTime(QDateTime::currentDateTime(), true) + " ");
  QTextStream* dateStream = new QTextStream(date, IO_ReadOnly);
  me_text->insertText(dateStream);
  delete dateStream;
  delete date;
}

void KJotsMain::updateMenu()
{
  bool activeBook = folderOpen;

  actionCollection()->action("del_folder")->setEnabled(activeBook);
  actionCollection()->action("del_page")->setEnabled(activeBook);
  actionCollection()->action("new_page")->setEnabled(activeBook);

  actionCollection()->action(KStdAction::stdName(KStdAction::Save))->setEnabled(activeBook);
  menuBar()->setItemEnabled(id_f_save_book_ascii, activeBook);
  menuBar()->setItemEnabled(id_f_save_page_ascii, activeBook);

  actionCollection()->action(KStdAction::stdName(KStdAction::Cut))->setEnabled(activeBook);
  actionCollection()->action(KStdAction::stdName(KStdAction::Copy))->setEnabled(activeBook);
  actionCollection()->action(KStdAction::stdName(KStdAction::Paste))->setEnabled(activeBook);

  menuBar()->setItemEnabled(id_fav_add, activeBook);
  menuBar()->setItemEnabled(id_fav_del, activeBook);
}
#include "KJotsMain.moc"
