#include "gtetris.h"

#include "piece.h"


GenericTetris::GenericTetris(uint width, uint height, bool withPieces,
                             bool graphic)
    : _nbClearLines(height), _nextPiece(0), _currentPiece(0), _main(0),
      _graphic(graphic), _matrix(width, height)
{
    if (withPieces) {
        _nextPiece = new Piece;
        _currentPiece = new Piece;
    }
    _matrix.fill(0);
}

void GenericTetris::copy(const GenericTetris &g)
{
    Q_ASSERT(_currentPiece);
	// copy to non graphic
	_score         = g._score;
	_level         = g._level;
	_nbRemoved     = g._nbRemoved;
	_nbClearLines  = g._nbClearLines;
	_currentCol    = g._currentCol;
	_currentLine   = g._currentLine;
	_nextPiece->copy(g._nextPiece);
	_currentPiece->copy(g._currentPiece);
	for (uint i=0; i<_matrix.width(); i++)
		for (uint j=0; j<_matrix.height(); j++) {
            Coord c(i, j);
            delete _matrix[c];
			if ( g._matrix[c] ) _matrix[c] = new Block(g._matrix[c]->value());
			else _matrix[c] = 0;
		}
}

void GenericTetris::clear()
{
	_currentCol   = 0;
	_currentLine  = -1;
	for (uint i=0; i<_matrix.width(); i++)
		for (uint j=0; j<_matrix.height(); j++) removeBlock(Coord(i, j));
    computeInfos();
}

GenericTetris::~GenericTetris()
{
    // everything should already be done by setBlockInfo(0, 0);
}

void GenericTetris::setBlockInfo(BlockInfo *main, BlockInfo *next)
{
	Q_ASSERT( _graphic );
    if (main) {
        _main = main;
        if (_currentPiece) {
            _nextPiece->setBlockInfo(next);
            _currentPiece->setBlockInfo(main);
        }
    } else { // before destruction
        clear();
        delete _currentPiece;
        delete _nextPiece;
    }
}

void GenericTetris::init(const GTInitData &data)
{
	Q_ASSERT( _graphic );
	_random.setSeed(data.seed);
	if (_nextPiece) _nextPiece->setRandomSequence(&_random);
	_initLevel = data.initLevel;
}

void GenericTetris::start()
{
	Q_ASSERT( _graphic );
	updateScore(0);
	updateLevel(_initLevel);
	updateRemoved(0);
	clear();
    if (_nextPiece) {
        _nextPiece->generateNext();
        newPiece();
    }
}

void GenericTetris::dropDown()
{
	uint dropHeight = moveTo(0, -_currentLine);
	pieceDropped(dropHeight);
}

void GenericTetris::oneLineDown()
{
	if ( moveTo(0, -1)==0 ) pieceDropped(0);
}

void GenericTetris::newPiece()
{
    Q_ASSERT(_currentPiece);
	_currentLine  = _matrix.height() - 1 + _nextPiece->minY();
    _currentCol   =	(_matrix.width() - _nextPiece->width())/2
		- _nextPiece->minX();
    if ( !canPosition(_currentCol, _currentLine, _nextPiece)) {
        _currentLine = -1;
        gameOver();
        return;
    }
	_currentPiece->copy(_nextPiece);
	if ( _graphic ) {
		_currentPiece->move(toX(_currentCol), toY(_currentLine));
		_currentPiece->show(true);
		updatePieceConfig();
		_nextPiece->generateNext();
		_nextPiece->moveCenter();
		_nextPiece->show(true);
		updateNextPiece();
	}
}

bool GenericTetris::canPosition(uint col, uint line, const Piece *piece) const
{
    for(uint k=0; k<piece->nbBlocks(); k++) {
        uint i = piece->col(k, col);
        uint j = piece->line(k, line);
        if ( i>=_matrix.width() || j>=_matrix.height()
             || _matrix[Coord(i, j)]!=0 )
            return false; // outside or something in the way
    }
    return true;
}

uint GenericTetris::moveTo(int decX, int decY)
{
    Q_ASSERT(_currentPiece);
	Q_ASSERT(decX==0 || decY==0);

	int newCol  = _currentCol;
	int newLine = _currentLine;
	int dx = 0, dy = 0;
	uint n, i;

	if (decX) {
		dx = (decX<0 ? -1 : 1);
		n  = (uint)(decX<0 ? -decX : decX);
	} else {
		dy = (decY<0 ? -1 : 1);
		n  = (uint)(decY<0 ? -decY : decY);
	}

	for (i=0; i<n; i++) {
		if ( !canPosition(newCol + dx, newLine + dy, _currentPiece) ) break;
		newCol  += dx;
		newLine += dy;
	}
	if ( i!=0 ) { // piece can be moved
		_currentCol  = newCol;
		_currentLine = newLine;
		if (_graphic) {
			_currentPiece->move(toX(newCol),toY(newLine));
            updatePieceConfig();
		}
	}
	return i;
}

bool GenericTetris::rotate(bool left)
{
    Q_ASSERT(_currentPiece);

	Piece tmp;
	tmp.copy(_currentPiece);
    tmp.rotate(left, 0, 0);
    if ( canPosition(_currentCol, _currentLine, &tmp) ) {
		int x = 0, y = 0;
		if (_graphic) {
			x = toX(_currentCol);
			y = toY(_currentLine);
		}
		_currentPiece->rotate(left, x, y);
		if (_graphic) updatePieceConfig();
        return true;
	}
    return false;
}

void GenericTetris::computeInfos()
{
	_nbClearLines = 0;
	for (uint j=_matrix.height(); j>0; j--) {
		for (uint i=0; i<_matrix.width(); i++)
			if ( _matrix[Coord(i, j-1)]!=0 ) return;
		_nbClearLines++;
	}
}

void GenericTetris::setBlock(const Coord &c, Block *b)
{
	Q_ASSERT( b && _matrix[c]==0 );
	_matrix[c] = b;
	if (_graphic) b->sprite()->move(toX(c.i), toY(c.j));
}

void GenericTetris::removeBlock(const Coord &c)
{
	delete _matrix[c];
	_matrix[c] = 0;
}

void GenericTetris::moveBlock(const Coord &src, const Coord &dest)
{
	Q_ASSERT( _matrix[dest]==0 );
	if ( _matrix[src] ) {
		setBlock(dest, _matrix[src]);
		_matrix[src] = 0;
	}
}

int GenericTetris::toX(uint i) const
{
	return _main->toX(i);
}

int GenericTetris::toY(uint j) const
{
	return _main->toY(_matrix.height() - 1 - j);
}

void GenericTetris::gluePiece()
{
    Q_ASSERT(_currentPiece);

	for(uint k=0; k<_currentPiece->nbBlocks(); k++) {
        Coord c(_currentPiece->col(k, currentCol()),
                _currentPiece->line(k, currentLine()));
		setBlock(c, _currentPiece->takeBlock(k));
	}
	computeInfos();
}

void GenericTetris::bumpCurrentPiece(int dec)
{
	Q_ASSERT( _graphic && _currentPiece );
	_currentPiece->move(toX(_currentCol), toY(_currentLine) + dec);
}

void GenericTetris::partialMoveBlock(const Coord &c, int xDec, int yDec)
{
	Q_ASSERT( _graphic && _matrix[c]!=0 );
	_matrix[c]->sprite()->move(toX(c.i) + xDec, toY(c.j) + yDec);
}
