/* $Id: linkey.c,v 1.4 1998/11/25 23:57:04 marcus Exp $
******************************************************************************

   Convert Linux keysyms into LibGII syms and labels

   Copyright (C) 1998  Andrew Apted  [andrew@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include "linkey.h"

static uint32 basic_trans(int sym)
{
	int typ = KTYP(sym);
	int val = KVAL(sym);
	
	switch (typ) {

		case KT_LATIN: 
		case KT_LETTER: 
			return val;

		case KT_FN:
			if (val < 20) {
				return GII_KEY(GII_KT_FN, val+1);
			} else if (val >= 30) {
				return GII_KEY(GII_KT_FN, val-9);
			}
			break;

		case KT_CONS: return GII_KEY(GII_KT_CONS, val);
		case KT_META: return GII_KEY(GII_KT_META, val);

		case KT_SPEC:
		case KT_PAD:
		case KT_CUR:
			break;

		default:  /* The rest not handled yet */
			return GIIK_VOID;
	}

	switch (sym) {
		case K_HOLE:		return GIIK_VOID;

		case K_FIND:		return GIIK_Home;
		case K_SELECT:		return GIIK_End;
		case K_INSERT:		return GIIK_Insert;
		case K_REMOVE:		return GIIK_Delete;
		case K_PGUP:		return GIIK_PageUp;
		case K_PGDN:		return GIIK_PageDown;
		case K_MACRO:		return GIIK_Macro;
		case K_HELP:		return GIIK_Help;
		case K_DO:		return GIIK_Do;
		case K_PAUSE:		return GIIK_Pause;
#ifdef K_UNDO
		case K_UNDO:		return GIIK_Undo;
#endif

		case K_ENTER:		return GIIK_Enter;
		case K_SH_REGS:		return GIIK_ShowRegs;
		case K_SH_MEM:		return GIIK_ShowMem;
		case K_SH_STAT:		return GIIK_ShowStat;
		case K_BREAK:		return GIIK_Break;
		case K_CONS:		return GIIK_LastConsole;
		case K_CAPS:		return GIIK_CapsLock;
		case K_NUM:		return GIIK_NumLock;
		case K_HOLD:		return GIIK_ScrollLock;
		case K_BOOT:		return GIIK_Boot;
		case K_CAPSON:		return GIIK_CapsLock;
		case K_COMPOSE:		return GIIK_Compose;
		case K_SAK:		return GIIK_SAK;
#ifdef K_CAPSSHIFT
		case K_CAPSSHIFT:	return GIIK_CapsLock;
#endif

		case K_SCROLLFORW:	return GIIK_ScrollForw;
		case K_SCROLLBACK:	return GIIK_ScrollBack;
		case K_DECRCONSOLE:	return GIIK_PrevConsole;
		case K_INCRCONSOLE:	return GIIK_NextConsole;
		case K_SPAWNCONSOLE:	return GIIK_SpawnConsole;
		case K_BARENUMLOCK:	return GIIK_NumLock;

		case K_P0:		return GIIK_P0;
		case K_P1:		return GIIK_P1;
		case K_P2:		return GIIK_P2;
		case K_P3:		return GIIK_P3;
		case K_P4:		return GIIK_P4;
		case K_P5:		return GIIK_P5;
		case K_P6:		return GIIK_P6;
		case K_P7:		return GIIK_P7;
		case K_P8:		return GIIK_P8;
		case K_P9:		return GIIK_P9;
		case K_PPLUS:		return GIIK_PPlus;
		case K_PMINUS:		return GIIK_PMinus;
		case K_PSTAR:		return GIIK_PStar;
		case K_PSLASH:		return GIIK_PSlash;
		case K_PENTER:		return GIIK_PEnter;
		case K_PCOMMA:		return GIIK_PComma;
		case K_PDOT:		return GIIK_PDot;
		case K_PPLUSMINUS:	return GIIK_PPlusMinus;
#ifdef K_PPARENL
		case K_PPARENL:		return GIIK_PParenLeft;
		case K_PPARENR:		return GIIK_PParenRight;
#endif

		case K_DOWN:		return GIIK_Down;
		case K_LEFT:		return GIIK_Left;
		case K_RIGHT:		return GIIK_Right;
		case K_UP:		return GIIK_Up;
	}

	/* Undo some Linux braindamage */

	if (sym >= 0x2000) {
		return sym ^ 0xf000;
	}

	return GIIK_VOID;
}
	
uint32 _gii_linkey_labeltrans(int keysym, int shift)
{
	uint32 label;
	
	switch (keysym) {
		case K_ALT:		return GIIK_AltL;
		case K_ALTGR:		return GIIK_AltR;
		case K_SHIFT:
		case K_SHIFTL:		return GIIK_ShiftL;
		case K_SHIFTR:		return GIIK_ShiftR;
		case K_CTRL:
		case K_CTRLL:		return GIIK_CtrlL;
		case K_CTRLR:		return GIIK_CtrlR;
		
#ifdef K_SHIFTLOCK
		case K_ALTLOCK:		return GIIK_AltL;
		case K_ALTGRLOCK:	return GIIK_AltR;
		case K_SHIFTLOCK:
		case K_SHIFTLLOCK:	return GIIK_ShiftL;
		case K_SHIFTRLOCK:	return GIIK_ShiftR;
		case K_CTRLLOCK:
		case K_CTRLLLOCK:	return GIIK_CtrlL;
		case K_CTRLRLOCK:	return GIIK_CtrlR;
#endif

#ifdef K_SHIFT_SLOCK
		case K_ALT_SLOCK:	return GIIK_AltL;
		case K_ALTGR_SLOCK:	return GIIK_AltR;
		case K_SHIFT_SLOCK:
		case K_SHIFTL_SLOCK:	return GIIK_ShiftL;
		case K_SHIFTR_SLOCK:	return GIIK_ShiftR;
		case K_CTRL_SLOCK:
		case K_CTRLL_SLOCK:	return GIIK_CtrlL;
		case K_CTRLR_SLOCK:	return GIIK_CtrlR;
#endif
	}
	
	label = basic_trans(keysym);

	if (('a' <= label && label <= 'z') ||
	    (GIIUC_agrave <= label && label <= GIIUC_thorn &&
	     label != GIIUC_Division)) {

		return label - 0x20;
	}

	return label;
}

uint32 _gii_linkey_symtrans(int keysym, int shift)
{
	switch (keysym) {
		case K_ALT:
		case K_ALTGR:		return GIIK_Alt;
		case K_SHIFT:
		case K_SHIFTL:
		case K_SHIFTR:		return GIIK_Shift;
		case K_CTRL:
		case K_CTRLL:
		case K_CTRLR:		return GIIK_Ctrl;
		
#ifdef K_SHIFTLOCK
		case K_ALTLOCK:
		case K_ALTGRLOCK:	return GIIK_Alt;
		case K_SHIFTLOCK:
		case K_SHIFTLLOCK:
		case K_SHIFTRLOCK:	return GIIK_Shift;
		case K_CTRLLOCK:
		case K_CTRLLLOCK:
		case K_CTRLRLOCK:	return GIIK_Ctrl;
#endif

#ifdef K_SHIFT_SLOCK
		case K_ALT_SLOCK:
		case K_ALTGR_SLOCK:	return GIIK_Alt;
		case K_SHIFT_SLOCK:
		case K_SHIFTL_SLOCK:
		case K_SHIFTR_SLOCK:	return GIIK_Shift;
		case K_CTRL_SLOCK:
		case K_CTRLL_SLOCK:
		case K_CTRLR_SLOCK:	return GIIK_Ctrl;
#endif
	}

	return basic_trans(keysym);
}

#ifdef WANT_TRANSLATE_SHIFT

#define SHIFT_MODS  ((1<<KG_SHIFT) | (1<<KG_SHIFTL) | (1<<KG_SHIFTR))
#define CTRL_MODS   ((1<<KG_CTRL)  | (1<<KG_CTRLL) | (1<<KG_CTRLR))
#define ALT_MODS    ((1<<KG_ALT)   | (1<<KG_ALTGR))

int _gii_linkey_shifttrans(int shift)
{
	return  ((shift & SHIFT_MODS) ? (1 << GII_KM_SHIFT) : 0) |
		((shift & CTRL_MODS)  ? (1 << GII_KM_CTRL)  : 0) |
		((shift & ALT_MODS)   ? (1 << GII_KM_ALT)   : 0);
}
#endif
